;+
; =================================  DELIA ===============================
;
;   The documentation for XAID (the XOP's XAFS toolbox) can be found at:
;   http://www.esrf.fr/computing/scientific/xop/xaid/doc/
;
;		
;   DELIA (Dispersive Exafs caLIbration and Analysis)  is a widget based 
;   graphical interface to perform the interpolation, calibration, 
;   extraction and Fourier transform of Dispersive Exafs spectra.
;
;   For the extraction and Fourier transform, it calls XAID_MU2CHI and 
;   XAID_FF, respectively, to perform the operations in a single spectrum. 
;   Once the parameters have been set from the analysis of that spectrum, 
;   the operations are replicated over the totallity of spectra. 
;
;		
; ============= Non interactive use of DELIA  ============================
;
;	NAME:
;		DELIA
;	PURPOSE:
;		 Dispersive Exafs caLIbration and Analysis
; 	CATEGORY:
;		Widgets. XAFS data analysis.
;	CALLING SEQUENCE:
;		DELIA 
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of XAID_Mu2Chi.
;	OUTPUTS:
;		Open the widget utility.
;	COMMON BLOCKS:
;		None
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Four steps: 1) interpolation, 2) calibration, 3) extraction 
;		and 4) Fourier transform. 
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 2 October 1998
;		98-10-13 srio@esrf.fr adds e0 alignment, checks for
;			duplicated points, improves xaid_calib and makes
;			other improvements. version 1.0Beta2
;		98-10-13 srio@esrf.fr adds "sort" kw to gete0, displays
;			shifted values in a separate window. 
;			Version 1.0Beta3
;		98-10-26 srio@esrf.fr problems with interpolation/align
;			fixed. Added Back Fourier Transform. Cosmetics.
;			Added tools menus. Version 1.0Beta4
;		98-11-02 srio@esrf.fr changes the alignment method:
;			now it uses GetEdge. Version 1.0Beta5.
;		98-11-20 srio@esrf.fr Adds FIle/load file with Chi/Mu
;			Version 1.0Beta6.
;		99-10-03 srio@esrf.fr adds binary file RAW data input.
;			Version 1.0Beta7.
;		00-07-20 srio@esrf.fr last touches for xop 2.0
;			Version 1.0
;
;-
;
;========================================================================
;
FUNCTION Delia_Make_Text_Array,a,group=Group,NLimit=nlimit, Header=header 
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,Dialog_Parent=group,$
	'DELIA_MAKE_TEXT_ARRAY: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,'<error converting numerical array to text>'
ENDIF

IF Keyword_Set(nlimit) EQ 0 THEN nlimit = 300
IF Keyword_Set(header) EQ 0 THEN header = ''

npoints = (Size(a))[1:2]
ntxt = (npoints[1] < 2*nlimit)
txt = strarr(1+ ntxt )
txt[0]=header
IF npoints[1] LE 2*nlimit THEN BEGIN
  FOR i=1L,ntxt do txt[i]=string(a[*,i-1],format='('+$
    strcompress(npoints[0],/rem)+'(G11.5,X))')
ENDIF ELSE BEGIN
  FOR i=1L,nlimit-2 do txt[i]=string(a[*,i-1],format='('+$
    strcompress(npoints[0],/rem)+'(G11.5,X))')
  txt[nlimit-1]='...'
  txt[nlimit]='...<skipped>...'
  j=nlimit
  FOR i=npoints[1]-nlimit+1,npoints[1] DO BEGIN
    j = j+1
    txt[j]=string(a[*,i-1],format='('+$
    strcompress(npoints[0],/rem)+'(G11.5,X))')
  ENDFOR
ENDELSE
IF Keyword_Set(header) EQ 0 THEN txt = txt[1:N_Elements(txt)-1]
RETURN,txt
END ; Delia_Make_Text_Array
;
;========================================================================
;
FUNCTION Delia_Read_Binary,file, Group =group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'DELIA_READ_BINARY: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

IF N_Params() EQ 0 THEN BEGIN
  pwd = 0
  CD,Current=pwd
  file = Dialog_PickFile(/Read,Dialog_Parent=group,Path=pwd)
  IF file EQ '' THEN RETURN,0
ENDIF

; file = 'test_200.out'
OpenR,unit,file,/Get_lun,/Swap_If_Little_Endian
fsize = (FStat(unit)).size
npoints=fsize/3/4

a=FltArr(3,npoints)
ReadU,unit,a
Free_Lun,unit
RETURN,a
END ; Delia_Read_Binary
;
;========================================================================
;
Function delia_reform_arrays,x,block,index,pp1,pp2,pp3,pp4,text=txt,$
 xplot=xplot,_Extra=extra,No_block=no_block,NLimit=nlimit

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'DELIA_REFORM_ARRAYS: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

; x : array of npts
; block = array of (nn,npts)
; index = array of nn [index]
; pp1 ... pp4  = array of npts
; 
; reform arrays
;
;message,/info,'... Rearranging arrays ...'
npts = n_elements(x)
nn=n_elements(block[*,0])
npars = n_params()
if npars LT 2 then message,'Usage: out=delia_reform_arrays(x,block)'
if n_elements(index) EQ 0 then begin
  index=findgen(nn)
  npars = npars+1
endif
;if n_elements(pp2) EQ 0 then pp2=fltarr(npts)

out2 = fltarr(npars>3,nn*npts)
for i=0L,nn-1 do begin
  out2[0,i*npts:(i*npts)+npts-1]=reform(x)
  out2[1,i*npts:(i*npts)+npts-1]=index[i]
  if n_elements(pp1) GT 0 then out2[2,i*npts:(i*npts)+npts-1]=reform(pp1)
  if n_elements(pp2) GT 0 then out2[3,i*npts:(i*npts)+npts-1]=reform(pp2)
  if n_elements(pp3) GT 0 then out2[4,i*npts:(i*npts)+npts-1]=reform(pp3)
  if n_elements(pp4) GT 0 then out2[5,i*npts:(i*npts)+npts-1]=reform(pp4)
  out2[npars-1,i*npts:(i*npts)+npts-1]=block[i,*]
endfor
;
;if keyword_set(txt) then begin
;  txt = strarr(nn*npts)
;  for i=0L,nn*npts-1 do begin
;    txt[i] = string(out2[*,i],format='('+strcompress(npars,/rem)+'(G11.5,X))')
;  endfor
;endif
txt = Delia_Make_Text_Array(out2,NLimit=nlimit)
message,/info,'... Done ...'
; 
; display
;
;widget_control,state.wids.mu.text,set_value=txt
if keyword_set(xplot) then begin
  xplot,out2,xcol=1,no_block=no_block,p=p,_extra=extra
  xplot_mesh,p,flag=1,col=2
endif

return,out2
end ; delia_reform_arrays
;
;========================================================================
;
PRO delia_mappanels, wids

widget_control,wids.panels, get_value=tmp
mapp = intarr(4)
mapp[tmp] = 1
for i=0,3 do widget_control,wids.panel_ids[i],map=mapp[i]
;
; FT appodization value
;
if tmp EQ 3 then begin
  win=0
  widget_control,wids.ft.filter,get_value=win
  win = 1+fix(win)
  if ( (win EQ 2) or (win EQ 4) or (win EQ 5) or (win EQ 6) ) then begin
    widget_control,wids.ft.app_base,map=1
    widget_control,wids.ft.app_label,set_value='Appodization width:'
  endif else if (win EQ 9) then begin
    widget_control,wids.ft.app_base,map=1
    widget_control,wids.ft.app_label,set_value='t-value:'
  endif else begin
    widget_control,wids.ft.app_base,map=0
  endelse
endif

end ; delia_mappanels

;
;========================================================================
;
PRO delia_event,event

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'DELIA: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   goto,out
endif

;
;
Widget_Control, event.id, get_UValue=eventUValue
Widget_Control, event.id, get_Value=Value
 
if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state, /No_Copy
delia_mappanels,state.wids
 
Case eventUValue OF
    'QUIT': BEGIN
	for i=0,n_elements(state.ptr)-1 do $
	  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	Widget_Control,event.handler,/destroy
	END
    'LOADFILE': BEGIN
	fileKind=''
	Widget_Control,event.id,Get_Value=fileKind
	pwd = 0
	CD,Current=pwd
	file = Dialog_PickFile(Group=event.top, Title=fileKind, $
	  Path=pwd)
	IF file EQ '' THEN GoTo,out
	data = RAscii(file)
	; works for idl 5.1 dimData=Size(data,/Dimensions)
	; IF SDep(/vf) EQ '5.0.2' THEN dimData=(Size(data))[1:2] $
	;  ELSE dimData=Size(data,/Dimensions)
	dimData=(Size(data))[1:2]
	IF N_Elements(dimData) NE 2 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Error, $
		'No good data found in file: '+file)
	  GoTo,out
	ENDIF
	IF dimData[0] LE 2 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Error, $
		'Input file ('+file+') must have at least 3 columns.')
	  GoTo,out
	ENDIF
	Widget_Control,/HourGlass
	;
	CASE fileKind OF
	  'Load file with Mu(E)...': BEGIN
		x=Reform(Data[0,*])
		mu=Data[1:dimData[0]-1,*]
		index = IndGen(dimData[0]-1)
		;
		; plot & update text widget
		;
		txt = 1
		tmp = delia_reform_arrays(x,mu,index,x,$
		  /xplot,group=event.top,text=txt,wtitle='Mu 3D plot',$
		  No_Block=state.no_block)
		txt = ['Xcalibrated      Index  Xoriginal         Mu', $
		  temporary(txt)]
		widget_control,state.wids.mu.text,set_value=txt
		;
		; update pointer
		;
		str = {x:x, energy:x, index:index, mu:mu}
		for i=1,n_elements(state.ptr)-1 do $
		if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
		state.ptr[1]=Ptr_New(str)
		widget_control,state.wids.panels,set_value=1
		END
	  'Load file with Chi(k)...': BEGIN
		k=Reform(Data[0,*])
		chi=Data[1:dimData[0]-1,*]
		index = IndGen(dimData[0]-1)
		;
		; plot & update text widget
		;
		txt=1
		tmp = delia_reform_arrays(k,chi,index,/xplot,wtitle=$
		 'Chi 3D plot',Text=txt,No_Block=state.no_block)
		txt = ['     k           Index    Chi     ', $
		  temporary(txt)]
		widget_control,state.wids.chi.text,Set_Value=txt
		;
		; update pointer
		;
		str = {k:k, index:index, chi:chi, spline:chi*0}
		for i=2,n_elements(state.ptr)-1 do $
		  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
		  state.ptr[2]=Ptr_New(str)
		widget_control,state.wids.panels,set_value=2
		END
	ENDCASE
	widget_control,state.wids.labelMain,Set_Value='Loaded file: '+file
	delia_mappanels,state.wids
	END
    'RAW_LOAD_ASCII': BEGIN
	pwd = 0
	CD,Current=pwd
	file = Dialog_PickFile(Group=event.top, Title= $
	  'Load RAW data (ASCII file)',Path=pwd)
	IF file EQ '' THEN GoTo,out
	if checkfile(file) NE 1 then begin
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Error,$
	     'File not found: '+file)
	  goto,out
	endif
	widget_control,/hourglass
	widget_control,state.wids.raw.text,set_value='<none>' 
	widget_control,state.wids.mu.text,set_value='<none>'
	widget_control,state.wids.labelMain,set_value=$
	 'Loading file... Please wait.'
	npoints=0
	a=rascii(file,npoints=npoints)
	widget_control,state.wids.labelMain,set_value=$
	 'File contains '+strcompress(npoints[1],/rem)+' lines '+$
	 'and '+strcompress(npoints[0],/rem)+' columns. Converting to text...'
	widget_control,/hourglass
	header=''
	for i=1L,npoints[0] do header=header+'   Column '+$
	  strcompress(i,/rem)+' '
	txt = Delia_Make_Text_Array(a,Header=header,Group=event.top,NLimit=300)
	widget_control,state.wids.raw.text,set_value=txt
	for i=0,n_elements(state.ptr)-1 do $
	  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[0]=Ptr_New(a)
	widget_control,state.wids.labelMain,set_value=$
	 'File loaded ('+strcompress(npoints[1],/rem)+' lines '+$
	 ', '+strcompress(npoints[0],/rem)+' columns).'
	END
    'RAW_LOAD_BINARY': BEGIN
        widget_control,state.wids.raw.col1,Set_value=2
        widget_control,state.wids.raw.col2,Set_value=3
        widget_control,state.wids.raw.col3,Set_value=1
	file=''
	widget_control,state.wids.raw.file,get_value=file
	if checkfile(file) NE 1 then begin
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Error,$
	     'File not found: '+file)
	  goto,out
	endif
	widget_control,/hourglass
	widget_control,state.wids.raw.text,set_value='<none>' 
	widget_control,state.wids.mu.text,set_value='<none>'
	widget_control,state.wids.labelMain,set_value=$
	 'Loading file... Please wait.'
	a=Delia_Read_Binary(file,Group=event.top)
	npoints = (Size(a))[1:2]
	widget_control,state.wids.labelMain,set_value=$
	 'File contains '+strcompress(npoints[1],/rem)+' lines '+$
	 'and '+strcompress(npoints[0],/rem)+' columns. Converting to text...'
	widget_control,/hourglass
	header =''
	for i=1L,npoints[0] do header=header+'   Column '+$
	  strcompress(i,/rem)+' '
	txt = Delia_Make_Text_Array(a,Header=header,Group=event.top,NLimit=300)
	widget_control,state.wids.raw.text,set_value=txt
	for i=0,n_elements(state.ptr)-1 do $
	  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[0]=Ptr_New(a)
	widget_control,state.wids.labelMain,set_value=$
	 'File loaded ('+strcompress(npoints[1],/rem)+' lines '+$
	 ', '+strcompress(npoints[0],/rem)+' columns).'
	END



    'RAW_VIEW#': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	tmp=''
	selected_index = 0
	add=0
	case event.id of
	  state.wids.raw.view: begin
		widget_control,event.id,get_value=selected_index
		widget_control,state.wids.raw.over,set_value=selected_index[0]
		end
	  state.wids.raw.over: begin
		widget_control,event.id,get_value=selected_index
		add=1
		end
	  else: begin  ; button pressed
		widget_control,event.id,get_value=tmp
		case tmp[0] of
		  'Go': widget_control,state.wids.raw.view,get_value=selected_index
		  'Add next' : BEGIN 
		    widget_control,state.wids.raw.over,get_value=selected_index
		    selected_index = selected_index+1
		    widget_control,state.wids.raw.over,set_value=selected_index
		    add=1
		    END
		endcase
		end
	endcase
	xcol=0 & ycol=0 & indexcol=0
	widget_control,state.wids.raw.col1,get_value=xcol
	widget_control,state.wids.raw.col2,get_value=ycol
	widget_control,state.wids.raw.col3,get_value=indexcol
	xcol = (xcol-1) > 0
	ycol = (ycol-1) > 0
	indexCol = (indexCol-1) > 0
	indices = where( a[indexCol,*] EQ selected_index)

	if indices[0] EQ -1 then begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Wrong index value.')
	  goto,out
	endif else begin
	  arr1 = a[xcol,indices] 
	  arr2 = a[indexcol,indices] 
	  arr3 = a[ycol,indices] 
	  arr = fltarr(3,n_elements(arr1))
	  arr[0,*] = arr1
	  arr[1,*] = arr2
	  arr[2,*] = arr3
	  if widget_info(state.wids.xplot,/Valid_Id) then begin
	    if add then xplot_savecurrent,state.wids.xplot else $
		xplot_plotmgr,state.wids.xplot,/delete_all
	    xplot_loadfile,state.wids.xplot, arr
	  endif else begin
	    xplot,p=p,arr,group=event.top,No_Block=state.no_block
	    state.wids.xplot=p
	  endelse
	endelse
	END


    'SWITCH': BEGIN
	switchindex=0
	widget_control,state.wids.panels,get_value=switchindex,/hourglass

	;if switchindex NE 0 then begin
	;  if not(Ptr_Valid(state.ptr[0]) ) then begin
	;    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	;    'Data not found. Please load a data file.')
	;    widget_control,state.wids.panels,set_value=0
	;    delia_mappanels,state.wids
	;    goto,out
	;  endif
	;endif

	if switchindex EQ 1 then begin
	  if Ptr_Valid(state.ptr[1]) then begin
            delia_mappanels,state.wids
	    widget_control,state.wids.labelMain,Set_Value= $
	     'Use 2D or 3D buttons to visualize data and chi(k) panel to process.'
	    GoTo,out
	  endif
	  if not(Ptr_Valid(state.ptr[0])) then begin
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Data not found. Please load a data file.')
	    widget_control,state.wids.panels,set_value=0
	    delia_mappanels,state.wids
	    goto,out
	  endif
	  a=*(state.ptr[0])
	  xcol=0 & ycol=0 & indexcol=0
	  widget_control,state.wids.raw.col1,get_value=xcol
	  widget_control,state.wids.raw.col2,get_value=ycol
	  widget_control,state.wids.raw.col3,get_value=indexcol
	  xcol = (xcol-1) > 0
	  ycol = (ycol-1) > 0
	  indexCol = (indexCol-1) > 0
	  ;
	  ; get the index values
	  ;
          tmp = reform(a[indexCol,*])
	  arrIndex = tmp[Uniq( tmp, Sort(tmp) )]
	  n_arrIndex = n_elements(arrIndex)
	  ;
	  ; get e0s if selected
	  ; 
	  e0s = fltarr(n_arrIndex)
	  e0shifted = fltarr(n_arrIndex)
	  index = lonarr(n_arrIndex) & delta_e0=fltarr(n_arrIndex)
	  ialign = 0
	  widget_control,state.wids.mu.align,get_value=ialign
	  if ialign EQ 1 then begin
	    for i=0L,n_arrIndex-1 do begin
	      tmp = a[[xcol,ycol],*]
	      tmp = a[ *, where(a[indexCol,*] EQ arrIndex[i]) ]
;	      e0=gete0(tmp,/sort)
	      e0=getedge(tmp,/sort)
	      e0s[i]=e0
	    endfor
	    e0shifted = e0s-e0s[0]
	    txt = ['Spectrum index     Shift    Half-jump at']
	    for i=0L,n_arrIndex-1 do txt = $
		[txt,string(arrIndex[i])+string(e0shifted[i])+$
		string(e0s[i])]
	    xdisplayfile1,text=txt,group=event.top,title='X shifts',$
		width=40
	  endif 
	  ;
	  ; get the single extrema of spectra
	  ; 
	  widget_control,state.wids.labelMain,Set_Value= 'Computing extrema...'
	  for i=0L,n_arrIndex-1 do begin
	    tmp = a[ *, where(a[indexCol,*] EQ arrIndex[i]) ]
	    if i EQ 0 then begin
	       x2=0
	       x1 = min(tmp[xcol,*],max=x2) 
	       x1=x1-e0shifted[i]
	       x2=x2-e0shifted[i]
	       ;x2 = tmp[xcol,n_elements(tmp[0,*])-1] 
	    endif else begin
	      tmpmax=0
	      tmpmin = min(tmp[xcol,*],max=tmpmax)
	      tmpmin=tmpmin-e0shifted[i]
	      tmpmax=tmpmax-e0shifted[i]
	      x1 = [x1,tmpmin]
	      x2 = [x2,tmpmax]
	    endelse
	  endfor
	  x1max=0
	  x1min=min(x1,max=x1max)
	  x2max=0
	  x2min=min(x2,max=x2max)
	  ; 
	  ; update widgets
	  ;
	  widget_control,state.wids.mu.labelInt1,Set_Value='from (minima) '+$
	     vect2string( [x1min,x1max] )
	  widget_control,state.wids.mu.labelInt2,Set_Value='to (maxima) '+$
	     vect2string( [x2min,x2max] )
	  widget_control,state.wids.mu.intMin,Set_Value=max([x1min,x1max])
	  widget_control,state.wids.mu.intMax,Set_Value=min([x2min,x2max])
	  widget_control,state.wids.labelMain,Set_Value= $
		'Please check/change parameters and press "RUN" button'
	endif


	if switchindex EQ 2 then begin ; chi panel
	  if Ptr_Valid(state.ptr[2]) then begin
            delia_mappanels,state.wids
	    widget_control,state.wids.labelMain,Set_Value= $
	     'Use 2D or 3D buttons to visualize data and Fourier panel to process.'
	    GoTo,out
	  endif
	  widget_control,state.wids.labelMain,Set_Value=$
	   'Extraction on a single spectrum and then replication'
	  if not(Ptr_Valid(state.ptr[1]) ) then begin
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Mu Data not found. Please process data uning "mu" panel.')
	    widget_control,state.wids.panels,set_value=0
	    delia_mappanels,state.wids
	    goto,out
	  endif
	endif

	if switchindex EQ 3 then begin ; ft panel
	  widget_control,state.wids.labelMain,Set_Value=$
	   'FT on a single spectrum and then replication'
	  if not(Ptr_Valid(state.ptr[2]) ) then begin
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Chi Data not found. Please process data uning "chi" panel.')
	    widget_control,state.wids.panels,set_value=0
	    delia_mappanels,state.wids
	    goto,out
	  endif
	endif


	END
    'MU_GO': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	widget_control,state.wids.mu.text,set_value='<none>'
	xcol=0 & yxol=0 & indexcol=0
	widget_control,state.wids.raw.col1,get_value=xcol
	widget_control,state.wids.raw.col2,get_value=ycol
	widget_control,state.wids.raw.col3,get_value=indexcol
	xcol = (xcol-1) > 0
	ycol = (ycol-1) > 0
	indexCol = (indexCol-1) > 0

	;
	; get the index values
	;
        tmp = reform(a[indexCol,*])
	arrIndex = tmp[Uniq( tmp, Sort(tmp) )]
	n_arrIndex = n_elements(arrIndex)
	tmpindex = a[indexcol,*]
	tmpx = a[xcol,*]
	tmpy = a[ycol,*]
	xmin=0 & xmax=0 & npts=0
	widget_control,state.wids.mu.intMin,get_value=xmin
	widget_control,state.wids.mu.intMax,get_value=xmax
	widget_control,state.wids.mu.intN,get_value=npts

	;
	; get e0
	; 
	e0s = fltarr(n_arrIndex)
	e0shifted = fltarr(n_arrIndex)
	ialign = 0
	widget_control,state.wids.mu.align,get_value=ialign
	if ialign EQ 1 then begin
	  for i=0L,n_arrIndex-1 do begin
	    tmp = a[[xcol,ycol],*]
	    tmp = a[ *, where(a[indexCol,*] EQ arrIndex[i]) ]
;	    e0=gete0(tmp,/sort)
	    e0=getedge(tmp,/sort)
	    e0s[i]=e0
	  endfor
	  e0shifted = e0s-e0s[0]
	endif 
	; 
	; interpolate 
	;
	tmp3x = xmin + ( findgen(npts)/float(npts-1) )*(xmax-xmin)
	tmp3x = tmp3x - e0shifted[0]
	widget_control,state.wids.labelMain,Set_Value= $
		'Calculating interpolation...'
	widget_control,/hourglass
	out = fltarr(n_elements(arrIndex),npts)
	for i=0,n_arrIndex-1 do begin
	  widget_control,state.wids.labelMain,Set_Value= $
		'Calculating interpolation spectrum '+$
		strcompress(i+1)+' of '+strcompress(n_arrIndex)+'...'
	  good=where(tmpindex EQ arrIndex[i])
	  tmp2x=tmpx[good] ;;;; - e0shifted[i]
	  tmp2y=tmpy[good]
	  ;
	  ; use only non-duplicated points
	  ;
	  good2 = Uniq(tmp2x, Sort(tmp2x))
	  tmp2x = tmp2x[good2] 
	  tmp2x=tmp2x - e0shifted[i]
	  tmp2y = tmp2y[good2]
	  tmp3y = interpol(tmp2y,tmp2x,tmp3x)
	  if ialign then begin
	    if i EQ 0 then begin
	      e0before = fltarr(n_arrIndex)
	      e0after = fltarr(n_arrIndex)
	    endif
	    set11=fltarr(2,n_elements(tmp2y))
	    set11[0,*]=tmp2x
	    set11[1,*]=tmp2y
	    set22=fltarr(2,n_elements(tmp3y))
	    set22[0,*]=tmp3x
	    set22[1,*]=tmp3y
	    tmp11=gete0(set11,/sort)
	    tmp22=gete0(set22,/sort)
	    e0before[i]=tmp11
	    e0after[i]=tmp22
	  endif
	  out[i,*] = reform(tmp3y)
	endfor
	; 
	; calibrate the X array
	; 
	widget_control,state.wids.labelMain,Set_Value= $
		'Calculating calibration...'
	a=0 & b=0 & c=0
	widget_control,state.wids.mu.cal_a,get_value=a
	widget_control,state.wids.mu.cal_b,get_value=B
	widget_control,state.wids.mu.cal_c,get_value=c
	if (a EQ 0 and b EQ 0 and c EQ 0) then begin
	  tmp3x_cal = tmp3x
	endif else begin
	  tmp = tmp3x-tmp3x[0]
	  tmp3x_cal = tmp*tmp*c + tmp*b + a
	endelse
	if ialign then begin
	  for i=0,n_arrIndex-1 do begin
	    if i EQ 0 then e0calibrated = fltarr(n_arrIndex)
	    set = fltarr(2,n_elements(tmp3x_cal))
	    set[0,*]=tmp3x_cal
	    set[1,*]=out[i,*]
	    e0=gete0(set,/sort)
	    e0calibrated[i]=e0
	  endfor
	  txt = ['Column 1: Spectrum index',$
		'Column 2: Inflexion point (X) before interpolation',$
		'Column 3: Inflexion point (X) after interpolation',$
		'Column 4: Inflexion point (E) after calibration',' ',$
		'     Column 1     Column 2     Column 3     Column 4']
	  for i=0L,n_arrIndex-1 do txt = $
	    [txt,string(arrIndex[i])+string(e0before[i])+$
	    string(e0after[i])+string(e0calibrated[i])]
	  xdisplayfile1,text=txt,group=event.top,title=$
	   'X shifts in interpolation and calibration',width=60
	endif
	widget_control,state.wids.labelMain,Set_Value=$
		'Preparing plot...'
	; 
	; reform arrays
	;
	txt = 1
	tmp = delia_reform_arrays(tmp3x_cal,out,arrIndex,tmp3x,$
	  /xplot,group=event.top,text=txt,wtitle='Mu 3D plot',$
	  No_Block=state.no_block)
	txt = ['Xcalibrated      Index  Xoriginal         Mu', $
	  temporary(txt)]
	widget_control,state.wids.mu.text,set_value=txt
	widget_control,state.wids.labelMain,Set_Value='Done'
	;
	; save data in Ptr
	;
	str = {x:tmp3x, energy:tmp3x_cal, index:arrIndex, mu:out}
        for i=1,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[1]=Ptr_New(str)
	END
    'MU_GET': BEGIN
	deliaid = state.wids
	xplotid = state.wids.xplot
	xaid_calib,deliaid=deliaid,xplotid=xplotid,group=event.top,$
	  No_Block=no_block
	END
    'MU_PLOT': BEGIN
	if Ptr_Valid(state.ptr[1]) then a=*(state.ptr[1]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  [' Data containing Mu not found.',$
	   ' Please load a data file and/or ',$
	   ' RUN interpolate & calibrate.'])
	  goto,out
	endelse
	val=''
	widget_control,event.id,get_value=val
	case val of
	  '2 D': BEGIN
		nnx = 1+n_elements(a.mu[*,0])
		nny = n_elements(a.mu[0,*])
		out2 = fltarr(nnx,nny)
		out2[0,*]=a.energy
		out2[1:nnx-1,0:nny-1]=a.mu
		xplot,out2,No_Block=state.no_block,$
		  coltitles=['Energy [eV]','Index: '+$
		  string(a.index)], group=event.top, Wtitle='Mu 2D plot'
		END
	  '3 D': BEGIN
		tmp = delia_reform_arrays(a.energy,a.mu,a.index,a.x,$
		  /xplot,group=event.top,wtitle='Mu 3D plot',No_Block=$
		  state.no_block)
		END
	endcase
	END
    'CHI_MU2CHI': BEGIN
	indexcol=0
	widget_control,state.wids.chi.mu2chi,get_value=indexcol,/hourglass
	str = *(state.ptr[1])
	myindex = where(str.index EQ indexcol)
	if myindex[0] EQ -1 then begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Spectrum with selected index('+strcompress(indexcol,/rem)+$
	    ') not found')
	  goto,out
	endif
	set = fltarr(2,n_elements(str.energy))
	set[0,*] = reform(str.energy)
	set[1,*] = reform(str.mu[myindex,*])
	xaid_mu2chi,set,group=event.top,No_Block=state.no_block,$
	  deliaid=state.wids
	END


    'CHI_GO': BEGIN
	str = *(state.ptr[1])
	; 
	; preedge
	;
	widget_control,state.wids.labelMain,Set_Value=$
	   'calculating preedge...'
	preMin=0 & preMax=0
	widget_control,state.wids.chi.preMin,get_value=preMin
	widget_control,state.wids.chi.preMax,get_value=preMax
	set1 = fltarr(2,n_elements(str.energy))
	set1[0,*] = reform(str.energy)
	nspectra = n_elements(str.mu[*,0])
	npts=long(n_elements(str.energy))
	fit1=0
	xrange=[preMin,preMax]
	out = str.mu
	if preMin NE preMax then begin
	  for i=0L,nspectra-1 do begin
	    set1[1,*] = str.mu[i,*]
	    fitset,set1,fit1,method=1,degree=1,xrange=xrange
	    set1[1,*] = set1[1,*]-fit1[1,*]
	    out[i,*]=reform(set1[1,*])
	  endfor
	endif
	;
	; energy -> k
	;
	widget_control,state.wids.chi.e0,get_value=e0
	set2 = e2k(set1,e0=e0)
	kk=reform(set2[0,*])

	;
	; postedge
	;
	widget_control,state.wids.labelMain,Set_Value=$
	   'calculating postedge...'
	tmp=' '
	widget_control,state.wids.chi.Npol,get_value=tmp
	spl_n = fix(tmp[0])
	widget_control,state.wids.chi.Degree,get_value=tmp
	spl_degree = 0
	itmp = execute('spl_degree = '+tmp[0])
	sknots=''
	widget_control,state.wids.chi.knots,get_value=sknots
	spl_knots = 0
	itmp = execute('spl_knots = '+sknots[0])
	if max(spl_knots) GT max(kk)+1e-3 then begin
	  itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
		['Knot value ('+strcompress(max(kk),/rem)+$
		') > maximum wavenumber ('+strcompress(max(kk),/rem)+'). ',$
		 'Please check knot values: must be in k (wavenumber) scale!',$
		 'Abort? '])
	  if itmp EQ 'Yes' then goto,out
	endif
	w=fltarr(npts+1)
	w=w*0.0+1.0
	c = dblarr(36)
	xl = 0.0D0
	xh = dblarr(10)
        ioerr = execute('xl = '+sknots[0])
	xl = [0.0,xl]
	nc = [0.0D0,spl_degree]
        ;
        for i=1,spl_n do begin
          xh[i] = xl[i+1]
          nc[i] = nc[i]+1
        endfor
	tmp = dblarr(10)
	tmp[0:n_elements(xl)-2]=xl[0:n_elements(xl)-2]
	xl=tmp
	xx=[0.0,kk]
	nr = fix(spl_n)
	fit=0
	set1[0,*]=kk
	out2=fltarr(nspectra,npts)
	widget_control,/hourglass
	xl_saved = xl
	xh_saved=xh
	for i=0L,nspectra-1 do begin
	  widget_control,state.wids.labelMain,Set_Value=$
	   'calculating postedge spectrum: '+strcompress(i+1)+' of '+$
	   strcompress(nspectra)
	  yy = [0.0, reform(out[i,*])]
	  c=dblarr(36)
	  ; this is necessary because polspl changes the xh and xl values!!
	  xl = xl_saved
	  xh = xh_saved
	  ;
	  polspl,xx,yy,w,npts,xl,xh,nr,c,nc
	  polspl_evaluate,set1,xl,xh,c,nc,nr,fit
	  out2[i,*] = reform(fit[1,*])
	endfor
	;
	; extraction
	;
	widget_control,state.wids.labelMain,Set_Value=$
	   'Calculating extraction...'
	jump=0
	widget_control,state.wids.chi.jump,get_value=jump
	out3=fltarr(nspectra,npts)
	for i=0L,nspectra-1 do begin
	  mu = reform(out[i,*])
	  yfit = reform(out2[i,*])
          chi  = (mu - yfit) / jump
	  out3[i,*] = reform(chi)
	endfor
	;
	;remove kk<= kkmin
	;
	kmin=0.0
	widget_control,state.wids.chi.kmin,get_value=kmin
	good = where(kk GT kmin)
	kk=kk(good)
	out3 = out3[*,good]
	out2 = out2[*,good]
	;
	; plot
	;
	widget_control,state.wids.labelMain,Set_Value=$
	   'Preparing plots...'
	txt=1
	tmp = delia_reform_arrays(kk,out3,str.index,/xplot,text=txt,$
	  wtitle='Chi 3D plot',No_Block=state.no_block)
	widget_control,state.wids.chi.text,Set_Value=txt
	widget_control,state.wids.labelMain,Set_Value='Done'
	;
	; save data in Ptr
	;
	str = {k:kk, index:str.Index, chi:out3, spline:out2}
	for i=2,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[2]=Ptr_New(str)

	END


    'CHI_PLOT': BEGIN
	if Ptr_Valid(state.ptr[2]) then a=*(state.ptr[2]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  ' Data containing Chi not found.')
	  goto,out
	endelse
	val=''
	widget_control,event.id,get_value=val
	case val of
	  '2 D': BEGIN
		nnx = 1+n_elements(a.chi[*,0])
		nny = n_elements(a.chi[0,*])
		out2 = fltarr(nnx,nny)
		out2[0,*]=a.k
		out2[1:nnx-1,0:nny-1]=a.chi
		xplot,out2,/no_block,coltitles=['Energy [eV]','Chi for: '+$
		  string(a.index)], group=event.top, Wtitle='Chi 2D plot'
		END
	  '2 D (splines)': BEGIN
		nnx = 1+n_elements(a.chi[*,0])
		nny = n_elements(a.chi[0,*])
		out2 = fltarr(nnx,nny)
		out2[0,*]=a.k
		out2[1:nnx-1,0:nny-1]=a.spline
		xplot,out2,/no_block,coltitles=['Energy [eV]','Spl for: '+$
		  string(a.index)], group=event.top, Wtitle='Splines 2D plot'
		END
	  '3 D': BEGIN
		tmp = delia_reform_arrays(a.k,a.chi,a.index,$
		  /xplot,group=event.top,wtitle='Chi 3D plot',$
		  No_Block=state.no_block)
		END
	endcase
	END
    'FT_CHI2FT': BEGIN
	indexcol=0
	widget_control,state.wids.ft.chi2ft,get_value=indexcol
	str = *(state.ptr[2])
	myindex = where(str.index EQ indexcol)
	if myindex[0] EQ -1 then begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Spectrum with selected index('+strcompress(indexcol,/rem)+$
	    ') not found')
	  goto,out
	endif
	set = fltarr(2,n_elements(str.k))
	set[0,*] = reform(str.k)
	set[1,*] = reform(str.chi[myindex,*])
	xaid_ff,set,group=event.top,No_Block=state.no_block,deliaid=state.wids
	END

    'FT_GO': BEGIN
	str = *(state.ptr[2])
	; 
	; preedge
	;
	widget_control,state.wids.labelMain,Set_Value=$
	   'calculating Fourier & Back F transforms...'
	kMin=0 & kMin=0 & k2Min=0 & k2Min=0
	widget_control,state.wids.ft.kMin,get_value=kMin
	widget_control,state.wids.ft.kMax,get_value=kMax
	widget_control,state.wids.ft.k2Min,get_value=k2Min
	widget_control,state.wids.ft.k2Max,get_value=k2Max
	rMin=0 & rMin=0 & r2Min=0 & r2Min=0
	widget_control,state.wids.ft.rMin,get_value=rMin
	widget_control,state.wids.ft.rMax,get_value=rMax
	widget_control,state.wids.ft.r2Min,get_value=r2Min
	widget_control,state.wids.ft.r2Max,get_value=r2Max
	filter=0 & kpower=0 & rshell=0
	widget_control,state.wids.ft.filter,get_value=filter
	widget_control,state.wids.ft.kpower,get_value=kpower
	widget_control,state.wids.ft.rshell,get_value=rshell
	filter=fix(filter)+1
	kpower=fix(kpower)
	app_value = 0
	widget_control,state.wids.ft.app_value,get_value=app_value
	set1 = fltarr(2,n_elements(str.k))
	set1[0,*] = reform(str.k)
	nspectra = n_elements(str.chi[*,0])
	npts=long(n_elements(str.k))

	widget_control,/hourglass
	;out = fltarr(nspectra,npoints)
	for i=0L,nspectra-1 do begin
	  widget_control,state.wids.labelMain,Set_Value=$
	   'calculating FT & BFT for spectrum: '+strcompress(i+1)+' of '+$
	   strcompress(nspectra)
	  ;
	  ; ft calculation
	  ;
	  ft=0
	  set1[1,*] = str.chi[i,*]
	  ftrset,set1,ft,method=1,xrange=[kMin,kMax],rrange=[rMin,rMax], $
	    kpower=kpower,window=filter,windpar=app_value,$ ; /plotfourier,$
	    npoint=4096,kstep=0.04
	  if i EQ 0 then begin
	    npts_out = n_elements(ft[0,*])
	    ; 5 columns: r index ft_real, ft_imag ft_mod
	    out = fltarr(5,npts_out*nspectra)
	  endif
	  out[0,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[0,*])
	  out[1,i*npts_out:(i*npts_out)+npts_out-1] = str.index[i]
	  out[2,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[2,*])
	  out[3,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[3,*])
	  out[4,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[1,*])
	  ;
	  ; bf calculation
	  ;
	  bftrset,ft,setbf,rrange=[r2Min,r2Max], krange=[k2Min,k2Max], $
	  	window=filter,wrange=[kMin,kMax],method=1,$
	  	rshell=rshell,windpar=app_value
	  if i EQ 0 then begin
	    npts_out2 = n_elements(setbf[0,*])
	    ; 5 columns: k index bft_real, bft_modulus bft_phase
	    out2 = fltarr(5,npts_out2*nspectra)
	  endif
	  out2[0,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[0,*])
	  out2[1,i*npts_out2:(i*npts_out2)+npts_out2-1] = str.index[i]
	  out2[2,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[2,*])
	  out2[3,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[3,*])
	  out2[4,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[1,*])
	endfor

	widget_control,state.wids.labelMain,Set_Value='Done'
	windowlist = 1  & window_ftr,names=windowlist

;	;
;	; plot ft
;	;
;	xplot,out,xcol=1,ycol=5,/no_block,p=p,wtitle='Fourier Transform',$
;	 coltitles= ['R [A]','Index','Real(FT)','Imag(FT)','Mod(FT)'],title=$
;	 'FT k='+vect2string([kMin,kMax])+'; Weight=k^'+$
;	 strcompress(kpower,/rem)+'; Window= '+windowlist[filter-1]
;	xplot_mesh,p,flag=1,col=2
;	;
;	; plot bft
;	;
;	xplot,out2,xcol=1,ycol=5,/no_block,p=p,wtitle='Back Fourier T',$
;	 coltitles=['K [A^-1]','Index','Modulus(BFT)','Phase(BFT)',$
;	 'real(BFT)'],title='BFT k='+vect2string([k2Min,k2Max])+$
;	 '; R='+vect2string([r2Min,r2Max])+$
;	 '; R shell= '+strcompress(rshell,/rem)
;	xplot_mesh,p,flag=1,col=2
	;
	; save data in Ptr
	;
        for i=3,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[3]=Ptr_New({npoints:npts_out,nspectra:nspectra,data:out,$
	  kmin:kmin,kmax:kmax,kpower:kpower,filter:filter})
	state.ptr[4]=Ptr_New({npoints:npts_out2,nspectra:nspectra,data:out2,$
	  k2min:k2min,k2max:k2max,r2min:r2min,r2max:r2max,rshell:rshell})
	END



    'FT_PLOT': BEGIN
	val=''
	widget_control,event.id,get_value=val
	case val of
	  'Fourier (2D)': BEGIN
		plotwhat='fourier'
		plotdim=2
		END
	  'Fourier (3D)': BEGIN
		plotwhat='fourier'
		plotdim=3
		END
	  'Back F  (2D)': BEGIN
		plotwhat='back'
		plotdim=2
		END
	  'Back F  (3D)': BEGIN
		plotwhat='back'
		plotdim=3
		END
	endcase

	case plotwhat of
	  'fourier': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		windowlist = 1  & window_ftr,names=windowlist
		if plotdim EQ 2 then begin
		  isel = wmenu2(['Calcel','Plot Real part of FT',$
		    'Plot Imaginary part of FT','Plot FT Modulus'])
		  case isel of
			0: goto,out
			1: icol=2
			2: icol=3
			3: icol=4
		  endcase
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  mydata[0,*] = a.data[0,0:a.npoints-1]
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  coltitles=['R [A]',strcompress(mydataindex,/rem)]
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  mydata[1:a.nspectra,*]=transpose(tmp)
		  xplot,mydata,group=event.top,/no_block, coltitles=coltitles,$
		  wtitle='Fourier Transform (2D)',$
		  title='FT k='+vect2string([a.kMin,a.kMax])+'; Weight=k^'+$
                  strcompress(a.kpower,/rem)+'; Window= '+windowlist[a.filter-1]
		endif else begin
		  xplot,a.data,xcol=1,ycol=5,/no_block,p=p,wtitle=$
		  'Fourier Transform (3D)',$
		  coltitles= ['R [A]','Index','Real(FT)',$
		  'Imag(FT)','Mod(FT)'],$
		  title='FT k='+vect2string([a.kMin,a.kMax])+'; Weight=k^'+$
		  strcompress(a.kpower,/rem)+'; Window= '+windowlist[a.filter-1]
		  xplot_mesh,p,flag=1,col=2
		endelse
		END
	  'back': BEGIN
		if Ptr_Valid(state.ptr[4]) then a=*(state.ptr[4]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Back Fourier Transform not found.')
		  goto,out
		endelse

		if plotdim EQ 2 then begin
		  isel = wmenu2(['Calcel','Plot Modulus of BFT',$
		    'Plot Phase of BFT','Plot real part of BFT'])
		  case isel of
			0: goto,out
			1: icol=2
			2: icol=3
			3: icol=4
		  endcase
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  mydata[0,*] = a.data[0,0:a.npoints-1]
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  coltitles=['K [A^-1]',strcompress(mydataindex,/rem)]
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  mydata[1:a.nspectra,*]=transpose(tmp)
		  xplot,mydata,group=event.top,/no_block, coltitles=coltitles,$
		  wtitle='Back Fourier Transform (2D)',$
		  title='BFT k='+vect2string([a.k2Min,a.k2Max])+$
		  ' r='+vect2string([a.r2Min,a.r2Max])+$
		  '; R shell= '+strcompress(a.rshell,/rem)

		endif else begin
		  xplot,a.data,xcol=1,ycol=5,/no_block,p=p,wtitle=$
		  'Back Fourier Transform (3D)',$
		  coltitles=['K [A^-1]','Index',$
		  'Modulus(BFT)','Phase(BFT)','real(BFT)'],$
		   title='BFT k='+vect2string([a.k2Min,a.k2Max])+$
		   ' r='+vect2string([a.r2Min,a.r2Max])+$
		  '; R shell= '+strcompress(a.rshell,/rem)
		  xplot_mesh,p,flag=1,col=2
		endelse


		END
	endcase
	END

    'ABOUT': XAID_Help,Group=event.top
    'HELP': BEGIN
	widget_control,event.id,get_value=tmp
	xhelp,tmp,GROUP=event.top
	END
    'DABAX_EXAFS': BEGIN
	Widget_Control,/Hourglass
	file = 'XAFS_DATA_EXAFS_CHI.dat'
	h=dabax_access(file(0))
	xplot,spec=h,wtitle=file(0),xtitle='-1',ytitle='-1',title='#S'
	END
    'DABAX_XANES': BEGIN
	Widget_Control,/Hourglass
	file = 'XAFS_DATA_XANES.dat'
	h=dabax_access(file(0))
	xplot,spec=h,wtitle=file(0),xtitle='-1',ytitle='-1',title='#S'
	END
    'XOP_WD': BEGIN
	Xop_WD,Group=event.top
	END
    else: BEGIN
	tmp=''
	Widget_Control,event.id,Get_UValue=tmp,/Hourglass
	command = tmp[0]+ $  ;',Group='+StrCompress(event.Id,/Rem)+'L'+$
	 ',No_Block='+StrCompress(state.no_block)
	Message,/info,'Executing: '+command
	itmp = Execute(command)
	END
EndCase


out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state, /No_Copy
end ; delia_event
;
;====================================================================
;

PRO delia, group=group, No_Block=no_Block

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'DELIA: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return
endif
IF N_Elements(no_block) EQ 0 THEN no_block=0

;
; define widgets
;
wbase=widget_base(/col,title='DELIA 1.01',MBar=wMenuBar)

wtmp = widget_base(wbase) ; to store state

wFileMenu = Widget_Button(wmenuBar,Value='File',/Menu)
  wtmp = Widget_Button(wFileMenu,Value='Load file with ASCII RAW data...',$
	UValue='RAW_LOAD_ASCII')
  wtmp = Widget_Button(wFileMenu,Value='Load file with Mu(E)...',UValue='LOADFILE')
  wtmp = Widget_Button(wFileMenu,Value='Load file with Chi(k)...',UValue='LOADFILE')
  wtmp = Widget_Button(wFileMenu,Value='Quit',UValue='QUIT',/Separator)
wExafsToolsMenu = Widget_Button(wmenuBar,Value='EXAFS_Tools',/Menu)
  wtmp = Widget_Button(wExafsToolsMenu,Value='DELIA',$
	UValue='DELIA')
  wtmp = Widget_Button(wExafsToolsMenu,Value='Main window (XAID)',$
	UValue='XAID')
  wtmp = Widget_Button(wExafsToolsMenu,Value='Calibration (XAID_CALIB)',$
	UValue='XAID_CALIB')
;  wtmp = Widget_Button(wExafsToolsMenu,Value='Calibration (XAID_CALIB)',$
;	UValue='XAID_CALIB')
;  wtmp = Widget_Button(wExafsToolsMenu,Value='Extraction (XAID_MU2CHI)',$
;	UValue='XAID_MU2CHI')
;  wtmp = Widget_Button(wExafsToolsMenu,Value='Fourier Filtering (XAID_FF)',$
;	UValue='XAID_FF')
;wDabaxMenu = Widget_Button(wmenuBar,Value='DABAX_Tools',/Menu)
;  wtmp = Widget_Button(wDabaxMenu,Value='f1 and f2 values',$
;	UValue='XF1F2')
;  wtmp = Widget_Button(wDabaxMenu,Value='McKale phases',$
;	UValue='XMCKALE')
;  wtmp = Widget_Button(wDabaxMenu,Value='EXAFS spectra database',$
;	/Separator,UValue='DABAX_EXAFS')
;  wtmp = Widget_Button(wDabaxMenu,Value='XANES spectra database',$
;	UValue='DABAX_XANES')
wToolsMenu = Widget_Button(wmenuBar,Value='Tools',/Menu)
  wtmp = Widget_Button(wToolsMenu,Value='Plotting tool...',$
	UValue='XPLOT')
  wtmp = Widget_Button(wToolsMenu,Value='Display file...',$
        UValue='XOP_FILE_DISPLAY')
  wtmp = Widget_Button(wToolsMenu,Value='Change working directory...',$
	UValue='XOP_WD')

wHelpMenu = Widget_Button(wmenuBar,Value='Help',/Help)
  wtmp = Widget_Button(wHelpMenu,Value='About XAID',UValue='ABOUT')
  wtmp = Widget_Button(wHelpMenu,Value='delia',UValue='HELP')
;
; sub panels
;
wpanels = cw_bgroup(wbase,/Exclusive,['Raw','mu(E)','chi(k)','Fourier'],$
	UVALUE='SWITCH',Col=5,/Frame,Set_Value=0,/No_Release,$
	LABEL_LEFT='Control panel: ')

wtmp0 = widget_base(wbase)
  wbase_raw = widget_base(wtmp0,/Col,/Frame)
  wbase_mu = widget_base(wtmp0,/Col,/Frame)
  wbase_chi = widget_base(wtmp0,/Col,/Frame)
  wbase_ft = widget_base(wtmp0,/Col,/Frame)
wlabelMain = widget_label(wbase,Value=$
;  '**********************************************************************')
  '*****Welcome to DELIA (Dispersive Exafs caLIbration and Analysis)******')
  ;/Dynamic_Resize)
;
; Raw data panel
;
wtmp0 = widget_base(wbase_raw,/Col,/Frame)
  wtmp = widget_label(wtmp0,Value='Raw data file [binary]: ',/Align_Left)
  wFileRaw = cw_pickfile(wtmp0,Title=' ',Value='<none>',$
	UValue='RAW_LOAD_BINARY',XSize=60)
wTextRaw = widget_text(wbase_raw,Value='<none>',XSIZE=20,YSIZE=20,/Scroll)
wtmp0 = widget_base(wbase_raw,/Row)
  wcol1 = cw_field(wtmp0,Title='Col. with X: ',Value='1',UValue='COL',$
        /Integer,XSize=2)
  wcol2 = cw_field(wtmp0,Title='Col. with Mu: ',Value='2',UValue='COL',$
        /Integer,XSize=2)
  wcol3 = cw_field(wtmp0,Title='Col. with Index: ',Value='3',UValue='COL',$
        /Integer,XSize=2)
wtmp0 = widget_base(wbase_raw,/Row)
  wview = cw_field(wtmp0,Title='Plot spectrum with index: ',Value='0',UValue=$
	'RAW_VIEW#',/Float,XSize=8,/Return_event)
  wtmp = widget_button(wtmp0,Value='Go',UValue='RAW_VIEW#')
wtmp0 = widget_base(wbase_raw,/Row)
  wover = cw_field(wtmp0,Title='Overplot spectrum with index: ',Value='0',$
	UValue='RAW_VIEW#',/Float,XSize=8,/Return_event)
  wtmp = widget_button(wtmp0,Value='Add next',UValue='RAW_VIEW#')
;
; Mu data panel
;
wbase_int = widget_base(wbase_mu,/Col,/Frame)
  wtmp = widget_label(wbase_int,Value=' Interpolation: ',/Align_Left)
 
  wtmp1 = widget_base(wbase_int,/Row)
    wtmp = widget_label(wtmp1,/Align_Left,Value=$
     'Align spectra to their half-jump value?: ')
    wAlign = cw_bgroup(wtmp1,/Exclusive,['No','Yes'],$
        Col=2,Set_Value=0, UValue='SWITCH', /No_Release)

  wtmp = widget_label(wbase_int,/Align_Left,Value=$
	'X extrema in the intervals : ')
  wlabelInt1 = widget_label(wbase_int,/Align_Left,Value=$
	'  from (minima) [0,0] ',/Dynamic_Resize)
  wlabelInt2 = widget_label(wbase_int,/Align_Left,Value=$
	'  to (maxima) [0,0]',/Dynamic_Resize)
  wtmp0 = widget_base(wbase_int,/Row)
    wIntMin = cw_field(wtmp0,Title='Min X:  ',$
	Value='0',/Float,XSize=8)
    wIntMax = cw_field(wtmp0,Title='Max X:  ',$
	Value='0',/Float,XSize=8)
    wIntN = cw_field(wtmp0,Title='Number of points:  ',$
	Value='512',/Integer,XSize=4)

wbase_cal = widget_base(wbase_mu,/Col,/Frame)
  wtmp = widget_label(wbase_cal,/Align_Left,Value=$
     'Calibration:  Xnew[i] = a + b*(X[i]-X[0]) + c*(X[i]-X[0])^2 ')
  wtmp0 = widget_base(wbase_cal,/Row)
    wcal_a = cw_field(wtmp0,Title='a: ',Value='0',/Float,XSize=8)
    wcal_b = cw_field(wtmp0,Title='b: ',Value='1',/Float,XSize=8)
    wcal_c = cw_field(wtmp0,Title='c: ',Value='0',/Float,XSize=8)
    wtmp = widget_button(wtmp0,Value='Get...',UValue='MU_GET')
 

wbase_res = widget_base(wbase_mu,/Col,/Frame)
  wtmp = widget_button(wbase_res,Value='RUN interpolate & calibrate',$
    UValue='MU_GO')
  wtmp = widget_label(wbase_res,Value='Result: ',/Align_Left)
  wTextMu = widget_text(wbase_res,Value='<none>',XSIZE=60,YSIZE=10,/Scroll)
  wtmp0 = widget_base(wbase_res,/Row)
    wtmp = widget_label(wtmp0,/Align_Left,Value=$
	'Plot results: ')
    wtmp = widget_button(wtmp0,Value='2 D',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='3 D',UValue='MU_PLOT')

;
; Chi data panel
;
wtmp0 = widget_base(wbase_chi,/Row,/Frame)
  wMu2Chi = cw_field(wtmp0,$
   Title='Prepare extraction using spectrum with index: ',$
   Value='0',/Float,XSize=12,UValue='CHI_MU2CHI',/Return_Events)
  wtmp = widget_button(wtmp0,Value='Go',UValue='CHI_MU2CHI')
wtmp0 = widget_base(wbase_chi,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,$
    Value='Parameters to calculate EXAFS signal')
  wtmp = widget_label(wtmp0,/Align_Left,$
    Value=' ')
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
      'PreEdge Linear Fit (Does not apply if Min=Max): ')

  wtmp1 = widget_base(wtmp0,/Row)
      wPreMin = cw_field(wtmp1,Title=$
      ' E Min:',Value='0',/Float,XSize=10)
      wPreMax = cw_field(wtmp1,Title='E Max:',Value='0',/Float,XSize=10)
      wE0 = cw_field(wtmp1,Title='Eo:',Value='0.0',/Float,XSize=10)
  wtmp1 = widget_base(wtmp0,/Row)
    wJump = cw_field(wtmp1,Title='Jump: ',Value='0.0',/Float,XSize=10)
    wKKMin = cw_field(wtmp1,Title='Calculate chi only for for k >  ',$
  	Value='2.0',/Float,XSize=10)

  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_label(wtmp1,/Align_Left,Value='PostEdge: N polynom.:')
    wNPol = widget_text(wtmp1,Value='3',XSize=2,/Edit)
    wtmp = widget_label(wtmp1,/Align_Left,Value='Degrees:')
    wDegree = widget_text(wtmp1,Value='[2,2,2]',XSize=15,/Edit)
  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_label(wtmp1,/Align_Left,Value='PostEdge: Knots (in k):')
    wknots = widget_text(wtmp1,Value='[1,3,4,6]',XSize=50,/Edit)
  ;wtmp1 = widget_base(wtmp0,/Row)
  ;  wKKMin = cw_field(wtmp1,Title='Calculate chi for k greater than: ',$
  ;	Value='2.0',/Float,XSize=10)
wtmp0 = widget_base(wbase_chi,/Col,/Frame)
  wtmp = widget_button(wtmp0,$
      Value='CALCULATE EXTRACTION OVER ALL SPECTRA',UValue='CHI_GO')
  wTextChi = widget_text(wtmp0,Value='<none>',XSIZE=60,YSIZE=10,/Scroll)
  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_label(wtmp1,/Align_Left,Value='Plot results: ')
    wtmp = widget_button(wtmp1,Value='2 D',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='2 D (splines)',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='3 D',UValue='CHI_PLOT')
;
; FT data panel
;

wtmp0 = widget_base(wbase_ft,/Row,/Frame)
  wChi2FT = cw_field(wtmp0,Title=$
   'Prepare FT & BFT using spectrum with index: ',Value='0',$
   /Float,XSize=4,UValue='FT_CHI2FT',/Return_Events)
  wtmp = widget_button(wtmp0,Value='Go',UValue='FT_CHI2FT')

wtmp0 = widget_base(wbase_ft,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
    'Parameters to calculate Fourier tramsform')

  wtmp1 = widget_base(wtmp0,/Row)
    wKmin = cw_field(wtmp1,Title=$
      'Interval in K (Full interval if Min=Max):  K Min:',Value='0',/Float,$
      XSize=4)
    wKmax = cw_field(wtmp1,Title='K Max:',Value='0',/Float,XSize=4)
  wtmp1 = widget_base(wtmp0,/Row)
    wRmin = cw_field(wtmp1,  Title='Interval in R:   Rmin',vALUE='0.0',$
      /Float,XSize=4)
    wRmax = cw_field(wtmp1,Title='Rmax:',Value='8.0',/Float,XSize=4)

  wtmp1 = widget_base(wtmp0,/Row)
    windowlist = 1 & window_ftr,names=windowlist
    wFilter = cw_droplist(wtmp1, $
	VALUE=['0',windowlist], $
	TITLE='Window: ',UValue='FT_FILTERDROP' )
    wapp_base = widget_base(wtmp1,/Row) ; to allow mapping
    wapp_label = widget_label(wapp_base,Value='Appodization width:')
    wapp_value = cw_field(wapp_base,Title=' ',Value='0.2',/Float,XSize=4)

  wKpower = cw_droplist(wtmp0, VALUE=['0','0','1','2','3','4'], $
	TITLE='Weight data with K Power: ')


;;;;;;;;;;;
wtmp0 = widget_base(wbase_ft,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
    'Parameters to calculate BACK Fourier tramsform')
  wtmp1 = widget_base(wtmp0,/Row)
    wR2min = cw_field(wtmp1,Title=$
      'R interval: R Min:',Value='0',/Float,XSize=4)
    wR2max = cw_field(wtmp1,Title='R Max:',Value='8',/Float,XSize=4)
  wtmp1 = widget_base(wtmp0,/Row)
    wK2min = cw_field(wtmp1,Title=$
      'K interval:  K Min:',$
	Value='0',/Float,XSize=4)
    wK2max = cw_field(wtmp1,Title='K Max:',Value='0',/Float,XSize=4)
  wRShell = cw_field(wtmp0,Title='Shell radius for the back phase:',$
	Value='0',/Float,XSize=4)
;;;;;;;;;;;

   
wtmp = widget_button(wbase_ft,Value='CALCULATE FT  & BFT OVER ALL SPECTRA',$
  UValue='FT_GO')
wtmp0 = widget_base(wbase_ft,/Row)
  wtmp = widget_label(wtmp0,Value='Plot results: ',/Align_Left)
  wtmp = widget_button(wtmp0,Value='Fourier (2D)',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Fourier (3D)',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Back F  (2D)',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Back F  (3D)',UValue='FT_PLOT')
;
;
;

; pointer array: 
;   index 0 : raw data
;   index 1 : mu (structure)
;   index 2 : chi (structure)
;   index 3 : fourier transform (matrix)
;   index 4 : back fourier transform (matrix)
ptr = PtrArr(5)


widsRaw = {file:wFileRaw, text:wTextRaw,$
  view:wview, over:wover, $
  col1:wcol1, col2:wcol2, col3:wcol3 }
widsMu = {labelInt1:wLabelInt1, labelInt2:wLabelInt2, intMin:wIntMin, $
  intmax:wIntMax, intN:wIntN, text:wTextMu, align:wAlign, $
  cal_a:wcal_a, cal_b:wcal_b, cal_c:wcal_c}
widsChi = {mu2chi:wMu2Chi, preMin:wPreMin, preMax:wPreMax, e0:wE0, $
  jump:wJump, nPol:wNPol, degree:wdegree, knots:wknots, text:wTextChi,$
  kMin:wKKMin}
widsFT = {chi2ft:wChi2FT, kMin:wkMin, kMax:wkMax, rMin:wrMin, rMax:wrMax, $
  filter:wFilter, app_base:wApp_base, app_label:wapp_label, $
  app_value:wapp_value, kPower:wkPower, $
  k2Min:wK2Min,k2Max:wK2Max, rShell:wRShell, r2Min:wR2Min, r2Max:wR2Max}

wids = {panels:wpanels, labelMain:wlabelMain,$
  panel_ids:[wbase_raw,wbase_mu,wbase_chi,wbase_ft],xplot:0L, $
  raw:widsRaw, mu:widsMu, chi:widsChi, ft:widsFT }
state = {wids:wids, ptr:ptr, no_block:no_block}

delia_mappanels,wids
;
;
widget_control,Widget_Info(wbase,/Child),set_uvalue=state,/no_copy

widget_control,wbase,/realize
xmanager,'delia',wbase,GROUP=group,No_Block=no_block
end
