;Pro XSh_Source,input, Group=group, BM=bm, NO_BLOCK=no_block, $
;  Only_Run=only_Run, Modal=modal

;+
;
;   ===========================  Xsh_Source   =========================
;
;   Xsh_Source is an widget based graphical interface to generate
;   the SHADOW geometrical and bending magnet sources.
;
; ------------- Non-interactive use of XSH_Source ----------------------
;
; NAME:        
;       XSH_SOURCE
; PURPOSE:              
;       To generate SHADOW bending magnet and geometrical sources
;               
; CALLING SEQUENCE: 
;       XSH_SOURCE [, input] 
;    
; OPTIONAL INPUT PARAMETERS:     
;	input: A structure or a pointer to a structure containing 
;		a g-structure (i.e., am IDL structure containing the 
;		information set in a g-file, like start.00)
;		If this input is not set, the g-structure is 
;		obtained from XSH_Defaults('XSH_SOURCE_G') or 
;		XSH_Defaults('XSH_SOURCE_BM') depending if the 
;		BM keyword is not or yes, respectively. 
;
; OUTPUTS:
;       Calls SHADOW gen_source when running. In input is a structure, 
;	this structure is modified.
;                              
; KEYWORDS:
;       BM = If set, starts the Bending magnet parameter window, 
;		otherwise, it starts the geometrical source.
;       ONLY_RUN = run SHADOW/Source for the given input without starting 
;		any window.
;       NO_BLOCK = IF set, this keyword is passed to XManager.
;       MODAL = Start the application in modal mode, the modifications are 
;		returned in the input structure.
;		
; MODIFICATION HISTORY:
;       Author: M. Sanchez del Rio (srio@esrf.fr), ESRF, Jun 2000
;       Modification history:
;	00/06/30 srio@esrf.fr first version. This applications
;		makes XSHGEOM and XSHBM obsolete.
;	01/06/12 srio@esrf.fr bug fixed when passing hor div to srcomp
;
;-

;		
;========================================================================
;

PRO XSH_Source_Event,event


CATCH, error_status
IF error_status NE 0 THEN BEGIN
   Message,/info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,$
	'XSH_SOURCE_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   GoTo,out
ENDIF

;
;
Widget_Control, event.id, get_UValue=eventUValue
; help,eventUValue
 
infoId = Widget_Info(event.handler,/Child)
Widget_Control, infoId, get_UValue=info, /No_Copy

 
;
; dynamically map panels
;
IF eventUValue EQ 'FGRID' THEN $
  XSh_Source_MapPanels,info.wids,Group=event.top

;
;
;
index=-1
tmp=0
Case eventUValue OF
    'Accept': BEGIN
	Widget_Control,event.handler,/destroy
	END
    'Cancel': BEGIN
	; retrieve the original input data
	*info.ptrStr=info.init_str
	Widget_Control,event.handler,/destroy
	END
    'Run Shadow/source': BEGIN
        CD,Current=path
        file=path+'/start.00'
        Write_GFile,*info.ptrStr,file
        Widget_Control,/HourGlass
        XSh_Run,"gen_source start.00"
	END
    'adv.loadg': BEGIN
        CD,Current=path
	file = Dialog_Pickfile(Dialog_Parent=event.top, $
	  Filter='*.00', /Must_Exist, Path=path, $
	  Title='Select GFile with source (e.g., start.00)' )
	IF file EQ '' THEN GoTo,out
	tmp = Read_GFile(file)
	IF Type(tmp) EQ 2 THEN BEGIN
	  itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
		['Error reading GFile: ',file,' ', $
		'It does not seems to be a GFile'])
	  GoTo,out
	ENDIF
	tNames = Tag_Names(tmp)
	wNames = Where (tNames EQ 'F_WIGGLER')
	IF wNames[0] EQ -1 THEN BEGIN
	  itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
		['Error loading data from GFile: ',file,' ', $
		'It does not seems to contain a SHADOW source'])
	  GoTo,out
	ENDIF
	f_wiggler = tmp.F_WIGGLER
	sType = 'geometric'
	CASE f_wiggler OF 
	  0: IF tmp.FDISTR EQ 4 OR tmp.FDISTR EQ 6 THEN $
		sType = 'bending magnet'
	  1: sType = 'wiggler'
	  2: sType = 'undulator'
	  3: sType = 'unknown'
	ENDCASE
	IF info.bm EQ 0 AND sType NE 'geometric' THEN BEGIN
	  itmp = Dialog_Message(/Question, Dialog_Parent=event.top, $
		['Error loading data from GFile: ',file,' ', $
		'The selected file source is not geometric, but : '+sType,$
		'This application (XSh_Source) is set to accept ', $
		'only geometric sources. ', $
		'However, the parameters corresponding to geometric ',$
		'source can be loaded. Do you want to continue ?'] )
	  IF itmp EQ 'No' THEN GoTo,out
	ENDIF ELSE IF info.bm EQ 1 AND sType NE 'bending magnet' THEN BEGIN
	  itmp = Dialog_Message(/Question, Dialog_Parent=event.top, $
		['Error loading data from GFile: ',file,' ', $
		'The selected file source is not a bending magnet, but : '+$
			sType,$
		'This application (XSh_Source) is set to accept ', $
		'only bending magnet sources. ', $
		'However, the parameters corresponding to bending magnet ',$
		'source can be loaded. Do you want to continue ?'] )
	  IF itmp EQ 'No' THEN GoTo,out
	ENDIF
	str = *info.ptrStr
	Copy_Structure,tmp,str,Group=event.top
	*info.ptrStr = str
	XSh_Source_UpdatePanels,info.wids,info.ptrStr,Group=group,/Map
	END
    'adv.writeg': BEGIN
	file = Dialog_Pickfile(Dialog_Parent=event.top, $
	  Filter='*.00', File='start_geom.00', Path=path, $
	  Title='Select file to write source' )
	IF file EQ '' THEN GoTo,out
	IF CheckFile(file) EQ 1 THEN BEGIN
	  itmp = Dialog_Message(/Question, Dialog_Parent=event.top, $
		['File already exists: ',file,'Overwrite it?'])
	  IF itmp EQ 'No' THEN GoTo,out
	ENDIF
	tmp = *info.ptrStr
	Write_GFile,tmp,file
	END
    'adv.viewg': BEGIN
	file = 'xsh_source_tmp.txt'
	tmp = *info.ptrStr
	Write_GFile,tmp,file
	XDisplayFile1,file,Title='GFile for source',Dialog_Parent=event.top,$
	  /Remove
	END


    ;
    ; COMMON ENTRIES (GEOMETRICAL AND BM SOURCES)
    ;
    'NPOINT': BEGIN
	;  npoint =          5000 - number of random rays (0-5000).
	Widget_Control,info.wids.NPOINT,Get_Value=tmp
	(*info.ptrStr).NPOINT = tmp
	END
    'ISTAR1': BEGIN
	;  istar1 =        478291 - seed for random number generator, odd.
	Widget_Control,info.wids.ISTAR1,Get_Value=tmp
	(*info.ptrStr).ISTAR1 = tmp
	END
    ;
    ; GEOMETRICAL SOURCE ENTRIES
    ;
    'FGRID': BEGIN
	; fgrid  =             0 - defines source modelling type --
	; spatial/momentum space.  Options are:
	; RANDOM/RANDOM (0), GRID/GRID (1), GRID/RANDOM (2), RANDOM/GRID (3), 
	; ELLIPSE in phase/RANDOM on the ellipse (4), 
	; ELLIPSE in phase/GRID on the ellipse (5).
	Widget_Control,info.wids.FGRID,Get_Value=index
	(*info.ptrStr).FGRID=Fix(index)
	IF index EQ '0' THEN GoTo,out
	; 
	IF index EQ '1' OR index EQ '2' OR index EQ '3' THEN BEGIN
	  ; 
	  data={ido_vx:(*info.ptrStr).IDO_VX, $
		ido_vz:(*info.ptrStr).IDO_VZ, $
		ido_x_s:(*info.ptrStr).IDO_X_S, $
		ido_y_s:(*info.ptrStr).IDO_Y_S, $
		ido_z_s:(*info.ptrStr).IDO_Z_S, $
		n_circle:(*info.ptrStr).N_CIRCLE, $
		n_cone:(*info.ptrStr).N_CONE } 
	  titles=["# grid points in X'", $
		  "# grid points in Z'", $
		  "# grid points in X", $
		  "# grid points in Y", $
		  "# grid points in Z", $
		  "# radial grid points ", $
		  "# concentrical grid points "]
;	  wtitle='FGRID='+index
	  wtitle='Grid sources'
	  nCol=3
	  flags=Replicate('1',7)
	  CASE index OF
	   '2': flags[0:1]='0' ; GRI-space/RAN-momentum
	   '3': flags[2:4]='0' ; RAN-space/GRI-momentum
	   else:
	  ENDCASE
	  CASE (*info.ptrStr).FDISTR OF
	    1: flags[5:6]='0'
	    2: flags[5:6]='0'
	    5: flags[0:4]='0' 
	    else:
	  ENDCASE
	  ; This is correct. It is commented because a bug in SHADOW:
	  ; GRID/GRID source: generates NX*NY*NZ*Nx'*NZ' rays
	  ; If FSOURCE_DEPTH=1 (OFF) NY must be 1, but it takes the value in
	  ;  ido_y_s
	  ;CASE (*info.ptrStr).FSOURCE_DEPTH OF
	  ;  1: flags[3]='0'
	  ;  else: 
	  ;ENDCASE

	  XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
	    Action=action, FieldLen=15, Dialog_Parent=event.top, $
	    NCol=nCol, Flags=flags
	  IF action EQ 'DONT' THEN GoTO,out
	  (*info.ptrStr).IDO_VX = data.IDO_VX 
	  (*info.ptrStr).IDO_VZ = data.IDO_VZ 
	  (*info.ptrStr).IDO_X_S = data.IDO_X_S 
	  (*info.ptrStr).IDO_Y_S = data.IDO_Y_S 
	  (*info.ptrStr).IDO_Z_S = data.IDO_Z_S 
	  (*info.ptrStr).N_CIRCLE = data.N_CIRCLE 
	  (*info.ptrStr).N_CONE = data.N_CONE 
	ENDIF ELSE BEGIN ; Ellipses in phase spaces.
	  IDO_XL = StrCompress( (*info.ptrStr).IDO_XL - 1,/Rem)
	  IDO_ZL = StrCompress( (*info.ptrStr).IDO_ZL - 1,/Rem)
	  data={sigmax:(*info.ptrStr).SIGMAX, $
	  	sigmaz:(*info.ptrStr).SIGMAZ, $
	  	sigdix:(*info.ptrStr).SIGDIX, $
	  	sigdiz:(*info.ptrStr).SIGDIZ, $
	  	tmp1:0, $
		ido_xl:[IDO_XL,'1','2','3','4','5','6','7','8','9','10'],$
		ido_zl:[IDO_ZL,'1','2','3','4','5','6','7','8','9','10'],$
	  	ido_xn:(*info.ptrStr).IDO_XN, $
	  	ido_zn:(*info.ptrStr).IDO_ZN, $
	  	tmp2:0, $
	  	sigxl1:(*info.ptrStr).SIGXL1, $
	  	  sigxl2:(*info.ptrStr).SIGXL2, $
	  	  sigxl3:(*info.ptrStr).SIGXL3, $
	  	  sigxl4:(*info.ptrStr).SIGXL4, $
	  	  sigxl5:(*info.ptrStr).SIGXL5, $
	  	  sigxl6:(*info.ptrStr).SIGXL6, $
	  	  sigxl7:(*info.ptrStr).SIGXL7, $
	  	  sigxl8:(*info.ptrStr).SIGXL8, $
	  	  sigxl9:(*info.ptrStr).SIGXL9, $
	  	  sigxl10:(*info.ptrStr).SIGXL10, $
	  	sigzl1:(*info.ptrStr).SIGZL1, $
	  	  sigzl2:(*info.ptrStr).SIGZL2, $
	  	  sigzl3:(*info.ptrStr).SIGZL3, $
	  	  sigzl4:(*info.ptrStr).SIGZL4, $
	  	  sigzl5:(*info.ptrStr).SIGZL5, $
	  	  sigzl6:(*info.ptrStr).SIGZL6, $
	  	  sigzl7:(*info.ptrStr).SIGZL7, $
	  	  sigzl8:(*info.ptrStr).SIGZL8, $
	  	  sigzl9:(*info.ptrStr).SIGZL9, $
	  	  sigzl10:(*info.ptrStr).SIGZL10 }

	  titles=[ $
	    "Sigma X [cm]","Sigma Z [cm]", $
		"Sigma X' [rad]","Sigma Z' [rad]","",$	; 0-4
	    'Ellipses in X','Ellipses in Z', $
		'Grid points in X','Grid points in Z','', $	; 5-9
	    'X level #1', 'X level #2', 'X level #3', $
		'X level #4', 'X level #5',  $			; 10-14
	    'X level #6', 'X level level #7', 'X level #8', $
		'X level #9', 'X level #10',  $			; 15-19
	    'Z level #1', 'Z level #2', 'Z level #3', $
		'Z level #4', 'Z level #5',  $			; 20-24
	    'Z level #6', 'Z level #7', 'Z level #8', $
		'Z level #9', 'Z level #10' ]			; 25-29
	  flags = ['1','1','1','1','0', $
	    '1','1','1','1','0', $
	    'w(5) GE 0','w(5) GE 1','w(5) GE 2','w(5) GE 3','w(5) GE 4', $
	    'w(5) GE 5','w(5) GE 6','w(5) GE 7','w(5) GE 8','w(5) GE 9', $
	    'w(6) GE 0','w(6) GE 1','w(6) GE 2','w(6) GE 3','w(6) GE 4', $
	    'w(6) GE 5','w(6) GE 6','w(6) GE 7','w(6) GE 8','w(6) GE 9' ]
	  CASE StrCompress(index,/Rem) OF
	    '4': BEGIN
		wtitle='Ellipse in phase space/Random on the ellipse'
	  	flags[7:8]='0'
		END
	    '5': BEGIN
		wtitle='Ellipse in phase space/Grid on the ellipse'
		END
	    else:
	  ENDCASE
	  nCol=6
	  XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
	    Action=action, FieldLen=15, Dialog_Parent=event.top, $
	    NCol=nCol, /Interp, Flags=flags
	  IF action EQ 'DONT' THEN GoTO,out
	  (*info.ptrStr).SIGMAX = data.SIGMAX 
	  (*info.ptrStr).SIGMAZ = data.SIGMAZ 
	  (*info.ptrStr).SIGDIX = data.SIGDIX 
	  (*info.ptrStr).SIGDIZ = data.SIGDIZ 
	  (*info.ptrStr).IDO_XL = Fix(data.IDO_XL[0])+1 
	  (*info.ptrStr).IDO_ZL = Fix(data.IDO_ZL[0])+1 
	  (*info.ptrStr).IDO_XN = data.IDO_XN 
	  (*info.ptrStr).IDO_ZN = data.IDO_ZN 
	  (*info.ptrStr).SIGXL1 = data.SIGXL1 
	    (*info.ptrStr).SIGXL2 = data.SIGXL2 
	    (*info.ptrStr).SIGXL3 = data.SIGXL3 
	    (*info.ptrStr).SIGXL4 = data.SIGXL4 
	    (*info.ptrStr).SIGXL5 = data.SIGXL5 
	    (*info.ptrStr).SIGXL6 = data.SIGXL6 
	    (*info.ptrStr).SIGXL7 = data.SIGXL7 
	    (*info.ptrStr).SIGXL8 = data.SIGXL8 
	    (*info.ptrStr).SIGXL9 = data.SIGXL9 
	    (*info.ptrStr).SIGXL10 = data.SIGXL10 
	  (*info.ptrStr).SIGZL1 = data.SIGZL1 
	    (*info.ptrStr).SIGZL2 = data.SIGZL2 
	    (*info.ptrStr).SIGZL3 = data.SIGZL3 
	    (*info.ptrStr).SIGZL4 = data.SIGZL4 
	    (*info.ptrStr).SIGZL5 = data.SIGZL5 
	    (*info.ptrStr).SIGZL6 = data.SIGZL6 
	    (*info.ptrStr).SIGZL7 = data.SIGZL7 
	    (*info.ptrStr).SIGZL8 = data.SIGZL8 
	    (*info.ptrStr).SIGZL9 = data.SIGZL9 
	    (*info.ptrStr).SIGZL10 = data.SIGZL10 
	ENDELSE
;	itmp = Dialog_Message('To be completed...',Dialog_Parent=event.top)

	END
    'FSOUR': BEGIN
	; fsour  =  0 - spatial source type/shape in X-Z plane.  Options
	;           are: point (0), rectangle (1), ellipse (2),
	;           gaussian (3).
	Widget_Control,info.wids.FSOUR,Get_Value=index
	(*info.ptrStr).FSOUR=Fix(index)
	CASE index OF
	 '0':	GoTo,out ; Point
	 '1':	BEGIN
		titles=['Width [cm]','Height [cm]']
		wtitle='Spatial(Recangular)'
		data={x:(*info.ptrStr).WXSOU,z:(*info.ptrStr).WZSOU}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).WXSOU = data.x & (*info.ptrStr).WZSOU = data.z
		END
	 '2':	BEGIN
		titles=['semi-axis X [cm]','semi-axis Z [cm]']
		wtitle='Spatial(Elliptical)'
		data={x:(*info.ptrStr).WXSOU,z:(*info.ptrStr).WZSOU}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).WXSOU = data.x & (*info.ptrStr).WZSOU = data.z
		END
	 '3':	BEGIN
		titles=['Sigma X [cm]','Sigma Z [cm]']
		wtitle='Spatial(Gaussian)'
		data={x:(*info.ptrStr).SIGMAX,z:(*info.ptrStr).SIGMAZ}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).SIGMAX = data.x & (*info.ptrStr).SIGMAZ = data.z
		END
	 else:
	ENDCASE
	END

    'PreFDISTR': BEGIN
	; fdistr =   2 - defines source angle distribution types:
	;	Available options are: flat(1),uniform(2),
	;	gaussian(3), synchrotron(4), conical(5), exact synchrotron(6).
	;
	Widget_Control,info.wids.PreFDISTR,Get_Value=index
	(*info.ptrStr).FDISTR=Fix(index)+1
	CASE index OF
	  '0': BEGIN ; Flat
		titles=['Horizontal Divergence X(+) [rad]', $
			'Horizontal Divergence X(-) [rad]', $
			'Vertical Divergence Z(+) [rad]', $
			'Vertical Divergence Z(-) [rad]']
		wtitle='Angle Distribution (flat)'
		data={x1:(*info.ptrStr).HDIV1,x2:(*info.ptrStr).HDIV2, $
		      z1:(*info.ptrStr).VDIV1,z2:(*info.ptrStr).VDIV2 }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).HDIV1 = data.x1
		(*info.ptrStr).HDIV2 = data.x2
		(*info.ptrStr).VDIV1 = data.z1
		(*info.ptrStr).VDIV2 = data.z2
		END
	  '1': BEGIN ; Uniform
		titles=['Horizontal Divergence X(+) [rad]', $
			'Horizontal Divergence X(-) [rad]', $
			'Vertical Divergence Z(+) [rad]', $
			'Vertical Divergence Z(-) [rad]']
		wtitle='Angle Distribution (uniform)'
		data={x1:(*info.ptrStr).HDIV1,x2:(*info.ptrStr).HDIV2, $
		      z1:(*info.ptrStr).VDIV1,z2:(*info.ptrStr).VDIV2 }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).HDIV1 = data.x1
		(*info.ptrStr).HDIV2 = data.x2
		(*info.ptrStr).VDIV1 = data.z1
		(*info.ptrStr).VDIV2 = data.z2
		END
	  '2': BEGIN ; Gaussian
		titles=['Horizontal Limit X(+) [rad]', $
			'Horizontal Limit X(-) [rad]', $
			'Vertical Limit Z(+) [rad]', $
			'Vertical Limit Z(-) [rad]', $
			'Horizontal Sigma [X] [rad]', $
			'Vertical  Sigma [Z] [rad]']
		wtitle='Angle Distribution (Gaussian)'
		data={x1:(*info.ptrStr).HDIV1,x2:(*info.ptrStr).HDIV2, $
		      z1:(*info.ptrStr).VDIV1,z2:(*info.ptrStr).VDIV2, $
		      sx:(*info.ptrStr).SIGDIX,sz:(*info.ptrStr).SIGDIZ }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).HDIV1 = data.x1
		(*info.ptrStr).HDIV2 = data.x2
		(*info.ptrStr).VDIV1 = data.z1
		(*info.ptrStr).VDIV2 = data.z2
		(*info.ptrStr).SIGDIX = data.sx
		(*info.ptrStr).SIGDIZ = data.sz
		END
	  '3': BEGIN
		itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ['This option is not used by SHADOWVUI. ', $
		   'Please select another option', $
		   '(It is reserved for FDIST=4 (synchrotron))'])
		END
	  '4': BEGIN
		titles=['Cone Internal Half-Aperture [rad???]', $
			'Cone External Half-Aperture [rad???]']
		wtitle='Angle Distribution (conical)'
		data={x1:(*info.ptrStr).CONE_MIN,x2:(*info.ptrStr).CONE_MAX}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).CONE_MIN = data.x1
		(*info.ptrStr).CONE_MAX = data.x2
		END
	  '5': BEGIN
		itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ['This option is not used by SHADOWVUI. ', $
		   'Please select another option', $
		   '(It is reserved for FDIST=6 (exact synchrotron))'])
		END
	  else:
	ENDCASE
	END

    'PreFSOURCE_DEPTH': BEGIN
	; fsource_depth  =   1 - defines the source depth (Y).  Options
	;	are:  no depth (1), flat depth (2), gaussian (3), 
	;	synchrotron depth (4).
	Widget_Control,info.wids.PreFSOURCE_DEPTH,Get_Value=index
	(*info.ptrStr).FSOURCE_DEPTH=Fix(index)+1
	CASE index OF
	  '0': GoTo,out
	  '1': BEGIN
		titles=['Source Depth [Y] [cm]']
		wTitle='Depth (uniform)'
		data={y:(*info.ptrStr).WYSOU}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).WYSOU = data.y
		END
	  '2': BEGIN
		titles=['Sigma Y [cm]']
		wTitle='Depth (Gaussian)'
		data={y:(*info.ptrStr).SIGMAY}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, FieldLen=15, Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).SIGMAY = data.y
		END
	ENDCASE
	END

;    'F_PHOT': BEGIN
;	; f_phot =   0 - defines whether the photon energy will be
;	;	specified in eV (0) or Angstroms (1).
;	Widget_Control,info.wids.F_PHOT,Get_Value=index
;	(*info.ptrStr).F_PHOT=Fix(index)
;	END

    'PreF_COLOR': BEGIN
	; f_color  =    1 - photon energy distribution type.
	;	Options are:  single energy (1), 
	;	mutliple discrete energies, up to 10 energies (2), 
	;	uniform energy distribution (3).
	; f_phot =   0 - defines whether the photon energy will be
	;	specified in eV (0) or Angstroms (1).
	Widget_Control,info.wids.PreF_COLOR,Get_Value=index
	(*info.ptrStr).F_COLOR=Fix(index)+1
	F_PHOT = StrCompress((*info.ptrStr).F_PHOT,/Rem)
	CASE index OF
	  '0': BEGIN ; Single line
		titles = ['Units:','Value:']
		wTitle='Single energy/wavelength'
		data={ f_phot:[F_PHOT,'Energy/eV' ,'Wavelength/A'], $
			ph1:(*info.ptrStr).PH1 }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).F_PHOT=Fix( (data.f_phot)[0] )
		(*info.ptrStr).PH1 = data.ph1 
		END
	  '1': BEGIN ; Multiple line
		titles = ['Units:','Number of lines',  $
		  'Line 1','Line 2','Line 3','Line 4','Line 5',$
		  'Line 6','Line 7','Line 8','Line 9','Line 10']
		wTitle='Several energy/wavelength lines'
		flags = ['1','1', $
			'w(1) GE 0','w(1) GE 1','w(1) GE 2','w(1) GE 3',$
			'w(1) GE 4','w(1) GE 5','w(1) GE 6','w(1) GE 7',$ 
			'w(1) GE 8','w(1) GE 9']
		n_color=(*info.ptrStr).N_COLOR
		n_color = StrCompress( (n_color-1) >0 <9 ,/Rem)
		data={ f_phot:[F_PHOT,'Energy/eV' ,'Wavelength/A'], $
			n_color:[n_color,'1','2','3','4','5','6','7',$
				'8','9','10'], $
			ph1:(*info.ptrStr).PH1, ph2:(*info.ptrStr).PH2, $
			ph3:(*info.ptrStr).PH3, ph4:(*info.ptrStr).PH4, $
			ph5:(*info.ptrStr).PH5, ph6:(*info.ptrStr).PH6, $
			ph7:(*info.ptrStr).PH7, ph8:(*info.ptrStr).PH8, $
			ph9:(*info.ptrStr).PH9, ph10:(*info.ptrStr).PH10 }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret, $
		  NCol=4, Flags=flags
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).F_PHOT=Fix( (data.f_phot)[0] )
		(*info.ptrStr).N_COLOR=Fix( (data.n_color)[0] ) + 1
		(*info.ptrStr).PH1 = data.ph1 & (*info.ptrStr).PH2 = data.ph2
		(*info.ptrStr).PH3 = data.ph3 & (*info.ptrStr).PH4 = data.ph4
		(*info.ptrStr).PH5 = data.ph5 & (*info.ptrStr).PH6 = data.ph6
		(*info.ptrStr).PH7 = data.ph7 & (*info.ptrStr).PH8 = data.ph8
		(*info.ptrStr).PH9 = data.ph9 & (*info.ptrStr).PH10 = data.ph10
		END
	  '2': BEGIN ; Continous
		titles = ['Units:','Minimum energy/wavelength',$
			'Maximum energy/wavelength']
		wTitle='Continous (uniform) energy spectrum'
		n_color=(*info.ptrStr).N_COLOR
		n_color = StrCompress( (n_color-1) >0 <9 ,/Rem)
		data={ f_phot:[F_PHOT,'Energy/eV' ,'Wavelength/A'], $
			ph1:(*info.ptrStr).PH1, ph2:(*info.ptrStr).PH2}
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret, NCol=1
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).F_PHOT=Fix( (data.f_phot)[0] )
		(*info.ptrStr).PH1 = data.ph1 & (*info.ptrStr).PH2 = data.ph2
		END
	  '3': BEGIN ; Relative intensities
		titles = ['Units:','Number of lines',  $
		  'Line 1','Line 2','Line 3','Line 4','Line 5',$
		  'Line 6','Line 7','Line 8','Line 9','Line 10',$
		  'Int 1','Int 2','Int 3','Int 4','Int 5',$
		  'Int 6','Int 7','Int 8','Int 9','Int 10' ]
		wTitle='Several lines with relative intensities'
		flags = ['1','1', $
			'w(1) GE 0','w(1) GE 1','w(1) GE 2','w(1) GE 3',$
			'w(1) GE 4','w(1) GE 5','w(1) GE 6','w(1) GE 7',$ 
			'w(1) GE 8','w(1) GE 9', $
			'w(1) GE 0','w(1) GE 1','w(1) GE 2','w(1) GE 3',$
			'w(1) GE 4','w(1) GE 5','w(1) GE 6','w(1) GE 7',$ 
			'w(1) GE 8','w(1) GE 9' ]
		n_color=(*info.ptrStr).N_COLOR
		n_color = StrCompress( (n_color-1) >0 <9 ,/Rem)
		data={ f_phot:[F_PHOT,'Energy/eV' ,'Wavelength/A'], $
			n_color:[n_color,'1','2','3','4','5','6','7',$
				'8','9','10'], $
			ph1:(*info.ptrStr).PH1, ph2:(*info.ptrStr).PH2, $
			ph3:(*info.ptrStr).PH3, ph4:(*info.ptrStr).PH4, $
			ph5:(*info.ptrStr).PH5, ph6:(*info.ptrStr).PH6, $
			ph7:(*info.ptrStr).PH7, ph8:(*info.ptrStr).PH8, $
			ph9:(*info.ptrStr).PH9, ph10:(*info.ptrStr).PH10, $
			rl1:(*info.ptrStr).RL1, rl2:(*info.ptrStr).RL2, $
			rl3:(*info.ptrStr).RL3, rl4:(*info.ptrStr).RL4, $
			rl5:(*info.ptrStr).RL5, rl6:(*info.ptrStr).RL6, $
			rl7:(*info.ptrStr).RL7, rl8:(*info.ptrStr).RL8, $
			rl9:(*info.ptrStr).RL9, rl10:(*info.ptrStr).RL10 }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret,  $
		  NCol=4, Flags=flags
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).F_PHOT=Fix( (data.f_phot)[0] )
		(*info.ptrStr).N_COLOR=Fix( (data.n_color)[0] ) + 1
		(*info.ptrStr).PH1 = data.ph1 & (*info.ptrStr).PH2 = data.ph2
		(*info.ptrStr).PH3 = data.ph3 & (*info.ptrStr).PH4 = data.ph4
		(*info.ptrStr).PH5 = data.ph5 & (*info.ptrStr).PH6 = data.ph6
		(*info.ptrStr).PH7 = data.ph7 & (*info.ptrStr).PH8 = data.ph8
		(*info.ptrStr).PH9 = data.ph9 & (*info.ptrStr).PH10 = data.ph10
		(*info.ptrStr).RL1 = data.rl1 & (*info.ptrStr).RL2 = data.rl2
		(*info.ptrStr).RL3 = data.rl3 & (*info.ptrStr).RL4 = data.rl4
		(*info.ptrStr).RL5 = data.rl5 & (*info.ptrStr).RL6 = data.rl6
		(*info.ptrStr).RL7 = data.rl7 & (*info.ptrStr).RL8 = data.rl8
		(*info.ptrStr).RL9 = data.rl9 & (*info.ptrStr).RL10 = data.rl10
		END
	ENDCASE
	END
    'F_POLAR': BEGIN
	; f_polar = 1 - flag defining whether or not to generate
	;	the A vectors:  yes (1), no (0).
	; f_coher = 0 - if generating the A vectors, defines
	;	whether the light is incoherent (0), or coherent (1).
	Widget_Control,info.wids.F_POLAR,Get_Value=index
	(*info.ptrStr).F_POLAR=Fix(index)
	f_coher = StrCompress((*info.ptrStr).F_COHER,/Rem)

	CASE index OF
	  '0': GoTo,out
	  '1': BEGIN
		titles = ['Coherent beam:', $
			'Phase diff [deg,0=linear,+90=ell/right]', $
			'Polarization degree [cos_s/(cos_s+sin_S)']
		wTitle='Polarization'
		data={ f_coher:[f_coher,'No','Yes'], $
		       pol_angle:(*info.ptrStr).POL_ANGLE, $
		       pol_deg:(*info.ptrStr).POL_DEG }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret
		IF action EQ 'DONT' THEN GoTO,out
		(*info.ptrStr).F_COHER = Fix( (data.f_coher)[0] )
		(*info.ptrStr).POL_ANGLE = data.pol_angle
		(*info.ptrStr).POL_DEG = data.pol_deg
		END
	ENDCASE
	END
    'F_OPD': BEGIN
	;  f_opd  =   0 - flag defining whether or not to save the
	;	optical paths (OPD):  yes (1), no (0).
	Widget_Control,info.wids.F_OPD,Get_Value=index
	(*info.ptrStr).F_OPD=Fix(index)
	END
    ;
    ; BM SOURCE ENTRIES
    ;
    'XSH_SRCOMP': BEGIN
	str = XSh_Defaults_Utils('XSH_SRCOMP')
help,/str,str
	str.parameters.Machine_name='Data from XSh_Source'
	str.parameters.RB_choice[0]='0' ; 'Magnetic Radius'
	str.parameters.Machine_R_m= Abs( (*info.ptrStr).R_MAGNET )
	str.parameters.BField_T=1.0
	str.parameters.Beam_energy_Gev=(*info.ptrStr).BENER
	str.parameters.Current_A=1.0
	str.parameters.Hor_div_mrad= $
	  (Abs( (*info.ptrStr).HDIV1 + (*info.ptrStr).HDIV2))*1e3
	str.parameters.Phot_energy_min= (*info.ptrStr).PH1
	str.parameters.Phot_energy_max= (*info.ptrStr).PH2
	str.parameters.Phot_energy_step=  $
	  ( (*info.ptrStr).PH2 - (*info.ptrStr).PH1)/200.0
	str.parameters.Vert_div_mrad= $
	  1000.0*( (*info.ptrStr).VDIV1 + (*info.ptrStr).VDIV2 )
	XSh_SRComp,str
	END
    'PH1': BEGIN
        Widget_Control,info.wids.PH1,Get_Value=tmp
        (*info.ptrStr).PH1 = tmp
	END
    'PH2': BEGIN
        Widget_Control,info.wids.PH2,Get_Value=tmp
        (*info.ptrStr).PH2 = tmp
	END
    'SIGMAX': BEGIN
        Widget_Control,info.wids.SIGMAX,Get_Value=tmp
        (*info.ptrStr).SIGMAX = tmp
	END
    'SIGMAZ': BEGIN
        Widget_Control,info.wids.SIGMAZ,Get_Value=tmp
        (*info.ptrStr).SIGMAZ = tmp
	END
    'EPSI_X': BEGIN
        Widget_Control,info.wids.EPSI_X,Get_Value=tmp
        (*info.ptrStr).EPSI_X = tmp
	END
    'EPSI_Z': BEGIN
        Widget_Control,info.wids.EPSI_Z,Get_Value=tmp
        (*info.ptrStr).EPSI_Z = tmp
	END
    'BENER': BEGIN
        Widget_Control,info.wids.BENER,Get_Value=tmp
        (*info.ptrStr).BENER = tmp
	END
    'Pre1R_MAGNET': BEGIN
	radius = 0
        Widget_Control,info.wids.Pre1R_MAGNET,Get_Value=radius
        (*info.ptrStr).R_MAGNET = Abs(radius)
        (*info.ptrStr).R_ALADDIN = radius*100.0 ; in CM
	bFieldMax = 3.334728*( (*info.ptrStr).BENER )/radius
        Widget_Control,info.wids.Pre2R_MAGNET,Set_Value=bFieldMax
	END
    'Pre2R_MAGNET': BEGIN
	bFieldMax = 0
        Widget_Control,info.wids.Pre2R_MAGNET,Get_Value=bFieldMax
	radius = 3.334728*( (*info.ptrStr).BENER )/bFieldMax
        (*info.ptrStr).R_MAGNET = Abs(radius)
        (*info.ptrStr).R_ALADDIN = radius*100.0 ; in CM
        Widget_Control,info.wids.Pre1R_MAGNET,Set_Value=radius
	END
    'HDIV1': BEGIN
        Widget_Control,info.wids.HDIV1,Get_Value=tmp
        (*info.ptrStr).HDIV1 = tmp
	END
    'HDIV2': BEGIN
        Widget_Control,info.wids.HDIV2,Get_Value=tmp
        (*info.ptrStr).HDIV2 = tmp
	END
    'VDIV1': BEGIN
        Widget_Control,info.wids.VDIV1,Get_Value=tmp
        (*info.ptrStr).VDIV1 = tmp
	END
    'VDIV2': BEGIN
        Widget_Control,info.wids.VDIV2,Get_Value=tmp
        (*info.ptrStr).VDIV2 = tmp
	END
    'Pre2FDISTR': BEGIN
        Widget_Control,info.wids.Pre2FDISTR,Get_Value=tmp
	CASE tmp OF
	  '0': (*info.ptrStr).FDISTR=4
	  else: (*info.ptrStr).FDISTR=6
	ENDCASE
	END
    'EPSI_DX': BEGIN
        Widget_Control,info.wids.EPSI_DX,Get_Value=tmp
        (*info.ptrStr).EPSI_DX = tmp
	END
    'EPSI_DZ': BEGIN
        Widget_Control,info.wids.EPSI_DZ,Get_Value=tmp
        (*info.ptrStr).EPSI_DZ = tmp
	END
    'F_SR_TYPE': BEGIN
        Widget_Control,info.wids.F_SR_TYPE,Get_Value=tmp
        (*info.ptrStr).F_SR_TYPE = Fix(tmp)
	END
    'PreF_POL': BEGIN
        Widget_Control,info.wids.PreF_POL,Get_Value=tmp
        (*info.ptrStr).F_POL = 1+Fix(tmp)
	END
	
;............................................................

    else: Print,'CASE not found'+eventUValue
EndCase


out:
IF Widget_Info(infoId,/Valid_ID) THEN $
  Widget_Control, infoId, Set_UValue=info, /No_Copy

END ; XSH_Source_Event


;		
;========================================================================
;
Pro XSh_Source,input, Group=group, BM=bm, NO_BLOCK=no_block, $
  Only_Run=only_Run, Modal=modal


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('XSh_Source: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF

IF XRegistered('XSH_Source') THEN BEGIN
  itmp = Dialog_Message(Dialog_Parent=group, $
    ['SHADOW VUI has detected an open window for Source',$
     'Please use it or close it before starting a new one'])
  RETURN
ENDIF

IF N_Elements(no_block) EQ 0 THEN no_block = 1
IF N_Elements(modal) EQ 0 THEN modal = 0
IF N_Elements(bm) EQ 0 THEN bm = 0

input_type = Type(input)
CASE input_type OF
  0: begin
	IF bm EQ 0 THEN $
	  str=XSh_Defaults('XSH_SOURCE_G')  ELSE $
	  str=XSh_Defaults('XSH_SOURCE_BM')  
	ptrStr = Ptr_New(str)
	end
  8: begin
	str=input		    ; structure
	ptrStr = Ptr_New(str)
	modal = 1
	no_block = 1
	end
 10: begin
	str=*input		    ; pointer
	ptrStr = input
	end
  else: message,'Invalid input'
endcase

IF Keyword_Set(only_Run) THEN BEGIN
  data=str
  CD,Current=path
  file=path+'/start.00'
  Write_GFile,data,file
  Widget_Control,/HourGlass
  XSh_Run,"gen_source start.00"
  RETURN
ENDIF

;
;*************************************************************
; define widgets 
;*************************************************************
;
CASE bm OF
  0: title='Geometrical source parameters (XSH_Source)'
  else: title = 'Bending Magnet parameters (XSH_Source)'
ENDCASE
wTLB=Widget_Base(Column=1,Title= title)

wtmp = Widget_Base(wTLB) ; to store state

;
; buttons
;
wtmpb=Widget_Base(wTLB,Row=1)
  ;wtmp=Widget_Button(wtmpb,Value='Accept',UValue='Accept')
  wtmp=Widget_Button(wtmpb,Value='Close',UValue='Accept')
  ;wtmp=Widget_Button(wtmpb,Value='Cancel',UValue='Cancel')
  wtmp=Widget_Button(wtmpb,Value='Discard changes',UValue='Cancel')
  wtmp=Widget_Button(wtmpb,Value='Run Shadow/source',UValue='Run Shadow/source')

IF bm EQ 1 THEN  $
  wtmp=Widget_Button(wtmpb,Value='Simulate spectrum with SRComp',UValue= $
   'XSH_SRCOMP')
;
;*************************************************************
; define widgets for GEOMETRICAL SOURCE
;*************************************************************
;
IF bm EQ 0 THEN BEGIN ; Start GEOMETRICAL SOURCE

;
; parameters
;
wpar=Widget_Base(wTLB,Column=1,Frame=0)

;
; Monte Carlo
;

;
; Monte Carlo and Sampling
;
wbMC = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbMC,Value='** Monte Carlo & Sampling **',/Align_Left)
  wtmpb =Widget_Base(wbMC,Row=1)
    wNPOINT = CW_Field(wtmpb,XSize=12,Value=0,Title='Number of random rays:', $
      UValue='NPOINT',/Long,/All_Events)
    wISTAR1 = CW_Field(wtmpb,XSize=12,Value=0,Title='Seed: [odd, 100-1e5]', $
      UValue='ISTAR1',/Long,/All_Events)
  wtmpb =Widget_Base(wbMC,Row=1)
    list=['0','Random/Random','Grid/Grid...', $
	'Grid/Random...', 'Random/Grid...',  $
	'Ellipse in phase/Random on the ellipse...',$
	'Ellipse in phase/Grid on the ellipse...']
    wFGRID=CW_DropList(wtmpb,Value=list,Title='Sampling (space/divergences):',$
      UValue='FGRID')

;
; Geometry
;
wbGeometry = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbGeometry,Value='** Geometry **',/Align_Left)
  wtmpb=Widget_Base(wbGeometry,Row=1)
    list=['0','Point','Rectangle...','Ellipse...','Gaussian...']
    wFSOUR=CW_DropList(wtmpb,Value=list,Title='Spatial Type:',$
      UValue='FSOUR')

    list=['0','Flat...','Uniform...','Gaussian...','<reserved>', $
	'Conical...','<reserved>']
    wPreFDISTR=CW_DropList(wtmpb,Value=list,Title='Angle Distr.:', $
      UValue='PreFDISTR')

    list=['0','Off','Uniform...','Gaussian...']
    wPreFSOURCE_DEPTH=CW_DropList(wtmpb,Value=list, Title='Depth', $
      UValue='PreFSOURCE_DEPTH')

;
; Energy
;
wbEnergy = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbEnergy,Value= $
	'** Energy/Wavelength and polarization  **',/Align_Left)
  wtmpb=Widget_Base(wbEnergy,Row=1)
    list=['0','Single line...','Several lines...','Uniform...', $
	'Relative Intensities...']
    wPreF_COLOR=CW_DropList(wtmpb,Value=list,  $
      Title='Photon Energy Distribution:',UValue='PreF_COLOR')

  wtmpb=Widget_Base(wbEnergy,Row=1)
    list=['0','No','Yes...']
      wF_POLAR=CW_DropList(wtmpb,Value=list,Title='Polarization ?', $
	UValue='F_POLAR')
    ; Optical path
    list=['0','No','Yes']
    wF_OPD=CW_DropList(wtmpb,Value=list,UValue='F_OPD', $
	Title='Store optical paths ?')

ENDIF
;
;*************************************************************
; END define widgets for GEOMETRICAL SOURCE
;*************************************************************
;



;
;*************************************************************
; define widgets for BENDING MAGNET
;*************************************************************
;
IF bm EQ 1 THEN BEGIN
  
;
; parameters
;
wpar=Widget_Base(wTLB,Column=1,Frame=0)

;
; Monte Carlo
;
wbMC = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbMC,Value='** Monte Carlo and Energy Spectrum **', $
    /Align_Left)
  wtmpb =Widget_Base(wbMC,Row=1)
    wNPOINT = CW_Field(wtmpb,XSize=12,Value=0,Title='Number of random rays:', $
      UValue='NPOINT',/Long,/All_Events)
    wISTAR1 = CW_Field(wtmpb,XSize=12,Value=0,Title='Seed: [odd, 100-1e5]', $
      UValue='ISTAR1',/Long,/All_Events)
  wtmpb =Widget_Base(wbMC,Row=1)
    wPh1=CW_Field(wtmpb,XSize=12,Value=0, UValue='PH1', $
      Title='Emin [eV]',/Float,/All_Events)
    wph2=CW_Field(wtmpb,XSize=12,Value=0, UValue='PH2', $
      Title='Emax [eV] (Emax=Emin for single line)',/Float,/All_Events)
  wtmpb =Widget_Base(wbMC,Row=1)
    ; Optical path
    list=['0','No','Yes']
    wF_OPD=CW_DropList(wtmpb,Value=list,UValue='F_OPD', $
        Title='Store optical paths ?')
    ; Distribution type
    list=['0','Photon','Power']
    wF_SR_TYPE=CW_DropList(wtmpb,Value=list,UValue='F_SR_TYPE', $
        Title='Sample distribution:')
  wtmpb =Widget_Base(wbMC,Row=1)
    ; Polarization
    list=['0','Only parallel','Only perpendicular','Total']
    wPreF_POL=CW_DropList(wtmpb,Value=list,UValue='PreF_POL', $
        Title='Generate polarization:')
;

;
; Machine
;
wbMachine = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbMachine,Value='** Machine Parameters **',/Align_Left)
  wtmpb =Widget_Base(wbMachine,Row=1)
    wSigmaX = CW_Field(wtmpb,XSize=12,Value=0,Title='Sigma X [cm]', $
      UValue='SIGMAX',/Float,/All_Events)
    wSigmaZ = CW_Field(wtmpb,XSize=12,Value=0,Title='Sigma Z [cm]', $
      UValue='SIGMAZ',/Float,/All_Events)
  wtmpb =Widget_Base(wbMachine,Row=1)
    wEpsi_X = CW_Field(wtmpb,XSize=12,Value=0,Title='Emittance X [rad.cm]', $
      UValue='EPSI_X',/Float,/All_Events)
    wEpsi_Z = CW_Field(wtmpb,XSize=12,Value=0,Title='Emittance Z [rad.cm]', $
      UValue='EPSI_Z',/Float,/All_Events)
  wtmpb =Widget_Base(wbMachine,Row=1)
    wBener = CW_Field(wtmpb,XSize=12,Value=0,Title='Energy [GeV]', $
      UValue='BENER',/Float,/All_Events)
    wEPSI_DX = CW_Field(wtmpb,XSize=12,Value=0,Title= $
      'Distance from waist [cm], X:', UValue='EPSI_DX',/Float,/All_Events)
    wEPSI_DZ = CW_Field(wtmpb,XSize=12,Value=0,Title= $
      ' Z:', UValue='EPSI_DZ',/Float,/All_Events)


;
; Bending Magnet specific parameters
;
wbMachine = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbMachine,Value='** Bending Magnet Parameters **', $
    /Align_Left)
  wtmpb =Widget_Base(wbMachine,Row=1)
    wPre1R_MAGNET=CW_Field(wtmpb,XSize=12,Value=0,Title= $
        'Magnetic radius [m]:',/Floating,/All_Events,UValue='Pre1R_MAGNET')
    wPre2R_MAGNET=CW_Field(wtmpb,XSize=12,Value=0,Title= $
        'or Magnetic Field [T]:',/Floating,/All_Events,UValue='Pre2R_MAGNET')
  wtmpb =Widget_Base(wbMachine,Row=1)
    wHDiv1=CW_Field(wtmpb,XSize=12,Value=0, UValue='HDIV1', Title= $
      'Horizontal half-divergence [rads] From[+]', /Floating,/All_Events)
    wHDiv2=CW_Field(wtmpb,XSize=12,Value=0, UValue='HDIV2', Title= $
      'To[-]', /Floating,/All_Events)
  wtmpb =Widget_Base(wbMachine,Row=1)
    wVDiv1=CW_Field(wtmpb,XSize=12,Value=0, UValue='VDIV1', Title= $
      'Max vertical half-divergence [rads] From[+]', /Floating,/All_Events)
    wVDiv2=CW_Field(wtmpb,XSize=12,Value=0, UValue='VDIV2', Title= $
      'To[-]', /Floating,/All_Events)
  wtmpb =Widget_Base(wbMachine,Row=1)
    tmp = Widget_Label(wtmpb,Value='(Note: Set vertical divergence to'+$
          ' something big (e.g., 1) for generating full emission)')

  wtmpb =Widget_Base(wbMachine,Row=1)
    bm_calc_options=['0','Precomputed','Exact']
    wPre2FDISTR=CW_Droplist(wtmpb,Value=bm_calc_options, $
       UValue='Pre2FDISTR',Title='Calculation mode:')
    
ENDIF
;
;*************************************************************
; END define widgets for BENDING MAGNET
;*************************************************************
;

;
; Advanced
;
wbAdvanced = Widget_Base(wpar,Column=1,/Frame)
  wtmp = Widget_Label(wbAdvanced,Value='** Advanced features **', $
    /Align_Left)
  wtmpb=Widget_Base(wbAdvanced,Row=1)
    wtmp=Widget_Button(wtmpb,Value='View GFile...',UValue='adv.viewg')
    wtmp=Widget_Button(wtmpb,Value='Load GFile...',UValue='adv.loadg')
    wtmp=Widget_Button(wtmpb,Value='Write GFile...',UValue='adv.writeg')

;
;*************************************************************
; END define widgets 
;*************************************************************
;

;
; store parameters
;
IF bm EQ 0 THEN BEGIN
  wids={npoint:wNPOINT,istar1:wISTAR1, fgrid:wFGRID, $
    fsour:wFSOUR, PreFdistr:wPreFDISTR, PreFsource_depth:wPreFSOURCE_DEPTH, $
    PreF_color:wPreF_COLOR, f_polar:wF_POLAR, f_opd:wF_OPD }
ENDIF ELSE BEGIN
  wids={npoint:wNPOINT,istar1:wISTAR1,  ph1:wPH1, ph2:wPH2,  $
        sigmaX:wSIGMAX, sigmaZ:wSIGMAZ, epsi_X:wEPSI_X, epsi_Z:wEPSI_Z,  $
        bener:wBENER, $
        pre1R_MAGNET:wPre1R_MAGNET, pre2R_MAGNET:wPre2R_MAGNET,  $
        hDiv1:wHDIV1, hDiv2:wHDIV2,  vDiv1:wVDIV1, vDiv2:wVDIV2,  $
        pre2FDISTR:wPre2FDISTR, PreF_Pol:wPreF_POL, F_SR_Type:wF_SR_Type, $
	F_OPD:wF_OPD, epsi_DX:wEPSI_DX, epsi_DZ:wEPSI_DZ }
ENDELSE

info={wids:wids, ptrStr:ptrStr, init_str:str, modal:modal, $
  no_block:no_block, bm:bm}

Widget_Control,Widget_Info(wTLB,/Child),Set_UValue=info,/no_copy 
Widget_Control,wTLB,/Realize

;
; update widgets
;
XSh_Source_UpdatePanels,wids,ptrStr,Group=group,/Map

XManager,'XSH_Source',wTLB,MODAL=modal,No_Block=no_block,Group=group

;
; retrieve data if modal
;
IF Keyword_Set(modal) THEN BEGIn
;  Handle_Value,pointix.ptr,retdata
;  case input_type of
;    0: input = retdata.res
;    8: input = retdata.res
;   10: *input = retdata.res
;   else:
;  endcase
ENDIF
END
