Function nsl_calc_mix,input,descriptor,wavelength,RETURN=FDES,density=density,$
   theta=theta,rough=rough,NAME=NAME,group=group,polar=pola, verbose=verbose

;+
; NAME:
;       NSL_CALC_MIX
;
; PURPOSE:
;       Evaluate the neutron scattering lengths, cross sections
;       and related quantities (refraction index, absorption coefficients
;       and mirror relfectivity) versus neutron wavelength for compounds 
;	and mixtures.
;       It uses data from DABAX database.

;
; CATEGORY:
;       Neutron optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = nsl_calc_mix(input,descriptor[,wavelength])
; INPUTS:
;	input: a dabax input file as accepted by dabax_access().
;	descriptor: a string  with either the descriptor or the name of the
;	  mixture. When using the descriptor, you must supply the density (g/cc)
;	  The descriptor sintax is the following:
;	  string with the chemical formula of the compounds with their weight
;	  fraction between parenthesis.
;               Descriptor examples:
;               SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)
;               H2O(0.99)Cl2(0.01)
;               H(0.11189834)O(0.88810166)
;	The name is one of the names defines in the DABAX file Compounds.dat
;	When using the name input option, the keyword NAME must be set.
;
; OPTIONAL INPUTS:
;	wavelength: The value (or array of values) with wavelength in Angstrom.
;		If undefined, use a 100 points array in the [0.02,100] interval.
;		When wavelength is defined, this value is used as a grid
;		for the results (it interpolates in the case of
;		tabulated data)
;
;	RETURN: flag for selecting the output:
;               0 (default) returns a 2-col array with Real(b) and Im(b)
;               1  returns Real(b)
;               2  returns Im(b)
;
;               3  returns coherent scattering cross section in barns
;               4  returns incoherent cross section in barns
;               5  returns coherent scattering cross section in barns
;               6  returns nuclear absorption cross section
;
;
;               7  returns delta; n, the refraction index n=1-delta-i*beta
;               8  returns beta
;               9  returns mu (Nuclear absorption coefficient)
;               10  returns mu/rho (Mass Nuclear absorption coefficient)
;
;               11  returns reflectivity
;               12  returns nuclear absorption cross section at ref wavelength
;               13  returns ????? for crystal????
;
;	THETA= the incident [grazing] angle in mrad (for reflectivity
;		calculations). It is set to 3.0 (default) when RET=11 and
;		THETA is undefined. It can be an array. In such a case, the
;		returned value os a matrix: out(theta,wavelength)
;       ROUGH= the rms roughness in Angstroms (for reflectivity
;               calculations. A Debye-Waller or Nevot-Croce model is used.)
;               Convention:
;                  If rough<0, a Nevot-Croce model is used
;                  If rough>0, a Debye-Waller model is used
;	NAME: Set this keyword to a valid mixture name in the file
;		Compounds.dat. If not set, it will interpret
;		"descriptor" as a mixture formula.
;	DENSITY= the density value to be used for the calculations. If
;		not defined, and NAME is not set, then it uses 1.0 except
;		in the case when there is a single element in the
;		compund. In such a case it calls atomic_constants to get
;		the value.
;       GROUP = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;	 VERBOSE = Flag to inform on isotope name and density used.
;
; OUTPUT:
;	out: an array with the values of the selected returm paramaeter.
;
; PROCEDURE:
;	Calls nsl_calc for each element of the compound and make the
;	weighted average in the following way:
;	For compounds:
;	  1) The compound's  coherent scattering length is the sum of the 
;	     atomic coherent scattering lengths.
;	  2) The compound's  coherent scattering cross section is the sum 
;	     of the atomic ones. 
;	  3) The compound's  nuclear absorption  cross section is the sum 
;	     of the atomic ones. 
;	  4) for the refraction index n=1-delta-i*beta
;		delta is calculated using the compound's  coherent 
;		      scattering length.
;		beta is calculated using the compound's  nuclear absorption 
;		      cross section.
;	  5) mu for the compounds use the compound's nuclear absorption cross
;		section.
;	  6) reflectivity calculations use the compound's delta and beta.
;
; WARNING:
;	All absorption cross sections are "nuclear", therefore the attenuation
;	produced by the Thermal Diffuse Scattering is not included. 
;
; EXAMPLES:
;	EXAMPLE1: 
;	Prints scattering lengths (real and imag) for Si3N4 at 1 A.
;	print,DeBe_calc('DeBe_NuclearSpin.dat','Si3N4',1.0)
;       49.892129  -1.2547931e+10
;
;	EXAMPLE2:  ????????????????
;
; MODIFICATION HISTORY:
;      2003-03-12 M. Sanchez del Rio (srio@esrf.fr)
;		Taken the version written by Ch. Boisseau, based on
;		f1f2_calc_mix.pro. Cleaning and documented. 
;
;      2003-02-26 alianell@ill.fr corrected mu, sxs, sigabs
;      and reflectivity calculation  
;-
on_error,2

molwt=0.0

IF N_Elements(verbose) EQ 0 THEN verbose=1
IF N_Elements(wavelength) EQ 0 THEN wavelength=makearray1(100,0.02,100)
wavelength1 = wavelength*1.d-10   ; in m


if keyword_set(name) then begin
  cfile = 'Compounds.dat'
  h = dabax_access(cfile,group=group)
  index = dabax_select(h,descriptor,/casematch,group=group)
  tmp = spec_data(h,index,/index)
  if not(keyword_set(density)) then begin
    rho1 = spec_headers(h,index,['#URHO'],/index)
    density = float(strmid(rho1,5,strlen(rho1)-1))
  endif
  z = fix(tmp(0,*))
  f = tmp(1,*)
  s = atomic_symbols(z)
  ;
  ; get the molecular weight of the mixture
  ;
  ; this is necessary for calculating the molecular weight. However
  ; the other parameters returned are redundant with what is obtained
  ; before. It is then recommended to include the molecular weight in
  ; the Compounds.dat file to avoid using parse_mixture here.
  ; To be done in the future. srio@esrf.fr 01/02/28
  ;
  for1 = spec_headers(h,index,['#UFOR'],/index)
  formula1 = strmid(for1,5,strlen(for1)-1)
  formula1 = strcompress(formula1,/remove_all)
  n = parse_mixture(formula1,s,f,z,fw,molwt=molwt,verbose=verbose)
endif else begin
  n = parse_mixture(descriptor,s,f,z,fw,molwt=molwt,verbose=verbose)
endelse
IF verbose THEN BEGIN 
   Message,/Info,'Using molecular weight: '+StrCompress(molwt)
ENDIF


if n_elements(s) LT 1 then begin
  message,/info,'Error in mixture descriptor/name (Returning 0): '+mixture
  return,0
endif


if keyword_set(fdes) then if fdes ge 7 then $
  if not(keyword_set(name)) and n_elements(density) eq 0 then begin
    IF n_elements(s) EQ 1 THEN BEGIN
      density = atomic_constants(s(0),return='Density')
    ENDIF ELSE BEGIN
      density = 1.0
      IF SDep(/W) EQ 1 THEN $
      itmp = Widget_Message(/Info,'Density undefined. Set to one.',$
	Dialog_Parent=group)
    ENDELSE
    Message,/info,'Density value undefined, set to '+strcompress(density)
  endif


if n_elements(density) GT 0 then $
IF verbose THEN message,/info,'Density ('+descriptor+'): '+strcompress(density)

;
; calculate global b
;
sigabs=0.d0
br=0.d0
sxs=0.d0
sigma_coh=0.d0
sigma_incoh=0.d0
aw=dblarr(n_elements(s))

for i=0,n_elements(s)-1 do begin
  ; moved back   aw(i)=atomic_constants(s(i),return='AtomicMass')
  descriptor1=s[i]
  b = nsl_calc(input,descriptor1,wavelength,RET=0,polar=pola, $
	new_descriptor=new_descriptor,verbose=verbose)
  sig_scat = nsl_calc(input,new_descriptor,wavelength,RET=3,polar=pola, $
	verbose=0)
  sig_incoh = nsl_calc(input,new_descriptor,wavelength,RET=4,polar=pola, $
	verbose=0)
  sig_coh = nsl_calc(input,new_descriptor,wavelength,RET=5,polar=pola, $
	verbose=0)
  siga = nsl_calc(input,new_descriptor,wavelength,RET=12,polar=pola, $
	verbose=0)
  aw[i]=atomic_weights(new_descriptor)

;  if i eq 0 then begin
;     br=br+b[0,*]*f(i) 
;     ; missing incoherent part?????? (srio)  sxs=sxs+4.d0*!pi*(b[0,*]^2)*f(i)*1.d-02
;     sxs=sxs+sig_scat*f(i)
;     sigabs=sigabs+siga*f(i)
;     sigma_coh=sigma_coh+sig_coh*f(i)
;     sigma_incoh=sigma_incoh+sig_incoh*f(i)
;  endif else begin
     br=br+b[0,*]*f(i) 
     ; idem (srio)          sxs=sxs+4.d0*!pi*(b[0,*]^2)*f(i)*1.d-02
     sxs=sxs+sig_scat*f(i)
     sigabs=sigabs+siga*f(i)
     sigma_coh=sigma_coh+sig_coh*f(i)
     sigma_incoh=sigma_incoh+sig_incoh*f(i)
; endelse
endfor

wlref=1.798d-10       ;m
;srio use the one at ref wavelength bi=-sigabs/2.d0*1.d-13/wavelength   ;fermi
bi=-sigabs/2.d0*1.d-13/wlref   ;fermi

; returns result( if user requested fdes)
sigabs_at_lambda = wavelength1*(sigabs/wlref)[0]

IF NOT(Keyword_Set(fdes)) THEN fdes=0
CASE Fix(fdes) OF
  0:begin
    fout = make_set(br,bi)
    return,fout
    end
  1:return,br
  2:return,bi
  3:return,sxs
  4:return,sigma_incoh
  5:return,sigma_coh
  6:return,sigabs_at_lambda
  12:return,sigabs
  else:
ENDCASE

IF verbose THEN Message,/Info,'Molecular weight is: '+StrCompress(molwt)
avogadro = physical_constants('avogadro')
tocm = physical_constants('hc')*1e-8

molecules_per_cc = density*avogadro/molwt

k = molecules_per_cc*1.d6*wavelength1*wavelength1/2.0/!pi

;
; calculation of refraction index
;

nbee=n_elements(wavelength1)
refi=dblarr(2,nbee)
debe=dblarr(2,nbee)
mu=dblarr(nbee)
for i=0,nbee-1 do begin
    debe[0,i]=k[i]*br*1.d-15 ;delta
    ; this is correct, but longer than the used one
    ; debe[1,i] = 0.5d0*k[i]*sigabs_at_lambda[i]*1.d-28/wavelength1 ;beta
    debe[1,i] = 0.5d0*k[i]*sigabs*1.d-28/wlref ;beta
    mu[i] = 4.0*!pi*debe[1,i]/(wavelength1[i]*100.)   ;cm-1
    endfor
refi[0,*]=1.d0-debe[0,*]
refi[1,*]=debe[1,*]

case fdes of
  7:return,debe[0,*] ;delta
  8:return,debe[1,*] ;beta
  9:return,mu
  10:return,mu/density
  ; ????????????     11:return,mu/molecules_per_cc*1.0d+24    ;barn
;special for NCrystal calculation
; return molecules per unit cell [m-3] and
; nuclear absorption cross-section [barn]
  13:begin
     nsig=mu/molecules_per_cc*1.0d+24     ;barn
     tem=make_set(molecules_per_cc*1.d6,nsig)
     return,tem
     end
  else:
endcase

;
; calculation of reflectivity (piece of code adapted from shadow/abrefc)
;

if n_elements(theta) eq 0 then theta1 = 3.0e-3 else theta1 = theta*1e-3
if n_elements(theta) eq 0 then sin_theta2 = 3.0e-3 else sin_theta2 = sin(theta)*1e-3  
if n_elements(rough) eq 0 then rough1 = 0.0 else rough1=rough
rough1 = rough1*1.0d-8 ; in cm
; epsi = 1 - alpha - i gamma
alpha = 2.0D0*debe[0,*]
gamma = -2.0D0*debe[1,*]
	for i=0,n_elements(theta1)-1 do begin

     	RHO	=   (SIN(theta1(i)))^2 - ALPHA
     	RHO	=   RHO + SQRT ((SIN(theta1(i))^2 - ALPHA)^2 + GAMMA^2)
     	RHO	=   SQRT(RHO/2)
;** Computes now the reflectivity for s-pol
     	RS1	=   4*(RHO^2)*(SIN(theta1(i))-RHO)^2 + GAMMA^2
     	RS2	=   4*(RHO^2)*(SIN(theta1(i))+RHO)^2 + GAMMA^2
     	RS	=   RS1/RS2

        DebyeWaller = exp( -(4.0*!pi*sin(theta1(i))*rough1/wavelength1)^2)


        ; snell lay
        cos_theta2 = cos(theta1[i])/(1.0D0-debe[0,*])
        sin_theta2_2 = (1.0D0-cos_theta2^2) > 0.0D0
        sin_theta2 = Sqrt(sin_theta2_2)
        NevotCroce = exp( -16.0/!pi^2*sin(theta1(i))*sin_theta2* (rough1/wavelength1)^2)
        ;;print,'DW, NC: cos, sin : ',DebyeWaller,NevotCroce,cos_theta2,sin_theta2
        ;;print,' '


        IF rough1 GE 0 THEN BEGIN   ; apply DebyeWaller
          RS = RS * DebyeWaller
        ENDIF ELSE BEGIN
          RS = RS * NevotCroce
        ENDELSE

        out = RS
	if n_elements(theta1) eq 1 then out1 = out else begin
          if i eq 0 then begin
            out1 = fltarr(n_elements(theta1),n_elements(out))
            out1(0,*) = reform(out)
          endif else out1(i,*) = reform(out)
        endelse
        endfor

print,'>>>>>>>>>>>>>>>>>',fdes,wavelength1
help,out1
return,out1
END
