;+
; NAME:
;
;       RD_SUPER
;       
; PURPOSE:
;
;       Read in diffraction data written by the program SUPER.
;       
; CATEGORY:
; 
;	IMD
;	
; CALLING SEQUENCE:
; 
;	RD_SUPER[,CTS,SECONDS]
;
; OPTIONAL OUTPUTS:
;	
;	CTS - Array of number of counts measured.
;	
;	SECONDS - Array of integration times.
;	
; KEYWORD PARAMETERS:
; 
;	FILE - A string or string array specifying the file(s) to read.  If
;              more than one file is specified, then the data arrays
;              are concatenated.
;	
;	H - h-momentum transfer vector.
;	
;	K - k-momentum transfer vector.
;	
;	L - l-momentum transfer vector.
;	
;	TTH - Array of two-theta values.
;	
;	TH - Array of theta values.
;	
;	PHI - Array of phi values.
;	
;	CHI - Array of chi values.
;	
;       CH4 - Array of counts recorded in CH4 (must set RD_CH4 keyword)
;   
;	RD_CH4 = set to read in additional data stored as CH4
;	
;       SILENT = set to inhibit printing file info.
;       
;       SET_IMD - Set to instruct RD_SUPER to convert the data
;                 contained in the file to IMD Measured Data
;                 variables, Reflectance vs. Theta.  The IMD angle
;                 pointer (IMD.X_M) will be set to 0 (angles).
;
;                 NOTE: Do not use this procedure directly!
;                 
;                 Use IMD_M_RD,RD_M_CMD='RD_SUPER,/SET_IMD'
;
;       INC_INT - Incident intensity: If SET_IMD is set, then the data
;                 will be normalized to the incident intensity.  If
;                 INC_INT is equal to zero, or undefined, then it will
;                 be assumed that the incidence intensity is equal to
;                 twice the value of the first element of CTS (as
;                 would be the case for reflectance vs angle data
;                 starting from TH=0., for a properly aligned sample.)
;
;       ATTENUTATION - An additional attenuation factor used to
;                      normalized the data: If SET_IMD is set, and
;                      ATTENUATION is non-zero, then the computed
;                      reflectance will be given by:
;                      Y_m=DETECTOR/INC_INT/ATTENUATION
;
;                      ATTENUATION can be an array, equal to
;                      the number of elements of FILE, in order
;                      to normalize several data sets at once.
;   
; COMMON BLOCKS:
;
;       COMMON IMD
;	COMMON IMD_M
;	
; SIDE EFFECTS:
; 
;       The IMD variables X_M, Y_M, SIGY_M, and IMD.X_M are
;       defined. (Set IMD.Y_M manually, according to the type of data:
;       0 for reflectance, 1 for transmittance, 2 for absorptance.)
;	
; RESTRICTIONS:
;
;       You'll have to fix this program up to work with your SUPER
;       files, if your files are configured differently than mine.
; 
;	Use in the IMD (V3.0B1 or higher) environment, but:
;
;       DO NOT USE THIS PROCEDURE DIRECTLY!
;                 
;           Use IMD_M_RD,RD_M_CMD='RD_SUPER,/SET_IMD'
;
; MODIFICATION HISTORY:
;
;       David L. Windt, Bell Laboratories, January 1990.
;       Added CH4 and RD_CH4 keywords, June 1995
;       Added SILENT keywords, August 1995
;       Upgraded for use with IMD V3, May 1997
;
;       Updated IMD common blocks to IMD version 4.0, May 1998
;
;       windt@bell-labs.com
;-
pro rd_super,file=file,h=h,k=k,l=l,cts,seconds,ch4=ch4,rd_ch4=rd_ch4, $
      tth=tth,th=th,phi=phi,chi=chi,silent=silent, $
      set_imd=set_imd, $
      inc_int=inc_int, $
      attenuation=attenuation

on_error,2

common imd, $
  imd,imdhome,imd_nkpath,imd_nkpath_ptr, $
  imdsplot,imdsplot_fc,imdsplot_bangpy, $
  imdfplot,imdfplot_fc, $
  imdcplot,imdcplot_fc, $
  imdprint

common imd_m,x_m,y_m,sigy_m

n_files=n_elements(file)

if n_files gt 1 then begin
    if n_elements(attenuation) ne n_files then  $
      attenuation=fltarr(n_files)+1.

    ;; define command to read file:
    command='rd_super,file=file(i),'+ $
      'cts,seconds,ch4=ch4,rd_ch4=keyword_set(rd_ch4),' + $
      'h=h,k=k,l=l,silent=keyword_set(silent),' + $
      'set_imd=keyword_set(set_imd),inc_int=inc_int,' + $
      'attenuation=attenuation(i),th=th,tth=tth,chi=chi,phi=phi'

    ;; initialize variables
    ctsI=0
    secondsI=0
    ch4I=0
    hI=0
    kI=0
    lI=0
    thI=0
    tthI=0
    chiI=0
    phiI=0
    x_mI=0
    y_mI=0
    sigy_mI=0

    ;; read each file and concatenate:
    for i=0,n_files-1 do begin
        ee=execute(command)
        ctsI=[ctsI,cts]
        secondsI=[secondsI,seconds]
        if n_elements(ch4) ne 0 then ch4I=[ch4I,ch4]
        hI=[hI,h]
        kI=[kI,k]
        lI=[lI,l]
        thI=[thI,th]
        tthI=[tthI,tth]
        chiI=[chiI,chi]
        phiI=[phiI,phi]
        x_mI=[x_mI,x_m]
        y_mI=[y_mI,y_m]
        sigy_mI=[sigy_mI,sigy_m]
    endfor

    ;; redefine keywords to contain arrays:
    cts=ctsI(1:*)
    seconds=secondsI(1:*)
    if n_elements(ch4I) gt 1 then ch4=ch4I(1:*)
    h=hI(1:*)
    k=kI(1:*)
    l=lI(1:*)
    th1=thI(1:*)
    tth1=tthI(1:*)
    chi1=chiI(1:*)
    phi1=phiI(1:*)
    x_m=x_mI(1:*)
    y_m=y_mI(1:*)
    sigy_m=sigy_mI(1:*)
endif else begin
    ;;
    ;; code to actually read the file:
    ;;
    if n_files eq 0 then begin
        if (!d.flags and 65536) ne 0 then begin
            file=pickfile(title='Read which SPEC data file?')
            if file eq '' then return
        endif else begin
            file=' ' 
            read,'Enter the name of the SPEC data file to read: ',file 
        endelse
    endif 

    if keyword_set(silent) eq 0 then begin
        print,' '
        print,'File name: ',file
        print,' '
    endif

    openr,lun,file,/get_lun     ; open file.
    dum=' ' & for i=0,13 do readf,lun,dum ; read in header

    nw=0                        ; count number of data points.
    while not eof(lun) do begin
        readf,lun,dum
        flag=strmid(dum,0,3) eq 'Sum' ; search for 'Sum'
        nw=nw+1
        if flag then goto,done
    endwhile

    done: point_lun,lun,0       ; go to start of file.
    nw=nw-1
    if keyword_set(silent) eq 0 then print,nw,' data points. '

    for i=0,13 do begin         ; read in header.
        readf,lun,dum	
    endfor

    ;; read in data...
    if keyword_set(rd_ch4) eq 0 then begin
        g=fltarr(9,nw)
        readf,lun,g
        close,lun
        free_lun,lun

        h=reform(g(0,*))
        k=reform(g(1,*))
        l=reform(g(2,*))
        seconds=reform(g(3,*))
        cts=reform(g(4,*))
        tth=reform(g(5,*))
        th=reform(g(6,*))
        phi=reform(g(7,*))
        chi=reform(g(8,*))
    endif else begin
        g=reform(fltarr(10,nw))
        readf,lun,g
        close,lun
        free_lun,lun

        h=reform(g(0,*))
        k=reform(g(1,*))
        l=reform(g(2,*))
        seconds=reform(g(3,*))
        cts=reform(g(4,*))
        ch4=reform(g(5,*))
        tth=reform(g(6,*))
        th=reform(g(7,*))
        phi=reform(g(8,*))
        chi=reform(g(9,*))
    endelse

    ;; fix up variables for IMD:
    if keyword_set(set_imd) then begin
        r=cts
        sigr=sqrt(r)

        ;; set inc_int and attenuation:
        if n_elements(inc_int) eq 0 then inc_int=cts(0)
        
        ;; don't divide by zero:
        inc_int=inc_int > 1

        if n_elements(attenuation) eq 0 then attenuation=1.

        ;; only use data where R ne 0:
        valid=where(r gt 0.,count)
        if count eq 0 then return
        th=th(valid)
        tth=tth(valid)
        r=r(valid)
        sigr=sigr(valid)

        ;; if imd.x_m is not set to angles, then set it to TH.
        if (imd.x_m ne 0) and (imd.x_m ne 2) then imd.x_m=0

        case imd.x_m of
            0: begin

                ;; sort:
                y_m=r(sort(th))/inc_int(0)/attenuation(0)
                sigy_m=sigr(sort(th))/inc_int(0)/attenuation(0)

                ;; angles, grazing incidence:
                x_m=th(sort(th))

                ;; deal with angle units:
                case imd.angleunits_ptr of
                    0: x_m=90.-x_m ; output is degrees normal
                    1: x_m=x_m  ; output is degrees grazing
                    2: x_m=(90.-x_m)*60. ; output is arc-min normal
                    3: x_m=x_m*60. ; output is arc-min grazing
                    4: x_m=(90.-x_m)*3600. ; output is arc-sec normal
                    5: x_m=x_m*3600. ; output is arc-sec grazing
                    6: x_m=(90.-x_m)*!dtor*1000. ; output is mrad normal
                    7: x_m=x_m*!dtor*1000. ; output is mrad grazing
                endcase
            end

            2: begin
                ;; sort:
                y_m=r(sort(tth))/inc_int(0)/attenuation(0)
                sigy_m=sigr(sort(tth))/inc_int(0)/attenuation(0)

                ;; angles, grazing incidence:
                x_m=tth(sort(tth))

                ;; deal with angle units:
                case imd.angleunits_ptr of
                    0: x_m=180.-x_m ; output is degrees normal
                    1: x_m=x_m  ; output is degrees grazing
                    2: x_m=(180.-x_m)*60. ; output is arc-min normal
                    3: x_m=x_m*60. ; output is arc-min grazing
                    4: x_m=(180.-x_m)*3600. ; output is arc-sec normal
                    5: x_m=x_m*3600. ; output is arc-sec grazing
                    6: x_m=(180.-x_m)*!dtor*1000. ; output is mrad normal
                    7: x_m=x_m*!dtor*1000. ; output is mrad grazing
                endcase
            end
        endcase
    endif
endelse
return
end
