;+
; NAME:
;
;       RD_IMC
;       
; PURPOSE:
;
;       Read in reflectance data written by the HP BASIC/UX program imc
;	
; CATEGORY:
; 
;	IMD
;	
; CALLING SEQUENCE:
; 
;	RD_IMC
;	
; KEYWORD PARAMETERS:
; 
;	FILE = the name of the reflectance data file.
;	
;	DP = n-element array of data point numbers.
;	
;	DSM = (n,6)-element array of data scan markeres.
;	
;	P = (n,7)-element array of positions:
;		   (j,0) = x
;		   (j,1) = y
;		   (j,2) = z
;		   (j,3) = tilt
;		   (j,4) = rotation
;		   (j,5) = detector
;		   (j,6) = wavelength
;		   
;	ID = (n,7)-element array of input defaults.
;	
;	INT = (n,3,2)-element array of intensities
;	
;	IINT = (n,3,2)-element array of incident intensities.
;	
;	RATIO = (n,2)-element array of ratios.
;	
;	SMOOTH = if non-zero, then the ratios are recomputed,
;	    as ratio=smooth(int,sm)/smooth(iint,sm), where sm
;	    is the non-zero value of smooth.
;	    
;	VSANGLE = if set, then the COMMON variable lambda_m will
;	    be set to the 0th element of the array of wavelengths.
;	    
;	VSWAVELENGTH = if set, then the COMMON variable theta_m will
;	    be set to the 0th element of the array of incidence angles.
;	    
;	SILENT = set to suppress printing comments.
;
; COMMON BLOCKS:
;
;       COMMON IMD
;	COMMON IMD_M
;	
; SIDE EFFECTS:
; 
;       The IMD variables X_M, Y_M, SIGY_M, and IMD.X_M are
;       defined. (Set IMD.Y_M manually, according to the type of data:
;       0 for reflectance, 1 for transmittance, 2 for absorptance.)
;	
; RESTRICTIONS:
; 
;	Use in the IMD (V3.0B1 or higher) environment, but:
;
;       Do not use this procedure directly!
;                 
;       Use IMD_M_RD,RD_M_CMD='RD_IMC'
;
; MODIFICATION HISTORY:
; 
;	David L. Windt, Bell Laboratories, March 1991.
;	Added PICKFILE usage, January 1992.
;	Upgraded to IMD V3, May 1997
;       Updated IMD common blocks to IMD version 4.0, May 1998
;	
;	windt@bell-labs.com
;-
pro rd_imc,file=file,dp=dp,dsm=dsm,id=id,p=p,int=int, $
    iint=iint,ratio=ratio,smooth=smooth,iismooth=iismooth,vsangle=vsangle, $
    vswavelength=vswavelength,iiaverage=iiaverage,silent=silent

on_error,2
common imd, $
  imd,imdhome,imd_nkpath,imd_nkpath_ptr, $
  imdsplot,imdsplot_fc,imdsplot_bangpy, $
  imdfplot,imdfplot_fc, $
  imdcplot,imdcplot_fc, $
  imdprint
common imd_m,x_m,y_m,sigy_m

if n_elements(file) eq 0 then begin
    if (!d.flags and 65536) ne 0 then begin
        file=pickfile(title='Read which IMC data file?')
        if file eq '' then return
    endif else begin
        file=' ' 
        read,'Enter the name of the IMC data file to read: ',file 
    endelse
endif 

openr,lun,file,/get_lun
if keyword_set(silent) eq 0 then print,'File: '+strtrim(file)+'...'
if keyword_set(silent) eq 0 then print,' '
dum=' '                         ; read header lines...
for i=0,10 do begin
    readf,lun,dum
    if i le 4 or i ge 9 then if keyword_set(silent) eq 0 then print,dum
    if i eq 4 then version=dum
endfor
n=fix(dum)                      ; number of data points.
dum=fltarr(13,3*n)              ; make new array.
readf,lun,dum                   ; read data.
free_lun,lun

dp=intarr(n)                    ; create array variables.
dsm=intarr(n,6)
id=intarr(n,7)
p=fltarr(n,7)
int=fltarr(n,3,2)
iint=fltarr(n,3,2)
ratio=fltarr(n,2)

for i=0,n-1 do for j=0,2 do begin ; extract variables from data array.
    case j of
        0:begin
            dp(i)=dum(0,i*3)
            dsm(i,*)=dum(1:6,i*3)
        end
        1: id(i,*)=dum(0:6,i*3+j)
        2: p(i,*)=dum(0:6,i*3+j)
    endcase
    int(i,j,*)=dum(7:8,i*3+j)
    iint(i,j,*)=dum(9:10,i*3+j)
    if j eq 2 then ratio(i,*)=dum(11:12,i*3+j)
endfor 

theta=p(*,4)                    ; incidence angle.
lambda=p(*,6)                   ; wavelength.
r=ratio(*,0)                    ; reflectance.
sigr=ratio(*,1)                 ; uncertainty.

roi=where(dsm(*,5) ne 1,n)      ; only take non-incident intensities.
if n gt 0 then begin
    i=int(roi,2,0)
    ii=iint(roi,2,0)
    theta=theta(roi)
    lambda=lambda(roi)
    r=r(roi)
    sigr=sigr(roi)
endif

roi=where(i lt 1.e300,n)        ; skip bad data points.
if n gt 0 then begin
    i=i(roi)
    ii=iint(roi)
    theta=theta(roi)
    lambda=lambda(roi)
    r=r(roi)
    sigr=sigr(roi)
endif

if keyword_set(smooth) or keyword_set(iismooth) or keyword_set(iiaverage) $
  then begin                    ; smooth data...
    if keyword_set(smooth) then i=smooth(i,smooth)
    if keyword_set(iismooth) then ii=smooth(ii,iismooth) else $
      if keyword_set(iiaverage) then $
      ii=total(ii)/n_elements(ii) else $
      if keyword_set(smooth) then ii=smooth(ii,smooth) 
    if keyword_set(iiaverage) eq 0 then begin
        roi=where(ii gt 0,n)
        if n gt 0 then begin
            r=i(roi)/ii(roi)
            sigr=ratio(roi,1)   ; don't smooth uncertainty.
            theta=theta(roi)
            lambda=lambda(roi)
        endif
    endif else r=i/ii
endif

if keyword_set(vsangle) then begin
    x_m=theta
    imd.x_m=0
    ;; deal with angle units:
    case imd.angleunits_ptr of
        0: x_m=x_m              ; output is degrees normal
        1: x_m=90.-x_m          ; output is degrees grazing
        2: x_m=x_m*60.          ; output is arc-min normal
        3: x_m=(90.-x_m)*60.    ; output is arc-min grazing
        4: x_m=x_m*3600.        ; output is arc-sec normal
        5: x_m=(90.-x_m)*3600.  ; output is arc-sec grazing
        6: x_m=x_m*!dtor*1000.  ; output is mrad normal
        7: x_m=(90.-x_m)*!dtor*1000. ; output is mrad grazing
    endcase
endif else begin
    x_m=lambda
    imd.x_m=1
    case imd.photonunits_ptr of
        0:                      ; do nothing - already in A
        1: x_m=x_m/10.d         ; convert to nm
        2: x_m=x_m/10000.d      ; convert to um
        3: x_m=1.d8/x_m         ; convert to cm(-1)
        4: x_m=imd.a2ev/x_m     ; convert is ev
        5: x_m=imd.a2ev/1000.d/x_m ; convert to kev
    endcase
endelse

y_m=r
sigy_m=sigr

roi=where(y_m gt 0.,n)
if n gt 0 then begin
    y_m=y_m(roi)
    sigy_m=sigy_m(roi)
    x_m=x_m(roi)
endif

return
end
