;+
; NAME:
;
;       RD_SPEC
;       
; PURPOSE:
;
;       Read data files written by the program SPEC.
;       
; CALLING SEQUENCE:
; 
;	RD_SPEC[,DETECTOR]
;
; OPTIONAL OUTPUTS:
;
;       DETECTOR - Array of detector counts.
;   
; KEYWORD PARAMETERS:
; 
;       FILE - A string or string array specifying the file(s) to
;              read.  If more than one file is specified, then the
;              data arrays are concatenated.
;	
;       SCAN - A scalar, or an array of integers, specifying which
;              scan(s) to be read; default is the 1st found if not
;              set.  You can concatenate multiple scans from the same
;              file by specifying a scalar value for FILE, and an
;              array for SCAN.
;   
;       MONITOR - Array of monitor counts.
;	
;       SECONDS - Array of integration times.
;	
;       EPOCH - Array of seconds since epoch time.
;   
;       H - h-momentum transfer vector.
;	
;       K - k-momentum transfer vector.
;	
;       L - l-momentum transfer vector.
;
;       THETA - Array of theta values.
;
;       TWOTHETA - Array of two-theta values.
;
;       PHI - Array of phi values.
;	
;       CHI - Array of chi values.
;	
;       COLUMN_TITLES - A string array that returns the data column
;                       titles in the file.
;
;       NAME_DETECTOR - Set this keyword to the name of the data
;                       column title corresponding to the detector
;                       counts if it is different than the default
;                       value of 'Detector'.
;
;       NAME_V1, NAME_V2, NAME_V3, V1, V2, V3 - If there are data
;                                               columns with column
;                                               titles other than the
;                                               standard ones
;                                               (detector, monitor,
;                                               seconds, epoch, h, k,
;                                               l theta, twotheta,
;                                               phi, and chi) then the
;                                               V1, V2, and V3 values
;                                               can be used in
;                                               conjunction with the
;                                               NAME_V1, NAME_V2, and
;                                               NAME_V3 to extract
;                                               these data.
;
;                    V1, V2, and V3 are the data arrays, while NAME_Vx
;                    are the column titles.  For example, if you have
;                    a SPEC file with the column title 'Ion Chamber',
;                    then set NAME_V1='Ion Chamber'; the values of
;                    'Ion Chamber' will be returned in the V1
;                    variable.
;   
;       SILENT = set to inhibit printing file info.
;
;       SET_IMD - Set to instruct RD_SPEC to convert the data
;                 contained in the file to IMD Measured Data
;                 variables, Reflectance vs. Theta.  It is assumed
;                 that V1=TH in this case.  The IMD angle pointer
;                 (IMD.X_M) will be set to 0 (angles).
;
;                 NOTE: Do not use this procedure directly!
;                 
;                 Use IMD_M_RD,RD_M_CMD='RD_SPEC,/SET_IMD'
;
;       INC_INT - Incident intensity: If SET_IMD is set, then the data
;                 will be normalized to the incident intensity.  If
;                 INC_INT is equal to zero, or undefined, then it will
;                 be assumed that the incidence intensity is equal to
;                 twice the value of the first element of DETECTOR (as
;                 would be the case for reflectance vs angle data
;                 starting from TH=0., for a properly aligned sample.)
;
;       ATTENUATION - An additional attenuation factor used to
;                     normalized the data: If SET_IMD is set, and
;                     ATTENUATION is non-zero, then the computed
;                     reflectance will be given by:
;                     Y_m=DETECTOR/INC_INT/ATTENUATION
;
;                      ATTENUATION can be an array, equal to the
;                      number of elements of FILE, in order to
;                      normalize several data sets at once.
;   
;       OFFSET - Often, when measuring the specular reflectance of a
;                film, i.e. in the TH-TTH geometry, it is desirable to
;                also perform an offset scan, i.e., a TH-TTH scan with
;                TTH offset by some small amount, in order to account
;                for diffuse scattering.  If an offset scan has been
;                performed, the diffuse intensity can be subtracted
;                from the specular intensity when computing
;                reflectance (assuming the SET_IMD keyword is set) by
;                setting this variable to an array (having the same
;                number of elements as SCAN) of 0s and 1s, where the
;                1s point to the offset scan data.  Make sure that the
;                ATTENUATION variable corresponding to the offset scan
;                data is set properly.
;
;                For example, suppose three scans were performed for a
;                film: 1. TH=0-1, with a 1000:1 attenuator, 2. TH=1-5
;                with no attenuator, and 3. an offset scan from TH=0-5
;                (with TTH=2*TH+0.3), also with no attenuator.  In
;                this case, the command to read the file and compute
;                the offset-corrected reflectance would be:
;
;                RD_SPEC,/SET_IMD,SCAN=[1,2,3],OFFSET=[0,0,1],ATTEN=[1.,1000.,1000.]
;
; COMMON BLOCKS:
;
;       COMMON IMD
;	COMMON IMD_M
;	
; SIDE EFFECTS:
;
;       If SET_IMD is set, then the IMD variables X_M, Y_M, SIGY_M,
;       and IMD.X_M are defined. (Set IMD.Y_M manually, according to
;       the type of data: 0 for reflectance, 1 for transmittance, 2
;       for absorptance.)
;	
; RESTRICTIONS:
; 
;	Use in the IMD (V3.0B1 or higher) environment, but:
;
;       DO NOT USE THIS PROCEDURE DIRECTLY!
;                 
;           Use IMD_M_RD,RD_M_CMD='RD_SPEC,/SET_IMD'
;
;
; MODIFICATION HISTORY:
; 
;	David L. Windt, Bell Laboratories, June 1996.
;	
;       Modified to work with IMD V3, (Added SET_IMD, INC_INT, and
;       ATTENUATION keywords) May 1997
;
;       Include code to read extra header line (G2) in spec V4.03 files,
;       Aug 1997
;
;       Added OFFSET keyword, December 1997.
;
;       Updated IMD common blocks to IMD version 4.0, May 1998
;
;       Almost completely re-written to handle arbitrary SPEC file formats.
;       The program now figures out which columns of data are which, and
;       returns the appropriate data vectors into keyword variables.
;
;	windt@bell-labs.com
;-

pro rd_spec,detector, $
      file=file, $
      scan=scan, $
      column_titles=column_titles, $
      name_detector=name_detector, $
      monitor=monitor, $
      seconds=seconds, $
      epoch=epoch, $
      h=h,k=k,l=l,$
      theta=theta,twotheta=twotheta,phi=phi,chi=chi, $
      v1=v1,v2=v2,v3=v3, $
      name_v1=name_v1,name_v2=name_v2,name_v3=name_v3, $
      silent=silent, $ 
      set_imd=set_imd, $
      inc_int=inc_int, $
      attenuation=attenuation, $
      offset=offset, $
      off_th=off_th,off_det=off_det

common imd, $
  imd,imdhome,imd_nkpath,imd_nkpath_ptr, $
  imdsplot,imdsplot_fc,imdsplot_bangpy, $
  imdnplot,imdnplot_fc, $
  imdfplot,imdfplot_fc, $
  imdcplot,imdcplot_fc, $
  imdprint
common imd_m,x_m,y_m,sigy_m

on_error,0

n_files=n_elements(file)
n_scans=n_elements(scan)

if (n_files eq 0) and (n_scans gt 1) then begin
    if (!d.flags and 65536) ne 0 then begin
        file=pickfile(title='Read which SPEC data file?')
        if file eq '' then return
    endif else begin
        file=' ' 
        read,'Enter the name of the SPEC data file to read: ',file 
    endelse
    n_files=n_elements(file)
endif 

;; are multiple scans from the same file specified?
if (n_files eq 1) and (n_scans gt 1) then begin
    file=strarr(n_scans)+file
    n_files=n_scans
endif

;; if there's more than one file, then read each file and concatenate
;; the data:
if n_files gt 1 then begin
    if n_elements(scan) ne n_files then scan=intarr(n_files)
    if n_elements(attenuation) ne n_files then attenuation=fltarr(n_files)+1.
    if n_elements(offset) ne n_files then offset=fltarr(n_files)

    ;; define command to read file:
    command='rd_spec,file=file(i),scan=scan(i),'+ $
      'detector,' + $
      'monitor=monitor,seconds=seconds,epoch=epoch,' + $
      'h=h,k=k,l=l,' + $
      'theta=theta,twotheta=twotheta,phi=phi,chi=chi,' + $
      'v1=v1,v2=v2,v3=v3,' + $
      'name_detector=name_detector,' + $
      'name_v1=name_v1,name_v2=name_v2,name_v3=name_v3,' + $
      'silent=keyword_set(silent),' + $
      'set_imd=keyword_set(set_imd),inc_int=inc_int,' + $
      'attenuation=attenuation(i),offset=offset(i),' + $
      'off_th=off_th,off_det=off_det'

    ;; initialize variables
    detectorI=0
    monitorI=0
    epochI=0
    secondsI=0
    hI=0
    kI=0
    lI=0
    thetaI=0
    twothetaI=0
    phiI=0
    chiI=0
    v1I=0
    v2I=0
    v3I=0
    x_mI=0
    y_mI=0
    sigy_mI=0
    off_th=0
    off_det=0

    ;; define file_index, i.e., the order that the files will be read,
    ;; since we need to read any offset scans first:
    file_index=0
    wh=where(offset eq 1,count)
    if count ne 0 then file_index=[file_index,wh]
    file_index=[file_index,where(offset eq 0)]
    file_index=file_index(1:*)

    ;; read each file and concatenate variables:
    for j=0,n_files-1 do begin
        i=file_index(j)
        ee=execute(command)
        if offset(i) then begin
            ;; if it's an offset scan, then append offset data:
            off_th=[off_th,v1]
            ;; attenuate offset data, so it's properly normalized
            ;; relative to the incident intensity:
            off_det=[off_det,detector/attenuation(i)]
        endif else begin
            ;; otherwise, append data:
            if n_elements(detector) gt 1 then detectorI=[detectorI,detector]
            if n_elements(monitor) gt 1 then monitorI=[monitorI,monitor]
            if n_elements(seconds) gt 1 then secondsI=[secondsI,seconds]
            if n_elements(epoch) gt 1 then epochI=[epochI,epoch]
            if n_elements(h) gt 1 then hI=[hI,h]
            if n_elements(k) gt 1 then kI=[kI,k]
            if n_elements(l) gt 1 then lI=[lI,l]
            if n_elements(theta) gt 1 then thetaI=[thetaI,theta]
            if n_elements(twotheta) gt 1 then twothetaI=[twothetaI,twotheta]
            if n_elements(phi) gt 1 then phiI=[phiI,phi]
            if n_elements(chi) gt 1 then chiI=[chiI,chi]
            if n_elements(v1) gt 1 then v1I=[v1I,v1]
            if n_elements(v2) gt 1 then v2I=[v2I,v2]
            if n_elements(v3) gt 1 then v3I=[v3I,v3]
            x_mI=[x_mI,x_m]
            y_mI=[y_mI,y_m]
            sigy_mI=[sigy_mI,sigy_m]
        endelse
    endfor

    ;; redefine keywords to contain arrays:
    if n_elements(detectorI) gt 1 then detector=detectorI(1:*)
    if n_elements(monitorI) gt 1 then monitor=monitorI(1:*)
    if n_elements(epochI) gt 1 then epoch=epochI(1:*)
    if n_elements(secondsI) gt 1 then seconds=secondsI(1:*)
    if n_elements(hI) gt 1 then h=hI(1:*)
    if n_elements(kI) gt 1 then k=kI(1:*)
    if n_elements(lI) gt 1 then l=lI(1:*)
    if n_elements(thetaI) gt 1 then theta=thetaI(1:*)
    if n_elements(twothetI) gt 1 then twotheta=twothetaI(1:*)
    if n_elements(phiI) gt 1 then phi=phiI(1:*)
    if n_elements(chiI) gt 1 then chi=chiI(1:*)
    if n_elements(v1I) gt 1 then v1=v1I(1:*)
    if n_elements(v2I) gt 1 then v2=v2I(1:*)
    if n_elements(v3I) gt 1 then v3=v3I(1:*)
    x_m=x_mI(1:*)
    y_m=y_mI(1:*)
    sigy_m=sigy_mI(1:*)

endif else begin
    ;;
    ;; code to actually read the file:
    ;;
    if n_files eq 0 then begin
        if (!d.flags and 65536) ne 0 then begin
            file=pickfile(title='Read which SPEC data file?')
            if file eq '' then return
        endif else begin
            file=' ' 
            read,'Enter the name of the SPEC data file to read: ',file 
        endelse
    endif 

    if not keyword_set(silent) then begin
        print,'File name: ',file
    endif

    openr,lun,file,/get_lun     ; open file.
    dum=' ' 

    scan_found=0.               ; set scan_found flag to false.
    scan_is_last=1.             ; set scan_is_last flag to true.
    on_ioerror,clean            ; goto clean when eof is found.
    while not eof(lun) do begin
        top: readf,lun,dum
        if strmid(dum,0,1) eq '#' then begin ; Comment or header line?
            case strmid(dum,1,1) of
                "F": begin      
                    ;; File Header
                    for i=0,4 do readf,lun,dum         
                end
                "S": begin      
                    ;; Scan Header
                    if scan_found eq 1 then begin 
                        ;; Already read the scan we're after?
                        scan_is_last=0 
                        ;; Desired scan is not last.
                        goto, clean
                    endif

                    ;; Initialize data variables:           
                    detector=0.
                    monitor=0.
                    seconds=0.
                    epoch=0.
                    h=0.
                    k=0.
                    l=0.
                    theta=0.
                    twotheta=0.
                    phi=0.
                    chi=0.
                    v1=0.
                    v2=0.
                    v3=0.
                    scan_number=fix(strmid(dum,3,3)) 
                    ;; Scan number. If no scan specified, then just
                    ;; read 1st one:
                    if keyword_set(scan) eq 0 then scan=scan_number 
                    ;; Is this the scan we're after?
                    if scan(0) eq scan_number then scan_found=1 
                end
                "N": begin      
                    ;; Number of data columns
                    n_columns=fix(strmid(dum,2,4)) 
                    ;; set up column titles
                    column_titles=strarr(n_columns) 
                end
                "L": begin
                    i1=3
                    for i=0,n_columns-1 do begin
                        i2=strpos(dum,"  ",i1)
                        if i2 eq -1 then i2=100
                        column_titles(i)=strmid(dum,i1,i2-i1)
                        i1=i2+2
                    endfor

                    ;; find positions of relevant data columns:
                    if keyword_set(name_detector) eq 0 then name_detector='detector'
                    detector_position=where(strlowcase(column_titles) eq name_detector)
                    monitor_position=where(strlowcase(column_titles) eq 'monitor')
                    seconds_position=where(strlowcase(column_titles) eq 'seconds')
                    epoch_position=where(strlowcase(column_titles) eq 'epoch')
                    h_position=where(strlowcase(column_titles) eq 'h')
                    k_position=where(strlowcase(column_titles) eq 'k')
                    l_position=where(strlowcase(column_titles) eq 'l')
                    theta_position=where(strlowcase(column_titles) eq 'theta')
                    twotheta_position=where(strlowcase(column_titles) eq 'twotheta')
                    if twotheta_position(0) eq -1 then  $
                      twotheta_position=where(strlowcase(column_titles) eq 'two theta')
                    phi_position=where(strlowcase(column_titles) eq 'phi')
                    chi_position=where(strlowcase(column_titles) eq 'chi')
                    if keyword_set(name_v1) then $
                      v1_position=where(strlowcase(column_titles) eq strlowcase(name_v1))
                    if keyword_set(name_v2) then $
                      v2_position=where(strlowcase(column_titles) eq strlowcase(name_v2))
                    if keyword_set(name_v3) then $
                      v3_position=where(strlowcase(column_titles) eq strlowcase(name_v3))
                end
                "C": if not keyword_set(silent) then print,dum ; Comment
                else: ;; ignore other header lines
            endcase
        endif else begin        
            ;; It's not a header line, so it must be data.

            ;; ignore blanks...
            if dum eq '' then goto,top 

            ;; setup dummy variable
            vars=fltarr(n_columns) 

            ;; extract numeric fields
            i1=0                
            for i=0,n_columns-1 do begin
                i2=strpos(dum," ",i1)
                if i2 eq -1 then i2=100
                vars(i)=float(strmid(dum,i1,i2-i1))
                i1=i2+1
            endfor

            ;; save data values:
            if detector_position(0) ne -1 then detector=[detector,vars(detector_position)]
            if monitor_position(0) ne -1 then monitor=[monitor,vars(monitor_position)]
            if seconds_position(0) ne -1 then seconds=[seconds,vars(seconds_position)]
            if epoch_position(0) ne -1 then epoch=[epoch,vars(epoch_position)]
            if h_position(0) ne -1 then h=[h,vars(h_position)]
            if k_position(0) ne -1 then k=[k,vars(k_position)]
            if l_position(0) ne -1 then l=[l,vars(l_position)]
            if theta_position(0) ne -1 then theta=[theta,vars(theta_position)]
            if twotheta_position(0) ne -1 then twotheta=[twotheta,vars(twotheta_position)]
            if phi_position(0) ne -1 then phi=[phi,vars(phi_position)]
            if chi_position(0) ne -1 then chi=[chi,vars(chi_position)]
            if keyword_set(name_v1) then if v1_position(0) ne -1 then v1=[v1,vars(v1_position)]
            if keyword_set(name_v2) then if v2_position(0) ne -1 then v2=[v2,vars(v2_position)]
            if keyword_set(name_v3) then if v3_position(0) ne -1 then v3=[v3,vars(v3_position)]

        endelse
    endwhile	

    ;; clean up data variables:
    clean:                     

    ;; close file.
    close,lun                  
    free_lun,lun

    ;; valid data subscript
    s=1 & e=n_elements(h)-1    
    if e eq 0 then e=n_elements(seconds)-1
    if e eq 0 then e=n_elements(monitor)-1

    ;; last point is invalid in this case
    if scan_is_last then e=e-1  

    ;; clean up data:
    if detector_position(0) ne -1 then detector=detector(s:e)
    if monitor_position(0) ne -1 then monitor=monitor(s:e)
    if seconds_position(0) ne -1 then seconds=seconds(s:e)
    if epoch_position(0) ne -1 then epoch=epoch(s:e)
    if h_position(0) ne -1 then h=h(s:e)
    if k_position(0) ne -1 then k=k(s:e)
    if l_position(0) ne -1 then l=l(s:e)
    if theta_position(0) ne -1 then theta=theta(s:e)
    if twotheta_position(0) ne -1 then twotheta=twotheta(s:e)
    if phi_position(0) ne -1 then phi=phi(s:e)
    if chi_position(0) ne -1 then chi=chi(s:e)
    if keyword_set(name_v1) then if v1_position(0) ne -1 then v1=v1(s:e)
    if keyword_set(name_v2) then if v2_position(0) ne -1 then v2=v2(s:e)
    if keyword_set(name_v3) then if v3_position(0) ne -1 then v3=v3(s:e)

    ;; fix up variables for IMD:
    if n_elements(offset) eq 0 then offset=0
    if keyword_set(set_imd) and (offset eq 0) then begin
        r=detector
        sigr=sqrt(r)
        th=theta

        ;; set inc_int and attenuation:
        if n_elements(inc_int) eq 0 then inc_int=detector(0)*2.
        
        ;; don't divide by zero:
        inc_int=inc_int > 1

        if n_elements(attenuation) eq 0 then attenuation=1.

        ;; subtract offset scan data:
        if n_elements(off_th) gt 0 then begin
            for i=0,n_elements(th)-1 do begin
                index=where(th(i) eq off_th,count)
                if count gt 0 then begin
                    r(i)=r(i)-attenuation*off_det(index)
                    sigr(i)=sigr(i)+sqrt(attenuation*off_det(index))
                endif
            endfor
        endif

        ;; only use data where R ne 0:
        valid=where(r gt 0.,count)
        if count eq 0 then return
        th=th(valid)
        r=r(valid)
        sigr=sigr(valid)

        ;; sort:
        y_m=r(sort(th))/inc_int(0)/attenuation(0)
        sigy_m=sigr(sort(th))/inc_int(0)/attenuation(0)

        ;; angles, grazing incidence:
        x_m=th(sort(th))

        ;; if imd.x_m is not set to angles, then set it to TH.
        if (imd.x_m ne 0) and (imd.x_m ne 2) then imd.x_m=0

        case imd.x_m of
            0: begin
                ;; deal with angle units:
                case imd.angleunits_ptr of
                    0: x_m=90.-x_m ; output is degrees normal
                    1: x_m=x_m  ; output is degrees grazing
                    2: x_m=(90.-x_m)*60. ; output is arc-min normal
                    3: x_m=x_m*60. ; output is arc-min grazing
                    4: x_m=(90.-x_m)*3600. ; output is arc-sec normal
                    5: x_m=x_m*3600. ; output is arc-sec grazing
                    6: x_m=(90.-x_m)*!dtor*1000. ; output is mrad normal
                    7: x_m=x_m*!dtor*1000. ; output is mrad grazing
                endcase
            end

            2: begin
                ;; (th is actually tth)
                ;; deal with angle units:
                case imd.angleunits_ptr of
                    0: x_m=180.-x_m ; output is degrees normal
                    1: x_m=x_m  ; output is degrees grazing
                    2: x_m=(180.-x_m)*60. ; output is arc-min normal
                    3: x_m=x_m*60. ; output is arc-min grazing
                    4: x_m=(180.-x_m)*3600. ; output is arc-sec normal
                    5: x_m=x_m*3600. ; output is arc-sec grazing
                    6: x_m=(180.-x_m)*!dtor*1000. ; output is mrad normal
                    7: x_m=x_m*!dtor*1000. ; output is mrad grazing
                endcase
            end
        endcase
    endif
endelse

return
end
