;+
; =================================  xaid_mu2chi ===============================
;
;
;
;   The documentation for XAID (the XOP's XAFS toolbox) can be found at:
;   http://www.esrf.fr/computing/scientific/xop/xaid/doc/
;
;   xaid_mu2chi is a widget based graphical interface to perform the
;   XAFS background substraction. Among several different  operations 
;   xaid_mu2chi may extract the Eo value from the maximum of the derivative, 
;   get automatically or manually the Jump value, perform polynomial 
;   and multispline fits, and allows three methods of normalization.
;   
;   The main window contains the following controls
;
;	FILE Menu:
;	  LOAD FILE: To Load a File with Data. The file must contain 
;		the data in ascii formatted in two columns. It
;		may content comments becoming by a non-numeric (nor + - .)
;		character. The program stores the data and reads 
;		automatically all the points and columns
;	  SAVE FILE: wites an ASCII file with the current data on the
;		display. The file may contain one or two data sets
;		depending on the number of curves plotted.
;	  PRINT:  Send a copy of the current set data to the printer.
;	  QUIT:	Exit or Quit the program
;
;	VIEW Menu:
;	  REFRESH: Updates the graphic display with the current data sets.
;	  DATA: shows the data set[s] numerical values.
;	  XPLOT: Send the data in the graphical display to the XPLOT
;		application, which allows more sophisticated analysis.
;
;	OPERATION 1 Menu:
;	  Must be called when only one curve is present in the display.
;	  These operations apply to a SINGLE CURVE
;	  CUT: Permits to reduce the horizontal limits of the data
;	  DERIVATIVES: Shows at the same plot the signal and
;		its two first derivatives
;	  INTERP: Allows Linear or Spline interpolation. (See IDL doc)
;	  GET Eo: Gets the Eo from the maximum of the derivative, and 
;		allows the user to change it or input another one.
;	  JUMP: Gets an initial Jump value and allows the user to 
;		change it.
;	  FIT POLYNOMIAL: Allows Least Squares od Singular Value 
;		Descomposition polynomial fits. (See IDL doc)
;	  FIT MULTISPLINE: allows multi-spline polynomial fits.
;	  FOURIER FILETERING: Performs Fourier Filtering by using the
;		xaid_ff application.
;
;	OPERATIONS 2 Menu:
;	  Makes operations between TWO data sets on the display
;	  SUBSTITUTE: Puts the set2 as set1 and deletes the set1
;	  SUBSTRACTION: performs the substractin of the ordinates
;		of the set1 minus the set2 and places the result
;		as a new set1. Set2 is deleted.
;	  EXTRACTION: Three tipes of EXAFS extraction or normalization.
;
;	TOOLS Menu: Calls several GUI applications (including this one).
;       HELP:   Shows documentation to several related XAFS routines.
;
;		
; ======== Non-interactive use of XAID_Mu2CHI ====================
;
;	NAME:
;		XAID_MU2CHI
;	PURPOSE:
;		 X user interface for cnpick. To analyse EXAFS data
; 	CATEGORY:
;		Widgets. XAFS data analysis.
;	CALLING SEQUENCE:
;		XAID_MU2CHI [,filename] 
;	OPTIONAL INPUT PARAMETERS:
;		filename: a string containing the file name to be
;		plotted or an idl variable [fltarr(2,n_points]
;		with data. The file must be ascii data arranged in 
;		columns. Lines starting with characters [except . + -]
;		are comments and not considered for the plot.
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of XAID_Mu2Chi.
;		NO_BLOCK = If set, starts Xmanager with the No_Block keyword.
;		WSIZE = The initial size of the graphical display in pixels
;	OUTPUTS:
;		Open a widget utility and present a graphic.
;	COMMON BLOCKS:
;		None
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Uses XAID data analysis IDL library.
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 11 October 1994
;		96-08-22 MSR creates a new version from the old xcnpick.
;		97-01-14 MSR fixes for Windows.
;		97-09-08 MSR adds NO_BLOCK keyword
;		97-09-30 srio@esrf.fr READ_ASCII renamed to RASCII
;		97-10-25 srio@esrf.fr small fixed in main window.
;		98-10-01 srio@esrf.fr adapts for delia. Use of catch.
;		99-02-18 srio@esrf.fr adapts XOP 2.0
;		00-02-15 srio@esrf.fr adapts for Exodus. Centers 
;			the screens. Adds *1000 and :1000 operations.
;			Allows to load a spline.
;
;-
PRO xaid_mu2chi_event, Event

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(Dialog_Parent=event.top,$
	/Error,'XAID_MU2CHI_EVENT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   goto,out
endif
uvalue = ''
Widget_Control, event.id, get_UValue=UValue


stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state, /No_Copy

wset,state.windownum
x_old = !x
y_old = !y
!x=state.x
!y=state.y

handle = handle_info(state.handle,/FIRST_CHILD)  & handle_value,handle,set1
handle = handle_info(handle,/SIBLING)  & handle_value,handle,set2
 

; exit with no data
if  ((n_elements(set1) eq 1) AND (uvalue NE 'Cursor') AND $
  (uvalue NE 'Load File') and  (uvalue NE 'Help') and  (uvalue NE 'Quit') $
  AND (uvalue NE 'Resize') AND (uvalue NE 'Execute') AND (uvalue NE 'About') ) $
  then begin
    Xtext,text='Please Load data.'
    goto,out 
endif



  CASE uvalue OF 


  'Load File': BEGIN
    ;PRINT, 'Event for Load File'
    file = Dialog_Pickfile(Dialog_Parent=event.top,/Read)
    if file eq '' then goto,out  
    Set1 = rascii(file)
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=set1,/FIRST_CHILD)
    plotfile,set1 & state.x=!x & state.y=!y
    handle = handle_create(handle,value=0,/SIBLING)
    WIDGET_CONTROL,state.Eo,SET_VAL='Eo: '
    WIDGET_CONTROL,state.Jump,SET_VAL='Jump: '
    END
  'Load Spline': BEGIN
    file = Dialog_Pickfile(Dialog_Parent=event.top,/Read)
    if file eq '' then goto,out  
    Set2 = rascii(file)
    ; Check if all abscissas are the same and calculate extrema
    IF N_Elements(set1) EQ 0 THEN BEGIN
      itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	'Error: Loaf first experimental data (Load File)')
      GoTo,out
    ENDIF
    iint = 0
    IF N_Elements(set1) NE N_Elements(set2) THEN iint = 1
    IF Total(Abs(set1[0,*]-set2[0,*])) NE 0 THEN iint = 1
    IF iint NE 0 THEN BEGIN  ; interpolation
	itmp = Widget_Message(/Info,Dialog_Parent=event.top,/Cancel, $
	    'Data abscissas are different. Data will be interpolated')
	yout = Interpol(set2[1,*],set2[0,*],set1[0,*])
	set2 = set1
	set2[1,*] = yout
    ENDIF 
    plotset,set1   &  state.x=!x  &  state.y=!y
    plotset,set2,/over
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=set1,/FIRST_CHILD)
    handle = handle_create(handle,value=set2,/SIBLING)
    END
  'Save File': BEGIN
    ;PRINT, 'Event for Save File'

    file = Dialog_Pickfile(/WRITE,file='xaid_mu2chi.out',$
	Dialog_Parent=event.top)
    if (file eq '') then goto,out
    if checkfile(file) eq 1 then begin
      tmp = Dialog_Message(Dialog_Parent=event.top,$
	['File '+ File+' already exists.', $
      'Overwrite it?'],/CANCEL,/DEFAULT_CANCEL)
      if tmp EQ 'Cancel' then goto,out
    endif
    if keyword_set(set2) then begin
        tmp = fltarr(3,n_elements(set2(0,*)))
        tmp(0:1,*) = set1
        tmp(2,*) = set2(1,*)
    endif  else begin
        tmp = set1
    endelse
    writeset,tmp,File
    ;tmp = Dialog_Message(Dialog_Parent=event.top,$
	;'File '+ File+' written to disk.',/INFO)

    END
  'Print': BEGIN
    ;PRINT, 'Event for Print'
    Xprint,BUFFER='plotfile,data',Data=state.handle,Group=Event.Top,/Color
    END
  'Quit': BEGIN
    ;PRINT, 'Event for Quit'
    handle_free,state.handle
    Widget_Control,Event.top,/Destroy
    return
    END
  'Refresh': BEGIN
    ;PRINT, 'Event for Refresh'
    plotfile,state.handle  &  state.x=!x  &  state.y=!y
    END
  'Data': BEGIN
    ;PRINT, 'Event for Data'
    xdisplayfile1,text=set1,title='XAID_mu2chi: set1',GROUP=event.top
    if n_elements(set2) gt 1 then $
    xdisplayfile1,text=set2,title='XAID_mu2chi: set2',GROUP=event.top
    END
  'Xplot': BEGIN
    ;PRINT, 'Event for Xplot'
    if keyword_set(set2) then begin
	tmp = fltarr(3,n_elements(set2(0,*)))
	tmp(0:1,*) = set1
	tmp(2,*) = set2(1,*)
	coltitles = ['X','set1','set2']
    endif  else begin
	tmp = set1
	coltitles = ['X','set1']
    endelse
    xplot,tmp,YCOL=2,GROUP=Event.Top,COLTITLES = coltitles,no_block=state.no_block
    END
  'Cut': BEGIN
    ;PRINT, 'Event for Cut'
    ;Xtext,text=['Click OK here','and select','two poins with','the mouse.']
    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,['This option permits to cut the ',$
	'spectrum to analyze only the desirable interval.',$
	'Please select two poins with the mouse.'],/CANCEL)
    if itmp EQ 'Cancel' then goto,out
    wait,0.1
    tmp1 = 1
    while (tmp1 EQ 1) do begin
      plotfile,state.handle  &  state.x=!x  &  state.y=!y
      cursor,x1,y1,3
      plotfile,[[x1,-1000000],[x1,1000000]],ki=2
      cursor,x2,y2,3
      plotfile,[[x2,-1000000],[x2,1000000]],ki=2
      plot,set1(0,*),set1(1,*),xrange=[x1,x2],/xstyle
      state.x = !x  &  state.y = !y
      tmp1 = Wmenu2(['Cancel','Repeat','Substitute'],Dialog_Parent=event.top)
      case tmp1 of
        0: begin
	   plotfile,state.handle  &  state.x=!x  &  state.y=!y
	   end
        1:
        2: begin
           cutset,set1,tmpout,xrange=[x1,x2]
           handle_free,state.handle
           state.handle = handle_create()
           handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
           handle = handle_create(handle,value=0,/SIBLING)
           state.zoom_sel=0
           plotfile,state.handle  &  state.x=!x  &  state.y=!y
	   goto,out
           end
      endcase
    endwhile
    END
  'derivatives': BEGIN
    ;PRINT, 'Event for derivatives'
    ;derivset,set1,plot=2
    derivset,set1,tmp
    tmp1=fltarr(5,n_elements(tmp(0,*)))
    tmp1(0:3,*) = tmp
    tmp1(4,*)=set1(1,*)
    xplot,tmp1,ycol=2,coltitles=['x',"f'(x)","f''(x)","f''''(x)",'f(x)'],$
	xtitle='-1',ytitle='-1',group=event.top,no_block=state.no_block
    END
  'Interp': BEGIN
    ;PRINT, 'Event for Interp'
    npoints = n_elements(set1(0,*))
    stdevval = stddev(set1(0,*))
    mean1 = mean(set1(0,*))
    vecttmp = shift(set1(0,*),-1) - set1(0,*)
    vecttmp = vecttmp(0:npoints-2)
    stdevval = stddev(vecttmp)
    mean1 = mean(vecttmp)
    text = [strcompress(npoints,/rem)+' points in current spectrum',$
    '------------- min -----  max -------  mean ------- stdev ---',$
    ' E  :     '+strcompress(min(set1(0,*)),/rem)+'       '+$
	strcompress(max(set1(0,*)),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' mu :     '+strcompress(min(set1(1,*)),/rem)+ '       '+ $
	strcompress(max(set1(1,*)),/rem)+'       '+ $
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' DE :     ' +strcompress(min(vecttmp),/rem)+'       '+$
	strcompress(max(vecttmp),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem), $
    '------------------------------------------------------------']

    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,/cancel,['This option can be used to',$
	'interpolate new points between the existing ones.',$
	'     ',text]) 
    if itmp EQ 'Cancel' then goto,out
    tmpmenu = { kind:['0','Linear','Spline'], npts:500 }
    XscrMenu,tmpmenu,/NoType,/Interp,Titles=['Interpolation: ',$
        'Number of points'],Wtitle='Interpolation',Action=action,$
	Dialog_Parent=event.top
    if action eq 'DO' then begin
      n_interp=tmpmenu.npts
      int_step = (max(set1(0,*))-min(set1(0,*)))/float(n_interp-1)
      tmpout = fltarr(2,n_interp)
      tmpout(0,*) = findgen(n_interp)*int_step + min(set1(0,*))
      if (n_interp LT npoints) then begin
        n_interp = npoints
        print,'XAID_Mu2Chi: Set interp number of points to minimum: ',npoints
      endif
      if (tmpmenu.kind(0) eq '0') then begin
        print,'XAID_Mu2Chi: Liner interp selected.'
        tmpout(1,*) = interpol(set1(1,*),set1(0,*),tmpout(0,*))
      endif else begin
        print,'XAID_Mu2Chi: Spline interp selected.'
        tmpout(1,*) = spline(set1(0,*),set1(1,*),tmpout(0,*))
      endelse
      plotset,tmpout   &  state.x=!x  &  state.y=!y
      stdevval = stddev(tmpout(0,*))
      stdevval = stddev(tmpout(1,*))
      vecttmp = shift(tmpout(0,*),-1) - tmpout(0,*)
      vecttmp = vecttmp(0:npoints-2)
      stdevval = stddev(vecttmp)
      mean1 = mean(vecttmp)

      print,n_interp,' points in current spectrum'
      print,'------------- min -----  max -------  mean ------- stdev ---'
      print,' E  :     ',min(tmpout(0,*)),max(tmpout(0,*)),mean1,stdevval
      print,' mu :     ',min(tmpout(1,*)),max(tmpout(1,*)),mean1,stdevval
      print,' DE :     ',min(vecttmp),max(vecttmp),mean1,stdevval
      print,'------------------------------------------------------------'


    text = [strcompress(n_interp,/rem)+' points in current spectrum',$
    '------------- min -----  max -------  mean ------- stdev ---',$
    ' E  :     '+strcompress(min(tmpout(0,*)),/rem)+'       '+$
	strcompress(max(tmpout(0,*)),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' mu :     '+strcompress(min(tmpout(1,*)),/rem)+ '       '+ $
	strcompress(max(tmpout(1,*)),/rem)+'       '+ $
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' DE :     ' +strcompress(min(vecttmp),/rem)+'       '+$
	strcompress(max(vecttmp),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem), $
    '------------------------------------------------------------']


      ;tmp1 = Wmenu2(['Cancel','Substitute'])
      tmp1 = Dialog_Message(Dialog_Parent=event.top,$
	/question,[text,' ','Do you want to '+$
	'substutute the interpolated spectrum?.'])
      case tmp1 of
        'No': BEGIN
		plotset,set1   &  state.x=!x  &  state.y=!y
		END
        'Yes': begin
           handle_free,state.handle
           state.handle = handle_create()
           handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
           handle = handle_create(handle,value=0,/SIBLING)
           end
      endcase
    endif
    END
  'kev2ev': BEGIN  ; *1000
	tmpout = set1
	tmpout[0,*]=tmpout[0,*]*1e3
	plotset,tmpout   &  state.x=!x  &  state.y=!y
	handle_free,state.handle
	state.handle = handle_create()
	handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
	handle = handle_create(handle,value=0,/SIBLING)
	END
  'ev2kev': BEGIN ; /1000
	tmpout = set1
	tmpout[0,*]=tmpout[0,*]*1e-3
	plotset,tmpout   &  state.x=!x  &  state.y=!y
	handle_free,state.handle
	state.handle = handle_create()
	handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
	handle = handle_create(handle,value=0,/SIBLING)
	END
  'Get Eo': BEGIN
    ;PRINT, 'Event for Get Eo'
    widget_control,state.wids.buttonsbase,get_value = tmp
    if tmp NE 0 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error,'X value must be E [Ev]')
      goto,out
    endif
    itmp=Dialog_Message(Dialog_Parent=event.top,$
	/info,/cancel,['Getting Eo from the maximum of the derivative.'])
    if itmp EQ 'Cancel' then goto,out
    e0 = gete0(set1,/sort)
    if type(state.wids.delia) EQ 8 then $
      widget_control,state.wids.delia.chi.e0,set_value=e0
    repeat begin
      plotset,set1   &  state.x=!x  &  state.y=!y
      plotset,[[e0,-1e5],[e0,1e5]],/over
      itmp = wmenu2([   'Ok, use this value: '+strcompress(e0), $
                        'Get a value with the cursor ',$
                        'Input an external value '],Dialog_Parent=event.top)
      case itmp of
        0: BEGIN
          WIDGET_CONTROL,state.Eo,SET_VAL='Eo: '+strcompress(e0,/rem)
          END
        1: BEGIN
          e0 = gete0(set1,/sort,method=1)
          print,' The selected Eo is: ',e0
          END
        2: BEGIN
          tmpmenu = { e0:e0 }
          XScrMenu,tmpmenu,/NOTYPE,TITLES='Eo value: ',ACTION=action,$
		Dialog_Parent=event.top,WTitle='E0'
          if action eq 'DO' then e0 = tmpmenu.e0
          END
      endcase
    endrep until (itmp eq 0)
    state.e0 = e0
    state.e0fork = e0
    END
  'Jump': BEGIN
    ;PRINT, 'Event for Jump'
    jump = getjump(set1,Dialog_Parent=event.top)
    if type(state.wids.delia) EQ 8 then $
      widget_control,state.wids.delia.chi.jump,set_value=jump
    plotset,set1   &  state.x=!x  &  state.y=!y
    state.Jumpval=jump
    WIDGET_CONTROL,state.Jump,SET_VALUE='Jump: '+strcompress(jump,/rem)
    END
  'Fourier Filter': BEGIN
    xaid_ff,set1,no_block=state.no_block
    END
  'Least Squares': BEGIN
    ;PRINT, 'Event for Least Squares'
    tmp = { n:1 }
    XscrMenu,tmp,/NoType,Titles='Polynomial Degree',Action=action,$
	Dialog_Parent=event.top,WTitle='LE degree'
    if action ne 'DO' then goto,out  
    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,/cancel,['Click the two limit'+ $
	' points for the fit.',' ','Then use the Operation 2 menu',$
	'to substract the fit from the spectrum.'])
    if itmp eq 'Cancel' then goto,out
    plotset,set1   &  state.x=!x  &  state.y=!y
    wait,0.1
    cursor,x1,y1,3
    oplot,[x1,x1],[y1,y1],psym=1
    cursor,x2,y2,3
    oplot,[x2,x2],[y2,y2],psym=1
    plotset,[[x1,-1000000],[x1,1000000]],/over
    plotset,[[x2,-1000000],[x2,1000000]],/over
    if type(state.wids.delia) EQ 8 then begin
      widget_control,state.wids.delia.chi.preMin,set_value=x1
      widget_control,state.wids.delia.chi.preMax,set_value=x2
    endif
    fitset,set1,fit1,method=1,degree=tmp.n,xrange=[x1,x2],/VERBOSE
    plotset,fit1,/over
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=set1,/FIRST_CHILD)
    handle = handle_create(handle,value=fit1,/SIBLING)
    END
  'Singular Value Descomposition': BEGIN
    ;PRINT, 'Event for Singular Value Descomposition'
    tmp = { n:1 }
    XscrMenu,tmp,/NoType,Titles='Polynomial Degree',Action=action,$
	Dialog_Parent=event.top, WTitle='SVD Degree'
    if action ne 'DO' then goto,out  
    plotset,set1   &  state.x=!x  &  state.y=!y
    ;Xtext,text=['Click the two limit','points for the ','fit.']
    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,/cancel,['Click the two limit'+ $
	' points for the fit.',' ','Then use the Operation 2 menu',$
	'to substract the fit from the spectrum.'])
    if itmp eq 'Cancel' then goto,out
    wait,0.1
    cursor,x1,y1,3
    plotset,set1   &  state.x=!x  &  state.y=!y
    oplot,[x1,x1],[y1,y1],psym=1
    cursor,x2,y2,3
    oplot,[x2,x2],[y2,y2],psym=1
    plotset,[[x1,-1000000],[x1,1000000]],/over
    plotset,[[x2,-1000000],[x2,1000000]],/over
    if type(state.wids.delia) EQ 8 then begin
      widget_control,state.wids.delia.chi.preMin,set_value=x1
      widget_control,state.wids.delia.chi.preMax,set_value=x2
    endif
    fitset,set1,fit1,method=2,degree=tmp.n,xrange=[x1,x2],/VERBOSE
    plotset,fit1,/over
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=set1,/FIRST_CHILD)
    handle = handle_create(handle,value=fit1,/SIBLING)
    END
  'Fit MultiSpline': BEGIN
    ;PRINT, 'Event for Fit MultiSpline'
    postedge,set1,fit1,deliaid=state.wids.delia,group=event.top,$
	Dialog_Parent=event.top
    if n_elements(fit1) eq 1 then goto,out  
    plotset,set1   &  state.x=!x  &  state.y=!y
    plotset,fit1,/over
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=set1,/FIRST_CHILD)
    handle = handle_create(handle,value=fit1,/SIBLING)
    END
  'Substitute': BEGIN
    ;PRINT, 'Event for Substitute'
    if n_elements(set2) EQ 1 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error,'Nothing to substitute.')
      goto,out
    endif
    tmpout=set2
    plotset,tmpout   &  state.x=!x  &  state.y=!y
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
    handle = handle_create(handle,value=0,/SIBLING)
    END
  'Substraction': BEGIN
    ;PRINT, 'Event for Substraction'
    if n_elements(set2) EQ 1 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error,'Nothing to substract.')
      goto,out
    endif
    tmpout=set1
    tmpout(1,*)=set1(1,*)-set2(1,*)
    plotset,tmpout   &  state.x=!x  &  state.y=!y
    handle_free,state.handle
    state.handle = handle_create()
    handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
    handle = handle_create(handle,value=0,/SIBLING)
    END
  'Experimental': BEGIN
    ;PRINT, 'Event for Experimental'
    if n_elements(set2) EQ 1 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error,'Nothing to extract.')
      goto,out
    endif
    ;if type(state.wids.delia) EQ 8 then $
    ;  widget_control,state.wids.delia.chi.method,set_value=0
    if state.i_scale_k EQ 1 then begin
      set22 = k2e(set1)
      fit22 = k2e(set2)
      localEo = 0.0
    endif else begin
      set22 = set1
      fit22 = set2
      localEo = state.e0fork
    endelse
    set3 = set22
    set3(1,*)  = (set22(1,*)  - fit22(1,*) ) / fit22(1,*)
    set4 = e2k(set3,e0=localEo)
    plotset,set4,xrange=[0.,max(set4(0,*))],yrange=[-.2,.2]
    state.x=!x  &  state.y=!y
    plotset,[[-100,0],[100,0]],/over
    itmp = wmenu2(['Cancel','Substitute','Better look (Xplot)'], $
	Dialog_Parent=event.top)
    case itmp of
      0: begin
         plotset,set1   &  state.x=!x  &  state.y=!y
         plotset,set2,/over
         end
      1: begin
         handle_free,state.handle
         state.handle = handle_create()
         handle = handle_create(state.handle,value=set4,/FIRST_CHILD)
         handle = handle_create(handle,value=0,/SIBLING)
	 widget_control,state.wids.buttonsBase,set_value = 2
	 state.i_scale_k = 1
         end
      2: begin
	   tmp = fltarr(6,n_elements(set4(0,*)))
	   tmp(0:1,*) = set4
	   for ii=1,3 do tmp(ii+1,*) = set4(1,*)*set4(0,*)^ii
           xplot,tmp,ycol=2,GROUP=Event.Top,xtitle='-1',ytitle='-1', $
		coltitles=['k[A]','chi(k)','chi(k)*k','chi(k)*k^2',$
		'chi(k)*k^3','base line'],no_block=state.no_block
           wset,state.windownum
           state.x=!x
           state.y=!y
           ;plotfile,state.handle
         end
    endcase

    END
  'Constant': BEGIN
    ;PRINT, 'Event for Constant'
    if n_elements(set2) EQ 1 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error,'Nothing to extract.')
      goto,out
    endif
    ;if type(state.wids.delia) EQ 8 then $
    ;  widget_control,state.wids.delia.chi.method,set_value=1
    if state.Jumpval EQ 0 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error, 'You must define the Jump...')
      goto,out
    endif
    if state.i_scale_k EQ 1 then begin 
      set22 = k2e(set1)
      fit22 = k2e(set2)
      localEo = 0.0
    endif  else begin
      set22 = set1
      fit22 = set2
      localEo = state.e0fork
    endelse
    set3 = set22
    set3(1,*)  = (set22(1,*)  - fit22(1,*) ) / state.jumpval
    set4 = e2k(set3,e0=localEo)
    plotset,set4,xrange=[0.,max(set4(0,*))],yrange=[-.2,.2]
    state.x=!x  &  state.y=!y
    plotset,[[-100,0],[100,0]],/over
    itmp = wmenu2(['Cancel','Substitute','Better look (Xplot)'], $
	Dialog_Parent=event.top)
    case itmp of
      0: begin
         plotset,set1   &  state.x=!x  &  state.y=!y
         plotset,set2,/over
         end
      1: begin
         handle_free,state.handle
         state.handle = handle_create()
         handle = handle_create(state.handle,value=set4,/FIRST_CHILD)
         handle = handle_create(handle,value=0,/SIBLING)
	 widget_control,state.wids.buttonsBase,set_value = 2
	 state.i_scale_k = 1
         end
      2: begin
	   tmp = fltarr(6,n_elements(set4(0,*)))
	   tmp(0:1,*) = set4
	   for ii=1,3 do tmp(ii+1,*) = set4(1,*)*set4(0,*)^ii
           xplot,tmp,ycol=2,GROUP=Event.Top,xtitle='-1',ytitle='-1', $
		coltitles=['k[A]','chi(k)','chi(k)*k','chi(k)*k^2',$
		'chi(k)*k^3','base line'],no_block=state.no_block
           wset,state.windownum
           state.x=!x
           state.y=!y
           ;plotfile,state.handle
         end
    endcase

    END
  'Lengeler-Eisenberg': BEGIN
    ;PRINT, 'Event for Lengeler-Eisenberg'
    if n_elements(set2) EQ 1 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error,'Nothing to extract.')
      goto,out
    endif
    ;if type(state.wids.delia) EQ 8 then $
    ;  widget_control,state.wids.delia.chi.method,set_value=2
    if state.Jumpval EQ 0 then begin
      itmp = Dialog_Message(Dialog_Parent=event.top,$
	/error, 'You must define the Jump...')
      goto,out
    endif
    if state.i_scale_k EQ 1 then begin 
      set22 = k2e(set1)
      fit22 = k2e(set2)
      set22(0,*) = set22(0,*) + state.e0
      fit22(0,*) = fit22(0,*) + state.e0
    endif  else begin
      set22 = set1
      fit22 = set2
      widget_control,state.wids.buttonsBase,get_value=tmp
     if tmp eq 1 then begin
        set22(0,*) = set22(0,*) + state.e0
        fit22(0,*) = fit22(0,*) + state.e0
      endif
    endelse
    set3 = set22
    e0leng = state.e0
    for i=0,n_elements(set22(0,*))-1 do begin
      set3(1,i) = (set22(1,i) - fit22(1,i)) / state.jumpval / $
                  (1. - (8./3.)*(set22(0,i)-e0leng)/e0leng)
    endfor
    set4 = e2k(set3,e0=state.e0)
    plotset,set4,xrange=[0.,max(set4(0,*))],yrange=[-.2,.2]
    state.x=!x  &  state.y=!y
    plotset,[[-100,0],[100,0]],/over
    itmp = wmenu2(['Cancel','Substitute','Better look (Xplot)'], $
	Dialog_Parent=event.top)
    case itmp of
      0: begin
         plotset,set1    &  state.x=!x  &  state.y=!y
         plotset,set2,/over
         end
      1: begin
         handle_free,state.handle
         state.handle = handle_create()
         handle = handle_create(state.handle,value=set4,/FIRST_CHILD)
         handle = handle_create(handle,value=0,/SIBLING)
	 state.i_scale_k = 1
	 widget_control,state.wids.buttonsBase,set_value=2
         end
      2: begin
	   tmp = fltarr(6,n_elements(set4(0,*)))
	   tmp(0:1,*) = set4
	   for ii=1,3 do tmp(ii+1,*) = set4(1,*)*set4(0,*)^ii
           xplot,tmp,ycol=2,GROUP=Event.Top,xtitle='-1',ytitle='-1', $
		coltitles=['k[A]','chi(k)','chi(k)*k','chi(k)*k^2',$
		'chi(k)*k^3','base line'],no_block=state.no_block
           wset,state.windownum
           state.x=!x
           state.y=!y
           ;plotfile,state.handle
         end
    endcase

    END
  'About': XAID_Help,Group=event.top
  'Help': BEGIN
    ;PRINT, 'Event for Help'
     widget_control,event.id,get_value=tmp
     xhelp,tmp,GROUP=event.top
    END
  'XVALUE': BEGIN
	widget_control,state.wids.buttonsBase,get_value = tmp
	if event.select NE 1 then goto,out  ; to avoid 3 events under Windows
        case tmp of
	0: begin   ; want E
	   if state.i_scale_k eq 0 then begin  ; E-Eo -> E
	     tmpout = set1
	     tmpout(0,*) = set1(0,*) + state.e0
	   endif else begin              ; k -> E
	     tmpout = k2e(set1)
	     tmpout(0,*) = tmpout(0,*) + state.e0
	   endelse
	   plotset,tmpout   &  state.x=!x  &  state.y=!y
	   handle_free,state.handle
	   state.handle = handle_create()
	   handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
	   handle = handle_create(handle,value=0,/SIBLING)
	   state.i_scale_k = 0
	   state.e0fork = state.e0
	   end
	1: begin   ; want E-Eo
	   if state.i_scale_k eq 0 then begin  ; E -> E-Eo
	     tmpout = set1
	     tmpout(0,*) = set1(0,*) - state.e0
	   endif else begin              ; k -> E-Eo
	     tmpout = k2e(set1)
	   endelse
	   plotset,tmpout   &  state.x=!x  &  state.y=!y
	   handle_free,state.handle
	   state.handle = handle_create()
	   handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
	   handle = handle_create(handle,value=0,/SIBLING)
	   state.i_scale_k = 0
	   state.e0fork = 0.0
	   end
	2: begin  ; want k
	   tmpout=e2k(set1,e0=state.e0fork)
	   plotset,tmpout   &  state.x=!x  &  state.y=!y
	   handle_free,state.handle
	   state.handle = handle_create()
	   handle = handle_create(state.handle,value=tmpout,/FIRST_CHILD)
	   handle = handle_create(handle,value=0,/SIBLING)
	   state.i_scale_k = 1
	   ;state.e0fork = 0.0
	   end
	else:
	endcase
	
	;widget_control,state.wids.buttonsBase,set_value = 0
    END
  'Resize': BEGIN
    ;widget_control,state.wids.wDr,scr_xsize=event.x-10,scr_ysize=event.y-70
    if !version.os_family eq 'Windows' then $
    widget_control,state.wids.wDr,scr_xsize=event.x-5,scr_ysize=event.y-68 $
    else $
    widget_control,state.wids.wDr,scr_xsize=event.x-5,scr_ysize=event.y-102
    uvalue = ''
    if  (n_elements(set1) ne 1) then begin
	plotfile,state.handle   &  state.x=!x  &  state.y=!y
        state.x=!x
        state.y=!y
    endif
    END
  'Cursor': BEGIN
     ;wset,state.windownum
     ;!x=state.x
     ;!y=state.y
     handle = handle_info(state.handle,/FIRST_CHILD)  
     handle_value,handle,tmp1
     coords_d = [event.x,event.y]
     ;if (coords_d(0) GT !d.x_size or coords_d(0) LT 0 or $
       ;coords_d(1) GT !d.y_size or coords_d(1) LT 0) then goto,out
     if n_elements(tmp1) gt 1 then $
       coords = convert_coord(event.x,event.y,/device,/to_data) else $
       coords = coords_d
     widget_control,state.XY,SET_VALUE='X: '+$
     strcompress(coords(0),/REM)+'   Y: '+$
     strcompress(coords(1),/REM)
     ;
     ; zoom
     ;
     if event.type EQ 0 then begin ;initialize zoom origin
       state.zoom_sel=1
       for ii=0,1 do state.zoomxy(ii)=coords_d(ii)
     endif
     if state.zoom_sel EQ 1 then begin ; zoom rectangle
       distx=abs(coords_d(0)-state.zoomxy(0))
       disty= abs(coords_d(1)-state.zoomxy(1))
       if distx NE 0.0 and disty NE 0.0 then begin
         tvbox,min([coords_d(0),state.zoomxy(0)]), $
         min([coords_d(1),state.zoomxy(1)]), $
         distx,disty,-2
       endif
     endif
     if ((event.type EQ 1) AND (state.zoom_sel EQ 1)) then begin 
       ;end zoom region
       state.zoom_sel=0
       coords0 = convert_coord(state.zoomxy(0),state.zoomxy(1),/device,/to_data)
       x0 = min([coords0(0),coords(0)])
       y0 = min([coords0(1),coords(1)])
       distx=abs(coords0(0)-coords(0))
       disty= abs(coords0(1)-coords(1))
       xrange = [x0,x0+distx]
       yrange = [y0,y0+disty]
       plotfile,state.handle,xrange=xrange,yrange=yrange
       state.x=!x
       state.y=!y
       ;state.zoomxy=coords
     endif
     ;state.x=!x
     ;state.y=!y
    END
    'Execute': BEGIN
      widget_control,event.id,get_value=tmp
      command = tmp+',No_Block='+StrCompress(state.no_block,/Rem)
      message,/info,'Executing: '+command & itmp = execute(command)
    END
   else:
  ENDCASE

out:
if n_elements(state) GT 0 then WIDGET_CONTROL,stateid,SET_UVALUE=state,  /No_Copy
!x = x_old
!y = y_old

END ; xaid_mu2chi_event

;=========================================================================

PRO xaid_mu2chi, setin, GROUP=Group, NODRAW=nodraw, NO_BLOCK=no_block, $
  deliaid=deliaid,WSize=wSize

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(Dialog_Parent=group,$
	/Error,'XAID_MU2CHI: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return
endif

  IF N_Elements(no_block) EQ 0 THEN no_block=0
  if type(setin) EQ 7 then setin = rascii(setin)
  IF N_ELEMENTS(Group) EQ 0 THEN GROUP=0


  wbase = widget_base(/COLUMN,TITLE='XAID_Mu2Chi 1.0 (XAFS Extraction)',$
  MBAR=wMenuBar,/TLB_SIZE_EVENTS,UVALUE='Resize')
  wtmp = widget_base(wbase) ; to store state

;Menu bar
wFileMenu =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Load File', Uvalue= 'Load File')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Load Spline', Uvalue= 'Load Spline')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Save File', Uvalue= 'Save File')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Print', Uvalue= 'Print')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Quit', Uvalue= 'Quit',/Separator)
wViewMenu =  WIDGET_BUTTON(wMenuBar, VALUE='View', /MENU)
  wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Refresh', Uvalue= 'Refresh')
  wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Data', Uvalue= 'Data')
  wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Xplot', Uvalue= 'Xplot')
wOperation1Menu =  WIDGET_BUTTON(wMenuBar, VALUE='Operation 1', /MENU)
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Cut', Uvalue= 'Cut')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='derivatives', $
	Uvalue= 'derivatives')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Interpolation', $
	Uvalue= 'Interp')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Abscissas*1000', $
	Uvalue= 'kev2ev')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Abscissas/1000', $
	Uvalue= 'ev2kev')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Get Eo', $
	Uvalue= 'Get Eo')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Jump', $
	Uvalue= 'Jump')
  wtmp1=WIDGET_BUTTON(wOperation1Menu, VALUE='Fit Polynomial',/MENU)
    wtmp = WIDGET_BUTTON(wtmp1, VALUE='Least Squares', $
	Uvalue='Least Squares')
    wtmp = WIDGET_BUTTON(wtmp1, VALUE='Singular Value Descomposition', $
	Uvalue='Singular Value Descomposition')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Fit MultiSpline', $
	Uvalue= 'Fit MultiSpline')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Fourier Filter', $
	Uvalue= 'Fourier Filter')

wOperation2Menu =  WIDGET_BUTTON(wMenuBar, VALUE='Operation 2', /MENU)
  wtmp=WIDGET_BUTTON(wOperation2Menu, VALUE='Substitute', $
	Uvalue= 'Substitute')
  wtmp=WIDGET_BUTTON(wOperation2Menu, VALUE='Substraction', $
	Uvalue= 'Substraction')
  wtmp1=WIDGET_BUTTON(wOperation2Menu, VALUE='Extraction',/MENU)
    wtmp = WIDGET_BUTTON(wtmp1, VALUE='Experimental', $
	Uvalue='Experimental')
    wtmp = WIDGET_BUTTON(wtmp1, VALUE='Constant', $
	Uvalue='Constant')
    wtmp = WIDGET_BUTTON(wtmp1, VALUE='Lengeler-Eisenberg', $
	Uvalue='Lengeler-Eisenberg')

wTools =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
  wtmp=WIDGET_BUTTON(wTools, VALUE='xaid_mu2chi', Uvalue= 'Execute')
  wtmp=WIDGET_BUTTON(wTools, VALUE='xaid', Uvalue= 'Execute')
  wtmp=WIDGET_BUTTON(wTools, VALUE='Xplot', Uvalue= 'Execute')

wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
   wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='About XAID', UVALUE='About')
   wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='xaid_mu2chi', UVALUE='Help')


if not(keyword_set(nodraw)) then $
IF N_Elements(wSize) EQ 0 THEN wSize=[500,400]
wDr = widget_draw(wBase, XSIZE=wSize[0], YSIZE=wSize[1], RETAIN=2, $
	MOTION_EVENTS=1,BUTTON_EVENTS=1,UVALUE='Cursor')

  BASE20 = WIDGET_BASE(wbase, ROW=1, MAP=1, UVALUE='BASE20')
  LABEL22 = WIDGET_LABEL( BASE20, UVALUE='LABEL22', VALUE=$
	'X:        Y:        ',/DYNAMIC_RESIZE)
  LABEL24 = WIDGET_LABEL( BASE20, UVALUE='LABEL24', VALUE='Eo: 0.0',$
	/DYNAMIC_RESIZE)
  LABEL26 = WIDGET_LABEL( BASE20, UVALUE='LABEL26', VALUE='Jump:',$
	/DYNAMIC_RESIZE)

  wButtons = 0L
  wButtonsBase = cw_bgroup(wbase,['E [Ev]', 'E-Eo', 'K [A^-1]'],$
	ids=wButtons, /exclusive, label_left='X value: ',$
	UVALUE = 'XVALUE', row=1, set_value = 0)


WIDGET_CONTROL, wbase, /REALIZE

   top = handle_create()
   npp = n_params()
   if npp EQ 0 then begin
     handle = handle_create(top,value=0,/first_child)
   endif else begin
     plotfile,setin,setout   
     handle = handle_create(top,value=setout,/first_child)
   endelse
   handle = handle_create(handle,value=0,/SIBLING)
   WIDGET_CONTROL, wDr, GET_VALUE=wDr_Id
   if not(keyword_set(deliaId)) then deliaId=0L
   wids = { wDr:wDr, buttonsBase:wButtonsBase, delia:deliaId} 
   state = { XY:LABEL22, Eo:LABEL24, Jump:LABEL26, Windownum:wDr_Id, $
	handle:top, e0:0.0, e0fork:0.0, i_scale_k:0, Jumpval:0.0, $
        x:!x, y:!y, wids:wids, zoom_sel:0, zoomxy:[0.0,0.0], $
	no_block:no_block}

widget_control,Widget_Info(wbase,/Child),set_uvalue=state,/no_copy
if keyword_set(no_block) then $
  xmanager, 'xaid_mu2chi', wbase, GROUP=group, /NO_BLOCK else $
  xmanager, 'xaid_mu2chi', wbase, GROUP=group

END
