Function nsl_calc,input,descriptor,wavelength,Return=F,density=density,$
   theta=theta,rough=rough,group=group,polar=polar,  $
   new_descriptor=new_descriptor, verbose=verbose, $
   full_match=full_match,Z=Z,molecules_per_cc=molecules_per_cc

;
;+
; NAME:
;       NSL_CALC  (Neutron Scattering Length calculations)
;
; PURPOSE:
;	Evaluate the neutron scattering lengths, cross sections
;	and related quantities (refraction index, absorption coefficients
;	and mirror relfectivity) versus neutron wavelength. 
;	It uses data from DABAX database.
;
; CATEGORY:
;       Neutron optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = nsl_calc(input,descriptor[,wavelength])
; INPUTS:
;	input: a dabax input file as accepted by dabax_access().
;	descriptor: a string  with a description of the material.
;		The string is searched in the scan titles #S in order
;		to find the appropiated data.
; OPTIONAL INPUTS:
;	wavelength: The value (or array of values) with wavelength in Angstrom.
;		If undefined, use a 100 points array in the [0.02,100] interval.
;		When wavelength is defined, this value is used as a grid
;		for the results (it interpolates in the case of
;		tabulated data). 
;
; KEYWORDS:
;	FULL_MATCH: If set, the "descriptor" word must match perfectly the 
;		scan entry in the DABAX file. If not set (default), the 
;		string "descriptor" is searched in the scan titles, and when
;		multiples matches are found, the user is asked to select one. 
;	RETURN: flag for selecting the output:
;		0 (default) returns a 2-col array with Real(b) and Im(b)
;		1  returns Real(b) (=br)
;		2  returns Im(b) (=bi)
;
;  		3  returns scattering cross section (=sigma_coh+sigma_incoh) 
;			in barns
;  		4  returns incoherent cross section (sigma_incoh) in barns
;  		5  returns coherent cross section (sigma_coh=4 pi br^2) 
;			in barns
;  		6  returns nuclear absorption cross section in barns
;
;
;  		7  returns delta; n, the refraction index n=1-delta-i*beta
;  		8  returns beta
;  		9  returns mu (Nuclear absorption coefficient)
;  		10  returns mu/rho (Mass Nuclear absorption coefficient)
;
;  		11  returns reflectivity
;  		12  returns nuclear absorption cross section in barns 
;			at ref. wavelength.
;
;	THETA= the incident [grazing] angle in mrad (for reflectivity
;		calculations). It is set to 3.0 (default) when RET=11 and
;		THETA is undefined. It can be an array. In such a case, the
;		returned value os a matrix: out(theta,wavelength)
;	ROUGH= the rms roughness in Angstroms (for reflectivity
;		calculations. A Debye-Waller or Nevot-Croce model is used.)
;		Convention: 
;		   If rough<0, a Nevot-Croce model is used
;		   If rough>0, a Debye-Waller model is used
;	
;	DENSITY= the density value to be used for the calculations. If
;		not defined, take the value given by atomic_constants().
;	GROUP = The widget ID of the widget that calls the routine.
;		This is passed to widget_message in its dialog_parent
;		keyword to center the window message.
;	POLAR = Flag for polarizarion: 
;			0 = Unpolairzed (default)
;			1 = Up(+1/2)
;			2 = Down(-1/2)
;		Note: For polarization calculations, the DABAX data file in use 
;		      must contain a good value in the column data 
;	              b+[fm] and b-[fm]. In this column is not present, or
;		      no value is present, the program ignores the polarization
;		      and uses the unpolarised case WITH NO WARNING. 
;	VERBOSE = Flag to inform on isotope name and density used. 
;	NEW_DESCRIPTOR = Set this keyword to a names variable to obtain the 
;		new descriptor name after the user selection (if needed). 
;	Z = Set this keyword to a names variable to obtain the atomic number.
;	MOLECULES_PER_CC = Set this keyword to a name variable to receive
;		the number of milecules per cm^3. 
; OUTPUT:
;	out: an array with the values of the selected return parameter.
;
; PROCEDURE:
;	Takes data from DABAX files. It also takes the
;	Atomic constants from DABAX, and performs the classical
;	operations to calculate the other parameters.
;
; EXAMPLES:
;	EXAMPLE1: 
;	Prints scattering lengths (real and imag) for Mn:
;	print,nsl_calc('Neutron_SLCS_NeutronNews.dat','Mn',Verbose=0)
;	-3.75000  -0.00369855
;
;	EXAMPLE2: reflectivity vs. theta and wavelength
;	; create 20 points in the wavelength range [0.1,100.] (eV)
;	wavelength = makearray1(100,0.1,100)
;	; create 20 points in the theta range [1.0,6.0] (mrad)
;	theta = makearray1(20,1.0,6.0)
;	out = nsl_calc('Neutron_SLCS_NeutronNews.dat','Mn',wavelength,RET=11,$
;		theta=theta)
;	;plot
;	surface,out,theta,wavelength,az=285
;
; MODIFICATION HISTORY:
;       2003-03-12 M. Sanchez del Rio (srio@esrf.fr)
;		Taken the version written by Ch. Boisseau, based on
;		f1f2_calc.pro. Cleaned and documented. 
;       2003-02-26 alianell@ill.fr corrected mu and reflectivity calculation
;       2003-03-13 srio@esrf.fr renamed to nsl_calc. Tons of cosmetics. 
;       2003-04-02 srio@esrf.fr uses atomic_weights instead
;		atomic_constants to implement atomic weights of isotopes. 
;-
on_error,2

h = dabax_access(input)

IF N_Elements(verbose) EQ 0 THEN verbose=1

if not(keyword_set(h)) then return,0
dataset = spec_headers(h,-1,'#F',/index,/all)
dataset = strcompress(strsubstitute(dataset,'#F',''),/rem)

igood = dabax_select(h,descriptor,/casematch,group=group, $
        new_descriptor=new_descriptor,full_match=full_match)

; added by srio (97/10/16) to avoid stopping the program when
; an element is not found (i.e. H in muscovite when using Brennan dataset)
if igood EQ 0 then return,0

IF verbose EQ 1 THEN message,/info, $
    'Using scan: '+strcompress(spec_name(h,igood,/index))
z = double(strsubstitute(spec_headers(h,igood,'#S',/index),'#S',''))

data = spec_data(h,igood,/index)
cols = spec_labels(h,igood,/index)


IF N_Elements(polar) EQ 0 THEN polar=0

; reference wavelenght absorption cross section is calculated at
wlref = 1.798d-10    ; in m


IF N_Elements(wavelength) EQ 0 THEN wavelength=makearray1(100,0.02,100)
wavelength1 = wavelength*1.d-10   ; in m

;
; obtain br, bi sigma_scat and sigma_abs from data file.
;

; get real(b) in column labelled "b_coh_real[fm]"
do_unpolarized = 1

IF Keyword_Set(polar) THEN BEGIN
  CASE polar OF
    1: col_label = 'b+[fm]'
    2: col_label = 'b-[fm]'
    else: Message,'Polarization flag nor valid' 
  ENDCASE
  tmp = strpos(cols,col_label)
  icol = where(tmp NE -1)
  IF icol[0] EQ -1 THEN BEGIN
    Message,/Info,'**** Column not found: '+ $
            col_label+'. Using unpolarized data.'
  ENDIF ELSE BEGIN
    br = data(icol)
    do_unpolarized = 0
    IF ((abs(br) eq 9999) or (abs(br) eq 0)) THEN  BEGIN 
      Message,/Info,'**** Polarized b value not good.  Using unpolarized data.'
      do_unpolarized=1
    ENDIF
  ENDELSE
ENDIF 

IF do_unpolarized EQ 1 THEN BEGIN 
  tmp = strpos(cols,'b_coh_real[fm]')
  icol = where(tmp NE -1)
  if icol[0] EQ -1 THEN Message,'Column b_coh_real[fm] not found'
  br = data(icol)
ENDIF

; get imag(b) in column labelled "b_coh_imag[fm]"
tmp = strpos(cols,'b_coh_imag[fm]')
icol = where(tmp NE -1)
alternative_way=0 
IF icol[0] EQ -1 THEN BEGIN 
	alternative_way=1 
ENDIF ELSE BEGIN
    bi = data(icol)
    IF ((abs(bi) eq 9999) or (abs(bi) eq 0)) THEN  alternative_way=1
ENDELSE

; get the four sigmas 

tmp = strpos(cols,'Sigma_Abs[barn]')
icol = where(tmp NE -1)
if icol[0] EQ -1 THEN Message,'Column Sigma_Abs[barn] not found'
sigma_abs = data(icol)

tmp = strpos(cols,'Sigma_Scat[barn]')
icol = where(tmp NE -1)
if icol[0] EQ -1 THEN Message,'Column Sigma_Scat[barn] not found'
sigma_scat = data(icol)

tmp = strpos(cols,'Sigma_Coh[barn]')
icol = where(tmp NE -1)
if icol[0] EQ -1 THEN Message,'Column Sigma_Coh[barn] not found'
sigma_Coh = data(icol)

tmp = strpos(cols,'Sigma_Incoh[barn]')
icol = where(tmp NE -1)
if icol[0] EQ -1 THEN Message,'Column Sigma_Incoh[barn] not found'
sigma_Incoh = data(icol)

; get imag(b) in an alternative way (using Sigma_Abs[barn] column)
IF alternative_way EQ 1 THEN BEGIN
  ;bi = -0.5d0*data(icol)*1.d-13/wlref   ; value in fm (fermi)
  bi = -0.5d0*sigma_abs*1.d-13/wlref   ; value in fm (fermi)
ENDIF

; returns result( if user requested f)
if not(keyword_set(f)) then f=0
case f of
  0:begin
    fout = fltarr(2,n_elements(br))
    fout(0,*) = br
    fout(1,*) = bi
    return,fout
    end
  1:return,br
  2:return,bi
  3:return,sigma_scat
  4:return,sigma_incoh
  5:return,sigma_coh
  6:return,wavelength1*(sigma_abs/wlref)[0]
;  12:return,sigma_abs
  else:
endcase

;
; calculation of refraction index
;

if n_elements(density) EQ 0 then $
	density = atomic_constants(z,return='density')
IF verbose EQ 1 THEN Message,/Info,'Using density for '+descriptor+' :  '+StrCompress(density)

;atwt = atomic_constants(z,return='atomicMass')
atwt = atomic_weights(new_descriptor,full_match=full_match)
; print,'OLD VALUE ATWT: ',atomic_constants(z,return='atomicMass')
; print,'NEW VALUE ATWT: ',atwt
IF Keyword_Set(verbose) THEN Message,/Info,'Atomic weight for '+ $
   new_descriptor+' is: '+StrCompress(atwt,/Rem)

avogadro = physical_constants('avogadro')
tocm = physical_constants('hc')*1e-8

molecules_per_cc = density*avogadro/atwt

case f of
  12:return,sigma_abs
  else:
endcase

k = molecules_per_cc*1.d6*wavelength1*wavelength1/2.0d0/!pi
nbee=n_elements(wavelength1)
refi=dblarr(2,nbee)
debe=dblarr(2,nbee)
mu=dblarr(nbee)
for i=0,nbee-1 do begin
    debe[0,i]=k[i]*br*1.d-15
    debe[1,i] = 0.5d0*k[i]*sigma_abs*1.d-28/wlref
    mu[i] = 4.0*!pi*debe[1,i]/(wavelength1[i]*100.)    ;cm-1
    endfor
;refi[0,*]=1.d0-debe[0,*]
;refi[1,*]=debe[1,*]

case f of
  7:return,debe[0,*]
  8:return,debe[1,*]
  9:return,mu
  10:return,mu/density
  else:
endcase

;
; calculation of reflectivity (piece of code adapted from shadow/abrefc)
;

if n_elements(theta) eq 0 then theta1 = 3.0e-3 else theta1 = theta*1e-3
;if n_elements(theta) eq 0 then sin_theta2 = 3.0e-3 else sin_theta2 = sin(theta)*1e-3  
if n_elements(rough) eq 0 then rough1 = 0.0 else rough1=rough
rough1 = rough1*1.0d-8 ; in cm
; epsi = 1 - alpha - i gamma
alpha = 2.0D0*debe[0,*]
gamma = -2.0D0*debe[1,*]
	for i=0,n_elements(theta1)-1 do begin

     	RHO	=   (SIN(theta1(i)))^2 - ALPHA
     	RHO	=   RHO + SQRT ((SIN(theta1(i))^2 - ALPHA)^2 + GAMMA^2)
     	RHO	=   SQRT(RHO/2)
;** Computes now the reflectivity for s-pol
     	RS1	=   4*(RHO^2)*(SIN(theta1(i))-RHO)^2 + GAMMA^2
     	RS2	=   4*(RHO^2)*(SIN(theta1(i))+RHO)^2 + GAMMA^2
     	RS	=   RS1/RS2
;
;	Calculate Debye-Waller and Nevot-Coce factors
; 

        DebyeWaller = exp( -(4.0*!pi*sin(theta1(i))*rough1/wavelength1)^2)
	; snell lay 
	cos_theta2 = cos(theta1[i])/(1.0D0-debe[0,*])
	sin_theta2_2 = (1.0D0-cos_theta2^2) > 0.0D0
	sin_theta2 = Sqrt(sin_theta2_2)
	NevotCroce = exp( -16.0/!pi^2*sin(theta1(i))*sin_theta2* (rough1/wavelength1)^2)      
	;; print,'DW, NC: cos, sin : ',DebyeWaller,NevotCroce,cos_theta2,sin_theta2
	;; print,' '
	

        IF rough1 GE 0 THEN BEGIN   ; apply DebyeWaller
	  RS = RS * DebyeWaller
        ENDIF ELSE BEGIN
	  RS = RS * NevotCroce
	ENDELSE
; Nevot-Croce check:
;         sin_theta2(i)=sqrt(1.-(cos(theta1(i))/(1.-debe(0,i)))^2)
;         DebyeWaller = exp( -(16.0/!pi^2*sin(theta1(i))*sin_theta2(i)*rough1/wavelength1)^2)      
;
;        out = RS*DebyeWaller    ;neutron case
        out = RS
	if n_elements(theta1) eq 1 then out1 = out else begin
          if i eq 0 then begin
            out1 = fltarr(n_elements(theta1),n_elements(out))
            out1(0,*) = reform(out)
          endif else out1(i,*) = reform(out)
        endelse

        endfor

return,out1   ; last case: reflectivity calculations

end
