;+
; NAME:
;
;       RD_SPEC
;       
; PURPOSE:
;
;       Read data files written by the program SPEC.
;       
; CALLING SEQUENCE:
; 
;	RD_SPEC[,DETECTOR,MONITOR,SECONDS]
;
; OPTIONAL OUTPUTS:
;
;       DETECTOR - Array of detector counts.
;   
;       MONITOR - Array of monitor counts.
;	
;       SECONDS - Array of integration times.
;	
; KEYWORD PARAMETERS:
; 
;       FILE - A string or string array specifying the file(s) to
;              read.  If more than one file is specified, then the
;              data arrays are concatenated.
;	
;       SCAN - A scalar, or an array of integers, specifying which
;              scan(s) to be read; default is the 1st found if not
;              set.  You can concatenate multiple scans from the same
;              file by specifying a scalar value for FILE, and an
;              array for SCAN.
;   
;       V1, V2, V3 - Array of motor positions, corresponding to the
;                    first one, two or three columns of data present
;                    in the file.
;   
;       H - h-momentum transfer vector.
;	
;       K - k-momentum transfer vector.
;	
;       L - l-momentum transfer vector.
;	
;       EPOCH - Array of seconds since epoch time.
;   
;       COLUMN_TITLES - A string array that contains the data column
;                       titles in the file.
;   
;       SILENT = set to inhibit printing file info.
;
;       SET_IMD - Set to instruct RD_SPEC to convert the data
;                 contained in the file to IMD Measured Data
;                 variables, Reflectance vs. Theta.  It is assumed
;                 that V1=TH in this case.  The IMD angle pointer
;                 (IMD.X_M) will be set to 0 (angles).
;
;                 NOTE: Do not use this procedure directly!
;                 
;                 Use IMD_M_RD,RD_M_CMD='RD_SPEC,/SET_IMD'
;
;       INC_INT - Incident intensity: If SET_IMD is set, then the data
;                 will be normalized to the incident intensity.  If
;                 INC_INT is equal to zero, or undefined, then it will
;                 be assumed that the incidence intensity is equal to
;                 twice the value of the first element of DETECTOR (as
;                 would be the case for reflectance vs angle data
;                 starting from TH=0., for a properly aligned sample.)
;
;       ATTENUATION - An additional attenuation factor used to
;                     normalized the data: If SET_IMD is set, and
;                     ATTENUATION is non-zero, then the computed
;                     reflectance will be given by:
;                     Y_m=DETECTOR/INC_INT/ATTENUATION
;
;                      ATTENUATION can be an array, equal to the
;                      number of elements of FILE, in order to
;                      normalize several data sets at once.
;   
;       OFFSET - Often, when measuring the specular reflectance of a
;                film, i.e. in the TH-TTH geometry, it is desirable to
;                also perform an offset scan, i.e., a TH-TTH scan with
;                TTH offset by some small amount, in order to account
;                for diffuse scattering.  If an offset scan has been
;                performed, the diffuse intensity can be subtracted
;                from the specular intensity when computing
;                reflectance (assuming the SET_IMD keyword is set) by
;                setting this variable to an array (having the same
;                number of elements as SCAN) of 0s and 1s, where the
;                1s point to the offset scan data.  Make sure that the
;                ATTENUATION variable corresponding to the offset scan
;                data is set properly.
;
;                For example, suppose three scans were performed for a
;                film: 1. TH=0-1, with a 1000:1 attenuator, 2. TH=1-5
;                with no attenuator, and 3. an offset scan from TH=0-5
;                (with TTH=2*TH+0.3), also with no attenuator.  In
;                this case, the command to read the file and compute
;                the offset-corrected reflectance would be:
;
;                RD_SPEC,/SET_IMD,SCAN=[1,2,3],OFFSET=[0,0,1],ATTEN=[1.,1000.,1000.]
;
; COMMON BLOCKS:
;
;       COMMON IMD
;	COMMON IMD_M
;	
; SIDE EFFECTS:
;
;       If SET_IMD is set, then the IMD variables X_M, Y_M, SIGY_M,
;       and IMD.X_M are defined. (Set IMD.Y_M manually, according to
;       the type of data: 0 for reflectance, 1 for transmittance, 2
;       for absorptance.)
;	
; RESTRICTIONS:
; 
;
;       You'll have to fix this program up to work with your SPEC
;       files, if your files are configured differently than mine.
; 
;	Use in the IMD (V3.0B1 or higher) environment, but:
;
;       DO NOT USE THIS PROCEDURE DIRECTLY!
;                 
;           Use IMD_M_RD,RD_M_CMD='RD_SPEC,/SET_IMD'
;
;
; MODIFICATION HISTORY:
; 
;	David L. Windt, Bell Laboratories, June 1996.
;	
;       Modified to work with IMD V3, (Added SET_IMD, INC_INT, and
;       ATTENUATION keywords) May 1997
;
;       Include code to read extra header line (G2) in spec V4.03 files,
;       Aug 1997
;
;       Added OFFSET keyword, December 1997.
;
;       Updated IMD common blocks to IMD version 4.0, May 1998
;
;	windt@bell-labs.com
;-

pro rd_spec,detector,monitor,seconds, $
      file=file, $
      scan=scan, $
      column_titles=column_titles, $
      v1=v1,v2=v2,v3=v3, $
      h=h,k=k,l=l,$
      epoch=epoch, $
      silent=silent, $ 
      set_imd=set_imd, $
      inc_int=inc_int, $
      attenuation=attenuation, $
      offset=offset, $
      off_th=off_th,off_det=off_det

common imd, $
  imd,imdhome,imd_nkpath,imd_nkpath_ptr, $
  imdsplot,imdsplot_fc,imdsplot_bangpy, $
  imdnplot,imdnplot_fc, $
  imdfplot,imdfplot_fc, $
  imdcplot,imdcplot_fc, $
  imdprint
common imd_m,x_m,y_m,sigy_m

on_error,0

n_files=n_elements(file)
n_scans=n_elements(scan)

if (n_files eq 0) and (n_scans gt 1) then begin
    if (!d.flags and 65536) ne 0 then begin
        file=pickfile(title='Read which SPEC data file?')
        if file eq '' then return
    endif else begin
        file=' ' 
        read,'Enter the name of the SPEC data file to read: ',file 
    endelse
    n_files=n_elements(file)
endif 

;; are multiple scans from the same file specified?
if (n_files eq 1) and (n_scans gt 1) then begin
    file=strarr(n_scans)+file
    n_files=n_scans
endif

;; if there's more than one file, then read each file and concatenate
;; the data:
if n_files gt 1 then begin
    if n_elements(scan) ne n_files then scan=intarr(n_files)
    if n_elements(attenuation) ne n_files then attenuation=fltarr(n_files)+1.
    if n_elements(offset) ne n_files then offset=fltarr(n_files)

    ;; define command to read file:
    command='rd_spec,file=file(i),scan=scan(i),'+ $
      'detector,monitor,seconds,epoch=epoch,' + $
      'h=h,k=k,l=l,silent=keyword_set(silent),' + $
      'set_imd=keyword_set(set_imd),inc_int=inc_int,' + $
      'attenuation=attenuation(i),offset=offset(i),' + $
      'off_th=off_th,off_det=off_det,v1=v1'
    if n_elements(v2) ne 0 then command=command+',v2=v2'
    if n_elements(v3) ne 0 then command=command+',v3=v3'

    ;; initialize variables
    detectorI=0
    monitorI=0
    epochI=0
    secondsI=0
    hI=0
    kI=0
    lI=0
    v1I=0
    if n_elements(v2) ne 0 then v2I=0
    if n_elements(v3) ne 0 then v3I=0
    x_mI=0
    y_mI=0
    sigy_mI=0
    off_th=0
    off_det=0

    ;; define file_index, i.e., the order that the files will be read,
    ;; since we need to read any offset scans first:
    file_index=0
    wh=where(offset eq 1,count)
    if count ne 0 then file_index=[file_index,wh]
    file_index=[file_index,where(offset eq 0)]
    file_index=file_index(1:*)

    ;; read each file and concatenate variables:
    for j=0,n_files-1 do begin
        i=file_index(j)
        ee=execute(command)
        if offset(i) then begin
            ;; if it's an offset scan, then append offset data:
            off_th=[off_th,v1]
            ;; attenuate offset data, so it's properly normalized
            ;; relative to the incident intensity:
            off_det=[off_det,detector/attenuation(i)]
        endif else begin
            ;; otherwise, append data:
            detectorI=[detectorI,detector]
            monitorI=[monitorI,monitor]
            epochI=[epochI,epoch]
            secondsI=[secondsI,seconds]
            hI=[hI,h]
            kI=[kI,k]
            lI=[lI,l]
            v1I=[v1I,v1]
            if n_elements(v2) ne 0 then v2I=[v2I,v2]
            if n_elements(v3) ne 0 then v3I=[v3I,v3]
            x_mI=[x_mI,x_m]
            y_mI=[y_mI,y_m]
            sigy_mI=[sigy_mI,sigy_m]
        endelse
    endfor

    ;; redefine keywords to contain arrays:
    detector=detectorI(1:*)
    monitor=monitorI(1:*)
    epoch=epochI(1:*)
    seconds=secondsI(1:*)
    h=hI(1:*)
    k=kI(1:*)
    l=lI(1:*)
    v1=v1I(1:*)
    if n_elements(v2) ne 0 then v2=v2I(1:*)
    if n_elements(v3) ne 0 then v3=v3I(1:*)
    x_m=x_mI(1:*)
    y_m=y_mI(1:*)
    sigy_m=sigy_mI(1:*)

endif else begin
    ;;
    ;; code to actually read the file:
    ;;
    if n_files eq 0 then begin
        if (!d.flags and 65536) ne 0 then begin
            file=pickfile(title='Read which SPEC data file?')
            if file eq '' then return
        endif else begin
            file=' ' 
            read,'Enter the name of the SPEC data file to read: ',file 
        endelse
    endif 

    if not keyword_set(silent) then begin
        print,'File name: ',file
    endif

    openr,lun,file,/get_lun     ; open file.
    dum=' ' 

    scan_found=0.               ; set scan_found flag to false.
    scan_is_last=1.             ; set scan_is_last flag to true.
    on_ioerror,clean            ; goto clean when eof is found.
    while not eof(lun) do begin
        top: readf,lun,dum
        if strmid(dum,0,1) eq '#' then begin ; Comment or header line?
            case strmid(dum,1,1) of
                "F": begin      ; File Header
                    for i=0,4 do begin
                        readf,lun,dum         
                                ; print,"File Header: ",dum
                    endfor
                end
                "S": begin      ; Scan Header
                    if scan_found eq 1 then begin ; Already read the scan
                                ; we're after?
                        scan_is_last=0 ; Desired scan is not last.
                        goto, clean
                    endif
                                ; Setup data variables:           
                    v1=0.
                    v2=0.
                    v3=0.
                    h=0.
                    k=0.
                    l=0.
                    epoch=0.
                    seconds=0.
                    monitor=0.
                    detector=0.
                    scan_number=fix(strmid(dum,3,3)) ; Scan number
                                ; if no scan specified, 
                                ; then just read 1st one:
                    if keyword_set(scan) eq 0 then scan=scan_number 
                    if scan(0) eq scan_number then scan_found=1 ; Is this the scan
                                ; we're after?
                    readf,lun,dum ; #D
                    readf,lun,dum ; #T | #M
                    time_flag=strmid(dum,1,1) eq 'T' ; Time or Monitor?
                    readf,lun,dum ; #G0
                    readf,lun,dum ; #G1
                    readf,lun,dum ; #Q
                    ;; spec V4.03 also has G2:
                    if strmid(dum,1,1) eq 'G' then readf,lun,dum ; #Q
                    readf,lun,dum ; #P
                    readf,lun,dum ; #N
                    n_columns=fix(strmid(dum,2,4)) ; Number of data columns
                    column_titles=strarr(n_columns) ; set up column titles
                    readf,lun,dum ; #L
                    i1=3
                    for i=0,n_columns-1 do begin
                        i2=strpos(dum," ",i1)
                        if i2 eq -1 then i2=100
                        column_titles(i)=strmid(dum,i1,i2-i1)
                        i1=i2+2
                    endfor
                end
                "C": if not keyword_set(silent) then print,dum ; Comment
                else:           ;
            endcase
        endif else begin        ; It's not a header line, so it must be data.
            if dum eq '' then goto,top ; don't be fooled by blanks...
            vars=fltarr(n_columns) ; setup dummy variable
            i1=0                ; extract numeric fields...
            for i=0,n_columns-1 do begin
                i2=strpos(dum," ",i1)
                if i2 eq -1 then i2=100
                vars(i)=float(strmid(dum,i1,i2-i1))
                i1=i2+1
            endfor
            v1=[v1,vars(0)]     ; insert fields into variables...
            case n_columns of
                8: j=1
                9: begin
                    v2=[v2,vars(1)]
                    j=2
                end
                10: begin
                    v2=[v2,vars(1)]
                    v3=[v3,vars(2)]
                    j=3
                end
                else:           ;
            endcase
            h=[h,vars(j)] & j=j+1
            k=[k,vars(j)] & j=j+1
            l=[l,vars(j)] & j=j+1
            epoch=[epoch,vars(j)] & j=j+1
            seconds=[seconds,vars(j)] & j=j+1
            monitor=[monitor,vars(j)] & j=j+1
            detector=[detector,vars(j)] & j=j+1
        endelse
    endwhile	

    clean:                      ; clean up data variables:
    close,lun                   ; close file.
    free_lun,lun

    s=1 & e=n_elements(v1)-1    ; valid data subscript
    if scan_is_last then e=e-1  ; last point is invalid in this case
    v1=v1(s:e)
    if n_columns gt 8 then v2=v2(s:e)
    if n_columns gt 9 then v3=v3(s:e)
    h=h(s:e)
    k=k(s:e)
    l=l(s:e)
    epoch=epoch(s:e)
    seconds=seconds(s:e)
    monitor=monitor(s:e)
    detector=detector(s:e)

    if not time_flag then begin ; switch monitor and seconds?
        dum=seconds
        seconds=monitor
        monitor=dum
    endif

    ;; fix up variables for IMD:
    if n_elements(offset) eq 0 then offset=0
    if keyword_set(set_imd) and (offset eq 0) then begin
        r=detector
        sigr=sqrt(r)
        th=v1

        ;; set inc_int and attenuation:
        if n_elements(inc_int) eq 0 then inc_int=detector(0)*2.
        
        ;; don't divide by zero:
        inc_int=inc_int > 1

        if n_elements(attenuation) eq 0 then attenuation=1.

        ;; subtract offset scan data:
        if n_elements(off_th) gt 0 then begin
            for i=0,n_elements(th)-1 do begin
                index=where(th(i) eq off_th,count)
                if count gt 0 then begin
                    r(i)=r(i)-attenuation*off_det(index)
                    sigr(i)=sigr(i)+sqrt(attenuation*off_det(index))
                endif
            endfor
        endif

        ;; only use data where R ne 0:
        valid=where(r gt 0.,count)
        if count eq 0 then return
        th=th(valid)
        r=r(valid)
        sigr=sigr(valid)

        ;; sort:
        y_m=r(sort(th))/inc_int(0)/attenuation(0)
        sigy_m=sigr(sort(th))/inc_int(0)/attenuation(0)

        ;; angles, grazing incidence:
        x_m=th(sort(th))

        ;; if imd.x_m is not set to angles, then set it to TH.
        if (imd.x_m ne 0) and (imd.x_m ne 2) then imd.x_m=0

        case imd.x_m of
            0: begin
                ;; deal with angle units:
                case imd.angleunits_ptr of
                    0: x_m=90.-x_m ; output is degrees normal
                    1: x_m=x_m  ; output is degrees grazing
                    2: x_m=(90.-x_m)*60. ; output is arc-min normal
                    3: x_m=x_m*60. ; output is arc-min grazing
                    4: x_m=(90.-x_m)*3600. ; output is arc-sec normal
                    5: x_m=x_m*3600. ; output is arc-sec grazing
                    6: x_m=(90.-x_m)*!dtor*1000. ; output is mrad normal
                    7: x_m=x_m*!dtor*1000. ; output is mrad grazing
                endcase
            end

            2: begin
                ;; (th is actually tth)
                ;; deal with angle units:
                case imd.angleunits_ptr of
                    0: x_m=180.-x_m ; output is degrees normal
                    1: x_m=x_m  ; output is degrees grazing
                    2: x_m=(180.-x_m)*60. ; output is arc-min normal
                    3: x_m=x_m*60. ; output is arc-min grazing
                    4: x_m=(180.-x_m)*3600. ; output is arc-sec normal
                    5: x_m=x_m*3600. ; output is arc-sec grazing
                    6: x_m=(180.-x_m)*!dtor*1000. ; output is mrad normal
                    7: x_m=x_m*!dtor*1000. ; output is mrad grazing
                endcase
            end
        endcase
    endif
endelse

return
end
