;+
; NAME:
;
;      FWHM
;      
; PURPOSE:
; 
;       Interactively measure the full-width-half-max of a region of a
;       curve that has been previously plotted.
;	
; CALLING SEQUENCE:
; 
;	RESULT=FWHM(XAXIS,YAXIS)
;	
; INPUTS:
; 
;	XAXIS - The x axis variable which has been plotted.
;	
;	YAXIS - The y axis variable which has been plotted.
;	
; OPTIONAL INPUT PARAMETERS:
; 
;       RANGE - Vector of subscripts, which refers to the range of X,Y
;               values over which the FWHM is to be determined.  If
;               not supplied, then GET_ROI is used to interactively
;               define the range.  To use FWHM with a non-interactive
;               graphics device, range MUST be supplied.
;	    
; KEYWORD PARAMETERS:
; 
;       YZERO - The zero point level. If not specified, the zero point
;               level is determined from the endpoints of the region
;               of interest of the curve.
;		
;       YHM - The value at which the full-width is computed.
;             Allowable range is 0. to 1. If not specified, .5 is
;             used.
;		
;       INVERT - Set to get width of 'absorption line' rather than
;                'emission line'.
;		
;       NOHIGHLIGHT - Set to inhibit highlighting the region of
;                     interest.
;		
;       H_COLOR - The color index for highlighting the region of
;                 interest. Default is 7 (Yellow.)
;		
;       H_THICK - The thickness for highlighting the region of
;                 interest.
;		
;	NOLABEL - Set to inhibit labelling the fwhm.
;	
;       L_HEADER - String specifying the label header. Default=''.
;	
;       L_COLOR - Color index for the label.
;	
;       L_FORMAT - Format string for label (eg. '(F4.2)').
;	
;       UNITS - String specifying units along x axis, used in label.
;	
;       CHARSIZE - Size of label text.
;	
;       PSYM - PSYM
;   
; OUTPUTS:
;
;        Result - The full-with-half-max of the region of interest 
;                 of the curve, in x-axis data units.
;		
; OPTIONAL OUTPUT PARAMETERS:
; 
;	ROI - The subscripts of the digitized region of interest.
;	
;       FWHM_ROI - The subscripts of the region between the fwhm
;                  points and the max (min) of the function.
;	    
;       LINE_PTS - A 4-element array containing the coordinates of
;                  the line drawn on the plot: [x0,x1,y0,y1]
;	    
;	LABEL - The label for the plot.
;	
;       L_POS - A two element array containing the x,y coordinates
;               of the label, in data coords.
;
; SIDE EFFECTS:
; 
;	TEK_COLOR is used to load in the tektronix colors.  The region of
;       interest of the curve is highlighted. The fwhm is labelled.
;	
; RESTRICTIONS:
; 
;	The data must be plotted prior to calling FWHM.
;	
; PROCEDURE:
; 
;       The user is asked to digitize the endpoints of the region of
;       interest with the mouse.  The region is highlighted, and the
;       fwhm is labelled.
;	
; MODIFICATION HISTORY:
; 
;	D. L. Windt, Bell Laboratories, November 1989
;
;	March 1998 - Removed MANUAL keyword. 
;
;	windt@bell-labs.com
;-

function fwhm,xaxis,yaxis,invert=invert,yzero=yzero,yhm=yhm, $
              roi=roi,fwhm_roi=fwhm_roi,psym=psym,line_pts=line_pts, $
              nohighlight=nohighlight,h_color=h_color,h_thick=h_thick, $
              nolabel=nolabel,manual=manual,units=units, $
              charsize=charsize,l_header=l_header,l_color=l_color,label=label, $
              l_pos=l_pos,l_format=l_format,range=range
on_error,0
if n_params() ne 2 then message,'usage: result=fwhm(x,y)'

; get the region of interest...

if keyword_set(h_color) eq 0 then h_color=7
if keyword_set(h_thick) eq 0 then h_thick=!p.thick
if keyword_set(psym) eq 0 then psym=!p.psym

if keyword_set(range) eq 0 then $
  roi=get_roi(xaxis,yaxis,nohighlight=keyword_set(nohighlight), $
              h_color=h_color,h_thick=h_thick,psym=psym) else roi=range
xroi=float(xaxis(roi))
yroi=float(yaxis(roi))

n_roi=n_elements(roi)		; get number of elements in roi.

; get zero point and 'half-max' point...
if n_elements(yzero) eq 0 then yzero=.5*(yroi(0)+yroi(n_roi-1))
if n_elements(yhm) eq 1 then yhm=(yhm>0.)<1. else yhm=.5

if keyword_set(invert) then begin ; look at inverted fwhm.
    ymin=min(yroi)              ; get min.
    ymax=yzero                  ; get max.
    fwhm_roi=where(yroi le ymax-yhm*(ymax-ymin))
endif else begin                ; look at 'normal' fwhm.
    ymax=max(yroi)              ; get max.
    ymin=yzero                  ; get min.
    fwhm_roi=where(yroi ge (yhm*(ymax-ymin)+ymin))
endelse
n=fwhm_roi(n_elements(fwhm_roi)-1)-fwhm_roi(0)+1
fwhm_roi=findgen(n)+fwhm_roi(0)

; get fwhm in data coordinates.
n_froi=n_elements(fwhm_roi)     ; number of points in fwhm roi.
fwhm=n_froi*(max(xaxis)-min(xaxis))/n_elements(xaxis) ; in data coords.

if keyword_set(nolabel) eq 0 then begin ; label the line...
    if keyword_set(l_color) eq 0 then l_color=1	
    if keyword_set(l_header) eq 0 then l_header=''
    if keyword_set(units) eq 0 then units=''
    if keyword_set(charsize) eq 0 then charsize=1 > !p.charsize

                                ; plot a line across the roi...
    x1=xroi(fwhm_roi(0)) & x2=xroi(fwhm_roi(n_froi-1))
    if keyword_set(invert) then y1=ymax-yhm*(ymax-ymin) $
    else y1=yhm*(ymax-ymin)+ymin
    line_pts=[x1,x2,y1,y1]
    plots,[x1,x2],[y1,y1],color=l_color
    

                                ; make label...
    if keyword_set(l_format) eq 0 then $
      label=l_header+strtrim(string(fwhm),2)+' '+units $
    else label=l_header+strtrim(string(fwhm,format=l_format),2)+' '+units
                                ; make label position and place the label...
;    if keyword_set(manual) then begin
;        make_arrow,label,x=xpos,y=ypos,color=l_color,size=charsize 
;        l_pos=[xpos,ypos]
;    endif else begin
        l_pos=[.1*(x2-x1)+x2,y1]
        xyouts,l_pos(0),l_pos(1),label,color=l_color,size=charsize
;    endelse
endif
fwhm_roi=roi(0)+fwhm_roi

return,fwhm	
end

