/* mount.cc

   Copyright 1996, 1997, 1998 Cygnus Solutions.

This file is part of Cygwin.

This software is a copyrighted work licensed under the terms of the
Cygwin license.  Please consult the file "CYGWIN_LICENSE" for
details. */

#include <stdio.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <mntent.h>
#include <windows.h>
#include <sys/cygwin.h>

static void show_mounts (void);
static void reset_mounts (void);

// static short create_missing_dirs = FALSE;
static short missing_dir_warning = TRUE;

static void
do_mount (const char *dev, const char *where, int flags)
{
  struct stat statbuf;
  char win32_path[MAX_PATH];

  cygwin_conv_to_win32_path (where, win32_path);

  if (stat (win32_path, &statbuf) == -1)
    {
      if (missing_dir_warning == TRUE)
	fprintf (stderr, "Warning: %s does not exist!\n", where);

#if 0
      /* FIXME: this'll fail if mount dir is missing any parent dirs */
      if (create_missing_dirs == TRUE)
	if (mkdir (where, 0755) == -1)
	  fprintf (stderr, "Warning: unable to create %s!\n", where);
#endif
    }
  else if (!(statbuf.st_mode & S_IFDIR))
    {
      if (missing_dir_warning == TRUE)
	fprintf (stderr, "Warning: %s is not a directory!\n", where);
    }    

  if (mount (dev, where, flags))
    {
      perror ("mount failed");
      exit (1);
    }

  exit (0);
}

static const char *progname;

static void
usage (void)
{
  fprintf (stderr, "Usage %s [-bf] <dospath> <unixpath>\n", progname);
  fprintf (stderr, "-b = text files are equivalent to binary files (newline = \\n)\n");
  //  fprintf (stderr, "-c = create missing mount point directories\n");
  fprintf (stderr, "-f = force mount, don't warn about missing mount point directories\n");
  fprintf (stderr, "\n");
  // fprintf (stderr, "-g = add mount point to global registry location\n");
  fprintf (stderr, "To reset the mount table use: mount --reset\n");
  exit(1);
}

int
main (int argc, const char **argv)
{
  int i;
  int flags = 0;

  progname = argv[0];

  if (argc == 1)
    {
      show_mounts ();
      exit (0);
    }

  for (i = 1; i < argc; ++i)
    {
     if (argv[i][0] != '-')
	break;

      if (strcmp (argv[i], "--reset") == 0)
	{
	  reset_mounts ();
	  exit (0);
	}

      if (strcmp (argv[i], "-b") == 0)
	flags |= MOUNT_BINARY;
#if 0
      else if (strcmp (argv[i], "-c") == 0)
	create_missing_dirs = TRUE;
      else if (strcmp (argv[i], "-g") == 0)
        flags |= MOUNT_GLOBAL;
#endif
      else if (strcmp (argv[i], "-f") == 0)
	missing_dir_warning = FALSE;
      else
	usage ();
    }

  if (i + 2 != argc)
    usage ();

  do_mount (argv[i], argv[i + 1], flags);

  /* NOTREACHED */
  return 0;
}

static void
show_mounts (void)
{
  FILE *m = setmntent ("/-not-used-", "r");
  struct mntent *p;
  char *format = "%-16s %-16s    %-10s  %s\n";

  printf (format, "Device", "Directory", "Type", "Flags");
  while ((p = getmntent (m)) != NULL)
    {
      printf (format,
	      p->mnt_fsname,
	      p->mnt_dir,
	      p->mnt_type,
	      p->mnt_opts);
    }
  endmntent (m);
}

static void
reset_mounts (void)
{
  FILE *m = setmntent ("/-not-used-", "r");
  struct mntent *p;

  /* Remove old mount points. */
  /* FIXME: is this really capable of deleting all mounts? */
  while ((p = getmntent (m)) != NULL)
    umount (p->mnt_dir);
  endmntent (m);

  /* Add new ones. */
  /* Slash defaults to the drive letter containing the system directory */
  char drivestring[MAX_PATH];
  GetSystemDirectory (drivestring, MAX_PATH);
  drivestring[2] = 0;   /* truncate drivestring path to "<drive>:" */
  mount (drivestring, "/", MOUNT_SILENT);

  /* No other default mounts. */
}
