/* registry.cc: registry interface

   Copyright 1996, 1997, 1998 Cygnus Solutions.

This file is part of Cygwin.

This software is a copyrighted work licensed under the terms of the
Cygwin license.  Please consult the file "CYGWIN_LICENSE" for
details. */

#include "winsup.h"
#include "registry.h"

char cygnus_class[] = "cygnus";

reg_key::reg_key (HKEY top, REGSAM access, ...)
{
  va_list av;
  va_start (av, access);
  build_reg (top, access, av);
  va_end (av);
}

reg_key::reg_key (REGSAM access, ...)
{
  va_list av;

  new (this) reg_key (HKEY_CURRENT_USER, access, "SOFTWARE",
		 CYGWIN_INFO_CYGNUS_REGISTRY_NAME,
		 CYGWIN_INFO_CYGWIN_REGISTRY_NAME, NULL);

  HKEY top = key;
  va_start (av, access);
  build_reg (top, KEY_READ, av);
  va_end (av);
  if (top != key)
    RegCloseKey (top);
}

reg_key::reg_key (REGSAM access)
{
  new (this) reg_key (HKEY_CURRENT_USER, access, "SOFTWARE",
		 CYGWIN_INFO_CYGNUS_REGISTRY_NAME,
		 CYGWIN_INFO_CYGWIN_REGISTRY_NAME,
		 CYGWIN_INFO_CYGWIN_MOUNT_REGISTRY_NAME, "mounts", NULL);
}

void
reg_key::build_reg (HKEY top, REGSAM access, va_list av)
{
  char *name;
  HKEY r = top;

  while ((name = va_arg (av, char *)) != NULL)
    {
      DWORD disp;
      int res = RegCreateKeyExA (r,
				 name,
				 0,
				 cygnus_class,
				 REG_OPTION_NON_VOLATILE,
				 access,
				 &sec_none_nih,
				 &key,
				 &disp);
      if (r != top)
	RegCloseKey (r);
      r = key;
      if (res != ERROR_SUCCESS)
	{
	  key = (HKEY) INVALID_HANDLE_VALUE;
	  strace_printf ("Failed to create key %s in the registry", name);
	  break;
	}
    }
}

int
reg_key::get_int (const char *name, int def)
{
  DWORD type;
  DWORD dst;
  DWORD size = sizeof (dst);

  LONG res = RegQueryValueExA (key,
			       name,
			       0,
			       &type,
			       (unsigned char *) &dst, &size);

  if (type != REG_DWORD || res != ERROR_SUCCESS)
    return def;

  return dst;
}

int
reg_key::set_int (const char *name, int val)
{
  DWORD value = val;
  return (int) RegSetValueExA (key, name, 0, REG_DWORD,
			       (unsigned char *) &value, sizeof (value));
}

int
reg_key::get_string (const char *name, char *dst, size_t max, const char * def)
{
  DWORD size = max;
  DWORD type;
  LONG res = RegQueryValueExA (key, name, 0, &type, (unsigned char *) dst,
								&size);

  if ((def != 0) && ((type != REG_SZ) || (res != ERROR_SUCCESS)))
    {
      strcpy (dst, def);
    }
  return (int) res;
}

int
reg_key::set_string (const char *name, const char *src)
{
  return (int) RegSetValueExA (key,name, 0, REG_SZ, (unsigned char*) src,
		  strlen (src) + 1);
}

int
reg_key::setone_string (const char *src, const char *name)
{
  return (int) RegSetValueExA (key, name, 0, REG_SZ,
			       (const unsigned char *) src, strlen (src) + 1);
}

#if 0
reg_key::reg_key ()
{
  key = (HKEY)INVALID_HANDLE_VALUE;
}
#endif

HKEY
reg_key::get_key ()
{
  return key;
}

void
reg_key::kill (const char *name)
{
  RegDeleteKeyA (key, name);
}

reg_key::~reg_key ()
{
  if (key != (HKEY) INVALID_HANDLE_VALUE)
    RegCloseKey (key);
  key = (HKEY) INVALID_HANDLE_VALUE;
}
