/* path.cc: path support.

   Copyright 1996, 1997, 1998 Cygnus Solutions.

This file is part of Cygwin.

This software is a copyrighted work licensed under the terms of the
Cygwin license.  Please consult the file "CYGWIN_LICENSE" for
details. */

/* This module's job is to
   - convert between posix and dos style filenames,
   - support the `mount' functionality,
   - support symlinks for files and directories

   Pathnames are handled as follows:

   - / is equivalent to \
   - Paths beginning with //<letter> (or \\<letter>) are translated to
   <letter>:  (i.e. This is how drives are specified.  According to
   xxx this is how Microsoft does it in the NT Posix subsystem).
   - Paths beginning with // (or \\) are otherwise not translated
   (i.e. looked up in the mount table or treated as drive specifiers).
   - Paths containing a : are not translated
   (paths like /foo/bar/har:har: don't make much sense but having the rule
   written this way allows one to use strchr).

   The goal in the above set of rules is to allow both posix and win32 flavors
   without either interfering.  The rules are intended to be as close to a
   superset of both as possible.

   A possible future enhancement would be to allow people to disable/enable
   the mount table and //<letter> handling to support pure win32 pathnames.
   Hopefully this won't be needed.  The suggested way to do this would be
   an env var because
   a) we need something that is inherited from parent to child,
   b) environment variables can be passed from the DOS shell to a cygwin app,
   c) it allows disabling the feature on an app by app basis within the
   same session (whereas playing about with the registry wouldn't - without
   getting too complicated).  Example: CYGWIN=pathrules[=@]{win32,posix}.
   If CYGWIN=pathrules=win32, mount table and //<letter> handling are
   disabled.  [The intent is to have CYGWIN be a catchall for tweaking
   various cygwin.dll features].

   Note that you can have more than one path to a file.  The mount table is
   always prefered when translating win32 paths to posix paths.  Mount table
   entries may not begin with //.

   Text vs Binary issues are not considered here in path style descisions.

   / and \ are treated as equivalent.  One or the other is prefered in certain
   situations (e.g. / is preferred in result of getcwd, \ is preferred in
   arguments to win32 api calls), but this code will translate as necessary.

   Apps wishing to translate to/from pure win32 and posix-like pathnames
   can use cygwin_foo.

   Removing mounted filesystem support would simplify things greatly, but
   having it gives us a mechanism of treating disk that lives on a unix machine
   as having unix semantics [it allows one to edit a text file on that disk
   and not have cr's magically appear and perhaps break apps running on unix
   boxes].  It also useful to be able to layout a hierarchy without changing
   the underlying directories.

   The semantics of mounting file systems is not intended to precisely follow
   normal unix systems.

   Each DOS drive is defined to have a current directory.
   Supporting this would complicate things so for now things are defined so
   that c: means c:\.
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/mount.h>
#include <mntent.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "winsup.h"
#include "registry.h"

static int symlink_check_one (const char *path, char *buf, int buflen,
			      DWORD& fileattr, int *exec,
			      const suffix_info *suffixes,
			      char *&found_suffix);
static int normalize_win32_path (const char *cwd, const char *src, char *dst);
static char *getcwd_inner (char *buf, size_t ulen, int posix_p);
static void slashify (const char *src, char *dst, int trailing_slash_p);
static void backslashify (const char *src, char *dst, int trailing_slash_p);
static void nofinalslash (const char *src, char *dst);
static int path_prefix_p_ (const char *path1, const char *path2, int len1);
static int slash_drive_prefix_p (const char *path);
static int slash_unc_prefix_p (const char *path);
static char *build_slash_drive_prefix (const char *path, char *buf);
static void slash_drive_to_win32_path (const char *path, char *buf,
				       int trailing_slash_p);

static NO_COPY const char escape_char = '^';

#define path_prefix_p(p1, p2, l1) \
       ((tolower(*(p1))==tolower(*(p2))) && \
       path_prefix_p_(p1, p2, l1))

#define SYMLINKATTR(x) \
  (((x) & (FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_DIRECTORY)) == \
   FILE_ATTRIBUTE_SYSTEM)

/* Return non-zero if P, a pure WIN32 path, points to a directory mounted
   with -b [binary mode].  */

int
mount_info::binary_win32_path_p (const char *p)
{
  for (int i = 0; i < nmounts; i++)
    {
      if (path_prefix_p (mount[i].device, p, mount[i].devicelen))
	return mount[i].binary;
    }
  return 0;
}

/* Return non-zero if PATH1 is a prefix of PATH2.
   Both are assumed to be of the same path style and / vs \ usage.
   Neither may be "".
   LEN1 = strlen (PATH1).  It's passed because often it's already known.

   Examples:
   /foo/ is a prefix of /foo  <-- may seem odd, but desired
   /foo is a prefix of /foo/
   / is a prefix of /foo/bar
   / is not a prefix of foo/bar
   foo/ is a prefix foo/bar
   /foo is not a prefix of /foobar
*/

static int
path_prefix_p_ (const char *path1, const char *path2, int len1)
{
  /* Handle case where PATH1 has trailing '/' and when it doesn't.  */
  if (SLASH_P (path1[len1 - 1]))
    len1--;

  if (len1 == 0)
    return SLASH_P (path2[0]) && !SLASH_P (path2[1]);

  if (!strncasematch (path1, path2, len1))
    return 0;

  return SLASH_P (path2[len1]) || path2[len1] == 0;
}

/* Convert an arbitrary path SRC to a pure WIN32 path, suitable for passing to
   Win32 API routines.

   Whether PATH is interpreted using WIN32 or Posix Rules is specified by a
   call to posix_path_p.

   If an error occurs, `error' is set to the errno value.
   Otherwise it is set to 0.

   follow_mode values:
	SYMLINK_FOLLOW	    - convert to PATH symlink points to
	SYMLINK_NOFOLLOW    - convert to PATH of symlink itself
	SYMLINK_IGNORE	    - do not check PATH for symlinks
	SYMLINK_CONTENTS    - just return symlink contents
*/

path_conv::path_conv (const char *src, symlink_follow follow_mode,
		      int use_full_path, const suffix_info *suffixes)
{
  /* This array is used when expanding symlinks.  It is MAX_PATH * 2
     in length so that we can hold the expanded symlink plus a
     trailer.  */
  char work_buf[MAX_PATH * 3 + 3];
  char tmp_buf[MAX_PATH];
  char path_buf[MAX_PATH];

  char *rel_path, *full_path;

  if ((error = check_null_empty_path (src)))
    return;

  if (use_full_path)
    rel_path = path_buf, full_path = this->path;
  else
    rel_path = this->path, full_path = path_buf;


  char *sym_buf = work_buf + MAX_PATH + 1;
  /* This loop handles symlink expansion.  */
  int loop = 0;
  symlink_p = 0;
  exec_p = 0;
  silent_p = 0;
  binary_p = 0;
  known_suffix = NULL;
  fileattr = (DWORD)-1;
  for (;;)
    {

      /* Must look up path in mount table, etc.  */

      error = cygwin_shared->mount.conv_to_win32_path (src, rel_path,
						       full_path,
						       devn, unit);
      if (error != 0)
	return;
      if (devn == FH_BAD)
	binary_p = cygwin_shared->mount.binary_win32_path_p (full_path);
      else
	{
	  binary_p = TRUE;		/* FIXME: probably not right */
	  fileattr = 0;
	  return;
	}

      /* Eat trailing slashes */
      char *tail = strchr (full_path, '\0');
      while (tail > full_path && (*--tail == '\\'))
	*tail = '\0';

      if (follow_mode == SYMLINK_IGNORE)
	{
	  fileattr = GetFileAttributesA (path);
	  return;
	}

      /* Make a copy of the path that we can munge up */
      char path_copy[strlen (full_path) + 2];
      strcpy (path_copy, full_path);

      tail = path_copy + 1 + (tail - full_path);	// Point to end of copy

      *sym_buf = '\0';			// Paranoid

      /* Scan path_copy from right to left looking either for a symlink
	 or an actual existing file.  If an existing file is found, just
	 return.  If a symlink is found exit the for loop.
	 Also: be careful to preserve the errno returned from
	 symlink_check_one as the caller may need it. */
      /* FIXME: Do we have to worry about multiple \'s here? */
      int component = 0;		// Number of translated components
      DWORD attr;
      for (;;)
	{
	  save_errno s (0);
	  int is_ex;
	  int *ep = component ? &is_ex : &exec_p;
	  const suffix_info *suff;
	  /* Don't allow symlink_check_one to set anything in the path_conv
	     class if we're working on an inner component of the path */
	  if (component)
	    {
	      ep = &is_ex;
	      suff = NULL;
	    }
	  else
	    {
	      ep = &exec_p;
	      suff = suffixes;
	    }
	  int len = symlink_check_one (path_copy, sym_buf, MAX_PATH, attr,
				       ep, suff, known_suffix);

	  /* If symlink_check_one found an existing non-symlink file, then
	     it returns a length of 0 and sets errno to EINVAL.  It also sets
	     any suffix found into `sym_buf'. */
	  if (!len && get_errno () == EINVAL)
	    {
	      if (component == 0)
		{
		  fileattr = attr;
		  if (follow_mode == SYMLINK_CONTENTS)
		    return;
		  else if (*sym_buf)
		    {
		      known_suffix = strchr (this->path, '\0');
		      strcpy (known_suffix, sym_buf);
		    }
		  else if (known_suffix)
		    known_suffix = this->path + (known_suffix - path_copy);
		}
	      return;			// file found
	    }
	  /* Found a symlink if len > 0.  If component == 0, then the
	     src path itself was a symlink.  If !follow_mode then
	     we're done.  Otherwise we have to insert the path found
	     into the full path that we are building and perform all of
	     these operations again on the newly derived path. */
	  else if (len > 0)
	    {
	      if (component == 0)
		{
		  if (follow_mode != SYMLINK_FOLLOW)
		    {
		      symlink_p = 1;	// last component of path was a symlink.
		      fileattr = attr;
		      if (follow_mode == SYMLINK_CONTENTS)
			  strcpy (path, sym_buf);
		      return;
		    }
		}
	      break;
	    }

	  s.reset ();      // remember errno from symlink_check_one

	  if (!(tail = strrchr (path_copy, '\\')) ||
	      (tail > path_copy && tail[-1] == ':'))
	    return;        // all done

	  /* Haven't found a valid pathname component yet.
	     Pinch off the tail and try again. */
	  *tail = '\0';
	  component++;
	}

      /* Arrive here if above loop detected a symlink. */

      if (++loop > MAX_LINK_DEPTH)
	{
	  error = ELOOP;   // Eep.
	  return;
	}

      tail = full_path + (tail - path_copy);
      int taillen = strlen (tail);
      int buflen = strlen (sym_buf);
      if (buflen + taillen > MAX_PATH)
	  {
	    error = ENAMETOOLONG;
	    strcpy (path, "::ENAMETOOLONG::");
	    return;
	  }

      /* Copy tail of full_path to discovered symlink. */
      char *p;
      for (p = sym_buf + buflen; *tail; tail++)
	*p++ = *tail == '\\' ? '/' : *tail;
      *p = '\0';

      /* If symlink referred to an absolute path, then we
	 just use sym_buf and loop.
	 Otherwise tack the head of path_copy before sym_buf
	 and translate it back from MS-DOS mode to POSIX. */
      if (isabspath (sym_buf))
	src = sym_buf;
      else if (!(tail = strrchr (path_copy, '\\')))
	system_printf ("problem parsing %s - '%s'", src, full_path);
      else
	{
	  int headlen = 1 + tail - path_copy;
	  p = sym_buf - headlen;
	  memcpy (p, path_copy, headlen);
	  error = cygwin_shared->mount.conv_to_posix_path (p, tmp_buf, 1);
	  if (error)
	    break;
	  src = tmp_buf;
	}
    }
}

#define deveq(s) (strcasematch (name, (s)))
#define deveqn(s, n) (strncasematch (name, (s), (n)))

static __inline int
digits (const char *name)
{
  char *p;
  int n = strtol(name, &p, 10);

  return p > name && !*p ? n : -1;
}

const char *windows_device_names[] =
{
  NULL,
  "\\dev\\console",
  "conin",
  "conout",
  "\\dev\\ttym",
  "\\dev\\tty%d",
  "\\dev\\ptym",
  "com%d",
  "\\dev\\pipe",
  "\\dev\\piper",
  "\\dev\\pipew",
  "\\dev\\socket",
  "\\dev\\windows",

  NULL, NULL, NULL,

  "\\dev\\disk",
  "\\dev\\fd%d",
  "\\dev\\st%d",
  "nul",
};

int
get_device_number (const char *name, int &unit)
{

  DWORD devn = FH_BAD;
  unit = 0;

  if ((*name == '/' && deveqn ("/dev/", 5)) ||
      (*name == '\\' && deveqn ("\\dev\\", 5)))
    {
      name += 5;
      if (deveq ("tty"))
	{
	  if (NOTSTATE (myself, PID_USETTY) && myself->ctty < 0)
	    devn = FH_CONSOLE;
	  else if (myself->ctty >= 0)
	    {
	      unit = myself->ctty;
	      devn = FH_TTYS;
	    }
	}
      else if (deveqn ("tty", 3) && (unit = digits (name + 3)) >= 0)
	devn = FH_TTYS;
      else if (deveq ("ttym"))
	devn = FH_TTYM;
      else if (deveq ("ptmx"))
	devn = FH_PTYM;
      else if (deveq ("windows"))
	devn = FH_WINDOWS;
      else if (deveq ("conin"))
	devn = FH_CONIN;
      else if (deveq ("conout"))
	devn = FH_CONOUT;
      else if (deveq ("null"))
	devn = FH_NULL;
      else if (deveqn ("fd", 2) && (unit = digits (name + 2)) >= 0)
	devn = FH_FLOPPY;
      else if (deveqn ("st", 2) && (unit = digits (name + 2)) >= 0)
	devn = FH_TAPE;
      else if (deveqn ("com", 3) && (unit = digits (name + 3)) >= 0)
	devn = FH_SERIAL;
      else if (deveq ("pipe") || deveq ("piper") || deveq ("pipew"))
	devn = FH_PIPE;
      else if (deveq ("tcp") || deveq ("udp") || deveq ("streamsocket")
	       || deveq ("dgsocket"))
	devn = FH_SOCKET;
    }
  else if (deveqn ("com", 3) && (unit = digits (name + 3)) >= 0)
    devn = FH_SERIAL;

  return devn;
}

BOOL
win32_device_name (const char *src_path, char *win32_path,
		   DWORD &devn, int &unit)
{
  const char *devfmt;

  devn = get_device_number (src_path, unit);

  if (devn == FH_BAD)
    return FALSE;

  if ((devfmt = windows_device_names[FHDEVN (devn)]) == NULL)
    return FALSE;
  __small_sprintf (win32_path, devfmt, unit);
  return TRUE;
}

/* Normalize a posix path.
   \'s are converted to /'s in the process.
   All duplicate /'s, except for 2 leading /'s, are deleted.
   The result is 0 for success, or an errno error value.  */

static __inline int
normalize_posix_path (const char *cwd, const char *src, char *dst)
{
  const char *src_start = src;
  char *dst_start = dst;

  if (! SLASH_P (src[0]))
    {
      if (strlen (cwd) + 1 + strlen (src) >= MAX_PATH)
	{
	  debug_printf ("ENAMETOOLONG = normalize_posix_path (%s)", src);
	  return ENAMETOOLONG;
	}
      strcpy (dst, cwd);
      dst = strchr (dst, '\0');
      if (dst > (dst_start + 1))
	*dst++ = '/';
    }
  /* Two leading /'s?  If so, preserve them.  */
  else if (SLASH_P (src[1]))
    {
      *dst++ = '/';
      *dst++ = '/';
      src += 2;
      if(SLASH_P(*src))
	{ /* Starts with three or more slashes - reset. */
	  dst = dst_start;
	  *dst++ = '/';
	  src = src_start + 1;
	}
    }

  while (*src)
    {
      /* Strip runs of /'s.  */
      if (SLASH_P (*src))
	{
	  *dst++ = '/';
	  src++;
	  while(SLASH_P(*src))
	    src++;
	}
      /* Ignore "./".  */
      else if (src[0] == '.' && SLASH_P (src[1])
	       && (src == src_start || SLASH_P (src[-1])))
	{
	  src += 2;
	  while(SLASH_P(src[0]))
	    src++;
	}
      /* Backup if "..".  */
      else if (src[0] == '.' && src[1] == '.'
	       /* dst must be greater than dst_start */
	       && isdirsep(dst[-1])
	       && (SLASH_P (src[2]) || src[2] == 0))
	{
	  /* Back up over /, but not if it's the first one.  */
	  if (dst > dst_start + 1)
	    dst--;
	  /* Now back up to the next /.  */
	  while (dst > dst_start + 1 && !isdirsep (dst[-1]))
	    dst--;
	  src += 2;
	  while (SLASH_P (*src))
	    src++;
	}
      /* Otherwise, add char to result.  */
      else
	{
	  if (*src == '\\')
	    *dst++ = '/';
	  else
	    *dst++ = *src;
	  ++src;
	}
    }
  *dst = 0;
  debug_printf ("%s = normalize_posix_path (%s)", dst_start, src_start);
  return 0;
}

/* Ensure SRC_PATH is a pure WIN32 path and store the result in WIN32_PATH.

   If WIN32_PATH != NULL, the relative path, if possible to keep, is
   stored in WIN32_PATH.  If the relative path isn't possible to keep,
   the full path is stored.

   If FULL_WIN32_PATH != NULL, the full path is stored there.

   The result is zero for success, or an errno value.

   {,FULL_}WIN32_PATH must have sufficient space (i.e. MAX_PATH bytes).  */

int
mount_info::conv_to_win32_path (const char *src_path, char *win32_path,
				char *full_win32_path, DWORD &devn, int &unit)
{
  int src_path_len = strlen (src_path);
  int trailing_slash_p = (src_path_len > 0
			  && SLASH_P (src_path[src_path_len - 1]));

  debug_printf ("conv_to_win32_path (%s)", src_path);

  if (src_path_len >= MAX_PATH)
    {
      debug_printf ("ENAMETOOLONG = conv_to_win32_path (%s)", src_path);
      return ENAMETOOLONG;
    }

  if (win32_device_name (src_path, full_win32_path ?: win32_path, devn, unit))
    return 0;

  /* The rule is :'s can't appear in [our] posix path names so this is a safe
     test; if ':' is present it already be in win32 form.  */
  if (strchr (src_path, ':') != NULL)
    {
      debug_printf ("%s already win32", src_path);
      if (win32_path != NULL)
	backslashify (src_path, win32_path, trailing_slash_p);
      if (full_win32_path != NULL)
	backslashify (src_path, full_win32_path, trailing_slash_p);
      return 0;
    }

  /* Normalize the path, taking out ../../ stuff, we need to do this
     so that we can move from one mounted directory to another with relative
     stuff.

     eg mounting c:/foo /foo
     d:/bar /bar

     cd /bar
     ls ../foo

     should look in c:/foo, not d:/foo.

     We do this by first getting an absolute Unix-style path and then
     converting it to a DOS-style path, looking up the appropriate drive
     in the mount table.  */

  char pathbuf[MAX_PATH];
  char cwd[MAX_PATH];

  /* No need to fetch cwd if path is absolute.  */
  if (! SLASH_P (*src_path))
    getcwd_inner (cwd, MAX_PATH, 1); /* FIXME: check rc */
  else
    strcpy (cwd, "/"); /* some innocuous value */
  int rc = normalize_posix_path (cwd, src_path, pathbuf);
  if (rc != 0)
    {
      debug_printf ("%d = conv_to_win32_path (%s)", rc, src_path);
      return rc;
    }
  nofinalslash (pathbuf, pathbuf);

  for (int i = 0; i < nmounts; ++i)
    {
      if (! path_prefix_p (mount[i].path, pathbuf, mount[i].pathlen))
	continue;

      /* SRC_PATH is in the mount table.  */

      /* Compute relative path if asked to and able to.  */
      int got_rel_p = 0;
      if (win32_path != NULL
	  && ! SLASH_P (*src_path)
	  /* Ensure the path didn't ../ out of cwd.
	     There are times when we could keep ../foo as is, but
	     normalize_posix_path strips all ../'s and it's [currently] too
	     much work to deal with them.  */
	  && path_prefix_p (cwd, pathbuf, strlen (cwd)))
	{
	  if (path_prefix_p (cwd, mount[i].path, strlen (cwd)))
	    {
	      if (strcmp (pathbuf, mount[i].path) == 0)
		strcpy (win32_path, ".");
	      else
		{
		  int n = strlen (cwd);
		  if (n > 1)
		    n++;
		  strcpy (win32_path, pathbuf + n);
		}
	    }
	  else
	    {
	      /* must be path_prefix_p (mount[i].path, cwd,
						  mount[i].pathlen) */
	      if (strcmp (pathbuf, cwd) == 0)
		strcpy (win32_path, ".");
	      else
		strcpy (win32_path, pathbuf + strlen (cwd) + 1);
	    }
	  backslashify (win32_path, win32_path, trailing_slash_p);
	  got_rel_p = 1;
	}

      /* If we've matched / and the path begins with the slash drive prefix,
	 or the UNC path prefix, break out.  If the slash drive path was
	 mounted, it would have matched already.  */
      if (! got_rel_p
	  && strcmp (mount[i].path, "/") == 0
	  && (slash_drive_prefix_p (pathbuf) || slash_unc_prefix_p(pathbuf)))
	break;

      /* Compute the full win32 path.
	 We go to some lengths to avoid unnecessary copying.  */
      char *p = NULL;
      if (full_win32_path != NULL)
	p = full_win32_path;
      else if (win32_path != NULL && ! got_rel_p)
	p = win32_path;
      if (p != NULL)
	{
	  int j = mount[i].devicelen;
	  memcpy (p, mount[i].device, j);
	  /* Do not add trailing \ to UNC device names like \\.\a: */
	  if (pathbuf[mount[i].pathlen] != '/' &&
	     !(strncmp (mount[i].device, "\\\\.\\", 4) == 0 &&
	       strncmp (mount[i].device + 4, "UNC\\", 4) != 0))
	    p[j++] = '\\';
	  strcpy (p + j, pathbuf + mount[i].pathlen);
	  backslashify (p, p, trailing_slash_p);
	}
      if (win32_path != NULL && ! got_rel_p && win32_path != p)
	strcpy (win32_path, p);

      debug_printf ("%s = conv_to_win32_path (%s)",
		    win32_path != NULL ? win32_path : full_win32_path,
		    src_path);
      return 0;
    }

  char *p;
  if (full_win32_path != NULL)
    p = full_win32_path;
  else
    p = win32_path;

  /* Not in the database.  Is it a //<letter> path?  */
  if (slash_drive_prefix_p (pathbuf))
    {
      slash_drive_to_win32_path (pathbuf, p, trailing_slash_p);
    }
  else
    {
      /* Can't translate it.  Just ensure no /'s are present.
	 Note that this should [theoretically] only happen if / isn't in
	 the data base, but that's ok [I think].
	 The use of posix_path and not pathbuf here is intentional.
	 We couldn't translate the path, so just ensure no \'s are present. */
      backslashify (src_path, p, trailing_slash_p);
    }
  if (win32_path != NULL && win32_path != p)
    strcpy (win32_path, p);

  debug_printf ("%s = conv_to_win32_path (%s)",
		win32_path != NULL ? win32_path : full_win32_path,
		src_path);
  return 0;
}

/* Ensure SRC_PATH is a POSIX_PATH.
   The result is zero for success, or an errno value.
   POSIX_PATH must have sufficient space (i.e. MAX_PATH bytes).
   If KEEP_REL_P is non-zero, relative paths stay that way.  */

int
mount_info::conv_to_posix_path (const char *src_path, char *posix_path,
				int keep_rel_p)
{
  int src_path_len = strlen (src_path);
  int trailing_slash_p = (src_path_len > 0
			  && SLASH_P (src_path[src_path_len - 1]));
  int relative_path_p = (! SLASH_P (*src_path)
			 && strchr (src_path, ':') == NULL);

  debug_printf ("conv_to_posix_path (%s, %s)", src_path,
		keep_rel_p ? "keep-rel" : "no-keep-rel");

  if (src_path_len >= MAX_PATH)
    {
      debug_printf ("ENAMETOOLONG");
      return ENAMETOOLONG;
    }

  /* FIXME: For now, if the path is relative and it's supposed to stay
     that way, skip mount table processing.  */
  if (keep_rel_p
      && relative_path_p)
    {
      slashify (src_path, posix_path, 0);
      debug_printf ("%s = conv_to_posix_path (%s)", posix_path, src_path);
      return 0;
    }

  char pathbuf[MAX_PATH];
  char cwd[MAX_PATH];

  /* No need to fetch cwd if path is absolute.  */
  if (relative_path_p)
    getcwd_inner (cwd, MAX_PATH, 0); /* FIXME: check rc */
  else
    strcpy (cwd, "/"); /* some innocuous value */
  int rc = normalize_win32_path (cwd, src_path, pathbuf);
  if (rc != 0)
    {
      debug_printf ("%d = conv_to_posix_path (%s)", rc, src_path);
      return rc;
    }
  nofinalslash (pathbuf, pathbuf);

  for (int i = 0; i < nmounts; ++i)
    {
      if (! path_prefix_p (mount[i].device, pathbuf, mount[i].devicelen))
	continue;

      /* SRC_PATH is in the mount table.  */

      if (mount[i].pathlen + strlen (pathbuf) - mount[i].devicelen >= MAX_PATH)
	return ENAMETOOLONG;
      strcpy (posix_path, mount[i].path);
      if (pathbuf[mount[i].devicelen])
	slashify (pathbuf + mount[i].devicelen,
		posix_path + (mount[i].pathlen == 1 ? 0 : mount[i].pathlen),
		trailing_slash_p);
      debug_printf ("%s = conv_to_posix_path (%s)", posix_path, src_path);
      return 0;
    }

  /* Not in the database.  This should [theoretically] only happen if either
     the path begins with //, or / isn't mounted, or the path has a drive
     letter not covered by the mount table.  If it's a relative path then the
     caller must want an absolute path (otherwise we would have returned
     above).  So we always return an absolute path at this point.  */
  if (isalpha (pathbuf[0]) && pathbuf[1] == ':')
    {
      slashify (pathbuf + 2,
		build_slash_drive_prefix (pathbuf, posix_path),
		trailing_slash_p);
    }
  else
    {
      /* The use of src_path and not pathbuf here is intentional.
	 We couldn't translate the path, so just ensure no \'s are present.  */
      slashify (src_path, posix_path, trailing_slash_p);
    }

  debug_printf ("%s = conv_to_posix_path (%s)", posix_path, src_path);
  return 0;
}

/* Normalize a win32 path.
   /'s are converted to \'s in the process.
   All duplicate \'s, except for 2 leading \'s, are deleted.

   The result is 0 for success, or an errno error value.
   FIXME: A lot of this should be mergeable with the posix critter.  */

static int
normalize_win32_path (const char *cwd, const char *src, char *dst)
{
  const char *src_start = src;
  char *dst_start = dst;

  if (! SLASH_P (src[0])
      && strchr (src, ':') == NULL)
    {
      if (strlen (cwd) + 1 + strlen (src) >= MAX_PATH)
	{
	  debug_printf ("ENAMETOOLONG = normalize_win32_path (%s)", src);
	  return ENAMETOOLONG;
	}
      strcpy (dst, cwd);
      dst += strlen (dst);
      *dst++ = '\\';
    }
  /* Two leading \'s?  If so, preserve them.  */
  else if (SLASH_P (src[0]) && SLASH_P (src[1]))
    {
      *dst++ = '\\';
      ++src;
    }

  while (*src)
    {
      /* Strip duplicate /'s.  */
      if (SLASH_P (src[0]) && SLASH_P (src[1]))
	src++;
      /* Ignore "./".  */
      else if (src[0] == '.' && SLASH_P (src[1])
	       && (src == src_start || SLASH_P (src[-1])))
	{
	  src += 2;
	}
#if 0
      /* Strip trailing "/.".  */
      else if (src[0] == '.' && src[1] == 0
	       /* dst must be greater than dst_start */
	       && dst[-1] == '\\')
	{
	  /* Only strip ".", not "\.", if path is only "\.".  */
	  if (dst - 1 > dst_start)
	    dst--;
	  src++;
	}
#endif
      /* Backup if "..".  */
      else if (src[0] == '.' && src[1] == '.'
	       /* dst must be greater than dst_start */
	       && dst[-1] == '\\'
	       && (SLASH_P (src[2]) || src[2] == 0))
	{
	  /* Back up over /, but not if it's the first one.  */
	  if (dst > dst_start + 1)
	    dst--;
	  /* Now back up to the next /.  */
	  while (dst > dst_start + 1 && dst[-1] != '\\' && dst[-2] != ':')
	    dst--;
	  src += 2;
	  if (SLASH_P (*src))
	    src++;
	}
      /* Otherwise, add char to result.  */
      else
	{
	  if (*src == '/')
	    *dst++ = '\\';
	  else
	    *dst++ = *src;
	  ++src;
	}
    }
  *dst = 0;
  debug_printf ("%s = normalize_win32_path (%s)", dst_start, src_start);
  return 0;
}

/* Build the entire mount table from the registry.  */

static
void
read_mounts (reg_key& key, REGSAM myaccess)
{
  for (int i = 0; i < MAX_MOUNTS; i++)
    {
      char key_name[10];
      char win32path[MAX_PATH];
      char unixpath[MAX_PATH];

      __small_sprintf (key_name, "%02x", i);

      reg_key k (key.get_key (), myaccess, key_name, NULL);

      /* The registry names are historical but useful so are left alone.  */
      k.get_string ("native", win32path, sizeof (win32path), "");
      k.get_string ("unix", unixpath, sizeof (unixpath), "");

      /* Does this entry contain something?  */
      if (*win32path != 0)
	{
	  int flags = 0;
	  if (k.get_int ("fbinary", 0))
	    flags |= MOUNT_BINARY;
	  if (k.get_int ("fsilent", 0))
	    flags |= MOUNT_SILENT;

	  cygwin_shared->mount.add_item (win32path, unixpath, flags);
	}
    }
}

void
mount_info::from_registry ()
{
  reg_key r;

  nmounts = 0;

  read_mounts (r, KEY_ALL_ACCESS);

  reg_key r1 (HKEY_LOCAL_MACHINE, KEY_READ, "SOFTWARE",
	      CYGWIN_INFO_CYGNUS_REGISTRY_NAME,
	      CYGWIN_INFO_CYGWIN_REGISTRY_NAME,
	      CYGWIN_INFO_CYGWIN_MOUNT_REGISTRY_NAME,
	      "mounts", NULL);
  read_mounts (r1, KEY_READ);

  sort ();
}

/* Store the mount table in the registry.  */
/* ??? We always read/write the mount table en masse, rather than only
   the entries that changed.  One benefit of doing things this way is that the
   registry will get cleaned up on a regular basis (say if someone deletes
   entry 02), but we may need to reassess the cost.  */
/* FIXME: Need a mutex to avoid collisions with other tasks.  */

void
mount_info::to_registry ()
{
  reg_key wrapper (KEY_ALL_ACCESS, CYGWIN_INFO_CYGWIN_MOUNT_REGISTRY_NAME, "mounts", NULL);

  for (int i = 0; i < MAX_MOUNTS; i++)
    {
      char key_name[10];

      __small_sprintf (key_name, "%02x", i);

      if (i < nmounts)
	{
	  mount_item *p = mount + i;
	  debug_printf ("%02x: %s, %s, %d, %d",
			i, p->device, p->path, p->binary, p->silent);
	  reg_key k = reg_key (wrapper.get_key (), KEY_ALL_ACCESS, key_name, NULL);
	  k.set_string ("native", p->device);
	  k.set_string ("unix", p->path);
	  k.set_int ("fbinary", p->binary);
	  k.set_int ("fsilent", p->silent);
	}
      else
	{
	  /* Mount entry doesn't exist.  Ensure it doesn't in the
	     registry either.  */
	  debug_printf ("deleting '%s'", key_name);
	  wrapper.kill (key_name);
	}
    }
}

/* Initialize the mount table.  */

void
mount_info::init ()
{
  nmounts = 0;

  /* Fetch the mount table from the registry.  */
  from_registry ();

  /* Add on defaults.  If the registry overrides these, these will fail
     with "already mounted".  */

  /* Slash defaults to the drive letter containing the system directory */
  char drivestring[MAX_PATH];
  GetSystemDirectory (drivestring, MAX_PATH);
  drivestring[2] = 0;   /* truncate drivestring path to "<drive>:" */
  add_item (drivestring, "/", 0);

  /* Reinitialize the registry.  */
  to_registry ();
}

struct mntent *
mount_item::getmntent ()
{
  static NO_COPY struct mntent ret;

  ret.mnt_fsname = device;
  ret.mnt_dir = path;
  ret.mnt_type = "native";

  /* We don't print `silent' here.  It's some magical internal thing.  */
  /* FIXME: The wording of mnt_opts needs to be more user friendly.  */
  if (binary)
    ret.mnt_opts = "text=binary";
  else
    ret.mnt_opts = "text!=binary";

  ret.mnt_freq = 1;
  ret.mnt_passno = 1;
  return &ret;
}

struct mntent *
mount_info::getmntent (int x)
{
  if (x < 0 || x >= nmounts)
    return NULL;

  return mount[x].getmntent ();
}

/* Fill in the fields of a mount table entry.  */

void
mount_item::init (const char *dev, const char *mountpoint, int flags)
{
  strcpy ((char *) device, dev);
  strcpy ((char *) path, mountpoint);

  devicelen = strlen (device);
  pathlen = strlen (path);
  binary = (flags & MOUNT_BINARY) != 0;
  silent = (flags & MOUNT_SILENT) != 0;

}

/* qsort callback to sort the mount entries */

static int
sort_by_name (const void *a, const void *b)
{
  mount_item *ap = (mount_item *) a;
  mount_item *bp = (mount_item *) b;

  return strcmp (bp->path, ap->path);
}

void
mount_info::sort ()
{
  /* Sort them into reverse length order, otherwise we won't
     be able to look for /foo in /.  */

  qsort (mount, nmounts, sizeof (mount[0]), sort_by_name);
}

/* Add an entry to the mount table.
   Returns 0 on success, -1 on failure and errno is set.

   This is where all argument validation is done.  It may not make sense to
   do this when called internally, but it's cleaner to keep it all here.  */

int
mount_info::add_item (const char *dev, const char *path, int flags)
{
  if (nmounts == MAX_MOUNTS)
    {
      set_errno (EMFILE);
      return -1;
    }

  /* Some argument validation.  */
  if (*path == 0)
    {
      set_errno (EINVAL);
      return -1;
    }

  /* Make sure both paths do not end in /.  */

  char devtmp[MAX_PATH];
  char pathtmp[MAX_PATH];

  backslashify (dev, devtmp, 0);
  nofinalslash (devtmp, devtmp);

  slashify (path, pathtmp, 0);
  nofinalslash (pathtmp, pathtmp);

  debug_printf ("%s[%s], %s[%s], %p",
		dev, devtmp, path, pathtmp, flags);

  /* Duplicate /'s in PATH are an error.  */
  for (char *p = pathtmp + 1; *p; ++p)
    {
      if (p[-1] == '/' && p[0] == '/')
	{
	  set_errno (EINVAL);
	  return -1;
	}
    }

  /* Is the path already mounted?  */
  for (int i = 0; i < nmounts; i++)
    {
      if (strcmp (mount[i].path, pathtmp) == 0)
	{
	  set_errno (EBUSY);
	  return -1;
	}
    }

  mount[nmounts++].init (devtmp, pathtmp, flags);
  sort ();
  return 0;
}

/* Delete a mount table entry where path is either a DOS or Unix name.
   Since the mount table is really just a table of aliases, deleting /
   is ok (although running without a slash mount is strongly discouraged
   because some programs may run erratically without one).  */

int
mount_info::del_item (const char *path)
{
  char pathtmp[MAX_PATH];

  slashify (path, pathtmp, 0);
  nofinalslash (pathtmp, pathtmp);

  for (int i = 0; i < nmounts; i++)
    {
      if (strcmp (mount[i].path, pathtmp) == 0
	  || strcmp (mount[i].device, pathtmp) == 0)
	{
	  /* Sort the names, which will put the new empty one to the end.
	     This is rather inefficient, but it's simple.  */
	  mount[i].init ("", "", 0);
	  sort ();
	  --nmounts;
	  return 0;
	}
    }
  set_errno (EINVAL);
  return -1;
}

/* Cache getcwd value.  FIXME: We need a lock for these in order to
   support multiple threads.  */

static char *current_directory_name;
static char *current_directory_posix_name;
static unsigned long current_directory_hash;

static int
get_current_directory_name ()
{
  DWORD dlen, len;

  for (dlen = 256; ; dlen *= 2)
    {
      current_directory_name = (char *) realloc (current_directory_name, dlen);
      if ((len = GetCurrentDirectoryA (dlen, current_directory_name)) < dlen)
	break;
    }

  if (len == 0)
    __seterrno ();
  else
    current_directory_hash = hash_path_name (0, current_directory_name);

  return len;
}

/* getcwd */

static char *
getcwd_inner (char *buf, size_t ulen, int posix_p)
{
  char *resbuf = NULL;
  size_t len = ulen;

  if (current_directory_name == NULL && !get_current_directory_name ())
    return NULL;

  if (!posix_p)
    {
      if (strlen (current_directory_name) >= len)
	set_errno (ERANGE);
      else
	{
	  strcpy (buf, current_directory_name);
	  resbuf = buf;
	}

      syscall_printf ("%p (%s) = getcwd_inner (%p, %d, win32) (cached)",
		      resbuf, resbuf ? resbuf : "", buf, len);
      return resbuf;
    }
  else if (current_directory_posix_name != NULL)
    {
      if (strlen (current_directory_posix_name) >= len)
	set_errno (ERANGE);
      else
	{
	  strcpy (buf, current_directory_posix_name);
	  resbuf = buf;
	}

      syscall_printf ("%p (%s) = getcwd_inner (%p, %d, posix) (cached)",
		      resbuf, resbuf ? resbuf : "", buf, len);
      return resbuf;
    }

  /* posix_p required and current_directory_posix_name == NULL */

  char temp[MAX_PATH];

  /* Turn from win32 style to our style.  */
  cygwin_shared->mount.conv_to_posix_path (current_directory_name, temp, 0);

  size_t tlen = strlen (temp);

  current_directory_posix_name = (char *) realloc (
				  current_directory_posix_name, tlen + 1);
  if (current_directory_posix_name != NULL)
    strcpy (current_directory_posix_name, temp);

  if (tlen >= ulen)
    {
      /* len was too small */
      set_errno (ERANGE);
    }
  else
    {
      strcpy (buf, temp);
      resbuf = buf;
    }

  syscall_printf ("%p (%s) = getcwd_inner (%p, %d, %s)",
		  resbuf, resbuf ? resbuf : "",
		  buf, len, posix_p ? "posix" : "win32");
  return resbuf;
}

char *
getcwd (char *buf, size_t ulen)
{
  char *res;

  if (buf == NULL || ulen == 0)
    {
      buf = (char *) alloca (MAX_PATH);
      res = getcwd_inner (buf, MAX_PATH, 1);
      res = strdup (buf);
    }
  else
    {
      res = getcwd_inner (buf, ulen, 1);
    }

  return res;
}

extern "C"
char *
getwd (char *buf)
{
  return getcwd (buf, MAX_PATH);
}

/* chdir: POSIX 5.2.1.1 */
extern "C"
int
chdir (const char *dir)
{
  path_conv path (dir);

  if (path.error)
    {
      set_errno (path.error);
      syscall_printf ("-1 = chdir (%s)", dir);
      return -1;
    }

  char *native_dir = path.get_win32 ();

  /* Check to see if path translates to something like C:.
     If it does, append a \ to the native directory specification to
     defeat the Windows 95 (i.e. MS-DOS) tendency of returning to
     the last directory visited on the given drive. */
  if (isalpha (native_dir[0]) && native_dir[1] == ':' && !native_dir[2])
    {
      native_dir[2] = '\\';
      native_dir[3] = '\0';
    }
  int res = SetCurrentDirectoryA (native_dir);
  if (!res)
    __seterrno ();

  sig_protect (here, 1);

  /* Clear the cache until we need to retrieve the directory again.  */
  if (current_directory_name != NULL)
    {
      free (current_directory_name);
      current_directory_name = NULL;
    }
  if (current_directory_posix_name != NULL)
    {
      free (current_directory_posix_name);
      current_directory_posix_name = NULL;
    }

  syscall_printf ("%d = chdir (%s) (dos %s)", res ? 0 : -1, dir, native_dir);
  return res ? 0 : -1;
}

/* Symbolic link support. */

/* Create a symlink from FROMPATH to TOPATH. */

extern "C"
int
symlink (const char *topath, const char *frompath)
{
  int fd;
  int res = -1;

  syscall_printf ("symlink (%s, %s)", topath, frompath);

  if (topath[0] == 0)
    {
      set_errno (EINVAL);
      goto done;
    }
  if (strlen (topath) >= MAX_PATH)
    {
      set_errno (ENAMETOOLONG);
      goto done;
    }

  fd = _open (frompath, O_WRONLY | O_CREAT | O_BINARY, 0);
  if (fd >= 0)
    {
      char buf[sizeof (SYMLINK_COOKIE) + MAX_PATH + 10];

      __small_sprintf (buf, "%s%s", SYMLINK_COOKIE, topath);
      int len = strlen (buf) + 1;

      /* Note that the terminating nul is written.  */
      if (_write (fd, buf, len) != len)
	{
	  int saved_errno = get_errno ();
	  _close (fd);
	  _unlink (frompath);
	  set_errno (saved_errno);
	}
      else
	{
	  _close (fd);
	  chmod (frompath, S_IFLNK | S_IRWXU | S_IRWXG | S_IRWXO);
	  res = 0;
	}
    }

done:
  syscall_printf ("%d = symlink (%s, %s)", res, topath, frompath);
  return res;
}

static __inline char *
has_suffix (const char *path, const suffix_info *suffixes)
{
  char *ext = strrchr (path, '.');
  if (ext)
    for (const suffix_info *ex = suffixes; ex->name != NULL; ex++)
      if (strcasematch (ext, ex->name))
	return ext;
  return NULL;
}

static int __inline
next_suffix (char *ext_here, const suffix_info *&suffixes)
{
  if (!suffixes)
    return 1;

  while (suffixes && suffixes->name)
    if (!suffixes->addon)
      suffixes++;
    else
      {
	strcpy (ext_here, suffixes->name);
	suffixes++;
	return 1;
      }
  return 0;
}

/* Check if PATH is a symlink.  PATH must be a valid win32 path name.

   If PATH is a symlink, put the value of the symlink--the file to
   which it points--into BUF.  The value stored in BUF is not
   necessarily null terminated.  BUFLEN is the length of BUF; only up
   to BUFLEN characters will be stored in BUF.  BUF may be NULL, in
   which case nothing will be stored.

   Set *SYML if PATH is a symlink.

   Set *EXEC if PATH appears to be executable.  This is an efficiency
   hack because we sometimes have to open the file anyhow.  *EXEC will
   not be set for every executable file.

   Return -1 on error, 0 if PATH is not a symlink, or the length
   stored into BUF if PATH is a symlink.  */

static int
symlink_check_one (const char *in_path, char *buf, int buflen, DWORD& fileattr,
		   int *exec, const suffix_info *suffixes, char *&known_suffix)
{
  HANDLE h;
  int res = 0;
  char extbuf[buflen + 5];
  char *ext_here;
  const char *path = in_path;

  if (!suffixes)
    ext_here = NULL;
  else if ((known_suffix = has_suffix (in_path, suffixes)) != NULL)
    {
      suffixes = NULL;
      ext_here = NULL;
    }
  else
    {
      path = strcpy (extbuf, in_path);
      ext_here = strchr (path, '\0');
    }

  *buf = '\0';
  do
    {
      if (!next_suffix (ext_here, suffixes))
	break;
      fileattr = GetFileAttributesA (path);
      if (fileattr == (DWORD) -1)
	{
	  /* The GetFileAttributesA call can fail for reasons that don't
	     matter, so we just return 0.  For example, getting the
	     attributes of \\HOST will typically fail.  */
	  debug_printf ("GetFileAttributesA (%s) failed", path);
	  __seterrno ();
	  continue;
	}

      /* Windows allows path\. even when `path' isn't a directory.
	 Detect this scenario and disallow it, since it is non-unix like. */
      char *p = strchr (path, '\0');
      if (p > path + 1 && p[-1] == '.' && SLASH_P (p[-2]) &&
	  !(fileattr & FILE_ATTRIBUTE_DIRECTORY))
	{
	  debug_printf ("\\. specified on non-directory");
	  set_errno (ENOTDIR);
	  return 0;
	}

      /* A symlink will have the `system' file attribute. */
      /* Only files can be symlinks (which can be symlinks to directories). */
      if (!SYMLINKATTR (fileattr))
	goto file_not_symlink;

      /* Check the file's extended attributes, if it has any.  */
      int unixattr;

      if (get_file_attribute (path, &unixattr) > 0)
	{
	  if ((unixattr & STD_XBITS) != 0)
	    *exec = 1;
	  if (! S_ISLNK (unixattr))
	    ;
	}

      /* Open the file.  */

      h = CreateFileA (path, GENERIC_READ, FILE_SHARE_READ, &sec_none_nih, OPEN_EXISTING,
		       FILE_ATTRIBUTE_NORMAL, 0);

      res = -1;
      if (h == INVALID_HANDLE_VALUE)
	__seterrno ();
      else
	{
	  char cookie_buf[sizeof (SYMLINK_COOKIE) - 1];
	  DWORD got;

	  if (! ReadFile (h, cookie_buf, sizeof (cookie_buf), &got, 0))
	    set_errno (EIO);
	  else if (got == sizeof (cookie_buf)
		   && memcmp (cookie_buf, SYMLINK_COOKIE,
			      sizeof (cookie_buf)) == 0)
	    {
	      /* It's a symlink.  */
	      *exec = 0;

	      res = ReadFile (h, buf, buflen, &got, 0);
	      if (!res)
		set_errno (EIO);
	      else
		{
		  /* Versions prior to b16 stored several trailing
		     NULs with the path (to fill the path out to 1024
		     chars).  Current versions only store one trailing
		     NUL.  The length returned is the path without
		     *any* trailing NULs.  We also have to handle (or
		     at least not die from) corrupted paths.  */
		  if (memchr (buf, 0, got) != NULL)
		    res = strlen (buf);
		  else
		    res = got;
		}
	    }
	  else
	    {
	      /* Not a symlink, see if executable.  */
	      *exec = (got >= 2
		       && cookie_buf[0] == '#'
		       && cookie_buf[1] == '!');
	      CloseHandle (h);
	      goto file_not_symlink;
	    }
	}
      CloseHandle (h);
      break;
    }
  while (suffixes);
  goto out;

file_not_symlink:
  set_errno (EINVAL);
  if (ext_here)
    strcpy (buf, ext_here);
  res = 0;

out:
  if (res >= 0)
    syscall_printf ("%d = symlink_check_one (%s, %p, %d) (%s)",
		    res, path, buf, buflen,
		    res > 0 ? "symlink"
		    : *exec ? "executable"
		    : "normal file");
  else
    syscall_printf ("%d = symlink_check_one (%s, %p, %d)",
		    res, path, buf, buflen);

  return res;
}

/* readlink system call */

extern "C"
int
readlink (const char *path, char *buf, int buflen)
{
  path_conv pathbuf (path, SYMLINK_CONTENTS);
  if (pathbuf.error)
    {
      set_errno (pathbuf.error);
      syscall_printf ("-1 = readlink (%s, %p, %d)", path, buf, buflen);
      return -1;
    }

  if (!pathbuf.symlink_p)
    {
      set_errno (EINVAL);
      return -1;
    }

  int len = strlen (pathbuf.get_win32 ());
  if (len > (buflen - 1))
    {
      set_errno (ENAMETOOLONG);
      return -1;
    }
  memcpy (buf, pathbuf.get_win32 (), len);
  buf[len] = '\0';

  /* errno set by symlink_check_one if error */
  return len;
}

/* Mount table system calls.
   Note that these are exported to the application.  */

extern "C"
int
mount (const char *dev, const char *path, int flags)
{
  int res = cygwin_shared->mount.add_item (dev, path, flags);

  if (res == 0)
    cygwin_shared->mount.to_registry ();

  syscall_printf ("%d = mount (%s, %s, %p)", res, dev, path, flags);
  return res;
}

extern "C"
int
umount (const char *path)
{
  int res = cygwin_shared->mount.del_item (path);

  if (res == 0)
    cygwin_shared->mount.to_registry ();

  syscall_printf ("%d = umount (%s)", res,  path);
  return res;
}

static int iteration;

extern "C"
FILE *
setmntent (const char *filep, const char *)
{
  iteration = 0;
  return (FILE *) filep;
}

extern "C"
struct mntent *
getmntent (FILE *)
{
  return cygwin_shared->mount.getmntent (iteration++);
}

extern "C"
int
endmntent (FILE *)
{
  return 1;
}

/* Some programs rely on st_dev/st_ino being unique for each file.
   Hash the path name and hope for the best.  The hash arg is not
   always initialized to zero since readdir needs to compute the
   dirent ino_t based on a combination of the hash of the directory
   done during the opendir call and the hash or the filename within
   the directory.  FIXME: Not bullet-proof. */
/* Cygwin internal */

unsigned long
hash_path_name (unsigned long hash, const char *name)
{
  if (!*name)
    return hash;

  /* Perform some initial permutations on the pathname if this is
     not "seeded" */
  if (!hash)
    {
      /* Simplistic handling of drives.  If there is a drive specified,
	 make sure that the initial letter is upper case.  If there is
	 no \ after the ':' assume access through the root directory
	 of that drive.
	 FIXME:  Should really honor MS-Windows convention of using
	 the environment to track current directory on various drives. */
      if (name[1] == ':')
	{
	  char *nn, *newname = (char *) alloca (strlen (name) + 2);
	  nn = strncpy (newname, name, 2);
	  if (islower (*nn))
	    *newname = toupper (*nn);
	  *(nn += 2) = '\0';
	  name += 2;
	  if (*name != '\\')
	    {
	      *nn = '\\';
	      *++nn = '\0';
	    }
	  strcpy (nn, name);
	  name = newname;
	  goto hashit;
	}

      /* Fill out the hashed path name with the current working directory if
	 this is not an absolute path and there is no pre-specified hash value.
	 Otherwise the inodes same will differ depending on whether a file is
	 referenced with an absolute value or relatively. */

      if (*name != '\\' && (current_directory_name == NULL ||
			    get_current_directory_name ()))
	{
	  hash = current_directory_hash;
	  if (name[0] == '.' && name[1] == '\0')
	    return hash;
	  hash = hash_path_name (hash, "\\");
	}
    }

hashit:
  /* Build up hash.  Ignore single trailing slash or \a\b\ != \a\b
     but allow a single \ if that's all there is. */
  do
    {
      hash += *name + (*name << 17);
      hash ^= hash >> 2;
    }
  while (*++name != '\0' && (*name != '\\' || name[1] != '\0'));
  return hash;
}

/* Cover functions to the path conversion routines.
   These are exported to the world as cygwin_foo by cygwin.din.  */

extern "C"
void
conv_to_win32_path (const char *path, char *win32_path)
{
  int unit_dummy;
  DWORD devn_dummy;
  if (check_null_empty_path_errno (path))
    return;
  cygwin_shared->mount.conv_to_win32_path (path, win32_path, NULL,
					   devn_dummy, unit_dummy);
}

extern "C"
void
conv_to_full_win32_path (const char *path, char *win32_path)
{
  int unit_dummy;
  DWORD devn_dummy;

  if (check_null_empty_path_errno (path))
    return;
  cygwin_shared->mount.conv_to_win32_path (path, NULL, win32_path,
					   devn_dummy, unit_dummy);
}

/* This is exported to the world as cygwin_foo by cygwin.din.  */

extern "C"
void
conv_to_posix_path (const char *path, char *posix_path)
{
  if (check_null_empty_path_errno (path))
    return;
  cygwin_shared->mount.conv_to_posix_path (path, posix_path, 1);
}

extern "C"
void
conv_to_full_posix_path (const char *path, char *posix_path)
{
  if (check_null_empty_path_errno (path))
    return;
  cygwin_shared->mount.conv_to_posix_path (path, posix_path, 0);
}

/* The realpath function is supported on some Unix systems.  */

extern "C"
char *
realpath (const char *path, char *resolved)
{
  int err;
  path_conv real_path (path);

  if (real_path.error)
    err = real_path.error;
  else
    {
      err = cygwin_shared->mount.conv_to_posix_path (real_path.get_win32 (), resolved, 0);
      if (err == 0)
	return resolved;
    }

  /* FIXME: on error, we are supposed to put the name of the path
     component which could not be resolved into RESOLVED.  */
  resolved[0] = '\0';

  set_errno (err);
  return NULL;
}

/* Return non-zero if PATH is a Posix path list.
   This is exported to the world as cygwin_foo by cygwin.din.

DOCTOOL-START
<sect1 id="add-func-cygwin-posix-path-list-p">
  <para>Rather than use a mode to say what the "proper" path list
  format is, we allow any, and give apps the tools they need to
  convert between the two.  If a ';' is present in the path list it's
  a win32 path list.  Otherwise, if the first path begins with
  [letter]: (in which case it can be the only element since if it
  wasn't a ';' would be present) it's a win32 path list.  Otherwise,
  it's a posix path list.</para>
</sect1>
DOCTOOL-END
  */

extern "C"
int
posix_path_list_p (const char *path)
{
  int posix_p = ! (strchr (path, ';')
		   || (isalpha (path[0]) && path[1] == ':'));
  return posix_p;
}

/* These are used for apps that need to convert env vars like PATH back and
   forth.  The conversion is a two step process.  First, an upper bound on the
   size of the buffer needed is computed.  Then the conversion is done.  This
   allows the caller to use alloca if it wants.  */

static int
conv_path_list_buf_size (const char *path_list, int to_posix_p)
{
  int i, num_elms, max_mount_path_len, size;
  const char *p;

  /* The theory is that an upper bound is
     current_size + (num_elms * max_mount_path_len)  */

  char delim = to_posix_p ? ';' : ':';
  p = path_list;
  for (num_elms = 1; (p = strchr (p, delim)) != NULL; ++num_elms)
    ++p;

  /* 7: strlen ("//c") + slop, a conservative initial value */
  for (max_mount_path_len = 7, i = 0; i < cygwin_shared->mount.nmounts; ++i)
    {
      int mount_len = (to_posix_p
		       ? cygwin_shared->mount.mount[i].pathlen
		       : cygwin_shared->mount.mount[i].devicelen);
      if (max_mount_path_len < mount_len)
	max_mount_path_len = mount_len;
    }

  /* 100: slop */
  size = strlen (path_list) + (num_elms * max_mount_path_len) + 100;
  return size;
}

extern "C"
int
win32_to_posix_path_list_buf_size (const char *path_list)
{
  return conv_path_list_buf_size (path_list, 1);
}

extern "C"
int
posix_to_win32_path_list_buf_size (const char *path_list)
{
  return conv_path_list_buf_size (path_list, 0);
}

/* Convert a list of path names to/from win32/posix.

   SRC is not a const char * because we temporarily modify it to ease
   the implementation.

   I believe win32 always has '.' in $PATH.   Posix obviously doesn't.
   We certainly don't want to handle that here, but it is something for
   the caller to think about.  */

static void
conv_path_list (const char *src, char *dst, int to_posix_p)
{
  char *s;
  char *d = dst;
  char src_delim = to_posix_p ? ';' : ':';
  char dst_delim = to_posix_p ? ':' : ';';
  void (*conv_fn) (const char *, char *) = (to_posix_p
					    ? conv_to_posix_path
					    : conv_to_win32_path);

  do
    {
      s = strchr (src, src_delim);
      if (s)
	{
	  *s = 0;
	  (*conv_fn) (src[0] != 0 ? src : ".", d);
	  d += strlen (d);
	  *d++ = dst_delim;
	  *s = src_delim;
	  src = s + 1;
	}
      else
	{
	  /* Last one.  */
	  (*conv_fn) (src[0] != 0 ? src : ".", d);
	}
    }
  while (s != NULL);
}

extern "C"
void
win32_to_posix_path_list (const char *win32, char *posix)
{
  conv_path_list (win32, posix, 1);
}

extern "C"
void
posix_to_win32_path_list (const char *posix, char *win32)
{
  conv_path_list (posix, win32, 0);
}

/* Various utilities.  */
static void
slashify (const char *src, char *dst, int trailing_slash_p)
{
  const char *start = src;

  while (*src)
    {
      if (*src == '\\')
	*dst++ = '/';
      else
	*dst++ = *src;
      ++src;
    }
  if (trailing_slash_p
      && src > start
      && !isdirsep (src[-1]))
    *dst++ = '/';
  *dst++ = 0;
}

static void
backslashify (const char *src, char *dst, int trailing_slash_p)
{
  const char *start = src;

  while (*src)
    {
      if (*src == '/')
	*dst++ = '\\';
      else
	*dst++ = *src;
      ++src;
    }
  if (trailing_slash_p
      && src > start
      && !isdirsep (src[-1]))
    *dst++ = '\\';
  *dst++ = 0;
}

/* Remove trailing / and \ from SRC (except for the first one).
   It is ok for src == dst.  */

static void
nofinalslash (const char *src, char *dst)
{
  int len = strlen (src);
  if (src != dst)
    memcpy (dst, src, len + 1);
  while (len > 1 && SLASH_P (dst[--len]))
    dst[len] = 0;
}

/* Return non-zero if PATH begins with //<letter>.  */

static int
slash_drive_prefix_p (const char *path)
{
  return (path[0] == '/'
	  && path[1] == '/'
	  && isalpha (path[2])
	  && (path[3] == 0 || path[3] == '/'));
}

/* Return non-zero if PATH begins with //UNC/SHARE */

static int
slash_unc_prefix_p (const char *path)
{
  char *p = NULL;
  int ret = (path[0] == '/'
	     && path[1] == '/'
	     && isalpha (path[2])
	     && path[3] != 0
	     && path[3] != '/'
	     && ((p = strchr(&path[3], '/')) != NULL));
  if (!ret || p == NULL)
    return ret;
  return ret && isalnum (p[1]);
}

/* Build "slash drive prefix" in BUF from PATH, which begins with <letter>:.
   The result is a pointer to the end of the string.  */

static char *
build_slash_drive_prefix (const char *path, char *buf)
{
  buf[0] = '/';
  buf[1] = '/';
  buf[2] = path[0];
  return buf + 3;
}

/* Convert PATH (for which slash_drive_prefix_p returns 1) to WIN32 form.  */

static void
slash_drive_to_win32_path (const char *path, char *buf, int trailing_slash_p)
{
  buf[0] = path[2];
  buf[1] = ':';
  if (path[3] == '0')
    strcpy (buf + 2, "\\");
  else
    backslashify (path + 3, buf + 2, trailing_slash_p);
}

/* Split a path into directory and file name parts.
   Buffers DIR and FILE are assumed to be big enough.

   Examples (path -> `dir' / `file'):
   / -> `/' / `'
   "" -> `.' / `'
   . -> `.' / `.' (FIXME: should be `.' / `')
   .. -> `.' / `..' (FIXME: should be `..' / `')
   foo -> `.' / `foo'
   foo/bar -> `foo' / `bar'
   foo/bar/ -> `foo' / `bar'
   /foo -> `/' / `foo'
   /foo/bar -> `/foo' / `bar'
   c: -> `c:/' / `'
   c:/ -> `c:/' / `'
   c:foo -> `c:/' / `foo'
   c:/foo -> `c:/' / `foo'
 */

extern "C"
void
split_path (const char *path, char *dir, char *file)
{
  int dir_started_p = 0;

  /* Deal with drives.
     Remember that c:foo <==> c:/foo.  */
  if (isalpha (path[0]) && path[1] == ':')
    {
      *dir++ = *path++;
      *dir++ = *path++;
      *dir++ = '/';
      if (! *path)
	{
	  *dir = 0;
	  *file = 0;
	  return;
	}
      if (SLASH_P (*path))
	++path;
      dir_started_p = 1;
    }

  /* Determine if there are trailing slashes and "delete" them if present.
     We pretend as if they don't exist.  */
  const char *end = path + strlen (path);
  /* path + 1: keep leading slash.  */
  while (end > path + 1 && SLASH_P (end[-1]))
    --end;

  /* At this point, END points to one beyond the last character
     (with trailing slashes "deleted").  */

  /* Point LAST_SLASH at the last slash (duh...).  */
  const char *last_slash;
  for (last_slash = end - 1; last_slash >= path; --last_slash)
    if (SLASH_P (*last_slash))
      break;

  if (last_slash == path)
    {
      *dir++ = '/';
      *dir = 0;
    }
  else if (last_slash > path)
    {
      memcpy (dir, path, last_slash - path);
      dir[last_slash - path] = 0;
    }
  else
    {
      if (dir_started_p)
	; /* nothing to do */
      else
	*dir++ = '.';
      *dir = 0;
    }

  memcpy (file, last_slash + 1, end - last_slash - 1);
  file[end - last_slash - 1] = 0;
}

#define CHXOR ('a' ^ 'A')
#define ch_case_eq(ch1, ch2) \
    ({ \
      unsigned char x; \
      !((x = ((unsigned char)ch1 ^ (unsigned char)ch2)) && \
       (x != CHXOR || !isalpha (ch1))); \
    })

int
strncasematch (const char *s1, const char *s2, size_t n)
{
  if (s1 == s2)
    return 1;

  n++;
  while (--n && *s1)
    {
      if (!ch_case_eq (*s1, *s2))
	return 0;
      s1++; s2++;
    }
  return !n || *s2 == '\0';
}

int
strcasematch (const char *s1, const char *s2)
{
  if (s1 == s2)
    return 1;

  while (*s1)
    {
      if (!ch_case_eq (*s1, *s2))
	return 0;
      s1++; s2++;
    }
  return *s2 == '\0';
}

char *
strcasestr (const char *searchee, const char *lookfor)
{
  if (*searchee == 0)
    {
      if (*lookfor)
	return NULL;
      return (char *) searchee;
    }

  while (*searchee)
    {
      int i = 0;
      while (1)
	{
	  if (lookfor[i] == 0)
	    return (char *) searchee;

	  if (!ch_case_eq (lookfor[i], searchee[i]))
	    break;
	  lookfor++;
	}
      searchee++;
    }

  return NULL;
}
