/* cygpath.c -- Tcl routines to convert paths under cygwin.

   Mumit Khan <khan@nanotech.wisc.edu>  */

#ifdef __CYGWIN__

#include <tcl.h>
#include <windows.h>

extern void cygwin_conv_to_win32_path (const char *, char *);
extern void cygwin_conv_to_full_win32_path (const char *, char *);
extern void cygwin_conv_to_posix_path (const char *, char *);
extern void cygwin_conv_to_full_posix_path (const char *, char *);
extern int cygwin_posix_path_list_p (const char *);
extern int cygwin_win32_to_posix_path_list_buf_size (const char *);
extern int cygwin_posix_to_win32_path_list_buf_size (const char *);
extern void cygwin_win32_to_posix_path_list (char *, char *);
extern void cygwin_posix_to_win32_path_list (char *, char *);
extern void cygwin_split_path (const char *, char *, char *);

/* This file declares a bunch of Tcl commands for Cygwin <--> Windows
   PATH conversion.

   cygwin_path_to_win32 ?-full? PATH
       Return PATH converted to a win32 pathname.

   cygwin_path_to_posix ?-full? PATH
       Return PATH converted to a POSIX pathname.

   The following command return a boolean value.

   cygwin_path_posix_path_list_p PATHLIST
       Return whether PATHLIST is a POSIX style path list.

   The following commands return strings.

   cygwin_path_posix_to_win32_path_list PATHLIST
       Return PATHLIST converted from POSIX style to win32 style.

   cygwin_path_win32_to_posix_path_list PATHLIST
       Return PATHLIST converted from win32 style to POSIX style.

   */

static int
cygwin_conv_path_to_win32 (ClientData cd, Tcl_Interp *interp, 
  int argc, char **argv)
{
  char buf[MAX_PATH];
  char *pathp = 0;
  int full = 0;

  if (argc < 2 || argc > 3)
  {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], 
                     " ?-full? PATH\"", NULL);
    return TCL_ERROR;
  }
  else if (argc == 3) 
  {  
    if (strcmp(argv[1], "-full") == 0)
      full = 1;
    else
    {
      Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], 
		       " ?-full? PATH\"", NULL);
      return TCL_ERROR;
    }
    pathp = argv[2];
  }
  else
    pathp = argv[1];

  if (full)
    cygwin_conv_to_full_win32_path (pathp, buf);
  else
    cygwin_conv_to_win32_path (pathp, buf);
  Tcl_SetResult (interp, buf, TCL_VOLATILE);
  return TCL_OK;
}

static int
cygwin_conv_path_to_posix (ClientData cd, Tcl_Interp *interp, 
  int argc, char **argv)
{
  char buf[MAX_PATH];
  char *pathp = 0;
  int full = 0;

  if (argc < 2 || argc > 3)
  {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], 
                     " ?-full? PATH\"", NULL);
    return TCL_ERROR;
  }
  else if (argc == 3) 
  {  
    if (strcmp(argv[1], "-full") == 0)
      full = 1;
    else
    {
      Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], 
		       " ?-full? PATH\"", NULL);
      return TCL_ERROR;
    }
    pathp = argv[2];
  }
  else
    pathp = argv[1];

  if (full)
    cygwin_conv_to_full_posix_path (pathp, buf);
  else
    cygwin_conv_to_posix_path (pathp, buf);
  Tcl_SetResult (interp, buf, TCL_VOLATILE);
  return TCL_OK;
}

static int
cygwin_conv_path_posix_path_list_p (ClientData cd, Tcl_Interp *interp, 
  int argc, char **argv)
{
  int ret;

  ret = cygwin_posix_path_list_p (argv[2]);
  Tcl_ResetResult (interp);
  Tcl_SetBooleanObj (Tcl_GetObjResult (interp), ret);
  return TCL_OK;
}

static int
cygwin_conv_path_posix_to_win32_path_list (ClientData cd, Tcl_Interp *interp, 
  int argc, char **argv)
{
  int size;
  char *buf;

  size = cygwin_posix_to_win32_path_list_buf_size (argv[2]);
  buf = Tcl_Alloc (size);
  cygwin_posix_to_win32_path_list (argv[2], buf);
  Tcl_SetResult (interp, buf, TCL_DYNAMIC);
  return TCL_OK;
}

static int
cygwin_conv_path_win32_to_posix_path_list (ClientData cd, Tcl_Interp *interp, 
  int argc, char **argv)
{
  int size;
  char *buf;

  size = cygwin_win32_to_posix_path_list_buf_size (argv[2]);
  buf = Tcl_Alloc (size);
  cygwin_win32_to_posix_path_list (argv[2], buf);
  Tcl_SetResult (interp, buf, TCL_DYNAMIC);
  return TCL_OK;
}

static char version_string[] = "1.0";

int __declspec(dllexport)
Cygpath_Init (Tcl_Interp *interp)
{
  if (Tcl_PkgRequire(interp, "Tcl", TCL_VERSION, 0) == NULL 
      || Tcl_PkgProvide (interp, "cygpath", version_string) != TCL_OK)
    return TCL_ERROR;

  if (Tcl_CreateCommand (interp, "cygwin_conv_path_to_win32", 
                         cygwin_conv_path_to_win32, NULL, NULL) == NULL) 
    return TCL_ERROR;

  if (Tcl_CreateCommand (interp, "cygwin_conv_path_to_posix", 
                         cygwin_conv_path_to_posix, NULL, NULL) == NULL) 
    return TCL_ERROR;

  if (Tcl_CreateCommand (interp, "cygwin_conv_path_posix_path_list_p", 
                         cygwin_conv_path_posix_path_list_p, NULL,
			 NULL) == NULL)
    return TCL_ERROR;

  if (Tcl_CreateCommand (interp, "cygwin_conv_path_posix_to_win32_path_list", 
                         cygwin_conv_path_posix_to_win32_path_list, NULL,
			 NULL) == NULL)
    return TCL_ERROR;

  if (Tcl_CreateCommand (interp, "cygwin_conv_path_win32_to_posix_path_list", 
                         cygwin_conv_path_win32_to_posix_path_list, NULL, 
			 NULL) == NULL)
    return TCL_ERROR;

  return TCL_OK;
}

#endif /* __CYGWIN__ */
