//
// bline_cmds.cc: Tcl commands related to manipulate the beamline/system.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994 Mumit Khan
//

#ifdef __BORLANDC__
# include <assert>
# include <stdlib>
# include <string>
#else
# include <cassert>
# include <cstdlib>
# include <cstring>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#if defined(NO_STRNCASECMP_PROTO)
extern "C" int strcasecmp(const char*, const char*);
extern "C" int strncasecmp(const char*, const char*, int);
#endif

#include "menu.h"
#include "beamline.h"
#include "utils.h"

#include "commands.h"
#include "xmenu.h"

// ================ Tcl commands created here ======================== //
/*
 *
 * ALL THE COMMANDS to "BEAMLINE" in this interpreter:
 *
 * beamline list system
 *
 * beamline selection copy
 * beamline selection paste
 * beamline selection clear
 *
 * beamline add source
 * beamline add oe <oe#>
 * beamline add screen <oe#> <scr#>
 * beamline add tool <toolname>
 *
 * beamline add inspector system <inspector> <id>
 * beamline add inspector source <inspector> <id>
 * beamline add inspector oe <oe#> <inspector> <id>
 * beamline add inspector screen <oe#> <scr#> <inspector> <id>
 *
 * beamline delete source
 * beamline delete oe <oe#>
 * beamline delete screen <oe#> <scr#>
 * beamline delete tool <tool>
 *
 * beamline delete inspector system <id>
 * beamline delete inspector source <id>
 * beamline delete inspector oe <oe#> <id>
 * beamline delete inspector screen <oe#> <scr#> <id>
 *
 * beamline select none
 * beamline select system
 * beamline select source
 * beamline select oe <oe#>
 * beamline select screen <oe#> <scr#>
 *
 * beamline select inspector system <id>
 * beamline select inspector source <id>
 * beamline select inspector oe <oe#> <id>
 * beamline select inspector screen <oe#> <scr#> <id>
 *
 * beamline edit none
 * beamline edit system
 * beamline edit source
 * beamline edit oe <oe#>
 * beamline edit screen <oe#> <scr#>
 *
 * beamline edit inspector system <id>
 * beamline edit inspector source <id>
 * beamline edit inspector oe <oe#> <id>
 * beamline edit inspector screen <oe#> <scr#> <id>
 *
 * beamline get current
 * beamline get source
 * beamline get oe <oe#>
 * beamline get screen <oe#> <scr#>
 * beamline get inspectors system
 * beamline get inspectors source
 * beamline get inspectors oe <oe#>
 * beamline get inspectors scr <oe#> <scr#>
 *
 * beamline vget source <var>
 * beamline vget source <var>
 * beamline vget oe <oe#> <var>
 * beamline vget screen <oe#> <scr#> <var>
 *
 * beamline vget inspector system <id#> var
 * beamline vget inspector source <id#> var
 * beamline vget inspector oe <oe#> <id#> var
 * beamline vget inspector scr <oe#> <scr#> <id#> var
 *
 * beamline vset source var value
 * beamline vset oe <oe#> var value
 * beamline vset screen <oe#> <scr#> var value
 *
 * beamline vset inspector system <id#> var value
 * beamline vset inspector source <id#> var value
 * beamline vset inspector oe <oe#> <id#> var value
 * beamline vset inspector scr <oe#> <scr#> <id#> var value
 *
 * beamline tget source 
 * beamline tget oe <oe#> 
 * beamline tget scr <oe#> <scr#> 
 *
 * beamline tget insector source <id>
 * beamline tget insector oe <oe#> <id>
 * beamline tget inspector scr <oe#> <scr#> <id>
 *
 * beamline tset source <title>
 * beamline tset oe <oe#> <title>
 * beamline tset scr <oe#> <scr#> <title>
 *
 * beamline tset insector source <id> <title>
 * beamline tset insector oe <oe#> <id> <title>
 * beamline tset inspector scr <oe#> <scr#> <id> <title>
 *
 * beamline load source <file>
 * beamline load oe <oe#> <file>
 * beamline load screen <oe#> <scr#> <file>
 * beamline load system <file>
 * beamline load tool <toolname> <file>
 *
 * beamline load inspector <id> <file>
 * beamline load inspector source <id> <file>
 * beamline load inspector oe <oe#> <id> <file>
 * beamline load inspector scr <oe#> <scr#> <id> <file>
 *
 * beamline store source <file>
 * beamline store oe <oe#> <file>
 * beamline store screen <oe#> <scr#> <file>
 * beamline store tool <toolname> <file>
 *
 * beamline store inspector <id> <file>
 * beamline store inspector source <id> <file>
 * beamline store inspector oe <oe#> <id> <file>
 * beamline store inspector scr <oe#> <scr#> <id> <file>
 *
 * beamline clear
 * beamline clear oe
 * beamline clear clipboard
 *
 * beamline state show
 * beamline state save
 * beamline state restore
 *
 * beamline dump system
 * beamline dump source
 * beamline dump clipboard
 * beamline dump selection
 * beamline dump oe <oe#>
 * beamline dump scr <oe#> <scr#>
 *
 * beamline dump inspector <id>
 * beamline dump inspector source <id>
 * beamline dump inspector oe <oe#> <id>
 * beamline dump inspector scr <oe#> <scr#> <id>
 */

// =================================================================== //

// =================================================================== //

static char g_buf[BUFSIZ];

// =================================================================== //

static inline char* string_to_charptr(const string& str) {
    return const_cast<char*>(str.c_str());
}

// =================================================================== //

static int dump_system(Tcl_Interp* interp) {
    Tcl_AppendResult(
	interp, "dumping system ",  (char*) NULL
    );
    GLB_beamline->dump();
    return TCL_OK;
}

static int dump_selection(Tcl_Interp* interp) {
    const BLToolInstance* selection = GLB_beamline->cur_selection();
    if (selection) {
	Tcl_AppendResult(
	    interp, "dumping selection ",  (char*) NULL
	);
	selection->dump();
    } else {
	Tcl_AppendResult(
	    interp, "Selection is empty",  (char*) NULL
	);
    }
    return TCL_OK;
}

static int dump_clipboard(Tcl_Interp* interp) {
    const BLToolInstance* clipboard = GLB_beamline->clipboard();
    if (clipboard) {
	Tcl_AppendResult(
	    interp, "dumping clipboard ",  (char*) NULL
	);
	clipboard->dump();
    } else {
	Tcl_AppendResult(
	    interp, "Clipboard is empty",  (char*) NULL
	);
    }
    return TCL_OK;
}

static int dump_source(Tcl_Interp* interp) {
    const Source* source = GLB_beamline->get_source();
    if (source == nil) {
	Tcl_AppendResult(interp, "No source defined",  (char*) NULL);
    } else {
	Tcl_AppendResult(
	    interp, "dumping source ",  (char*) NULL
	);
	source->dump();
    }
    return TCL_OK;
}

static int dump_oe(Tcl_Interp* interp, int oe) {
    const OE* oelem = GLB_beamline->get_oe(oe);
    if (oelem == nil) {
	Tcl_AppendResult(interp, "No such OE defined",  (char*) NULL);
    } else {
	Tcl_AppendResult(
	    interp, "dumping oe ",  (char*) NULL
	);
	oelem->dump();
    }
    return TCL_OK;
}

static int dump_scr(Tcl_Interp* interp, int oe, int scr) {
    const Screen* screen = GLB_beamline->get_scr(oe, scr);
    if (scr == nil) {
	Tcl_AppendResult(interp, "No such SCREEN defined",  (char*) NULL);
    } else {
	Tcl_AppendResult(
	    interp, "dumping screen ",  (char*) NULL
	);
	screen->dump();
    }
    return TCL_OK;
}

/*
 * beamline dump inspector system <id>
 * beamline dump inspector source <id>
 * beamline dump inspector oe <oe#> <id>
 * beamline dump inspector scr <oe#> <scr#> <id>
 */

static int dump_inspector(Tcl_Interp* interp, int argc, char* argv[]) {
    if (argc < 5 || argc > 7) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
	    " dump inspector [args]\"", (char *) NULL
	);
	return TCL_ERROR;
    }

    int id = atoi(argv[argc-1]);
    Inspector const* inspector = nil;

    if (strncasecmp(argv[3], "sy", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		" dump inspector system <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	inspector = GLB_beamline->get_inspector(id);
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		" dump inspector source <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	inspector = GLB_beamline->get_source_inspector(id);
    } else if (argv[3][0] == 'o') {
	if (argc != 6) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		" dump inspector oe <oe#> <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	int oe = atoi(argv[4]);
	inspector = GLB_beamline->get_oe_inspector(oe, id);
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
	if (argc != 7) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		" dump inspector scr <oe#> <scr#> <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	int oe = atoi(argv[4]);
	int scr = atoi(argv[5]);
	inspector = GLB_beamline->get_scr_inspector(oe, scr, id);
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[3], 
	    "\": should be source, oe, scr or <id>", (char *) NULL
	);
	return TCL_ERROR;
    }

    Tcl_AppendResult(interp, "dumping inspector ",  (char*) NULL);
    inspector->dump();

    return TCL_OK;
}

static int copy_selection_to_clipboard (Tcl_Interp* interp) {
    if (!GLB_beamline->cur_selection()) {
        Tcl_AppendResult(interp, "Nothing selected to copy", (char*)NULL);
	return TCL_ERROR;
    }
    int errcode = GLB_beamline->copy_selection_to_clipboard();
    return (errcode == 0) ? TCL_OK : TCL_ERROR;
}

static int paste_selection_from_clipboard (Tcl_Interp* interp) {
    if (!GLB_beamline->clipboard()) {
        Tcl_AppendResult(interp, "Nothing in clipboard to paste", (char*)NULL);
	return TCL_ERROR;
    }
    int retcode = TCL_ERROR;
    switch (GLB_beamline->paste_selection_from_clipboard()) {
        case 0:
	    retcode = TCL_OK;
	    break;
	case -1:
	    Tcl_AppendResult(interp, 
	        "Either nothing selected or clipboard empty", (char*)NULL
	    );
	    break;
	case -2:
	    Tcl_AppendResult(interp, 
	        "Cannot paste to incompatible type", (char*)NULL
	    );
	    break;
	case -3:
	    Tcl_AppendResult(interp, 
	        "Can't paste onto itself", (char*)NULL
	    );
	    break;
	default:
	    Tcl_AppendResult(interp, 
	        "Unknown error in pasting", (char*)NULL
	    );
	    break;
    }
    return retcode;
}

/*
 * returns the CURRENT OE
 *
 */
static int get_current(Tcl_Interp* interp) {
    sprintf(g_buf, "%d %d %d", 
        GLB_beamline->cur_oe(), GLB_beamline->cur_scr(),
        GLB_beamline->cur_inspector()
    );
    Tcl_AppendElement(interp, g_buf);
    return TCL_OK;
}


/*
 * returns a list with SOURCE config.
 *
 * for now the only element is either SOURCE or {} depending on if the
 * source is defined or not.
 *
 */
static int get_source(Tcl_Interp* interp) {
    const Source* source = GLB_beamline->get_source();
    if(source) {
	sprintf(g_buf, "SOURCE %d", source->num_inspectors());
    } else {
	g_buf[0] = '\0';
    }
    Tcl_AppendElement(interp, g_buf);
    return TCL_OK;
}

/*
 * returns a list with OE config.
 *
 * for now the only element is, which is the number of SCREENS in it, or
 * empty if the OE is not defined.
 *
 */
static int get_oe(Tcl_Interp* interp, int oe) {
    const OE* oelem = GLB_beamline->get_oe(oe);
    if(oelem) {
	int num_inspectors = oelem->num_inspectors();
	sprintf(g_buf, "%d %d", oelem->num_scr(), num_inspectors);
	for (int scr = 1; scr <= oelem->num_scr(); ++scr) {
	    char buf[10];
	    const Screen* screen = oelem->get_screen(scr);
	    sprintf(buf, " %d", screen->num_inspectors());
	    strcat(g_buf, buf);
	}
    } else {
	g_buf[0] = '\0';
    }
    Tcl_AppendElement(interp, g_buf);
    return TCL_OK;
}

/*
 * returns a list with SCREEN config.
 *
 * returns empty list if not defined.
 */
static int get_scr(Tcl_Interp* interp, int oe, int scr) {
    const Screen* screen = GLB_beamline->get_scr(oe, scr);
    Tcl_AppendElement(interp, (screen) ? "SCREEN" : "");
    return TCL_OK;
}

/*
 * returns a list with INSPECTOR config.
 */
static int do_get_inspectors(Tcl_Interp* interp, const InspectorMgr* obj) {
    if (obj == nil) {
	Tcl_AppendResult(interp, "Internal: Getting inspectors for NULL",
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    int num_inspectors = obj->num_inspectors();
    for (int id = 1; id <= num_inspectors; ++id) {
	const Inspector* inspector = obj->get_inspector(id);
	assert(inspector != nil);
	sprintf (g_buf, "%s %d", 
	    inspector->name().c_str(), inspector->get_id()
	);
	Tcl_AppendElement(interp, g_buf);
    }
    return TCL_OK;
}

/*
 * returns a list with INSPECTOR config.
 */
static int get_beamline_inspectors(Tcl_Interp* interp) {
    return do_get_inspectors(interp, GLB_beamline);
}

/*
 * returns a list with INSPECTOR config.
 */
static int get_source_inspectors(Tcl_Interp* interp) {
    const Source* source = GLB_beamline->get_source();
    return (source) ? do_get_inspectors(interp, source) : TCL_OK;
}

/*
 * returns a list with INSPECTOR config.
 */
static int get_oe_inspectors(Tcl_Interp* interp, int oe) {
    const OE* oelem = GLB_beamline->get_oe(oe);
    return (oelem) ? do_get_inspectors(interp, oelem) : TCL_OK;
}

/*
 * returns a list with INSPECTOR config.
 */
static int get_scr_inspectors(Tcl_Interp* interp, int oe, int scr) {
    const Screen* screen = GLB_beamline->get_scr(oe, scr);
    return (screen) ? do_get_inspectors(interp, screen) : TCL_OK;
}

/*
 * returns a list with INSPECTOR config.
 */
static int get_inspectors(Tcl_Interp* interp, int argc, char** argv) {
    if (strncasecmp(argv[3], "sy", 2) == 0) {
	if (argc == 4) {
	    return get_beamline_inspectors(interp);
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" get inspectors systems\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
	if (argc == 4) {
	    return get_source_inspectors(interp);
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" get inspectors source\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (argv[3][0] == 'o') {
	if (argc == 5) {
	    return get_oe_inspectors(interp, atoi(argv[4]));
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" get inspectors oe <oe#>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
	if (argc == 6) {
	    return get_scr_inspectors(interp, atoi(argv[4]), atoi(argv[5]));
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" get inspectors scr <oe#> <scr#>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    }
    return TCL_ERROR;
}

/*
 * returns a list with beamline config.
 *
 * element#1: {SOURCE}
 * element#2: current OE (cur_oe == 0 when none defined)
 * element#3 ... element#n: Number of screens for this OE (1...n).
 *
 * NOTE: SCREEN info is not given here, only the number of screens.
 *
 * example:
 *
 *    OLD FORMAT: {SOURCE} {2} {{2} {3} {0} {5} {0} {0} {0}}
 *
 *    {SOURCE <#i>} {2 1 i} {2 <#i> <#i> <#i>} {1 <#i> <#i> <#i>}
 *
 * this beamline has a SOURCE and 2 OEs, and each corresponding element
 * in the list tells you how many SCREENs that OE has defined as well
 * as how many Inspectors each SOURCE, OE or SCREEN has.

 *    {SOURCE <#i>} {2 1 i} {2 <#i> <#i> <#i>} {1 <#i> <#i> <#i>}
 *       1     2     3 4 5   6  7    8    9     10 11   12   13
 *
 *  1-2  SOURCE and # of SOURCE inspectors (empty list if no source)
 *  3-5  Current OE, SCREEN and Inspector (0 if not applicable)
 *  6    Number of Screens in this OE
 *  7    Number of Inspectors in this OE
 *  8-9  Number of Inspectors in SCREEN 1 of OE 1 and SCREEN 2 of OE 1
 *
 */
static int list_system(Tcl_Interp* interp) {
    get_source(interp);
    get_current(interp);
    int num_oe = GLB_beamline->num_oe();
    for(int oe = 1; oe <= num_oe; oe++) {
	get_oe(interp, oe);
    }
    return TCL_OK;
}

static int get_source_value(Tcl_Interp* interp, char* varname) {
    int retcode = TCL_OK;
    const char* value = GLB_beamline->get_source_value(varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such variable \"", varname, 
	    "\" defined for SOURCE.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

static int get_oe_value(Tcl_Interp* interp, int oe, char* varname) {
    int retcode = TCL_OK;
    const char* value = GLB_beamline->get_oe_value(oe, varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such variable \"", varname, 
	    "\" defined for OE.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

static int get_scr_value(Tcl_Interp* interp, int oe, int scr, char* varname) {
    int retcode = TCL_OK;
    const char* value = GLB_beamline->get_scr_value(oe, scr, varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such variable \"", varname, 
	    "\" defined for SCREEN.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

static int get_beamline_inspector_value (Tcl_Interp* interp, 
    int id, char* varname
) {
    int retcode = TCL_OK;
    const char* value = GLB_beamline->get_inspector_value(id, varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such inspector or no such variable \"", 
	    varname, "\" defined for BEAMLINE.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

static int get_source_inspector_value (Tcl_Interp* interp, 
    int id, char* varname
) {
    int retcode = TCL_OK;
    const char* value = GLB_beamline->get_source_inspector_value(id, varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such inspector or no such variable \"", 
	    varname, "\" defined for SOURCE.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

static int get_oe_inspector_value (Tcl_Interp* interp, int oe,
    int id, char* varname
) {
    int retcode = TCL_OK;
    const char* value = GLB_beamline->get_oe_inspector_value(oe, id, varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such inspector or no such variable \"", 
	    varname, "\" defined for this OE.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

static int get_scr_inspector_value (Tcl_Interp* interp, int oe, int scr,
    int id, char* varname
) {
    int retcode = TCL_OK;
    const char* value = 
	GLB_beamline->get_scr_inspector_value(oe, scr, id, varname);
    if (value == nil) {
	Tcl_AppendResult(interp, "No such inspector or no such variable \"", 
	    varname, "\" defined for this SCREEN.", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	Tcl_AppendResult(interp, value, (char*) NULL);
    }
    return retcode;
}

/*
 * beamline vget inspector system <id#> var
 * beamline vget inspector source <id#> var
 * beamline vget inspector oe <oe#> <id#> var
 * beamline vget inspector scr <oe#> <scr#> <id#> var
 */
static int get_inspector_value (Tcl_Interp* interp, int argc, char** argv){
    if (argc < 6 || argc > 8) {
	Tcl_AppendResult(interp, 
	    "wrong # args: should be \"beamline vget inspector [args]\"",
	    (char *) NULL
	);
	return TCL_ERROR;
    }
    if (strncasecmp(argv[3], "sy", 2) == 0) {
	if (argc == 6) {
	    return get_beamline_inspector_value (interp, 
		atoi(argv[4]), argv[5]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector system <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
	if (argc == 6) {
	    return get_source_inspector_value (interp, 
		atoi(argv[4]), argv[5]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector source <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (argv[3][0] == 'o') {
	if (argc == 7) {
	    return get_oe_inspector_value (interp, 
		atoi(argv[4]), atoi(argv[5]), argv[6]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector oe <oe#> <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
	if (argc == 8) {
	    return get_scr_inspector_value (interp, 
		atoi(argv[4]), atoi(argv[5]), atoi(argv[6]), argv[7]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector screen <oe#> <scr#> <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, 
	    "bad option \"", argv[3], "\": should be system, source, ",
	    "oe, screen.", (char*) NULL
	);
	return TCL_ERROR;
    }
    return TCL_ERROR;
}

static int set_value_msg(Tcl_Interp* interp, 
    int msgcode, const char* varname, const char* value
) {
    int retcode = TCL_ERROR;
    switch (msgcode) {
	case Beamline::OK:
	    retcode = TCL_OK;
	    break;
	case Beamline::BAD_VARIABLE:
	    Tcl_AppendResult(interp, 
		"ERROR: bad variable name \"", varname, "\" in namelist",
		(char*) NULL
	    );
	    break;
	case Beamline::BAD_ENUM_VALUE:
	    Tcl_AppendResult(interp, 
		"ERROR: bad value \"", value, "\" for ENUM Namelist item \"", 
		varname, "\".", (char*) NULL
	    );
	    break;
	case Beamline::BAD_VALUE:
	    Tcl_AppendResult(interp, 
		"ERROR: bad value \"", value, "\" for Namelist item \"", 
		varname, "\".", (char*) NULL
	    );
	    break;
	default:
	    Tcl_AppendResult(interp, 
		"ERROR: Cannot set value \"", value, "\" for Namelist item \"", 
		varname, "\".", (char*) NULL
	    );
	    break;
    }
    return retcode;
}

static int set_source_value(Tcl_Interp* interp, char* varname, char* value) {
    int retcode = GLB_beamline->set_source_value(varname, value);
    switch (retcode) {
	case Beamline::OK:
	    return get_source_value(interp, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}

static int set_oe_value(Tcl_Interp* interp, 
    int oe, char* varname, char* value
) {
    int retcode = GLB_beamline->set_oe_value(oe, varname, value);
    switch (retcode) {
	case Beamline::OK:
	    return get_oe_value(interp, oe, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}

static int set_scr_value(Tcl_Interp* interp, 
    int oe, int scr, char* varname, char* value
) {
    int retcode = GLB_beamline->set_scr_value(oe, scr, varname, value);
    switch (retcode) {
	case Beamline::OK:
	    return get_scr_value(interp, oe, scr, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}

static int set_beamline_inspector_value (Tcl_Interp* interp, 
    int id, char* varname, char* value
) {
    int retcode = GLB_beamline->set_inspector_value(id, varname, value);
    switch (retcode) {
	case Beamline::OK:
	    return get_beamline_inspector_value(interp, id, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}

static int set_source_inspector_value (Tcl_Interp* interp, 
    int id, char* varname, char* value
) {
    int retcode = GLB_beamline->set_source_inspector_value(id, varname, value);
    switch (retcode) {
	case Beamline::OK:
	    return get_source_inspector_value(interp, id, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}


static int set_oe_inspector_value (Tcl_Interp* interp, 
    int oe, int id, char* varname, char* value
) {
    int retcode = GLB_beamline->set_oe_inspector_value(oe, id, varname, value);
    switch (retcode) {
	case Beamline::OK:
	    return get_oe_inspector_value(interp, oe, id, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}

static int set_scr_inspector_value (Tcl_Interp* interp, 
    int oe, int scr, int id, char* varname, char* value
) {
    int retcode = GLB_beamline->set_scr_inspector_value (
	oe, scr, id, varname, value
    );
    switch (retcode) {
	case Beamline::OK:
	    return get_scr_inspector_value(interp, oe, scr, id, varname);
	    break;
	default:
	    return set_value_msg(interp, retcode, varname, value);
	    break;
    }
    return TCL_ERROR;
}

/*
 * beamline vset inspector system <id#> var value
 * beamline vset inspector source <id#> var value
 * beamline vset inspector oe <oe#> <id#> var value
 * beamline vset inspector scr <oe#> <scr#> <id#> var value
 */
static int set_inspector_value (Tcl_Interp* interp, int argc, char** argv){
    if (argc < 7 || argc > 9) {
	Tcl_AppendResult(interp, 
	    "wrong # args: should be \"beamline vset inspector [args]\"",
	    (char *) NULL
	);
	return TCL_ERROR;
    }
    if (strncasecmp(argv[3], "sy", 2) == 0) {
	if (argc == 7) {
	    return set_beamline_inspector_value (interp, 
		atoi(argv[4]), argv[5], argv[6]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector system <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
	if (argc == 7) {
	    return set_source_inspector_value (interp, 
		atoi(argv[4]), argv[5], argv[6]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector source <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (argv[3][0] == 'o') {
	if (argc == 8) {
	    return set_oe_inspector_value (interp, 
		atoi(argv[4]), atoi(argv[5]), argv[6], argv[7]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector oe <oe#> <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
	if (argc == 9) {
	    return set_scr_inspector_value (interp, 
		atoi(argv[4]), atoi(argv[5]), atoi(argv[6]), argv[7], argv[8]
	    );
	} else {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" inspector screen <oe#> <scr#> <id> <var>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, 
	    "bad option \"", argv[3], "\": should be system, source, ",
	    "oe, screen.", (char*) NULL
	);
	return TCL_ERROR;
    }
}

// ==================================================================== //
//
// "Beamline tget" commands
//

static int get_source_title(Tcl_Interp* interp) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_source();
    if (ti == nil) {
	Tcl_AppendResult(interp, "SOURCE not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

static int get_oe_title(Tcl_Interp* interp, int oe) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_oe(oe);
    if (ti == nil) {
	Tcl_AppendResult(interp, "OE not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

static int get_scr_title(Tcl_Interp* interp, int oe, int scr) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_scr(oe, scr);
    if (ti == nil) {
	Tcl_AppendResult(interp, "SCR not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

static int get_beamline_inspector_title(Tcl_Interp* interp, int id) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_inspector(id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

static int get_source_inspector_title(Tcl_Interp* interp, int id) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_source_inspector(id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for SOURCE",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

static int get_oe_inspector_title(Tcl_Interp* interp, int oe, int id) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_oe_inspector(oe, id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for OE",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

static int get_scr_inspector_title (Tcl_Interp* interp,
    int oe, int scr, int id
) {
    int retcode = TCL_OK;
    const ToolInstance* ti = GLB_beamline->get_scr_inspector(oe, scr, id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for SCR",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	char* title = string_to_charptr(ti->instance_name());
	Tcl_AppendResult(interp, title, (char*) NULL);
    }
    return retcode;
}

/*
 * beamline tget inspector <id#> 
 * beamline tget inspector source <id#> 
 * beamline tget inspector oe <oe#> <id#> 
 * beamline tget inspector scr <oe#> <scr#> <id#> 
 */
static int get_inspector_title (Tcl_Interp* interp, int argc, char** argv){
    int retcode = TCL_OK;
    switch (argc) {
	case 4:
	    retcode = get_beamline_inspector_title (interp, atoi(argv[3]));
	    break;

	case 5:
	    retcode = get_source_inspector_title (interp, atoi(argv[4]));
	    break;

	case 6:
	    retcode = get_oe_inspector_title (interp, 
		atoi(argv[4]), atoi(argv[5])
	    );
	    break;

	case 7:
	    retcode = get_scr_inspector_title (interp, 
		atoi(argv[4]), atoi(argv[5]), atoi(argv[6])
	    );
	    break;
	
	default:
	    retcode = TCL_ERROR;
    }
    return retcode;
}

// ==================================================================== //
//
// "Beamline tset" commands
//

static int set_source_title(Tcl_Interp* interp, char* title) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_source();
    if (ti == nil) {
	Tcl_AppendResult(interp, "SOURCE not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

static int set_oe_title(Tcl_Interp* interp, int oe, char* title) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_oe(oe);
    if (ti == nil) {
	Tcl_AppendResult(interp, "OE not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

static int set_scr_title(Tcl_Interp* interp, int oe, int scr, char* title) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_scr(oe, scr);
    if (ti == nil) {
	Tcl_AppendResult(interp, "SCR not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

static int set_beamline_inspector_title(Tcl_Interp* interp, 
    int id, char* title
) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_inspector(id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for Beamline",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

static int set_source_inspector_title(Tcl_Interp* interp, 
    int id, char* title
) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_source_inspector(id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for SOURCE",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

static int set_oe_inspector_title(Tcl_Interp* interp, 
    int oe, int id, char* title
) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_oe_inspector(oe, id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for OE",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

static int set_scr_inspector_title (Tcl_Interp* interp,
    int oe, int scr, int id, char* title
) {
    int retcode = TCL_OK;
    ToolInstance* ti = GLB_beamline->get_scr_inspector(oe, scr, id);
    if (ti == nil) {
	Tcl_AppendResult(interp, "INSPECTOR not defined for SCR",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    } else  {
	ti->set_instance_name(title);
    }
    return retcode;
}

/*
 * beamline tset inspector <id#> <title>
 * beamline tset inspector source <id#> <title>
 * beamline tset inspector oe <oe#> <id#> <title>
 * beamline tset inspector scr <oe#> <scr#> <id#> <title>
 */
static int set_inspector_title (Tcl_Interp* interp, int argc, char** argv){
    int retcode = TCL_OK;
    switch (argc) {
	case 5:
	    retcode = set_beamline_inspector_title (interp, atoi(argv[3]),
		argv[4]
	    );
	    break;

	case 6:
	    retcode = set_source_inspector_title (interp, 
		atoi(argv[4]), argv[5]
	    );
	    break;

	case 7:
	    retcode = set_oe_inspector_title (interp, 
		atoi(argv[4]), atoi(argv[5]), argv[6]
	    );
	    break;

	case 8:
	    retcode = set_scr_inspector_title (interp, 
		atoi(argv[4]), atoi(argv[5]), atoi(argv[6]), argv[7]
	    );
	    break;
	
	default:
	    retcode = TCL_ERROR;
    }
    return retcode;
}

// ==================================================================== //
//
// "Beamline load" commands
//

static int load_source(Tcl_Interp* interp, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->load_source(file)) {
	Tcl_AppendResult(interp, "Error loading SOURCE from \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int load_oe(Tcl_Interp* interp, int oe, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->load_oe(file, oe)) {
	Tcl_AppendResult(interp, "Error loading OE from \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int load_scr(Tcl_Interp* interp, int oe, int scr, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->load_scr(file, oe, scr)) {
	Tcl_AppendResult(interp, "Error loading SCR from \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int load_system(Tcl_Interp* interp, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->load_system(file)) {
	Tcl_AppendResult(interp, "Error loading SYSTEM from \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

/*
 *
 * beamline load inspector <id> <file>
 * beamline load inspector source <id> <file>
 * beamline load inspector oe <oe#> <id> <file>
 * beamline load inspector scr <oe#> <scr#> <id> <file>
 */

static int load_inspector(Tcl_Interp*, int argc, char** argv) {
    int retcode = 0;
    Inspector* inspector = nil;
    if (argc == 5) {
	int id = atoi(argv[3]);
        inspector = GLB_beamline->get_inspector(id);
    } else if (argc == 6) {
	int id = atoi(argv[4]);
        inspector = GLB_beamline->get_source_inspector(id);
    } else if (argc == 7) {
	int oe = atoi(argv[4]);
	int id = atoi(argv[5]);
        inspector = GLB_beamline->get_oe_inspector(oe, id);
    } else if (argc == 8) {
	int oe = atoi(argv[4]);
	int scr = atoi(argv[5]);
	int id = atoi(argv[6]);
        inspector = GLB_beamline->get_scr_inspector(oe, scr, id);
    } else {
        retcode = 1;
    }

    if (inspector) {
        retcode = GLB_beamline->load_nml(argv[argc-1], inspector);
    }
    return (retcode == 0) ? TCL_OK : TCL_ERROR;
}

static int store_source(Tcl_Interp* interp, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->store_source(file)) {
	Tcl_AppendResult(interp, "Error storing SOURCE to \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int store_oe(Tcl_Interp* interp, int oe, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->store_oe(file, oe)) {
	Tcl_AppendResult(interp, "Error storing OE to \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int store_scr(Tcl_Interp* interp, int oe, int scr, char* file) {
    int retcode = TCL_OK;
    if (GLB_beamline->store_scr(file, oe, scr)) {
	Tcl_AppendResult(interp, "Error storing SCR to \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int store_system(Tcl_Interp* interp, char* file, char* startprefix) {
    int retcode = TCL_OK;
    if (GLB_beamline->store_system(file, startprefix)) {
	Tcl_AppendResult(interp, "Error storing SYSTEM to \"",
	    file, "\".", (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

/*
 *
 * beamline store inspector <id> <file>
 * beamline store inspector source <id> <file>
 * beamline store inspector oe <oe#> <id> <file>
 * beamline store inspector scr <oe#> <scr#> <id> <file>
 */

static int store_inspector(Tcl_Interp*, int argc, char** argv) {
    int retcode = 0;
    Inspector const* inspector = nil;
    if (argc == 5) {
	int id = atoi(argv[3]);
        inspector = GLB_beamline->get_inspector(id);
    } else if (argc == 6) {
	int id = atoi(argv[4]);
        inspector = GLB_beamline->get_source_inspector(id);
    } else if (argc == 7) {
	int oe = atoi(argv[4]);
	int id = atoi(argv[5]);
        inspector = GLB_beamline->get_oe_inspector(oe, id);
    } else if (argc == 8) {
	int oe = atoi(argv[4]);
	int scr = atoi(argv[5]);
	int id = atoi(argv[6]);
        inspector = GLB_beamline->get_scr_inspector(oe, scr, id);
    } else {
        retcode = 1;
    }

    if (inspector) {
        retcode = GLB_beamline->store_nml(argv[argc-1], inspector);
    }
    return (retcode == 0) ? TCL_OK : TCL_ERROR;
}

static int add_src(Tcl_Interp* interp) {
    int retcode = TCL_OK;
    if(GLB_beamline->get_source() != nil) {
	Tcl_AppendResult(interp, "SOURCE already exists. Delete first.", 
	    (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    else {
	if (GLB_beamline->add_source() == -1) {
	    Tcl_AppendResult(interp, "ERROR adding SOURCE. Reason unknown", 
		(char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
    }
    return retcode;
    Tcl_AppendResult(interp, "Cannot ADD SOURCE's yet.", (char*)NULL);
    return TCL_ERROR;
}

static int delete_src(Tcl_Interp* interp) {
    int retcode = TCL_OK;
    if (GLB_beamline->remove_source()) {
	Tcl_AppendResult(interp, "Error deleting SOURCE.", (char*)NULL);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int add_oe(Tcl_Interp* interp, int oe) {
    int retcode = TCL_OK;
    int actual = oe;
    if(GLB_beamline->add_oe(oe, actual) == -1) {
	Tcl_AppendResult(interp, "Error adding OE to SYSTEM", (char*)NULL);
	retcode = TCL_ERROR;
    }
    //
    // CAVEAT: DO NOT ADD THIS OE as the current OE. That's upto the
    // interface to decide.
    //
    return retcode;
}

static int delete_oe(Tcl_Interp* interp, int oe) {
    int retcode = TCL_OK;
    if (GLB_beamline->remove_oe(oe)) {
	Tcl_AppendResult(interp, "Error deleting OE.", (char*)NULL);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int add_scr(Tcl_Interp* interp, int oe, int scr) {
    int retcode = TCL_OK;
    int actual = scr;
    if(GLB_beamline->add_scr(oe, scr, actual) == -1) {
	Tcl_AppendResult(interp, "Error adding SCR to OE", (char*)NULL);
	retcode = TCL_ERROR;
    }
    //
    // CAVEAT: DO NOT ADD THIS SCREEN as the current OE. That's upto the
    // interface to decide.
    //
    return retcode;
}

static int delete_scr(Tcl_Interp* interp, int oe, int scr) {
    int retcode = TCL_OK;
    if (GLB_beamline->remove_scr(oe, scr)) {
	Tcl_AppendResult(interp, "Error deleting SCREEN.", (char*)NULL);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int add_tool(Tcl_Interp* interp, const char* toolname) {
    int retcode = TCL_OK;
    if(GLB_beamline->add_tool(toolname) == -1) {
	Tcl_AppendResult (interp, 
	    "Error adding TOOL \"", toolname, "\".", (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int delete_tool(Tcl_Interp* interp, const char* toolname) {
    int retcode = TCL_OK;
    if(GLB_beamline->remove_tool(toolname) == -1) {
	Tcl_AppendResult (interp, 
	    "Error deleting TOOL \"", toolname, "\".", (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int delete_beamline_inspector(Tcl_Interp*, int id) { 
    return (GLB_beamline->remove_inspector(id) == 0) ? TCL_OK : TCL_ERROR;
}

static int delete_source_inspector(Tcl_Interp*, int id) { 
    return (GLB_beamline->remove_source_inspector(id) == 0) ? 
        TCL_OK : TCL_ERROR;
}

static int delete_oe_inspector(Tcl_Interp*, int oe, int id) { 
    return (GLB_beamline->remove_oe_inspector(oe, id) == 0) ? 
        TCL_OK : TCL_ERROR;
}

static int delete_scr_inspector(Tcl_Interp*, int oe, int scr, int id) { 
    return (GLB_beamline->remove_scr_inspector(oe, scr, id) == 0) ? 
        TCL_OK : TCL_ERROR;
}

/*
 * beamline delete inspector system <id>
 * beamline delete inspector source <id>
 * beamline delete inspector oe <oe#> <id>
 * beamline delete inspector screen <oe#> <scr#> <id>
 */
static int delete_inspector(Tcl_Interp* interp, int argc, char** argv) {
    if (strncasecmp(argv[3], "sy", 2) == 0) {
        if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" remove inspector system <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return delete_beamline_inspector(interp, atoi(argv[5]));
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
        if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" remove inspector source <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return delete_source_inspector(interp, atoi(argv[4]));
    } else if (argv[3][0] == 'o') {
        if (argc != 6) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" remove inspector oe <oe#> <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return delete_oe_inspector(interp, atoi(argv[4]), atoi(argv[5]));
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
        if (argc != 7) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" remove inspector screen <oe#> <scr#> <inspector> <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return delete_scr_inspector(interp,
	    atoi(argv[4]), atoi(argv[5]), atoi(argv[6])
	);
    } else {
	Tcl_AppendResult(interp, 
	    "wrong # args: should be \"", argv[0], 
	    " remove inspector ?source|oe|screen? [args]\"", 
	    (char *) NULL
	);
	return TCL_ERROR;
    }
    return TCL_ERROR;
}

static int add_beamline_inspector (Tcl_Interp* interp, 
    const char* inspector, int index
) {
    int retcode = TCL_OK;
    int actual = index;
    if(GLB_beamline->add_inspector(inspector, index, actual) == -1) {
	Tcl_AppendResult(interp, 
	    "Error adding BEAMLINE INSPECTOR \"", inspector, "\".", 
	    (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int add_source_inspector (Tcl_Interp* interp, 
    const char* inspector, int index
) {
    int retcode = TCL_OK;
    int actual = index;
    if(GLB_beamline->add_source_inspector(inspector, index, actual) == -1) {
	Tcl_AppendResult(interp, 
	    "Error adding SOURCE INSPECTOR \"", inspector, "\".", (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int add_oe_inspector (Tcl_Interp* interp, 
    int oe, const char* inspector, int index
) {
    int retcode = TCL_OK;
    int actual = index;
    if(GLB_beamline->add_oe_inspector(inspector, oe, index, actual) == -1) {
	Tcl_AppendResult(interp, 
	    "Error adding OE INSPECTOR \"", inspector, "\".", (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int add_scr_inspector (Tcl_Interp* interp,
    int oe, int scr, const char* inspector, int index
) {
    int retcode = TCL_OK;
    int actual = index;
    if (GLB_beamline->add_scr_inspector(
	inspector, oe, scr, index, actual) == -1
    ) {
	Tcl_AppendResult(interp, 
	    "Error adding SCREEN  INSPECTOR \"", inspector, "\".", (char*)NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

/*
 * beamline add inspector system <inspector> <id>
 * beamline add inspector source <inspector> <id>
 * beamline add inspector oe <oe#> <inspector> <id>
 * beamline add inspector screen <oe#> <scr#> <inspector> <id>
 */
static int add_inspector(Tcl_Interp* interp, int argc, char** argv) {
    if (strncasecmp(argv[3], "sy", 2) == 0) {
        if (argc != 6) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" add inspector system <inspector> <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return add_beamline_inspector(interp, argv[4], atoi(argv[5]));
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
        if (argc != 6) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" add inspector source <inspector> <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return add_source_inspector(interp, argv[4], atoi(argv[5]));
    } else if (argv[3][0] == 'o') {
        if (argc != 7) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" add inspector oe <oe#> <inspector> <id>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return add_oe_inspector(interp, atoi(argv[4]), argv[5], atoi(argv[6]));
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
        if (argc != 8) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" add inspector screen <oe#> <scr#> <inspector> <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return add_scr_inspector(interp,
	    atoi(argv[4]), atoi(argv[5]), argv[6], atoi(argv[7])
	);
    } else {
	Tcl_AppendResult(interp, 
	    "wrong # args: should be \"", argv[0], 
	    " add inspector ?source|oe|screen? [args]\"", 
	    (char *) NULL
	);
	return TCL_ERROR;
    }
    return TCL_ERROR;
}

//
// HACK/FIXME: Get rid this crap with cur_oe and so on.
//
static int select_none(Tcl_Interp* /*interp*/) {
    GLB_beamline->cur_oe(0);
    GLB_beamline->cur_scr(0);
    GLB_beamline->cur_inspector(0);
    GLB_beamline->cur_selection(nil);
    return TCL_OK;
}

static int select_system(Tcl_Interp*) {
    return TCL_OK;
}

static int select_source(Tcl_Interp* interp) {
    select_none(interp);
    GLB_beamline->cur_selection(GLB_beamline->get_source());
    return TCL_OK;
}

static int select_oe(Tcl_Interp* interp, int oe) {
    int retcode = TCL_OK;
    if (oe == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_oe(oe);
    } else {
	const OE* oelem = GLB_beamline->get_oe(oe);
	if (oelem == nil) {
	    sprintf(g_buf, "%d", oe);
	    Tcl_AppendResult(interp, 
		"requested OE ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_oe(oe);
	    GLB_beamline->cur_selection(oelem);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_oe());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int select_screen(Tcl_Interp* interp, int oe, int scr) {
    int retcode = TCL_OK;
    if (scr == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_oe(oe);
	GLB_beamline->cur_scr(scr);
    } else {
	const Screen* screen = GLB_beamline->get_scr(oe, scr);
	if (screen == nil) {
	    sprintf(g_buf, "SCREEN %d of OE %d", scr, oe);
	    Tcl_AppendResult(interp, 
		"Requested ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	} else {
	    GLB_beamline->cur_oe(oe);
	    GLB_beamline->cur_scr(scr);
	    GLB_beamline->cur_selection(screen);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_scr());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int select_beamline_inspector(Tcl_Interp* interp, int id) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(0);
    } else {
	const Inspector* inspector = GLB_beamline->get_inspector(id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_selection(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int select_source_inspector(Tcl_Interp* interp, int id) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(id);
    } else {
	const Inspector* inspector = GLB_beamline->get_source_inspector(id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_selection(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int select_oe_inspector(Tcl_Interp* interp, int oe, int id) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(id);
    } else {
	const Inspector* inspector = GLB_beamline->get_oe_inspector(oe, id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_selection(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int select_screen_inspector(
    Tcl_Interp* interp, int oe, int scr, int id
) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(id);
    } else {
	const Inspector* inspector = 
	    GLB_beamline->get_scr_inspector(oe, scr, id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_selection(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

/* 
 * beamline select inspector system <id>
 * beamline select inspector source <id>
 * beamline select inspector oe <oe#> <id>
 * beamline select inspector screen <oe#> <scr#> <id>
 */
static int do_select_inspector(Tcl_Interp* interp, int argc, char** argv) {
    if (strncasecmp(argv[3], "sy", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select inspector system <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_beamline_inspector(interp, atoi(argv[4]));
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select inspector source <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_source_inspector(interp, atoi(argv[4]));
    } else if (argv[3][0] == 'o') {
	if (argc != 6) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select inspector oe <oe#> <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_oe_inspector(interp, atoi(argv[4]), atoi(argv[5]));
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
	if (argc != 7) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select inspector screen <oe#> <scr#> <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_screen_inspector (interp, 
	    atoi(argv[4]), atoi(argv[5]), atoi(argv[6])
	);
    } 
    return TCL_ERROR;
}

static int do_select(Tcl_Interp* interp, int argc, char** argv) {
    if (argv[2][0] == 'n') {				// none
	if (argc != 3) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select none\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_none(interp);
    } else if (strncasecmp(argv[2], "sy", 2) == 0) {	// system
	if (argc != 3) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select system\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_system(interp);
    } else if (strncasecmp(argv[2], "so", 2) == 0) {	// source
	if (argc != 3) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select source\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_source(interp);
    } else if (argv[2][0] == 'o') {			// OE
	if (argc != 4) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select oe <oe#>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_oe(interp, atoi(argv[3]));
    } else if (strncasecmp(argv[2], "sc", 2) == 0) {	// Screen
	if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select screen <oe#> <scr#>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return select_screen(interp, atoi(argv[3]), atoi(argv[4]));
    } else if (argv[2][0] == 'i') {
	if (argc < 5 || argc > 7) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" select inspector [args]\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return do_select_inspector(interp, argc, argv);
    } else {
	Tcl_AppendResult(interp, 
	    "bad option \"", argv[2], 
	    "\": should be beamline, source, oe, scr, inspector [args]",
	    (char *) NULL
	);
	return TCL_ERROR;
    }
    return TCL_ERROR;
}

static int edit_none(Tcl_Interp* /*interp*/) {
    GLB_beamline->cur_oe(0);
    GLB_beamline->cur_scr(0);
    GLB_beamline->cur_inspector(0);
    GLB_beamline->cur_editing(nil);
    return TCL_OK;
}

static int edit_system(Tcl_Interp* interp) {
    edit_none(interp);
    return TCL_OK;
}

static int edit_source(Tcl_Interp* interp) {
    edit_none(interp);
    GLB_beamline->cur_editing(GLB_beamline->get_source());
    return TCL_OK;
}

static int edit_oe(Tcl_Interp* interp, int oe) {
    int retcode = TCL_OK;
    if (oe == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_oe(oe);
    } else {
	const OE* oelem = GLB_beamline->get_oe(oe);
	if (oelem == nil) {
	    sprintf(g_buf, "%d", oe);
	    Tcl_AppendResult(interp, 
		"requested OE ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_oe(oe);
	    GLB_beamline->cur_editing(oelem);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_oe());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int edit_screen(Tcl_Interp* interp, int oe, int scr) {
    int retcode = TCL_OK;
    if (scr == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_oe(oe);
	GLB_beamline->cur_scr(scr);
    } else {
	const Screen* screen = GLB_beamline->get_scr(oe, scr);
	if (screen == nil) {
	    sprintf(g_buf, "SCREEN %d of OE %d", scr, oe);
	    Tcl_AppendResult(interp, 
		"Requested ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	} else {
	    GLB_beamline->cur_oe(oe);
	    GLB_beamline->cur_scr(scr);
	    GLB_beamline->cur_editing(screen);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_scr());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int edit_beamline_inspector(Tcl_Interp* interp, int id) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(0);
    } else {
	const Inspector* inspector = GLB_beamline->get_inspector(id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_editing(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int edit_source_inspector(Tcl_Interp* interp, int id) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(id);
    } else {
	const Inspector* inspector = GLB_beamline->get_source_inspector(id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_editing(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int edit_oe_inspector(Tcl_Interp* interp, int oe, int id) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(id);
    } else {
	const Inspector* inspector = GLB_beamline->get_oe_inspector(oe, id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_editing(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

static int edit_screen_inspector(
    Tcl_Interp* interp, int oe, int scr, int id
) {
    int retcode = TCL_OK;
    if (id == 0) {			/* special case:  UNSELECT */
	GLB_beamline->cur_inspector(id);
    } else {
	const Inspector* inspector = 
	    GLB_beamline->get_scr_inspector(oe, scr, id);
	if (inspector == nil) {
	    sprintf(g_buf, "%d", id);
	    Tcl_AppendResult(interp, 
		"requested INSPECTOR ", g_buf, " does not exist.", (char*)NULL
	    );
	    retcode = TCL_ERROR;
	}
	else {
	    GLB_beamline->cur_inspector(id);
	    GLB_beamline->cur_editing(inspector);
	}
    }
    if (retcode == TCL_OK) {
	sprintf(g_buf, "%d", GLB_beamline->cur_inspector());
	Tcl_AppendElement(interp, g_buf);
    }
    return retcode;
}

/* 
 * beamline edit inspector system <id>
 * beamline edit inspector source <id>
 * beamline edit inspector oe <oe#> <id>
 * beamline edit inspector screen <oe#> <scr#> <id>
 */
static int do_edit_inspector(Tcl_Interp* interp, int argc, char** argv) {
    if (strncasecmp(argv[3], "sy", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit inspector system <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_beamline_inspector(interp, atoi(argv[4]));
    } else if (strncasecmp(argv[3], "so", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit inspector source <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_source_inspector(interp, atoi(argv[4]));
    } else if (argv[3][0] == 'o') {
	if (argc != 6) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit inspector oe <oe#> <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_oe_inspector(interp, atoi(argv[4]), atoi(argv[5]));
    } else if (strncasecmp(argv[3], "sc", 2) == 0) {
	if (argc != 7) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit inspector screen <oe#> <scr#> <id>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_screen_inspector (interp, 
	    atoi(argv[4]), atoi(argv[5]), atoi(argv[6])
	);
    } 
    return TCL_ERROR;
}

static int do_edit(Tcl_Interp* interp, int argc, char** argv) {
    if (argv[2][0] == 'n') {
	if (argc != 3) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit none\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_none(interp);
    } else if (strncasecmp(argv[2], "sy", 2) == 0) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit system\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_system(interp);
    } else if (strncasecmp(argv[2], "so", 2) == 0) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit source\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_source(interp);
    } else if (argv[2][0] == 'o') {
	if (argc != 4) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit oe <oe#>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_oe(interp, atoi(argv[3]));
    } else if (strncasecmp(argv[2], "sc", 2) == 0) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit screen <oe#> <scr#>\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return edit_screen(interp, atoi(argv[3]), atoi(argv[4]));
    } else if (argv[2][0] == 'i') {
	if (argc < 5 || argc > 7) {
	    Tcl_AppendResult(interp, 
		"wrong # args: should be \"", argv[0], 
		" edit inspector [args]\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return do_edit_inspector(interp, argc, argv);
    }
    return TCL_ERROR;
}

static int clear_system(Tcl_Interp* interp, bool clear_src) {
    int retcode = TCL_OK;
    if (GLB_beamline->clear(clear_src)) {
	Tcl_AppendResult(interp, "Error clearing SYSTEM.", (char*)NULL);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int clear_clipboard(Tcl_Interp*) {
    GLB_beamline->clear_clipboard();
    return TCL_OK;
}

static int show_beamline_state(Tcl_Interp* interp) {
    Tcl_AppendResult (interp,
        ((GLB_beamline->saved_state()) ? "1" : "0"), (char*)NULL
    );
    return TCL_OK;
}

static int save_beamline_state(Tcl_Interp*) {
    GLB_beamline->save_state();
    return TCL_OK;
}

static int restore_beamline_state(Tcl_Interp* interp) {
    int retcode = TCL_OK;
    if (GLB_beamline->saved_state() == nil) {
        Tcl_AppendResult (interp, 
	    "No saved state! State unchanged", (char*) NULL
	);
	retcode = TCL_ERROR;
    } else {
	retcode = (GLB_beamline->restore_state() == 0) ? TCL_OK : TCL_ERROR;
    }
    return retcode;
}

inline static int bline_arg_error(Tcl_Interp* interp, 
    const char* cmd, const char* arg
) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", cmd,
	" ", arg, "\"", (char *) NULL
    );
    return TCL_ERROR;
}


/***********************************************************************/

/*
 *--------------------------------------------------------------
 *
 * XMenu_BeamlineCmd:
 *
 *      This procedure is invoked to process the "beamline" Tcl
 *      command.  
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *--------------------------------------------------------------
 */

int XMenu_BeamlineCmd (ClientData, Tcl_Interp* interp, int argc, char** argv) {
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " option [args..]\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * beamline list
     */
    if (argv[1][0] == 'l' && strncasecmp(argv[1], "li", 2) == 0) {
        if (argc < 3) {
	    Tcl_AppendResult (interp, "wrong # args: should be \"",
		argv[0], " option [args]\"", (char*) NULL
	    );
	    return TCL_ERROR;
	}
	if (strncasecmp(argv[2], "sy", 2) == 0) {
	    if (argc == 3) {
		return list_system(interp);
	    } else {
		Tcl_AppendResult (interp, "wrong # args: should be \"",
		    argv[0], " system\"", (char*) NULL
		);
		return TCL_ERROR;
	    }
	}
	Tcl_AppendResult(interp, "bad option \"", argv[2],
	    "\": should be system. ", (char*) NULL
	);
	return TCL_ERROR;
    }
    /*
     * beamline add source
     * beamline add oe <oe#>
     * beamline add screen <oe#> <scr#>
     * beamline add tool <toolname>
     *
     * beamline add inspector system <inspector> <id>
     * beamline add inspector source <inspector> <id>
     * beamline add inspector oe <oe#> <inspector> <id>
     * beamline add inspector screen <oe#> <scr#> <inspector> <id>
     */
    if (argv[1][0] == 'a' && strcasecmp(argv[1], "add") == 0) {
        if (argc < 3 || argc > 8) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " add type ?typid?", (char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so",  2) == 0) {
	    if (argc == 3)
		return add_src(interp);
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " add source\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 4) {
		return add_oe(interp, atoi(argv[3]));
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " add oe <oe#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 5) {
		return add_scr(interp, atoi(argv[3]), atoi(argv[4]));
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " add screen <oe#> <scr#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 't') {		// add tool
	    if (argc == 4) {
		return add_tool(interp, argv[3]);
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " add tool <toolname>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {		// add inspector
	    if (argc >= 6 || argc <= 8) {
		return add_inspector(interp, argc, argv);
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " add inspector [args]\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, 
		"wrong element type:  should be \"", 
		argv[0], " add source|oe|screen ?oe_id? ?screen_id?\"",
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
    }
    /*
     * beamline delete source
     * beamline delete oe <oe#>
     * beamline delete screen <oe#> <scr#>
     * beamline delete tool <tool>
     *
     * beamline delete inspector system <id>
     * beamline delete inspector source <id>
     * beamline delete inspector oe <oe#> <id>
     * beamline delete inspector screen <oe#> <scr#> <id>
     */
    if (argv[1][0] == 'd' && strcasecmp(argv[1], "delete") == 0) {
        if (argc < 3 || argc > 7) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " delete type ?typid?", (char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so",  2) == 0) {
	    if (argc == 3)
		return delete_src(interp);
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " delete source\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 4) {
		return delete_oe(interp, atoi(argv[3]));
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " delete oe <oe#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 5) {
		return delete_scr(interp, atoi(argv[3]), atoi(argv[4]));
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " delete screen <oe#> <scr#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 't') {
	    if (argc == 4) {
		return delete_tool(interp, argv[3]);
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " delete tool <toolname>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {		// delete inspector
	    if (argc >= 5 || argc <= 7) {
		return delete_inspector(interp, argc, argv);
	    }
	    else {
		Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		    argv[0], " delete inspector [args]\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, 
		"wrong element type:  should be \"", 
		argv[0], " delete source|oe|screen ?oe_id? ?screen_id?\"",
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
    }
    /*
     * beamline select none
     * beamline select system
     * beamline select source
     * beamline select oe <oe#>
     * beamline select screen <oe#> <scr#>
     *
     * beamline select inspector system <id>
     * beamline select inspector source <id>
     * beamline select inspector oe <oe#> <id>
     * beamline select inspector screen <oe#> <scr#> <id>
     */
    if (argv[1][0] == 's' && strcasecmp(argv[1], "select") == 0) {
        if (argc < 3 || argc > 7) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " select [system|source|oe|scr|inspector] [args]\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return do_select(interp, argc, argv);
    }
    /*
     * beamline edit none
     * beamline edit system
     * beamline edit source
     * beamline edit oe <oe#>
     * beamline edit screen <oe#> <scr#>
     *
     * beamline edit inspector system <id>
     * beamline edit inspector source <id>
     * beamline edit inspector oe <oe#> <id>
     * beamline edit inspector screen <oe#> <scr#> <id>
     */
    if (argv[1][0] == 'e' && strcasecmp(argv[1], "edit") == 0) {
        if (argc < 3 || argc > 7) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " edit [system|source|oe|scr|inspector] [args]\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	return do_edit(interp, argc, argv);
    }
    /*
     * beamline get current
     * beamline get source
     * beamline get oe <oe#>
     * beamline get screen <oe#> <scr#>
     * beamline get inspectors system
     * beamline get inspectors source
     * beamline get inspectors oe <oe#>
     * beamline get inspectors scr <oe#> <scr#>
     */
    if (argv[1][0] == 'g' && strcasecmp(argv[1], "get") == 0) {
        if (argc < 3 || argc > 6) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " get current|source|oe|screen ?oe_id? ?screen_id?", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (argv[2][0] == 'c') {
	    if (argc == 3) {
		return get_current(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " get source\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 3) {
		return get_source(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " get source\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 4) { 
		return get_oe(interp, atoi(argv[3]));
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " get oe <oe#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 5) { 
		return get_scr(interp, atoi(argv[3]), atoi(argv[4]));
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " get screen <oe#> <scr#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {
	    return get_inspectors(interp, argc, argv);
	}
    }
    /*
     * beamline vget source <var>
     * beamline vget oe <oe#> <var>
     * beamline vget screen <oe#> <scr#> <var>
     *
     * beamline vget inspector system <id#> var
     * beamline vget inspector source <id#> var
     * beamline vget inspector oe <oe#> <id#> var
     * beamline vget inspector scr <oe#> <scr#> <id#> var
     */
    if (argv[1][0] == 'v' && strncasecmp(argv[1], "vg", 2) == 0) {
        if (argc < 4 || argc > 8) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " vget source|oe|scr|inspector [args] var", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 4) {
		return get_source_value(interp, argv[3]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vget source <var>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 5) {
		return get_oe_value(interp, atoi(argv[3]), argv[4]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vget oe <oe#> <var>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 6) { 
		return get_scr_value(
		    interp, atoi(argv[3]), atoi(argv[4]), argv[5]
		);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vget screen <oe#> <scr#> <var>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {			// inspector
	    if (argc >= 6 && argc <= 8) {
		return get_inspector_value (interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vget inspector [args] var\"",
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    /*
     * beamline vset source var value
     * beamline vset oe <oe#> var value
     * beamline vset screen <oe#> <scr#> var value
     *
     * beamline vset inspector system <id#> var value
     * beamline vset inspector source <id#> var value
     * beamline vset inspector oe <oe#> <id#> var value
     * beamline vset inspector scr <oe#> <scr#> <id#> var value
     */
    if (argv[1][0] == 'v' && strncasecmp(argv[1], "vs", 2) == 0) {
        if (argc < 5 || argc > 9) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " vset source|oe|scr|inspector [args] var val\"",
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 5) {
		return set_source_value(interp, argv[3], argv[4]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vset source <var> <val>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 6) {
		return set_oe_value(interp, atoi(argv[3]), argv[4], argv[5]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vset oe <oe#> <var> <val>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 7) { 
		return set_scr_value(
		    interp, atoi(argv[3]), atoi(argv[4]), argv[5], argv[6]
		);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vset screen <oe#> <scr#> <var> <val>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {			// inspector
	    if (argc >= 7 && argc <= 9) {
		return set_inspector_value (interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " vset inspector [args]\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    /*
     * beamline tget source
     * beamline tget oe <oe#> 
     * beamline tget scr <oe#> <scr#>
     *
     * beamline tget insector source <id>
     * beamline tget insector oe <oe#> <id>
     * beamline tget inspector scr <oe#> <scr#> <id>
     *
     */
    if (argv[1][0] == 't' && strncasecmp(argv[1], "tg", 2) == 0) {
        if (argc < 3 || argc > 7) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " tget source|oe|scr|inspector [args] var", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 3) {
		return get_source_title(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tget source\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 4) {
		return get_oe_title(interp, atoi(argv[3]));
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tget oe <oe#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 5) { 
		return get_scr_title(
		    interp, atoi(argv[3]), atoi(argv[4])
		);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tget screen <oe#> <scr#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {			// inspector
	    if (argc >= 5 && argc <= 7) {
		return get_inspector_title (interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tget inspector [args] var\"",
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    /*
     * beamline tset source <title>
     * beamline tset oe <oe#>  <title>
     * beamline tset scr <oe#> <scr#> <title>
     *
     * beamline tset insector source <id> <title>
     * beamline tset insector oe <oe#> <id> <title>
     * beamline tset inspector scr <oe#> <scr#> <id> <title>
     *
     */
    if (argv[1][0] == 't' && strncasecmp(argv[1], "ts", 2) == 0) {
        if (argc < 4 || argc > 8) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " tset source|oe|scr|inspector [args] var", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 4) {
		return set_source_title(interp, argv[3]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tset source <title>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 5) {
		return set_oe_title(interp, atoi(argv[3]), argv[4]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tset oe <oe#> <title>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 6) { 
		return set_scr_title(
		    interp, atoi(argv[3]), atoi(argv[4]), argv[5]
		);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tset screen <oe#> <scr#> <title>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {			// inspector
	    if (argc >= 5 && argc <= 8) {
		return set_inspector_title (interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " tset inspector [args] var\"",
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    /*
     * beamline load source <file>
     * beamline load oe <oe#> <file>
     * beamline load screen <oe#> <scr#> <file>
     * beamline load system <file>
     * beamline load tool <toolname> <file>
     *
     * beamline load inspector <id> <file>
     * beamline load inspector source <id> <file>
     * beamline load inspector oe <oe#> <id> <file>
     * beamline load inspector scr <oe#> <scr#> <id> <file>
     */
    if (argv[1][0] == 'l' && strncasecmp(argv[1], "lo", 2) == 0) {
        if (argc < 4 || argc > 6) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " load source|oe|scr ?oe_id? ?screen_id? file", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 4) {
		return load_source(interp, argv[3]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " load source <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 5) {
		return load_oe(interp, atoi(argv[3]), argv[4]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " load oe <oe#> <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 6) { 
		return load_scr(
		    interp, atoi(argv[3]), atoi(argv[4]), argv[5]
		);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " load screen <oe#> <scr#> <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sy", 2) == 0) {
	    if (argc == 4) {
		return load_system(interp, argv[3]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " load system <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {
	    if (argc >= 5 && argc <= 8) {
		return load_inspector(interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " load inspector [args]\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    /*
     * beamline store source <file>
     * beamline store oe <oe#> <file>
     * beamline store screen <oe#> <scr#> <file>
     * beamline store tool <toolname> <file>
     *
     * beamline store inspector <id> <file>
     * beamline store inspector source <id> <file>
     * beamline store inspector oe <oe#> <id> <file>
     * beamline store inspector scr <oe#> <scr#> <id> <file>
     *
     */
    if (argv[1][0] == 's' && strncasecmp(argv[1], "sto", 3) == 0) {
        if (argc < 4 || argc > 8) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " store source|oe|scr ?oe_id? ?screen_id? file", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 4) {
		return store_source(interp, argv[3]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " store source <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 5) {
		return store_oe(interp, atoi(argv[3]), argv[4]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " store oe <oe#> <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 6) { 
		return store_scr(
		    interp, atoi(argv[3]), atoi(argv[4]), argv[5]
		);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " store screen <oe#> <scr#> <file>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sy", 2) == 0) {
	    if (argc == 4) {
		return store_system(interp, argv[3], "start");
	    } else if (argc == 5) {
		return store_system(interp, argv[3], argv[4]);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " store system <file> [startprefix]\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {
	    if (argc >= 5 && argc <= 8) {
		return store_inspector(interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " store inspector [args]\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    /*
     * beamline clear
     * beamline clear oe
     */
    if (argv[1][0] == 'c' && strcasecmp(argv[1], "clear") == 0) {
        if (argc == 3) {
	    if (argv[2][0] == 'o')
		return clear_system(interp, false);
	    else if (argv[2][0] == 'c')
	        return clear_clipboard(interp);
	    else
		return bline_arg_error(interp, argv[0], argv[1]);
	} else if (argc == 2) {
	    return clear_system(interp, true);
	} else
	    return bline_arg_error(interp, argv[0], argv[1]);
    }
    /*
     * beamline selection copy
     * beamline selection paste
     * beamline selection clear
     */
    if (argv[1][0] == 's' && strcasecmp(argv[1], "selection") == 0) {
        if (argc != 3) {
	    Tcl_AppendResult (interp, 
		"wrong # args: should be \"", argv[0], " copy selection\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "co", 2) == 0) {	// copy
	    return copy_selection_to_clipboard(interp);
	} else if (strncasecmp(argv[2], "pa", 2) == 0) {	// paste
	    return paste_selection_from_clipboard(interp);
	} else if (strncasecmp(argv[2], "cl", 2) == 0) {	// clear
	    return clear_clipboard(interp);
	}
    }

    /* 
     * beamline state show
     * beamline state save
     * beamline state restore
     *
     */
    if (argv[1][0] == 's' && strncasecmp(argv[1], "sta", 3) == 0) {
        if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " state show|save|restore\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	} else if (strncasecmp(argv[2], "sh", 2) == 0) {
	    return show_beamline_state(interp);
	} else if (strncasecmp(argv[2], "sa", 2) == 0) {
	    return save_beamline_state(interp);
	} else if (argv[2][0] == 'r') {
	    return restore_beamline_state(interp);
	} else {
	    Tcl_AppendResult (interp, "bad option \"", argv[2],
		"\":  should be show, save or restore", (char*) NULL
	    );
	    return TCL_ERROR;
	}
    }

    /*
     * beamline dump system
     * beamline dump source
     * beamline dump oe <oe#>
     * beamline dump scr <oe#> <scr#>
     *
     * beamline dump inspector system <id>
     * beamline dump inspector source <id>
     * beamline dump inspector oe <oe#> <id>
     * beamline dump inspector scr <oe#> <scr#> <id>
     */

    if (argv[1][0] == 'd' && strcasecmp(argv[1], "dump") == 0) {
        if (argc < 3 || argc > 7) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " dump [args]\"", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	if (strncasecmp(argv[2], "sy", 2) == 0) {
	    if (argc == 3) {
		return dump_system(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], " dump system\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "so", 2) == 0) {
	    if (argc == 3) {
		return dump_source(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], " dump source\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'o') {
	    if (argc == 4) {
		return dump_oe(interp, atoi(argv[3]));
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], " dump oe <oe#>\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "sc", 2) == 0) {
	    if (argc == 5) { 
		return dump_scr(interp, atoi(argv[3]), atoi(argv[4]));
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " dump screen <oe#> <scr#>\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "se", 2) == 0) {
	    if (argc == 3) {
		return dump_selection(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], " dump election\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (strncasecmp(argv[2], "cl", 2) == 0) {
	    if (argc == 3) {
		return dump_clipboard(interp);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], " dump clipboard\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	} else if (argv[2][0] == 'i') {
	    if (argc >= 5 && argc <= 7) {
		return dump_inspector(interp, argc, argv);
	    } else {
		Tcl_AppendResult(interp, 
		    "wrong # args: should be \"", argv[0], 
		    " dump inspector [args]\"", (char *) NULL
		);
		return TCL_ERROR;
	    }
	}
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
	"\":  should be list, select, add, delete, clear, get, vget, vset, ",
	"edit, tget, tset, load, store, dump", (char*) NULL
    );
    return TCL_ERROR;
}
