//
// synonym.cc: manage NAME->VALUE synonyms
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

/***************************************************************************/

#ifdef __GNUG__
# include <cassert>
# include <cstdio>
# include <cstdlib>
# include <cstring>
#else
# include <assert>
# include <stdio>
# include <stdlib>
# include <string>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#if defined(NO_STRNCASECMP_PROTO)
extern "C" int strncasecmp(const char*, const char*, int);
#endif

#include "synonym.h"
#include "utils.h"

/***********************************************************************/

class Synonym {
public:
    Synonym(const char* variable, const char* symbol = nil, int value = 0);
    virtual ~Synonym();
    int add(const char* symbol, int value);
    bool exists(const char* symbol) const;
    bool exists(int value) const;
    bool exists(const char* symbol, SynonymTable::MatchType&) const;
    bool exists(int value, SynonymTable::MatchType&) const;
    int translate(const char* symbol) const;
    const char* translate(int value) const;
    int translate(const char* symbol, SynonymTable::MatchType&) const;
    const char* translate(int value, SynonymTable::MatchType&) const;
    const Stq* symbols() const;

    void dump() const;
private:
    const char* unambiguate(const char* abbrev, SynonymTable::MatchType&)const;

private:
    char* variable_;
    Stq* sym_names_;
    Hashtable* sym_tab_;
};

inline const Stq* Synonym::symbols() const { return sym_names_; }

Synonym::Synonym(const char* variable, const char* symbol, int value) {
    sym_names_ = new Stq;
    sym_tab_ = new Hashtable(7);
    variable_ = strcpy_with_alloc(variable);
    if (symbol) {
	add(symbol, value);
    }
}

Synonym::~Synonym() {
    unsigned nsymbols = sym_names_->size();
    for(unsigned i = 0; i < nsymbols; i++) {
	char* symbol = (char*)sym_names_->pop();
	int* valptr = (int*)sym_tab_->find(symbol);
	assert(valptr != nil);
	delete[] symbol;
	delete valptr;
    }
    delete sym_names_;
    delete sym_tab_;
}

int Synonym::add(const char* symbol, int value) {
    int retcode = -1;
    if (sym_tab_->find(symbol) == nil) {
	const char* lsymbol = strcpy_with_alloc(symbol);
	sym_tab_->insert(lsymbol, new int(value));
	sym_names_->enq(lsymbol);
	retcode = 0;
    }
    return retcode;
}

bool Synonym::exists(const char* symbol) const {
    return sym_tab_->find(symbol) != nil;
}

//
// inverse transform. 
// gotcha: returns the first one that matches the value.
//
bool Synonym::exists(int value) const {
    bool found = false;
    unsigned nsymbols = sym_names_->size();
    for(unsigned i = 1; i <= nsymbols && !found; i++) {
	char* symbol = (char*)sym_names_->get(i);
	int* valptr = (int*)sym_tab_->find(symbol);
	assert(valptr != nil);
	found = *valptr == value;
    }
    return found;
}

const char* Synonym::unambiguate(
    const char* abbrev, SynonymTable::MatchType& match
) const {
    unsigned matches = 0;
    unsigned nsymbols = sym_names_->size();
    unsigned abbrev_len = strlen(abbrev);
    char* s_unique = nil;
    for(unsigned i = 0; i < nsymbols; i++) {
	char* sym_name = (char*)sym_names_->cycle();
	if (strncasecmp(abbrev, sym_name, abbrev_len) == 0) {
	    if (matches == 0)
		s_unique = sym_name;
	    matches++;
	}
    }
    if (matches == 0) {
	match = SynonymTable::NONE;
    } else if (matches == 1) {
	match = SynonymTable::UNIQUE;
    } else if (matches > 1) {
	match = SynonymTable::AMBIGUOUS;
    }
    return s_unique;
}


bool Synonym::exists(
    const char* symbol, SynonymTable::MatchType& match
) const {
    return unambiguate(symbol, match) != nil;
}

//
// inverse transform. 
//
bool Synonym::exists(int value, SynonymTable::MatchType& match) const
{
    unsigned matches = 0;
    unsigned nsymbols = sym_names_->size();
    for(unsigned i = 0; i < nsymbols; i++) {
	const char* symbol = (char*)sym_names_->cycle();
	int* valptr = (int*)sym_tab_->find(symbol);
	assert(valptr != nil);
	if (*valptr == value) {
	    ++matches;
	}
    }
    if (matches == 0) {
	match = SynonymTable::NONE;
    } else if (matches == 1) {
	match = SynonymTable::UNIQUE;
    } else if (matches > 1) {
	match = SynonymTable::AMBIGUOUS;
    }
    return (matches != 0);
}

//
// ASSUME existence.
//
int Synonym::translate(const char* symbol) const {
    int* valptr = (int*)sym_tab_->find(symbol);
    assert(valptr != nil);
    return *valptr;
}

//
// ASSUME existence.
//
const char* Synonym::translate(int value) const {
    bool found = false;
    char const* symbol = nil;
    unsigned nsymbols = sym_names_->size();
    for(unsigned i = 1; i <= nsymbols && !found; i++) {
	char* sym = (char*)sym_names_->get(i);
	int* valptr = (int*)sym_tab_->find(sym);
	assert(valptr != nil);
	if (*valptr == value) {
	    found = true;
	    symbol = sym;
	}
    }
    assert(found == true);
    return symbol;
}

//
// ASSUME existence?
//
int Synonym::translate(
    const char* symbol, SynonymTable::MatchType& match
) const {
    const char* s_unique = unambiguate(symbol, match);
    int value = 0;
    if (s_unique) {
	int* valptr = (int*)sym_tab_->find(s_unique);
	assert(valptr != nil);
	value = *valptr;
    }
    return value;
}

//
// ASSUME existence.
//
const char* Synonym::translate(
    int value, SynonymTable::MatchType& match
) const {
    unsigned matches = 0;
    char const* symbol = nil;
    unsigned nsymbols = sym_names_->size();
    for(unsigned i = 0; i < nsymbols; i++) {
	const char* sym = (char*)sym_names_->cycle();
	int* valptr = (int*)sym_tab_->find(sym);
	assert(valptr != nil);
	if (*valptr == value) {
	    if (matches++ == 0)
		symbol = sym;
	}
    }
    if (matches == 0) {
	match = SynonymTable::NONE;
    } else if (matches == 1) {
	match = SynonymTable::UNIQUE;
    } else if (matches > 1) {
	match = SynonymTable::AMBIGUOUS;
    }
    return symbol;
}

void Synonym::dump() const {
    printf("\tSYNONYM: %s\n", variable_);
    unsigned nsymbols = sym_names_->size();
    for(unsigned i = 0; i < nsymbols; i++) {
	const char* sym = (char*)sym_names_->cycle();
	int* valptr = (int*)sym_tab_->find(sym);
	assert(valptr != nil);
	printf("\t\t%s ==> %d\n", sym, *valptr);
    }
}

/***********************************************************************/

SynonymTable::SynonymTable(unsigned hash_tabsize) {
    syn_list_ = new Stq;
    syn_tab_ = new Hashtable(hash_tabsize);
}

SynonymTable::~SynonymTable() {
    unsigned nsyns = syn_list_->size();
    for (unsigned i = 0; i < nsyns; i++) {
	Synonym* item = (Synonym*) syn_list_->pop();
	delete item;
    }
    delete syn_list_;
    delete syn_tab_;
}

int SynonymTable::add(const char* variable, const char* symbol, int value) {
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    if (synonym == nil) {
	synonym = new Synonym(variable);
	const char* lvariable = strcpy_with_alloc(variable);
	syn_tab_->insert(lvariable, synonym);
	syn_list_->enq(lvariable);
    }
    return synonym->add(symbol, value);
}

int SynonymTable::translate(const char* variable, const char* symbol) const {
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    assert(synonym);
    return synonym->translate(symbol);
}

const char* SynonymTable::translate(const char* variable, int value) const {
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    assert(synonym);
    return synonym->translate(value);
}

int SynonymTable::translate(
    const char* variable, const char* symbol, MatchType& match
) const {
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    assert(synonym);
    return synonym->translate(symbol, match);
}

const char* SynonymTable::translate(
    const char* variable, int value, MatchType& match
) const {
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    assert(synonym);
    return synonym->translate(value, match);
}


const Stq* SynonymTable::get_symbols(const char* variable) const {
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    assert(synonym);
    return synonym->symbols();
}

bool SynonymTable::exists(const char* variable, const char* symbol) const {
    bool does_exist = false;
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    if (synonym)
	does_exist = synonym->exists(symbol);
    return does_exist;
}

bool SynonymTable::exists(const char* variable, int value) const {
    bool does_exist = false;
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    if (synonym)
	does_exist = synonym->exists(value);
    return does_exist;
}

bool SynonymTable::exists(
    const char* variable, const char* symbol, MatchType& match
) const {
    bool does_exist = false;
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    if (synonym)
	does_exist = synonym->exists(symbol, match);
    else
	match = NONE;
    return does_exist;
}

bool SynonymTable::exists(
    const char* variable, int value, MatchType& match
) const {
    bool does_exist = false;
    Synonym* synonym = (Synonym*)syn_tab_->find(variable);
    if (synonym)
	does_exist = synonym->exists(value, match);
    else
	match = NONE;
    return does_exist;
}

void SynonymTable::dump() const {
    printf("DUMPING synonym table ...\n");
    unsigned nsyns = syn_list_->size();
    for(unsigned i = 0; i < nsyns; i++) {
	const char* variable = (char*)syn_list_->cycle();
	assert(variable != nil);
	const Synonym* synonym = (Synonym*)syn_tab_->find(variable);
	assert(synonym != nil);
	synonym->dump();
    }
}

/***********************************************************************/
