//
// beamline.h: SHADOW beamline interface
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

#ifndef shadow_beamline_h
#define shadow_beamline_h

// ======================================================================= //

#include <string>
#include <deque>

#include "xrtypes.h"
#include "tool.h"

// ======================================================================= //

class SynonymTable;
class Inspector;

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

// ======================================================================= //

//
// class BLToolInstance: used to  store an instance of a beamline tool.
// This is really only for lack of usable RTTI on various platforms.
//
class BLToolInstance : public ToolInstance {
public:
    enum {unspecified, source, oe, screen, inspector};
public:
    BLToolInstance(const Tool* tool, int instance, 
	int tool_type = BLToolInstance::unspecified
    );
    BLToolInstance(const BLToolInstance&);
    virtual ~BLToolInstance();

    virtual BLToolInstance* clone() const;
    virtual int copy(const BLToolInstance& from);

    int tool_type() const;

    virtual bool is_source() const;
    virtual bool is_oe() const;
    virtual bool is_screen() const;
    virtual bool is_inspector() const;
    virtual bool is_unspecified() const;

    virtual void dump() const;

private:
    int tool_type_;
};

class Inspector : public BLToolInstance {
public:
    Inspector(const Tool* tool, int id, const BLToolInstance* target);
    Inspector(const Inspector&);
    virtual ~Inspector();

    virtual Inspector* clone() const;
    virtual int copy(const BLToolInstance& from);

    virtual const BLToolInstance* get_target() const;
    virtual int get_id() const;
    virtual void set_id(int id);
    virtual void dump() const;

private:
    int id_;
    const BLToolInstance* target_;
};

class InspectorMgr {
public:
    InspectorMgr();
    InspectorMgr(const InspectorMgr&);
    virtual ~InspectorMgr();

    virtual InspectorMgr* clone() const;
    virtual int copy(const InspectorMgr& from);

    virtual int num_inspectors() const;
    virtual int add_inspector(Inspector*, int id);
    virtual int remove_inspector(int id);
    virtual Inspector* get_inspector(int index) const;
    virtual Inspector* get_inspector(const string& toolname, int id) const;

    virtual void dump() const;

private:
    void clear();
    virtual int renumber_inspectors();

private:
    deque<void*> inspectors_;
};

class Source : public BLToolInstance, public InspectorMgr {
public:
    Source(const Tool* tool);
    Source(const Source& source);
    virtual ~Source();

    virtual Source* clone() const;
    virtual int copy(const BLToolInstance& from);

    virtual void dump() const;
};

class Screen : public BLToolInstance, public InspectorMgr {
public:
    Screen(const Tool* tool, int oe, int scr);
    Screen(const Screen& screen);
    virtual ~Screen();

    virtual Screen* clone() const;
    virtual int copy(const BLToolInstance& from);

    void get_id(int& oe, int& scr) const;
    void set_id(int oe, int scr);

    virtual void dump() const;

private:
    int oe_;
    int scr_;
};

class OE : public BLToolInstance, public InspectorMgr {
public:
    OE(const Tool* tool, int id);
    OE(const OE& oe);
    virtual ~OE();

    virtual OE* clone() const;
    virtual int copy(const BLToolInstance& from);

    int get_id() const;
    int num_scr() const;
    Screen* get_screen(int scr) const;

    void set_id(int);
    int add_scr(const Tool*, int req, int& actual);
    int remove_scr(int scr);

    virtual void dump() const;

private:
    int clear();
    int reset_scr_variables();

private:
    int id_;
    deque<void*> scr_list_;
};

// ======================================================================= //

class Beamline : public InspectorMgr {
public:
    // various error codes from public members.
    enum {
        ERROR = -1,
        OK = 0,
        BAD_VARIABLE,
        BAD_VALUE,
        BAD_ENUM_VALUE
    };

public:
    Beamline(const ToolMgr& toolmgr, const SynonymTable& syn_table);
    Beamline(const Beamline&);
    virtual ~Beamline();

    Beamline* clone() const;
    int copy(const Beamline&);

    int add_source();
    int add_oe(int req_id, int& actual_id);
    int add_scr(int oe, int req_id, int& actual_id);
    int add_tool(const string& toolname);
    int remove_source();
    int remove_oe(int oe);
    int remove_scr(int oe, int scr);
    int remove_tool(const string& toolname);
    int replace_oe(int pos, OE*);

    int add_inspector(const string& toolname, int req_id, int& actual_id);
    int add_source_inspector(
	const string& toolname, int req_id, int& actual_id
    );
    int add_oe_inspector(
	const string& toolname, int oe_id, int req_id, int& actual_id
    );
    int add_scr_inspector(
	const string& toolname, int oe_id, int scr_id, 
	int req_id, int& actual_id
    );
    int remove_inspector(int id);
    int remove_source_inspector(int id);
    int remove_oe_inspector(int oe_id, int id);
    int remove_scr_inspector(int oe_id, int scr_id, int id);

    int clear(bool clear_src = true);

    int cur_oe() const;
    int cur_scr() const;
    int cur_inspector() const;
    const BLToolInstance* cur_selection() const;
    const BLToolInstance* cur_editing() const;
    void cur_oe(int);
    void cur_scr(int);
    void cur_inspector(int);
    void cur_selection(const BLToolInstance*);
    void cur_editing(const BLToolInstance*);

    int num_oe() const;
    int num_scr(int oe) const;
    int num_source_inspectors() const;
    int num_oe_inspectors(int oe) const;
    int num_scr_inspectors(int oe, int scr) const;

    const Var* get_var(const char* record, const char* name) const;
    const Var* get_var(const string& record, const string& name) const;

    int set_var(const char* record, const char* name, const char* value);
    int set_var(const string& record, const string& name, const string& value);

    int set_source_value(const char* varname, const char* value);
    int set_oe_value(int oe, const char* varname, const char* value);
    int set_scr_value(int oe, int scr, const char* varname, const char* value);
    int set_tool_value (
        const char* toolname, const char* varname, const char* value
    );
    int set_inspector_value (int id, const char* varname, const char* value);
    int set_source_inspector_value ( 
	int id, const char* varname, const char* value
    );
    int set_oe_inspector_value (
	int oe, int id, const char* varname, const char* value
    );
    int set_scr_inspector_value (
	int oe, int scr, int id, const char* varname, const char* value
    );

    const char* get_source_value(const char* varname) const;
    const char* get_oe_value(int oe, const char* varname) const;
    const char* get_scr_value(int oe, int scr, const char* varname) const;
    const char* get_inspector_value(int id, const char* varname) const;
    const char* get_source_inspector_value(int id, const char* varname) const;
    const char* get_oe_inspector_value (
	int oe, int id, const char* varname
    ) const;
    const char* get_scr_inspector_value (
	int oe, int scr, int id, const char* varname
    ) const;

    Source* get_source() const;
    OE* get_oe(int oe) const;
    Screen* get_scr(int oe, int scr) const;
    BLToolInstance* get_tool(const string& toolname) const;

    Inspector* get_inspector(int id) const;
    Inspector* get_source_inspector(int id) const;
    Inspector* get_oe_inspector(int oe_id, int id) const;
    Inspector* get_scr_inspector(int oe_id, int scr_id, int id) const;

    int load_source(const string& file);
    int load_oe(const string& file, int oe);
    int load_scr(const string& file, int oe, int scr);
    int load_system(const string& file);
    int load_tool(const string& toolname, const string& file);

    int store_source(const string& file) const;
    int store_oe(const string& file, int oe) const;
    int store_scr(const string& file, int oe, int scr) const;
    int store_system (
	const string& file, const string& startprefix = "start"
    ) const;
    int store_tool(const string& toolname, const string& file) const;

    int load_nml(const string& file, BLToolInstance* ti);
    int store_nml(const string& file, const BLToolInstance* ti) const;

    //
    // copy/cut/paste members.
    //
    int copy_selection_to_clipboard() const;
    int paste_selection_from_clipboard();
    const BLToolInstance* clipboard() const;
    void clear_clipboard() const;

    void save_state() const;
    int restore_state();
    const Beamline* saved_state() const;

    void dump() const;

private:
    int load_scr_from_oe(const OE* oelem, Screen* screen);
    const char* get_value(const BLToolInstance*, const string& varname) const;
    int set_value(BLToolInstance*, const string& varname, const char* value);

    int renumber_all_oe();

private:
    const ToolMgr& toolmgr_;
    const SynonymTable& syn_table_;
    Source* src_;
    deque<void*> oe_list_;
    map<string, void*, less<string> > tool_map_;
    int cur_oe_;
    int cur_scr_;
    int cur_inspector_;
    BLToolInstance* cur_selection_;
    BLToolInstance* cur_editing_;
    BLToolInstance* clipboard_;
    Beamline* saved_state_;
};

// ======================================================================= //

#endif/*shadow_beamline_h*/
