//
// reader.cc: reader/manipulator for MENU files.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

/***************************************************************************/

#ifdef __GNUG__
# include <cassert>
# include <cctype>
# include <cstdio>
# include <cstdlib>
# include <cstring>
# include <strstream.h>
#else
# include <assert>
# include <ctype>
# include <stdio>
# include <stdlib>
# include <string>
# include <strstream>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#if defined(NO_STRNCASECMP_PROTO)
extern "C" int strcasecmp(const char*, const char*);
extern "C" int strncasecmp(const char*, const char*, int);
#endif

#include <common.h>
#include <namelist.h>

#include "global.h"
#include "menu.h"
#include "reader.h"
#include "relation.h"
#include "synonym.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

/***********************************************************************/

static char* trim(char* str) {
    if (str == nil)
	return nil;
    else if (!*str)
	return str;
    char* p = str;
    for(; *p && (*p == ' ' || *p == '\t'); ++p)
	;
    char* q = &str[strlen(p)-1];
    q = (q > p) ? q : p;
    for(; q >= p && *q && (*q == ' ' || *q == '\t'); --q)
	;
    *(++q) = '\0';
    return p;
}

static char* trimright(char* str) {
    if (str == nil)
	return nil;
    else if (!*str)
	return str;
    char* p = str;
    char* q = &str[strlen(p)-1];
    q = (q > p) ? q : p;
    for(; q >= p && *q && (*q == ' ' || *q == '\t'); --q)
	;
    *(++q) = '\0';
    return p;
}

/***********************************************************************/

static const int num_sections = 7;
static const int defaults_section = 1;
static const int synonym_section = 2;
static const int pseudo_defaults_section = 3;
static const int command_section = 4;
static const int page_section = 5;
static const int variables_section = 6;
static const int tool_alias_section = 7;

MenuReader::MenuReader(ToolMgr& toolmgr, Menu* menu, bool debug) : 
    toolmgr_(toolmgr),
    menu_(menu),
    debug_(debug),
    menu_buf_(new char[1024]),
    line_no_(0)
{ }

MenuReader::~MenuReader() {
    delete[] menu_buf_;
}

#define CHECK_RETURN(val)	\
    {				\
	if ((val) == -1) {	\
	    return -1;		\
	}			\
    }

int MenuReader::load(const string& file) {
    ifstream ifp(file.c_str());
    if (!ifp) {
	return nil;
    }

    line_no_ = 0;

    CHECK_RETURN(load_tool_aliases(ifp));
    CHECK_RETURN(load_synonyms(ifp));
    CHECK_RETURN(load_defaults(ifp));
    CHECK_RETURN(load_pseudo_defaults(ifp));
    CHECK_RETURN(load_commands(ifp));
    CHECK_RETURN(load_pages(ifp));
    CHECK_RETURN(load_variables(ifp));
    return 0;
}

int MenuReader::skip_sections(ifstream& ifp, int nskip) {
    int nskipped = 0;
    for(bool done = false; !done && nskipped < nskip;) {
	++line_no_;
	if (!ifp.getline(menu_buf_, BUFSIZ)) {
	    done = true;
	}
	else {
	    if (MenuReader::is_eos(menu_buf_)) {
		++nskipped;
	    }
	}
    }
    return nskipped;
}

int MenuReader::goto_section(ifstream& ifp, int nsection) {
    assert(nsection <= num_sections);
    ifp.seekg(0, ios::beg);
    line_no_ = 0;
    int nskipped = skip_sections(ifp, nsection-1);
    return (nskipped == nsection-1) ? nsection : -1;
}

const char* MenuReader::get_menu_statement(ifstream& ifp) {
    while (ifp.getline(menu_buf_, BUFSIZ) && is_comment(menu_buf_)) {
        ++line_no_;
    }
    return (ifp) ? &menu_buf_[0] : nil;
}

int MenuReader::load_synonyms(ifstream& ifp) { 
    if (debug_)
	cerr << "reading synonyms..." << endl;

    int errors = 0;
    goto_section(ifp, synonym_section);
    SynonymTable* syntab = menu_->get_synonym_table();

    char varname[BUFSIZ]; 			// name of variable
    char symbol[BUFSIZ]; 			//   and a symbol
    int value;					//     equals this value

    for(;;) {
	const char* line = get_menu_statement(ifp);
	if (line == nil || MenuReader::is_eos(line)) 
	    break;

	memset(varname, 0, BUFSIZ);
	memset(symbol, 0, BUFSIZ);

	if (sscanf(line, "%16c%16c%2d", varname, symbol, &value) != 3)
	    errors++;
	syntab->add(trim(varname), trim(symbol), value);
	if (debug_) cerr << "\t" << trim(varname) << " ==> ("
	    << trim(symbol) << ", " << value << ")" << endl;
    }
    if (debug_) {
	cerr << "done reading synonyms" << endl;
	syntab->dump();
    }
    return errors;
}
    
int MenuReader::load_defaults(ifstream& ifp) {
    if (debug_)
	cerr << "reading defaults..." << endl;

    int errors = 0;
    char p_record[BUFSIZ]; char p_varname[BUFSIZ]; 
    char p_type[BUFSIZ]; char p_defval[BUFSIZ];
    char* record; char* varname; 
    char* type; char* defval;
    goto_section(ifp, defaults_section);
    for(;;) {
	const char* line = get_menu_statement(ifp);
	if (line == nil || MenuReader::is_eos(line)) 
	    break;

	memset(p_record, 0, BUFSIZ);
	memset(p_varname, 0, BUFSIZ);
	memset(p_type, 0, BUFSIZ);
	memset(p_defval, 0, BUFSIZ);

	if (
	    sscanf (line, "%3c%*5c%16c%1c%*7c%20c", 
	        p_record, p_varname, p_type, p_defval
	    ) != 4
	) {
	    errors++;
	}
	record = trim(p_record);
	varname = trim(p_varname);
	type = trim(p_type);
	defval = trim(p_defval);

	if (debug_) cerr << "\t(" << record << ", "
	    << varname << ") ==> (" << type << ", " << defval
	    << ")" << endl;

	Tool* tool = toolmgr_.find(record);
	if (tool == nil)
	    tool = toolmgr_.add(record);
	
	Value* value = nil;
	char vtype[BUFSIZ];
	SynonymTable* syn_table = menu_->get_synonym_table();
	switch(*type) {
	    case 'R':
		value = new RealValue(0.0);
		strcpy(vtype, "real");
		break;
	    case 'E':
		{
		    SynonymTable::MatchType match;
		    if (!syn_table->exists(varname, defval, match)) {
			cerr << "SYNONYM (" << varname << ", "
			    << defval << ") illegal!"
			    << endl;
			assert(0);
		    } else if (match == SynonymTable::AMBIGUOUS) {
			cerr << "SYNONYM (" << varname << ", "
			    << defval << ") ambiguous!"
			    << endl;
			assert(0);
		    }
		}
		// CHECK/FIXME: who deletes the char*?
		value = new EnumValue(
		    syn_table->get_symbols(varname)->clone()
		);
		strcpy(vtype, "enumerated");
		break;
	    case 'Y':
		// CHECK: special case of enumerated variables.
		//
		// add all of the usual YES/NO type of variables, since
		// SHADOW menu sometimes uses a Y/N variable and gives
		// it values like ON/OFF (cf: SCR/I_SLIT). We need to
		// be backward compatible here!
		//
		if (
		    strcasecmp(defval, "on") == 0 ||
		    strcasecmp(defval, "off") == 0
		) {
		    syn_table->add(varname, "OFF", 0);
		    syn_table->add(varname, "ON", 1);
		} else if (
		    strcasecmp(defval, "true") == 0 ||
		    strcasecmp(defval, "false") == 0 
		) {
		    syn_table->add(varname, "FALSE", 0);
		    syn_table->add(varname, "TRUE", 1);
		} else if (
		    strcasecmp(defval, "yes") == 0 ||
		    strcasecmp(defval, "no") == 0
		) {
		    syn_table->add(varname, "NO", 0);
		    syn_table->add(varname, "YES", 1);
		} else {
		    cerr << "BAD Y/N variable/value pair (" << record
		        << ") \"" 
		        << varname << "/" << defval << "\"" 
			<< endl;
		    exit(1);
		}

		// CHECK/FIXME: who deletes the char*?
		value = new EnumValue(
		    syn_table->get_symbols(varname)->clone()
		);
		strcpy(vtype, "enumerated");
		break;
	    case 'I':
		value = new IntegerValue(0);
		strcpy(vtype, "integer");
		break;
	    case 'F':
		value = new FilenameValue("");
		strcpy(vtype, "filename");
		break;
	    case 'A':
		value = new TextValue("");
		strcpy(vtype, "text");
		break;
	    default:
		cerr << "MenuReader::Bad variable type \""
		    << type << "\"...assuming text." << endl;
		value = new TextValue("");
		strcpy(vtype, "text");
		errors++;
		break;
	}
	if (value->setval(defval) == 0) {
	    cerr << "MenuReader::Bad value \"" << defval
	        << "\" for variable \"" << varname << "\"." << endl;
	    errors++;
	}
	Var* var = new Var(varname, vtype, value);
	tool->add_var(var);
    }
    if (debug_)
	cerr << "done reading defaults" << endl;
    return errors;
}

int MenuReader::load_pseudo_defaults(ifstream& /*ifp*/) {
    if (debug_)
	cerr << "reading (actually ignoring) PSEUDO defaults..." << endl;

    return 0;
}

int MenuReader::load_commands(ifstream& /*ifp*/) {
    if (debug_)
	cerr << "reading (actually ignoring) COMMANDS/VERBS..." << endl;

    return 0;
}

int MenuReader::load_pages(ifstream& ifp) { 
    if (debug_)
	cerr << "reading MENU PAGES..." << endl;

    int errors = 0;
    MenuPage* cur_page = nil;
    Relation* relation = nil;
    int ntext = 0;
    int nskip = 0;

    goto_section(ifp, page_section);
    for(;;) {
	const char* line = get_menu_statement(ifp);
	if (line == nil || MenuReader::is_eos(line)) 
	    break;
	if (strncmp(line, "HLPC", 4) == 0 || strncmp(line, "HLPM", 4) == 0)
	    continue;
	
	if (strncmp(line, "MENU", 4) == 0) {
	    // CHECK: ignore OE/SCR connections for now.
	    char mname[BUFSIZ]; char mtitle[BUFSIZ]; char mtype[BUFSIZ];
	    memset(mname, 0, BUFSIZ);
	    memset(mtitle, 0, BUFSIZ);
	    memset(mtype, 0, BUFSIZ);
	    sscanf(line, "%*5c%3c%*8c%16c%*8c%40c", mtype, mname, mtitle);
	    cur_page = new MenuPage(
		menu_, trim(mname), trim(mtitle), trim(mtype)
	    );
	    relation = new Relation;
	    cur_page->set_relation(relation);
	    menu_->add_page(cur_page);
	    ntext = 0;
	    nskip = 0;
	    if (debug_)
		cerr << "reading MENU page " 
		    << cur_page->name() << "..." << endl;
	} else if (strncmp(line, "SKIP", 4) == 0) {
	    assert(cur_page != nil);
	    char mname[BUFSIZ];
	    memset(mname, 0, BUFSIZ);
	    sprintf(mname, "%s:SKIP%d", cur_page->name(), ++nskip);
	    MenuItem* item = new SkipMenuItem(trim(mname));
	    cur_page->add_item(item);
	    continue;
	} else if (strncmp(line, "TEXT", 4) == 0) {
	    assert(cur_page != nil);
	    char mtext[BUFSIZ];
	    char mname[BUFSIZ];
	    char msubmenu[BUFSIZ];
	    memset(mname, 0, BUFSIZ);
	    memset(mtext, 0, BUFSIZ);
	    memset(msubmenu, 0, BUFSIZ);
	    sprintf(mname, "%s:TEXT%d", cur_page->name(), ++ntext);
	    sscanf(line, "%*40c%33c%16c", mtext, msubmenu);
	    MenuItem* item = new TextMenuItem(
	        trim(mname), trim(mtext), trim(msubmenu)
	    );
	    cur_page->add_item(item);
	} else if (strncmp(line, "DATA", 4) == 0) {
	    assert(cur_page != nil);
	    char record[BUFSIZ];
	    char name[BUFSIZ];
	    char prompt[BUFSIZ];
	    char submenu[BUFSIZ];
	    char flag[BUFSIZ];
	    char force, readonly;
	    int own1 = 0, own2 = 0;
	    memset(record, 0, BUFSIZ);
	    memset(name, 0, BUFSIZ);
	    memset(prompt, 0, BUFSIZ);
	    memset(submenu, 0, BUFSIZ);
	    memset(flag, 0, BUFSIZ);
	    if (
		sscanf(line, "%*5c%3c%*1c%1c%1c%1c%*2c%*2c%16c%*8c%33c%16c",
		   record, flag, &force, &readonly, /* &own1, &own2, */ name,
		   prompt, submenu
		) != 9
	    )
		errors++;
	    
	    bool do_add = true;
	    //
	    // HACK/CHECK: special hack to get rid of GOTO_OE ...
	    //
	    if (
	        strcmp(trim(name), "GOTO_OE") == 0 ||
	        strcmp(trim(name), "GOTO_SCR") == 0
	    ) {
	        do_add = false;
	    }
	    
	    if (do_add) {
		char* flagptr = (flag[0] == ' ') ? nil : flag;
		int i_flags = 0;
		if (readonly == 'Y' ||  readonly == 'y')
		    i_flags |= DataMenuItem::READ_ONLY; 
		if (force == 'Y' ||  force == 'y')
		    i_flags |= DataMenuItem::FORCE_UPDATE; 
		MenuItem* item = new DataMenuItem(
		    trim(name), trim(record), trimright(prompt), trim(submenu), 
		    trim(flagptr), i_flags, own1, own2
		);
		cur_page->add_item(item);
	    }
	} else if (strncmp(line, "SET", 3) == 0) {
	    assert(cur_page != nil);
	    char result[BUFSIZ];
	    char rec1[BUFSIZ];
	    char name1[BUFSIZ];
	    char rel[BUFSIZ];
	    char rec2[BUFSIZ];
	    char name2[BUFSIZ];
	    memset(result, 0, BUFSIZ);
	    memset(rec1, 0, BUFSIZ);
	    memset(name1, 0, BUFSIZ);
	    memset(rel, 0, BUFSIZ);
	    memset(rec2, 0, BUFSIZ);
	    memset(name2, 0, BUFSIZ);
	    if (
		sscanf(line, "%*8c%2c%*6c%3c%*5c%16c%3c%*5c%3c%*5c%16c",
		    result, rec1, name1, rel, rec2, name2
		) != 6
	    )
		errors++;

	    RelationItem* rel_item = new RelationItem(trim(result), 
		trim(rec1), trim(name1), 
		trim(rel), 
		trim(rec2), trim(name2)
	    );
	    relation->add(rel_item);
	}
    }
    if (debug_)
	cerr << "done reading MENU PAGES" << endl;
    return errors;
}

//
// Load from a SHADOW namelist (the original kind that's defined in 
// namelist.c into this ShadowNamelist object. This is to make sure
// that *all* the variables are defined that are not specified in the
// DEFAULTS section of the MENU file. I guess this is a hack until we
// add *all* the variables in the MENU file. This is the only connection
// to anything SHADOW related here (ie., with the "real" SHADOW), rest
// are completely self-contained.
//
static int load_nml (ToolMgr& toolmgr, const string& tname, Namelist* nml) {
    Tool* tool = toolmgr.find(tname);
    if (tool == nil) {
	tool = toolmgr.add(tname);
    }

    for (int i = 0; i < nml->size; ++i) {
        const NamelistData& shadow_nml_data = nml->nml_data[i];
	const char* orig_varname = shadow_nml_data.name;
	char varname[1024]; 
	strcpy(varname, orig_varname);
	if (tool->exists_var(varname))
	    continue;

	Value* value = nil;
	char vtype[BUFSIZ];
	switch (shadow_nml_data.data_type) {
	    case DT_dbl:
	    case DT_flt:
		value = new RealValue(0.0);
		strcpy(vtype, "real");
		break;
	    case DT_int:
		value = new IntegerValue(0);
		strcpy(vtype, "integer");
		break;
	    case DT_str:
		value = new TextValue("");
		strcpy(vtype, "text");
		break;
	}
	Var* var = new Var(varname, vtype, value);
	tool->add_var(var);
    }
    return 0;
}

int MenuReader::load_variables(ifstream& /*ifp*/) {
    int errors = 0;

    Namelist* src_nml = ::get_namelist(NML_source);
    load_nml(toolmgr_, "SRC", src_nml);

    Namelist* oe_nml = ::get_namelist(NML_oe);
    load_nml(toolmgr_, "OE", oe_nml);

    return errors;
}

int MenuReader::load_tool_aliases(ifstream& ifp) { 
    if (debug_)
	cerr << "reading MENU TOOL ALIASES ... " << endl;

    int errors = 0;
    char alias[BUFSIZ]; 
    char toolname[BUFSIZ]; 

    goto_section(ifp, tool_alias_section);
    for(;;) {
	const char* line = get_menu_statement(ifp);
	if (line == nil || MenuReader::is_eos(line)) 
	    break;

#ifdef BC5_ISTRSTREAM_BUG
	istrstream istr(const_cast<char*>(line));
#else
	istrstream istr(line);
#endif
	if (!(istr >> alias >> toolname)) {
	    cerr << "MenuReader:: Error reading Tool alias in line "
	        << line_no_ << "." << endl;
	    ++errors;
	} else {
	    toolmgr_.add_alias(alias, toolname);
	}
    }
    if (debug_)
	cerr << "done reading MENU TOOL ALIASES" << endl;
    return errors;
}

/***********************************************************************/
