/* 
 * groutines - a set of library routines for reading parameters.
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * Source: src/utils/io/groutines.h
 *
 * Log:	groutines.h
 * Revision 1.20  90/08/30  18:03:01  khan
 * About to add my own hashing routines.
 * 
 * Revision 1.19  90/06/16  12:43:22  khan
 * routine
 * 
 * Revision 1.18  90/05/28  10:17:57  khan
 * Hashing added, but trouble ahead.
 * 
 * Revision 1.17  90/05/09  12:01:27  khan
 * Adding Hashing.
 * 
 * Revision 1.16  90/05/01  14:32:03  khan
 * routine
 * 
 * Revision 1.15  90/04/12  21:36:17  khan
 * VMS support cut. Cleaned up code.
 * 
 * Revision 1.14  90/04/12  20:48:32  khan
 * Cleanup
 * 
 */

#ifndef shadow_groutines_h
#define shadow_groutines_h

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>

#if defined(SVR4) || defined(SYSV) || defined(__hpux) || defined(_AIX)
#include <sys/types.h>
#if defined(SVR4) || defined(SYSV) 
#include <sys/fcntl.h>
#else
#include <fcntl.h>
#endif /*!hpux*/
#endif /* SYSV */

#if defined (_WIN32) && ! defined (__CYGWIN32__)
#include <fcntl.h>
#else
#include <sys/file.h>
#endif
#include "libcxrl.h"

#include "hash.h"

htable hash_table = NULL;

#define TRUE 1
#define FALSE 0
#ifndef NULL
# define NULL 0
#endif
#define MAXLINELEN BUFSIZ
#define MAXTRY 10

/*
 * Define the number of symbols expected on average. This is to initialize
 * the hash table, but can be changed with a runtime option thru the 
 * gset interface.
 */
#define MAX_SYM_ENTRIES 107
#define HASH_TABLE_SIZE 107

/* 
 * To install the interrupt handler, define INT_HANDLER. Probably not very
 * useful under any circumstances.
 */
/* #define INT_HANDLER */

#define STATIC static
#define PRIVATE static
#define PUBLIC 

#if defined (_WIN32) || defined(SVR4) || defined(SYSV) || defined(__hpux) || defined(_AIX)
# define bcopy(s1, s2, n) memcpy(s2, s1, n)
#endif

/* 
 * define the characters that are considered special :
 * o SEPARATOR (e.g., resistance = 100)
 * o COMMENT leader (e.g., # This is a comment line)
 * o ESCAPE character (e.g., "\#" treats '#' as a literal character)  
 */

#define SEPARATOR '='		/* between the parameter and value */
#define COMMENT   '#'		/* the comment leader  */
#define ESCAPE    '\\'		/* the escape character */

/*
 * Having problems with the Ultrix 3.0 malloc (it causes segmentation
 * violation for some unknown reason). As an alternative, using GNU
 * malloc. Utility routine, xmalloc (), uses GNU_malloc () if 
 * USE_GNU_MALLOC is defined. Better defined in the Makefile.
 *
 */

/*
#define USE_GNU_MALLOC
*/

/* 
 * All the parameters and the associated values are stored in the hash
 * table, but the option names are also stored in a linear linked list
 * so we can traverse the list and print it out.
 */

typedef struct OPTIONS {
    char *key;
    struct OPTIONS *next;
} OPTIONS;


/*
 * Convenience enumerations for return types.
 */

typedef enum {
    no = 0,
    yes = 1
} yesno;

/*
 * Policy issue: Should the options and query be case insensitive?
 * Well, I'm providing a general purpose routine gcase (), to make
 * sensitive, and gnocase (), otherwise. 
 *
 * The default is to make the fortran interface INSENSITIVE, and
 * C interface sensitive.
 */

static enum {
    default_case = 0,           /* User hasn't specified. Use default */
    sensitive,
    insensitive
} case_sense = default_case;    /* Supports CASE-SENSE in searches */


/* What interface are we using? Important for case searches */

static enum {
    f_lang = 0,
    c_lang
} interface = c_lang;           /* Note: Default is C */


STATIC FILE *istream;       	    /* The stream we're reading from */

STATIC char routine[128];	    /* used for error reporting */
STATIC char line[MAXLINELEN];       /* To read in a parameter line */
STATIC char err_msg[MAXLINELEN];    /* error message to be printed */

STATIC int abort_on_error = FALSE;  /* Abort or warn on syntax error? */
STATIC yesno usr_debug = no;
				    /* User wants debugging info? */
STATIC hash_table_inited = FALSE; 	
				/* has the hash table been initialized */

/*
 * There seems to be bug in VMS C memory management. When I try to free
 * pointers allocated by malloc, it does it fine for some of them, but
 * gags at some giving a access violation error, even though the pointer
 * exists, ie., non-null (!!). I checked with the debugger. But if I use
 * VAXC$MALLOC_OPT and VAXC$FREE_OPT, everything's cool.
 */

#ifdef VMS
#    define malloc(size_t) VAXC$MALLOC_OPT(size_t)
#    define free(pointer) VAXC$FREE_OPT(pointer)
#endif

/*
 * if parameters are not found in the data file, the user must be prompted
 * for it; but STDIO routines report EOF, and refuse to read any more. To
 * this problem, the user's terminal is opened for reading and writing to
 * write the prompt and read the response.
 */

#ifdef VMS
#define TTY_DEV "tt:"
#else
#define TTY_DEV "/dev/tty"
#endif

STATIC int tty;		        /* file handle for /dev/tty. */
STATIC int file_open = FALSE;

/*
 *  Various interrupt handling stuff.
 */

STATIC int intr = FALSE;
STATIC jmp_buf env;             /* for longjmp, assuming setjmp.h included */
STATIC int jmp_val;


/* 
 *  The library routines themselves
 */

/* FORTRAN interface */

#if !defined (NO_FORT_UNDERSCORE)
# define F77_NAME(f) f##_
#else
# define F77_NAME(f) f
#endif

/*
 * G77 (GNU FORTRAN, upto and including v.0.5.18) adds a SECOND underscore
 * to names that already contain an underscore. Annoying as well, and
 * hence the G77_HACK_NAME(f) hack.
 */

#define G77_HACK_NAME(f) f##__

PUBLIC int F77_NAME(gpeek) ();
PUBLIC int F77_NAME(gset) ();
PUBLIC int F77_NAME(gabort) ();
PUBLIC int F77_NAME(gnoabort) ();
PUBLIC int F77_NAME(gdebug) ();
PUBLIC int F77_NAME(gprint) ();
PUBLIC int F77_NAME(gcase) ();
PUBLIC int F77_NAME(gnocase) ();
PUBLIC int F77_NAME(gstring) ();
PUBLIC double F77_NAME(gnumber) ();
PUBLIC int F77_NAME(gint) ();
PUBLIC int F77_NAME(gfile) ();
PUBLIC int F77_NAME(gclose) ();
PUBLIC int F77_NAME(gcleanup) ();

/* C interface */

PUBLIC int g_peek ();
PUBLIC int g_set ();
PUBLIC int g_abort ();
PUBLIC int g_noabort ();
PUBLIC int g_debug ();
PUBLIC int g_print ();
PUBLIC int g_case ();
PUBLIC int g_nocase ();
PUBLIC char *g_string ();
PUBLIC double g_number ();
PUBLIC int g_int ();
PUBLIC int g_file ();
PUBLIC int g_close ();
PUBLIC int g_cleanup ();

/*
 *  Various routines used in the library.
 */

STATIC hbucket match_opt ();        /* matches the query with stored */
STATIC char *prompt ();             /* prompts user for input */
PUBLIC char *trim_leading_white (); /* skips white from the front */
PUBLIC char *trim_trailing_white ();/* skips white from the back */
PUBLIC char *xmalloc ();            /* a little smarter malloc */
PUBLIC int error ();

STATIC reset_buffer_read ();
STATIC clear_option_list ();

#endif/*shadow_groutines_h*/
