C +++
C
C Source: src/trace/screen_ext.F
C
C ----------------------------------------------
C                SHADOW
C      Center for X-ray Lithography
C     University of Wisconsin-Madison
C  3731 Schneider Dr., Stoughton, WI, 53589
C ----------------------------------------------
C
C Author: Mumit Khan <khan@xraylith.wisc.edu>
C 
C ---

#if defined(unix) || HAVE_F77_CPP
#	include		<header.txt>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:HEADER.TXT/LIST'
#endif

C+++
C	SUBROUTINE	SCREEN_EXTERNAL
C
C	PURPOSE		This routine handles the screen slit/stops when
C			an external file with the coordinates is
C			specified (K_SLIT = 2).
C
C	ARGUMENTS	
C			[ I ]   I_SCR	The screen index
C			[ I ]   I_ELEMENT OE number
C                       [I/O]	RAY   	The beam description as computed
C					by RESTART or MIRROR
C                       [I/O]	RAY_OUT	The local copy of RAY in SCREEN.
C
C	Author:		Mumit Khan
C
C---
	SUBROUTINE 	SCREEN_EXTERNAL (I_SCR,I_ELEMENT,RAY,RAY_OUT)
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
#endif
	DIMENSION 	RAY(12,N_DIM),RAY_OUT(12,N_DIM)
C
C Local variables
C
C
C The following two parameters limit the polygons that can be read in.
C There can be upto MAX_NPOLY polygons, and an accumulated total of 
C MAX_NPOINTS points summed over all the polygons. The accumulated sum
C includes polygon closure (eg., a rectangle has *5*, not 4 points after
C closure). To up the limits, modify the following and recompile.
C
	INTEGER		MAX_NPOLY, MAX_NPOINTS
	PARAMETER	(MAX_NPOLY = 50)
	PARAMETER	(MAX_NPOINTS = 1000)

	INTEGER		N_POLYS, N_POINTS
	INTEGER		IVEC1, IVEC2
	DOUBLE PRECISION XVEC, ZVEC, PX, PZ
	DIMENSION	XVEC(MAX_NPOINTS), ZVEC(MAX_NPOINTS)
	DIMENSION	IVEC1(MAX_NPOLY), IVEC2(MAX_NPOLY)
	LOGICAL		RAY_LOST, HIT_FOUND
C
C Load the external polygon file that describe the patterns on the screen.
C The file format is described in screen_load_external().
C
C Note that the two vectors, xvec and zvec, contain the points for all
C the polygons. Two aux vectors, ivec1 and ivec2, contain the starting
C indices (into xvec and zvec) and number of points per polygon.
C 
	IFLAG = 0
#if HP_F77_BUGGY_NAMELIST
# define filename FILSCR(I_SCR)
#else
# define filename FILE_SCR_EXT(I_SCR)
#endif
	CALL SCREEN_LOAD_EXTERNAL(filename, MAX_NPOLY, 
     $	   			MAX_NPOINTS,XVEC,ZVEC,IVEC1,IVEC2,
     $				N_POLYS,N_POINTS,IFLAG)
	IF (IFLAG .EQ. -1) THEN
	  CALL MSSG ('SCREEN_EXTERNAL',
     $			'Error reading EXTERNAL file', -1)
	  STOP 1
	ELSE IF (IFLAG .EQ. -2) THEN
	  CALL MSSG ('SCREEN_EXTERNAL',
     $			'Error in External polygon description', -1)
	  STOP 1
	ENDIF

C
C Algorithm: For each ray, see if it hits any of the polygons; if it does,
C check if the polygon is a Aperture or Obstruction, and set the RAY_LOST
C logical accordingly.
C
      	DO 200 IRAY=1,NPOINT
	  HIT_FOUND = .FALSE.
 	  PX = RAY_OUT(1,IRAY)
 	  PZ = RAY_OUT(3,IRAY)
	  DO 11 IPOLY = 1, N_POLYS
	    ISTART = IVEC1(IPOLY)
	    INP = IVEC2(IPOLY)
	    IFLAG = 0
	    CALL PNPOLY (PX, PZ, XVEC(ISTART), ZVEC(ISTART),
     $			INP, IFLAG)
C
C IFLAG = -1 implies point *outside* polygon, 0 vertex, 1 inside.
C I_STOP = 1 for obstruction, and 0 for aperture.
C
C We just need a single hit (inside+vertex) for either type of slit.
C
	    IF (IFLAG.EQ.1 .OR. IFLAG.EQ.0) HIT_FOUND = .TRUE.
 11	  CONTINUE
C
C In case of aperture, a ray is lost if does NOT hit any of the polys;
C in case of obstruction, a ray is lost if it hits *ANY* of the polys.
C 
	  RAY_LOST = .FALSE.
	  IF (I_STOP(I_SCR) .EQ. 0 .AND. .NOT. HIT_FOUND) THEN
	      RAY_LOST = .TRUE.
	  ELSE IF (I_STOP(I_SCR) .EQ. 1 .AND. HIT_FOUND) THEN
	      RAY_LOST = .TRUE.
	  ENDIF

	  IF (RAY_LOST) THEN
	    RAY     (10,IRAY) = - 1.0D2*I_ELEMENT - 1.0D0*I_SCR
	    RAY_OUT (10,IRAY) = - 1.0D2*I_ELEMENT - 1.0D0*I_SCR
	  END IF
 200	CONTINUE
C
	RETURN
	END
C+++
C	SUBROUTINE	SCREEN_LOAD_EXTERNAL
C
C	PURPOSE		This routine load the polygon(s) from a file.
C
C	ARGUMENTS	
C			[ I ] FILE	Polygon file name
C			[ I ] MAX_NPOLYS Maximum number of polygons
C			[ I ] MAX_NPOINTS Maximum *total* number of points
C                       [ O ] XVEC   	X points
C                       [ O ] ZVEC   	Z points
C                       [ O ] IVEC1   	Index of poly starting points
C                       [ O ] IVEC2   	Number of points in each poly
C                       [ O ] NPOLY   	Number of polygons actually read
C                       [ O ] NPOINT   	Number of *total* points actually read
C                       [ O ] IFLAG	Returned value
C					-1: Can't read file
C					-2: Bad format, too many points
C					 0: All ok.
C
C       FORMAT
C					N_POLYGONS
C					N_POINTS
C					X1 Z1
C					X2 Z2
C					.....
C					XN ZN
C					N_POINTS
C					X1 Z1
C					....
C					XN ZN
C
C	 EXAMPLE FILE
C					2
C					4
C					-37.5	-12.5
C					 37.5	-12.5
C					 37.5	 12.5
C					-37.5	 12.5
C					5
C					-20.0	-30.0
C					 20.0	-30.0
C					 20.0	-20.0
C					-20.0	-20.0
C					-20.0	-30.0
C
C Note that in the 2nd polygon, the closure is explicit, while implicit in
C the first. The points can be in clockwise or counter-clockwise.
C		
C	AUTHOR 		Mumit Khan
C---
	SUBROUTINE 	SCREEN_LOAD_EXTERNAL (FILENAME, MAX_NPOLYS,
     $		MAX_NPOINTS,XVEC,ZVEC,IVEC1,IVEC2,NPOLY,NPOINT,IFLAG)
C
	IMPLICIT NONE
	CHARACTER*(*)	FILENAME
	INTEGER		MAX_NPOLYS,MAX_NPOINTS,NPOLY,NPOINT,IFLAG
	INTEGER		IVEC1, IVEC2
	DOUBLE PRECISION XVEC, ZVEC
	DIMENSION	XVEC(MAX_NPOINTS), ZVEC(MAX_NPOINTS)
	DIMENSION	IVEC1(MAX_NPOLYS),IVEC2(MAX_NPOLYS)
C
	INTEGER		IO_UNIT, NP1, POLY_INDEX, PT_INDEX
	INTEGER		START_INDEX
	DATA		IO_UNIT / 21 /
C
	IFLAG = 0
	OPEN(IO_UNIT, FILE=FILENAME, status='OLD', ERR=199)
	READ(IO_UNIT, *, ERR=299, END=299) NPOLY
	IF (NPOLY .GT. MAX_NPOLYS) THEN
	  CALL MSSG ('SCREEN_EXTERNAL','Too many polygons',-1)
	  GOTO 299
	ENDIF
C
C read all the polygons as a compound one (ie., in the same vectors).
C
	NPOINT = 0
	DO 10 POLY_INDEX = 1, NPOLY
	  READ(IO_UNIT, *, ERR=299, END=299) NP1
	  DO 20 PT_INDEX = 1, NP1
	    NPOINT = NPOINT + 1
	    IF (NPOINT .GT. MAX_NPOINTS) THEN
	      CALL MSSG ('SCREEN_EXTERNAL','Too many total points',-1)
	      GOTO 299
	    ENDIF
	    READ(IO_UNIT,*,ERR=299,END=299) XVEC(NPOINT),ZVEC(NPOINT)
 20	  CONTINUE
C
C Make sure the polygon is "closed", so check the first and last points.
C
	  START_INDEX = NPOINT - NP1 + 1
	  IF (XVEC(START_INDEX) .NE. XVEC(NPOINT) .OR.
     $	      ZVEC(START_INDEX) .NE. ZVEC(NPOINT)) THEN
	    NPOINT = NPOINT + 1
	    NP1 = NP1 + 1
	    XVEC(NPOINT) = XVEC(START_INDEX)
	    ZVEC(NPOINT) = ZVEC(START_INDEX)
	  ENDIF
	  IVEC1(POLY_INDEX) = START_INDEX
	  IVEC2(POLY_INDEX) = NP1
C
 10	CONTINUE
	CLOSE(IO_UNIT)
	GOTO 300

 199	CONTINUE
	IFLAG = -1
	GOTO 300

 299	CONTINUE
	CLOSE(IO_UNIT)
	IFLAG = -1
	GOTO 300

 300	CONTINUE
	RETURN
	END
