/*****************************************************************************
 *                                                                           *
 *  PRIMVS - Graphic Interpreter for X-Ray Lithography
 *                                                                           *
 ***************************************************************************** 
 *
 *  Author:   Luigi Capodieci
 *  Project:  PLPLOT based graphic interpreter
 *
 *  $Source: 
 *  $Revision:
 *  $Date: 
 *  $Log: command.c,v $
 *  Revision 1.2  2001/04/07 00:59:26  khan
 *  2001-04-06  Mumit Khan  <khan@nanotech.wisc.edu>
 *
 *  	* command.c (extract_number): New function.
 *  	(read_image): Use.
 *  	(read_vect): Use.
 *  	(read_xyc): Use.
 *
 *  Revision 1.1.1.1  1996/07/11 13:52:43  khan
 *  Initial checkin of SHADOW 2.1 distribution sources
 *
 * Revision 1.1.1.1  1996/01/02  17:07:38  khan
 * Toolset 2.0 release
 *
 *****************************************************************************/ 

#include	"allinclude.h"


/*** GLOBAL VARIABLES ********************************************************/ 

/*** Variables for FORMAT,FORMAT3D ***/
int	xcol=1;
int	ycol=2;
int	zcol=3;
int	ccol=3;
int	bnp=0; /* Beginning line of data file */
int	fnp=0; /* End line of data file */

/*** Variables for READING FORMATTED DATA ***/
char	num_buffer[MAX_WORD_LENGTH];
char	line_buffer[MAX_LINE_LENGTH];
int	kc,maxline;


/*** ASSIGNMENT **************************************************************/

void	exec_assign()

{

switch(avar) {
	case CXMIN:	{cxmin=(double) lex_number; break;}
	case CXMAX:	{cxmax=(double) lex_number; break;}
	case CYMIN:	{cymin=(double) lex_number; break;}
	case CYMAX:	{cymax=(double) lex_number; break;}

	default:	{fprintf(outfile,"Unknown IDENTIFIER\n"); break;}
	}

}

/*** FORMAT ******************************************************************/

void	exec_format()

{
	xcol=temp_xcol;
	ycol=temp_ycol;
	ccol=temp_ccol;

	bnp=temp_bnp;
	fnp=temp_fnp;

#if defined DEBUG
	printf("FORMAT executed with:\n");
	printf("xcol= %d, ycol= %d, ccol= %d bnp= %d, fnp= %d\n",
		xcol,ycol,ccol,bnp,fnp);
#endif

}


/*** Utilities for reading data *********************************************/

/*** Copy a line if the datafile in a line buffer ***/
int	getline()
{

	int	k;
	char	rch;

	k=0;
	if (!feof(functionfile)) rch=fgetc(functionfile); /* Read first char */
	while ((rch!='\n')&&(!feof(functionfile))) {
	   line_buffer[k]=rch;
	   k++;
	   rch=fgetc(functionfile);
	   }
	line_buffer[k]='\0'; /* The string is null terminated */
	return(k);
}


/*** Advance the buffer pointer kc skipping all blanks ***/
void	skipblank()

{

	while ((isspace(line_buffer[kc]))&&(kc<maxline)) {
		kc++;
		}
}

/*** Advance the buffer pointer kc skipping all non-blanks ***/
void	skipcolumn()

{

	while ((!isspace(line_buffer[kc]))&&(kc<maxline)) {
		kc++;
		}
}

/*** Copy a set of characters (potentially a number) in a buffer ***/
void	getnumber()

{
	int	k;

	k=0;
	while ((!isspace(line_buffer[kc]))&&(kc<maxline)) {
	   num_buffer[k]=line_buffer[kc];
	   k++;
	   kc++;
	   }
	num_buffer[k]='\0';
}

double 
extract_number(const char *str)
{
  double val;
  double power;
  int sign;

  while (isspace((int)*str))
    ++str;

  sign = (*str == '-') ? -1 : 1;

  if (*str == '+' || *str == '-')
    ++str;
  val = 0.0;
  while (isdigit((int)*str))
  {
    val = 10 * val + (*str-'0');
    ++str;
  }
  if (*str == '.')
    ++str;

  power = 1.0;
  while (isdigit((int)*str))
  {
    val = 10 * val + (*str-'0');
    power *= 10.0;
    ++str;
  }
  if (*str == 'e' || *str == 'E' || *str == 'd' || *str == 'D')
  {
    int expsign = (*++str == '-') ? -1 : 1;
    int expval;
    if (*str == '+' || *str == '-') 
      ++str;

    expval = 0;
    while (isdigit((int)*str)) 
    {
      expval = 10 * expval + (*str-'0');
      ++str;
    }
    while (expval--) 
    {
      if (expsign == 1)
        power /= 10.0;
      else
        power *= 10.0;
    }
  }
  return sign * (val / power);
}


/*** Read_xyc procedure ******************************************************/

int	read_xyc(xcol,ycol,ccol,bnp,fnp)

	int	xcol,ycol,ccol,bnp,fnp;

{

	float	xval,yval,cval;
	int	nlines,col;

	nlines=0;

/* Initialize values for MIN and MAX (x,y) */
	xxxmin=(float) RANDOM_MAX; xxxmax=(float) (- RANDOM_MAX);
	yyymin=(float) RANDOM_MAX; yyymax=(float) (- RANDOM_MAX);


	while (!feof(functionfile)&&((maxline=getline())!=0)) {
	   nlines++;
	   
	   /*** Read the column for x value ***/
	   if (xcol>0) {
	     kc=0;
	     col=1;
	     skipblank();
	     while ((col<xcol)&&(kc<maxline)) {
		skipcolumn();
		skipblank();
		col++;
		}
	     if ((col!=xcol)||(kc==maxline)) {
		error_msg("Error: column exceeding line boundary");
		fprintf(stderr,"No column %d at line %d\n",xcol,nlines);
		return(0);
		}
	       else {
		getnumber();
		xval = (float) extract_number(num_buffer);
		if (xval<xxxmin) xxxmin=xval;
		if (xval>xxxmax) xxxmax=xval;
		if (nlines<MAX_X) xx[nlines-1]=xval;
		  else {
		    error_msg("Too many X values");
		    return(0);
		    }
		}
	   }

	   /*** Read the column for y value ***/
	   if (ycol>0) {
	     kc=0;
	     col=1;
	     skipblank();
	     while ((col<ycol)&&(kc<maxline)) {
		skipcolumn();
		skipblank();
		col++;
		}
	     if ((col!=ycol)||(kc==maxline)) {
		error_msg("Error: column exceeding line boundary");
		fprintf(stderr,"No column %d at line %d\n",ycol,nlines);
		return(0);
		}
	       else {
		getnumber();
		yval = (float) extract_number(num_buffer);
		if (yval<yyymin) yyymin=yval;
		if (yval>yyymax) yyymax=yval;
		if (nlines<MAX_Y) yy[nlines-1]=yval;
		  else {
		    error_msg("Too many Y values");
		    return(0);
		    }
		}
	   }


	  }
	datain=TRUE;
	return(nlines);
}

/*** Read_vect procedure *****************************************************/

int	read_vect()


{

	int	xcol,ycol;
	float	xval,yval,dxval,dyval;
	int	nlines,col;

	nlines=0;

/* Initialize values for MIN and MAX (x,y) */
	xxxmin=(float) RANDOM_MAX; xxxmax=(float) (- RANDOM_MAX);
	yyymin=(float) RANDOM_MAX; yyymax=(float) (- RANDOM_MAX);


	while (!feof(functionfile)&&((maxline=getline())!=0)) {
	   nlines++;
	   
	   xcol=1;	/*** Read the column for x value ***/
	   if (xcol>0) {
	     kc=0;
	     col=1;
	     skipblank();
	     while ((col<xcol)&&(kc<maxline)) {
		skipcolumn();
		skipblank();
		col++;
		}
	     if ((col!=xcol)||(kc==maxline)) {
		error_msg("Error: column exceeding line boundary");
		fprintf(stderr,"No column %d at line %d\n",xcol,nlines);
		return(0);
		}
	       else {
		getnumber();
		xval = (float) extract_number(num_buffer);
		if (xval<xxxmin) xxxmin=xval;
		if (xval>xxxmax) xxxmax=xval;
		if (nlines<MAX_X) xx[nlines-1]=xval;
		  else {
		    error_msg("Too many X values");
		    return(0);
		    }
		}
	   }

	   ycol=2;	/*** Read the column for y value ***/
	   if (ycol>0) {
	     kc=0;
	     col=1;
	     skipblank();
	     while ((col<ycol)&&(kc<maxline)) {
		skipcolumn();
		skipblank();
		col++;
		}
	     if ((col!=ycol)||(kc==maxline)) {
		error_msg("Error: column exceeding line boundary");
		fprintf(stderr,"No column %d at line %d\n",ycol,nlines);
		return(0);
		}
	       else {
		getnumber();
		yval = (float) extract_number(num_buffer);
		if (yval<yyymin) yyymin=yval;
		if (yval>yyymax) yyymax=yval;
		if (nlines<MAX_Y) yy[nlines-1]=yval;
		  else {
		    error_msg("Too many Y values");
		    return(0);
		    }
		}
	   }

	   xcol=3;	/*** Read the column for dx value ***/
	   if (xcol>0) {
	     kc=0;
	     col=1;
	     skipblank();
	     while ((col<xcol)&&(kc<maxline)) {
		skipcolumn();
		skipblank();
		col++;
		}
	     if ((col!=xcol)||(kc==maxline)) {
		error_msg("Error: column exceeding line boundary");
		fprintf(stderr,"No column %d at line %d\n",xcol,nlines);
		return(0);
		}
	       else {
		getnumber();
		xval = (float) extract_number(num_buffer);
		if (nlines<MAX_X) dx[nlines-1]=xval;
		  else {
		    error_msg("Too many DX values");
		    return(0);
		    }
		}
	   }

	   ycol=4;	/*** Read the column for dy value ***/
	   if (ycol>0) {
	     kc=0;
	     col=1;
	     skipblank();
	     while ((col<ycol)&&(kc<maxline)) {
		skipcolumn();
		skipblank();
		col++;
		}
	     if ((col!=ycol)||(kc==maxline)) {
		error_msg("Error: column exceeding line boundary");
		fprintf(stderr,"No column %d at line %d\n",ycol,nlines);
		return(0);
		}
	       else {
		getnumber();
		yval = (float) extract_number(num_buffer);
		if (nlines<MAX_Y) dy[nlines-1]=yval;
		  else {
		    error_msg("Too many DY values");
		    return(0);
		    }
		}
	   }


	  }
	datain=TRUE;
	return(nlines);
}


/*** Read_image procedure ****************************************************/

int	read_image()

{
	float	xstep, ystep;
	int	i,j,k;

/* Deallocate the memory for xxx yyy zzz vectors (if already allocated) */
	if (alloc3d) {
		free(xxx); free(yyy); 
		for(i=0;i<nx;i++) free(zzz[i]);
		free(zzz);
		}

	if (!feof(functionfile))
	   maxline=getline(); /* Skip the first line (value ignored) */
	  else {
	   error_msg("Error reading DATA FILE");
	   exit(-2);
	   }

	if (!feof(functionfile)) {
	   maxline=getline(); /* Read the DELTA X */
	   xstep = (float) extract_number(line_buffer);
	   }
	  else {
	   error_msg("Error reading DATA FILE");
	   exit(-2);
	   }

	if (!feof(functionfile)) {
	   maxline=getline(); /* Read the DELTA Y */
	   ystep = (float) extract_number(line_buffer);
	   }
	  else {
	   error_msg("Error reading DATA FILE");
	   exit(-2);
	   }

	if (!feof(functionfile)) {
	   maxline=getline(); /* Read the NUMBER of X points */
	   nx = (int) extract_number(line_buffer);
	   }
	  else {
	   error_msg("Error reading DATA FILE");
	   exit(-2);
	   }

	if (!feof(functionfile)) {
	   maxline=getline(); /* Read the NUMBER of Y points */
	   ny = (int) extract_number(line_buffer);
	   }
	  else {
	   error_msg("Error reading DATA FILE");
	   exit(-2);
	   }

	if (!feof(functionfile))
	   maxline=getline(); /* Skip the sixth line (value ignored) */
	  else {
	   error_msg("Error reading DATA FILE");
	   exit(-2);
	   }

/* Allocate new memory for the data */

	xxx=(float *) malloc(nx*sizeof(float));
	yyy=(float *) malloc(ny*sizeof(float));
	zzz=(float **) malloc(nx*sizeof(float));

	if ((xxx==0)||(yyy==0)||(zzz==0)) {
		error_msg_nl("No more memory to allocate");
		exit(-3);
		}

/* Initialize X values and allocate Z row vectors*/
	xxx[0] = 0.0;
	zzz[0]=(float *) malloc(ny*sizeof(float));
	if (zzz[0]==0) {
		error_msg_nl("No more memory to allocate");
		exit(-4);
		}
	for (i=1;i<nx;i++) {
	   xxx[i]=xxx[i-1]+xstep; 
	   zzz[i]=(float *) malloc(ny*sizeof(float));
	   if (zzz[i]==0) {
		error_msg_nl("No more memory to allocate");
		exit(-5);
		}
	   }


/* Initialize Y values */
	yyy[0] = 0.0;
	for (j=1;j<ny;j++) 
	   yyy[j]=yyy[j-1]+ystep; 

/* Initialize values for MIN and MAX (x,y) */
	xxxmin=xxx[0]; xxxmax=xxx[nx-1];
	yyymin=yyy[0]; yyymax=yyy[ny-1];
	zzzmin=(float) RANDOM_MAX; zzzmax=(float) (- RANDOM_MAX);

/* Read Z values */
	for (j=0;j<ny;j++) {
	   for (i=0;i<nx;i++) {
		if (!feof(functionfile)) {
	   	   maxline=getline(); 
	   	   zzz[i][j] = (float) extract_number(line_buffer);
		   if ((zzz[i][j])<zzzmin) zzzmin=zzz[i][j];
		   if ((zzz[i][j])>zzzmax) zzzmax=zzz[i][j];
	   	   }
	  	  else {
	   	   error_msg("Error reading DATA FILE");
	   	   exit(-2);
	   	   }
		}
	   }

	alloc3d=TRUE;
	datain=TRUE;
	return(nx*ny);
}

	   
/*****************************************************************************/
