//
// tool.h: Interface to EXCON tool manager
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#ifndef excon_batch_tool_h
#define excon_batch_tool_h

#include <cstdio>

/**************************************************************************/

class Hashtable;
class Pmu;
class Stq;
class Value;
class Var;

/***************************************************************************
 *
 *  TOOLS
 *
 ***************************************************************************/

// USED TO STORE DEFINITION OF TOOL (1 FOR EACH TOOL)
class Tool {
public:
    Tool(const char* name);
    ~Tool();

    int setexec(const char* str);
    const char* getexec() const;

    int setpath(const char* str);
    const char* getpath() const;

    int setruntime(long secs);
    long getruntime() const;

    const char* getname() const;
    const Stq* getvarnamelist() const;
    int nvars() const;
    int addvar(Var* var); 
    Var* getvar(const char* name) const;
    Var* getvar(int ith) const;
    Stq* getlinks();

private:
    char* name_;		// TOOL NAME
    char* exec_;		// EXEC STRING
    char* path_;		// EXEC PATH
    long runtime_;		// AVERAGE RUN TIME (seconds)
    Hashtable* vartable_;	// TABLE OF VARIABLES AND THEIR DEFAULT VALUES
    Stq* links_;		// LINKS TO OTHER PROGRAMS: LIST OF Link*
    Stq* varnlist_;		// LIST OF VARIABLE NAMES (for iteration)
};

inline const char* Tool::getexec() const { return exec_; }
inline const char* Tool::getpath() const { return path_; }
inline int Tool::setruntime(long secs) { runtime_ = secs; return 0; }
inline long Tool::getruntime() const { return runtime_; }
inline const char* Tool::getname() const { return name_; }
inline Stq* Tool::getlinks() { return links_; }
inline const Stq* Tool::getvarnamelist() const { return varnlist_; }


/***************************************************************************
 *
 *  TOOL INSTANCE
 *
 ***************************************************************************/

class ToolInstance {
// USED TO  STORE AN INSTANCE OF A TOOL IN AN EXPERIMENT
public:
    ToolInstance(Tool *tool, int number, int iteration = 0);
    ~ToolInstance();

    const char* getname() const;
    const char* getinstname() const;
    Tool* gettool() const;
    int getnumber() const;
    int getiteration() const;
    void setiteration(int iteration);
    Var* getvar(const char* name) const;
    int addvar(Var* var);
    const Value* getvarval(const char* name) const;
    int setvarval(const char* name, Value* val);
    int setvarval(Var* var, Value* val);
    int genexec();
    const char* getexec() const;
    int resolveunique();
    int setgpath();
    const char* getgpath() const;
    int readgfile(const char*);
    int writegfile() const;
    ToolInstance* clone() const;

    const Stq* get_resolved_links() const;
    void set_resolved_links(Stq*);
    bool needs_to_run() const;
    void needs_to_run(bool);

private:
    Tool* tool_;
    int number_;
    int iteration_;		// iteration counter for free variables.
    char* instname_;		// TOOL NAME WITH number APPENDED
    Hashtable* instvars_;	// INSTANCE SPECIFIC VARIABLE VALUES 
				// (MAY BE SAME VAL AS DEFAULTS)
    char* gpath_;		// PATHNAME OF INSTANCE gfile IF RELEVENT OR nil
    char* exec_;		// TOOL INSTANCE EXEC PROGRAM

    Stq* resolved_links_;

    Stq* freevars_;		// list of free variables for this instance
    bool needs_to_run_;		// run tree optimization.
};

inline const char* ToolInstance::getname() const { return tool_->getname(); }
inline const char* ToolInstance::getinstname() const { return instname_; }
inline Tool* ToolInstance::gettool() const { return tool_; }
inline int ToolInstance::getnumber() const { return number_; }
inline int ToolInstance::getiteration() const { return iteration_; }
inline const char* ToolInstance::getexec() const { return exec_; }
inline const char* ToolInstance::getgpath() const { return gpath_; }

inline const Stq* ToolInstance::get_resolved_links() const { 
    return resolved_links_; 
}

inline bool ToolInstance::needs_to_run() const { return needs_to_run_; }
inline void ToolInstance::needs_to_run(bool yn) { needs_to_run_ = yn; }

/****************************************************************************
 *
 *  TOOLS MANAGER
 *
 ****************************************************************************/

class ToolsMgr {
public:
    ToolsMgr ();
    ~ToolsMgr ();
    int load(const char* path, int nesting = 0);
    Tool* gettool(const char* toolname) const;
    const Stq* gettools() const;

private:
    int loadinputs (std::FILE* f, int& line);
    int loadlinks (std::FILE* f, int& line);
    int loadoutputs (std::FILE* f, int& line);

private:
    void errormissingtool(const char* file);
    void errorfilenotfound(const char* file);

private:
    Hashtable* tools_;
    Stq* toollist_;		// List of Tool*'s.
    // VARIABLES USED DURING load
    Pmu* pmu_;
    char buf_[1024];
    char const* currentloadfile_;
    Tool* current_tool_;	// CURRENT TOOL BEING PARSED IN TOOL TABLE
};

#endif/*excon_batch_tool_h*/
