//
// report.cc: automated reporting facility
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <cassert>
#include <cctype>
#include <fcntl.h>
#include <iostream>
#include <strstream>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#if HAVE_SYSENT_H
# include <sysent.h>			// USL cfront 3.0 specific?? (for open)
#endif
#include <ctime>
#include <unistd.h>
#include <sys/wait.h>


#include "exp.h"
#include "global.h"
#include "logger.h"
#include "misc.h"
#include "pmu.h"
#include "report.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

using namespace std;

/****************************************************************************
 *
 *  REPORTERS
 *
 ****************************************************************************/

Reporter::Reporter (const char* tname, const char* varname) {
    tname_ = strcpy_with_alloc(tname);
    varname_ = strcpy_with_alloc (varname);
    runtime_ = 0;
}

Reporter::~Reporter() {
    delete[] tname_; tname_ = 0;
    delete[] varname_; varname_ = 0;
}

/*************************************************************************/

IVReporter::IVReporter(
    const char* tname, const char* varname
) : Reporter(tname, varname) { }

IVReporter::~IVReporter() { }

Reporter* IVReporter::clone() const {
    return new IVReporter(gettoolname(), getvarname());
}

const Value* IVReporter::reportval(int iteration) const {
    ToolInstance* ti = Globals::expmgr->gettoolinstance(tname_, iteration);
    assume(ti != 0);
    return ti->getvarval(varname_);
}

/*************************************************************************/

RUNReporter::RUNReporter(
    const char* tname, const char* varname
) : Reporter(tname, varname), program_text_(0) {
    // CREATE ERROR VALUES
    errorfopen_  = new TextValue ("\"ERROR Opening reporter file\"!");
    errorfork_   = new TextValue ("\"ERROR Running reporter\"!");
    errorgetres_ = new TextValue ("\"ERROR Getting value from reporter!\"");
    // GET THE ORIGINAL TOOL INSTANCE. INDEPENDENT OF THE ITERATION
    const ToolInstance* ti = Globals::expmgr->gettoolinstance(tname); 
    assume(ti != 0);
    // CREATE A VALUE OF THE RIGHT TYPE
    const Var* cvar = ti->getvar(varname);  assume (cvar != 0);
    val_ = cvar->genval();
    // THE VALUE OF varname IS THE program...EXPAND THE program TEXT
    // EX: tshow -i ttp $GPRED 
    const Value* cval = ti->getvarval (varname);  assume (cval != 0);

    char token[1024];  
    char buf[1024];
    strcpy (buf, cval->gettext());
    int pos = 0;  

    // get the first token, the program to run. rest of text is also saved.

    pos = Globals::pmu->match(buf, "<name>");
    assert(pos != 0);
    Globals::pmu->gettextfield(1, token);

    // IF ITS THE FIRST TOKEN ITS A PROGRAM NAME...FIND ITS FULL PATHNAME
    const Tool* tool = Globals::toolsmgr->gettool (token);  assume (tool != 0);
    program_ = strcpy_with_alloc(tool->getpath());

    // ESTIMATE OF GETTING RESULTS IS ESTIMATE PROGRAM RUN TIME
    runtime_ = tool->getruntime();

    // rest of the text is saved for expansion at runtime.
    program_args_ = strcpy_with_alloc(&buf[pos]);
}

RUNReporter::~RUNReporter() {
    delete errorfopen_;
    delete errorfork_;
    delete errorgetres_;
    delete val_;
    delete[] program_; program_ = 0;
    delete[] program_args_; program_args_ = 0;
    delete[] program_text_; program_text_ = 0;
}

Reporter* RUNReporter::clone() const {
    return new RUNReporter(gettoolname(), getvarname());
}

void RUNReporter::genexec(int iteration) {
    char tmpbuf[1024];

    delete[] program_text_; program_text_ = 0;

    strcpy(tmpbuf, program_args_);

    const ToolInstance* ti = Globals::expmgr->gettoolinstance(tname_, iteration);
    assume(ti != 0);

    // substitute for the key words '$KEYWORD'. Highly inefficient scanning,
    // but what the hell ... this is what computers are for, isn't it?
    // FIX the scanning and substitution algorithm. Take it from tcsh/csh.

    // proposed algorithm:
    //    char buf[tmpbuf];
    //    strcpy(buf, program_args_);
    //    while (token = get_keyword(program_args_)) {
    //        check_keyword(token);
    //        substitute_in_place(buf, token);
    //    }
    //


    // current scanning strategy. First scan and substitute for all simple
    // keywords like $CPU $PATH and weird ones like $GPRED $GPREDNS, which
    // don't have any qualifiers such as tool name.

    char tmpbuf2[1024];
    // SUBSTITUTE FOR $GPRED 
    if (strstr (tmpbuf, "$GPRED")) {
	char tmpbuf3[1024];

	tmpbuf3[0] = '\0';
	const Stq* job = Globals::expmgr->get_job_sequence(iteration); 
	assume (job != 0);
	int append = 1;
	for (int i=1;i<=job->size();i++) {
	    const ToolInstance *temp_ti = (ToolInstance*)job->cycle();
	    if (temp_ti == ti) 
		append = 0;
	    if (append) { 
		if (temp_ti->getgpath() != 0) { 
		    strcat (tmpbuf3, temp_ti->getgpath()); 
		    strcat (tmpbuf3, " "); 
		}
	    }
	}

	strsub("$GPRED", tmpbuf3, tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }

    if (strstr (tmpbuf, "$GPREDNS")) {
	char tmpbuf3[1024];

	tmpbuf3[0] = '\0';
	const Stq* job = Globals::expmgr->get_job_sequence(iteration); 
	assume (job != 0);
	// SCAN THE EXPERIMENT SEQUENCE TILL YOU GET TO ti...APPENDING 
	// gfile PATHS
	for (int i=1;i<=job->size();i++) {
	    int append = 1;
	    const ToolInstance *temp_ti = (ToolInstance*)job->cycle();
	    if (temp_ti == ti) 
		append = 0;
	    const char* toolname = temp_ti->getname();
	    if (strcmp(toolname,"sync")==0) 
		append = 0;
	    if (append) {
		if (temp_ti->getgpath() != 0) { 
		    strcat (tmpbuf3, temp_ti->getgpath()); 
		    strcat (tmpbuf3, " "); 
		}
	    }
	}
	strsub("$GPREDNS", tmpbuf3, tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }

    if (strstr (tmpbuf, "$ITERATION")) {
	char tmpbuf3[1024];

	tmpbuf3[0] = '\0';
	sprintf(tmpbuf3, "%d", iteration);
	strsub("$ITERATION", tmpbuf3, tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }


    // now the qualified ones. All variables have to be fully qualified
    // (such as xlith.gap), else these are simply passed off as strings
    // to the output driver.  $GFILE is a special case, where by itself
    // it is assumed to be for the current tool, whatever that may be.

    // use the global PMU instance to scan.
    int pos = 0;  
    char tmpbuf3[1024];
    strcpy(tmpbuf3, tmpbuf);
    char const* rest = tmpbuf3;
    char toolname[256];
    char varname[256];
    char match[256];
    while((pos = Globals::pmu->match(rest, "<name>"))) {
	rest = &rest[pos];
	Globals::pmu->gettextfield(1, match);
	if (Globals::pmu->match(match, "<ident>", ".", "<variable>")) {
	    Globals::pmu->gettextfield(1, toolname);
	    Globals::pmu->gettextfield(3, varname);

	    // check sanity. The only tool allowed is the current one.
	    const char* tname = ti->getname();
	    if(strcmp(tname, toolname) == 0) {
		const Value* value = ti->getvarval(varname);
		if(value != 0) {
		    strsub(match, value->getstrval(), tmpbuf, tmpbuf2);
		    strcpy(tmpbuf, tmpbuf2);
		} 
		else {
		    cerr 
			<< "RUNReport: bad variable name in tools file: ``"
			<< match << "''. Passed onto the output driver.\n";
		}
	    }
	    else {
		cerr 
		    << "RUNReport: bad tool name in tools file: ``"
		    << match << "''. Passed onto the output driver.\n";
	    }
	}
    }

    // catch the sole $GFILE if it's still remains.
    if (strstr (tmpbuf, "$GFILE")) {
	strsub("$GFILE", ti->getgpath(), tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }

    // buf NOW CONTAINS THE EXPANDED PROGRAM
    strcpy(tmpbuf2, program_);
    strcat(tmpbuf2, " ");
    strcat(tmpbuf2, tmpbuf);
    program_text_ = strcpy_with_alloc (tmpbuf2);
}

const Value *RUNReporter::reportval(int iteration) const {
    char path[1024];
    char buf[1024];

    // pretend to a non-const member.
    ((RUNReporter*)this)->genexec(iteration);		// UNCONST

    EXCON_LOG2(collect)
	<< "   RUN Reporter for `" << getvarname() << "'\n";

    // GET UNUSED FILE NAME TO COPY PROGRAM INTO AND WRITE RETURN VALUE
    ::getunusedpath (path, "exp.", 1);  // 1 AT END MEANS CREATE THE FILE
    EXCON_LOG2(collect)
	<< "      Unused path `" << path << "' returned\n";

    int infd = -1, outfd = -1;
    //
    // Run the job...waiting for it to finish. The child process return
    // status is kept in the status variable.
    //
    int status = 0;
    pid_t pid = 0;
    switch (pid = fork()) {
	case -1 : 
	    if (Globals::abort_on_error) {
		EXCON_LOG_ALWAYS
		    << "EXCON: Cannot execute Job " << program_text_
		    << "(" << iteration << ") due to lack of VM." 
		    << endl
		    << "    Aborting due to --abort-on-error flag." << endl;
		exit(1);
	    }
	    return errorfork_;		// Too many processes or out VM
	    status = -1;
	    break;

	case  0 : 
	    EXCON_LOG2(collect)
		<< "      Command " << program_text_ 
		<< " written to `" << path << "'\n";

	    // set up redirection for child process.
	    if ((outfd = open (path, O_WRONLY | O_CREAT)) == -1) {
		return errorfopen_;
	    }
	    if ((infd = open ("/dev/null", O_RDONLY)) == -1) {
		return errorfopen_;
	    }

	    close (fileno(stdin)); dup2(infd, fileno(stdin));
	    close (fileno(stdout)); dup2(outfd, fileno(stdout));
	    close (fileno(stderr)); dup2(outfd, fileno(stderr));
	    execlp ("/bin/sh", "/bin/sh", "-c", program_text_, (char*)0); 
	    perror("CANNOT EXEC CHILD");
	    _exit (1);
	    break;

	default : 
	    wait(&status); 
	    break;
    }
    if (Globals::abort_on_error && status != 0) {
	EXCON_LOG_ALWAYS
	    << "EXCON: $RUN operation `" << program_text_ << "' ("
	    << iteration << ") return non-0 status (" << status << ")"
	    << endl
	    << "    Aborting due to --abort-on-error flag." << endl;
	exit(1);
    }

    Globals::stats->ranprogram();
    //
    // Extract the result from PATH, package and return the result
    //
    FILE* fp = fopen (path, "r");
    if (fp == 0)
	return errorgetres_;
    if (fgets (buf, sizeof(buf), fp) == 0) 
	return errorgetres_;

    EXCON_LOG2(collect)
	<< "      Read return value `" << buf << "'\n";

    val_->setval (buf);
    fclose (fp);
    Globals::stats->searchedoutputfile();
    // GET RID OF THE FILE
    unlink (path);
    return val_;
}

/****************************************************************************
 *
 *  REPORTS
 *
 ***************************************************************************/

void Report::errorreportfile(const char* file) {
    cerr
	<< "Unable to open and append to the report file `" << file 
	<< "'.\nYou may not have write permission.\n"
	<< "Check the file and try running the experiment again.\n";
    exit (1);
}

Report::Report(const char* name, const char* path, const char* sep) {
    name_ = strcpy_with_alloc (name);
    path_ = strcpy_with_alloc (path);
    separator_ = strcpy_with_alloc(sep);
    headers_ = new Stq();
    reporters_ = new Stq ();
    row_= new Stq ();
    runtime_ = 0;
}

Report::~Report() {
    delete[] name_; name_ = 0;
    delete[] path_; path_ = 0;
    delete[] separator_;
    delete headers_;
    delete reporters_; reporters_ = 0;
    delete row_; row_ = 0;
}

Report* Report::clone() const {
    Report* newreport = new Report(getname(), getpath(), separator_);
    unsigned size = reporters_->size();
    while(size--) {
	const Reporter* reporter = (Reporter*)reporters_->cycle();
	Reporter* newreporter = reporter->clone();
	newreport->enqreporter(newreporter);
    }
    return newreport;
}

void Report::separator(const char* sep) {
    delete[] separator_;
    separator_ = strcpy_with_alloc(sep);
}

void Report::enqheader(const char* header) { 
    headers_->enq(strcpy_with_alloc(header)); 
}

void Report::enqreporter(const Reporter* reporter) { 
    reporters_->enq(reporter); 
    runtime_ += reporter->getruntime(); 
}

void Report::reset() { row_->reset(); }

//
// create the output file name for this report. The default name of
// the output file is the same as the one specified in the EXP file
// (eg., `table1.columns = ... ' ==> table1 as the output file name),
// but the final say comes from EXCON main driver via command line
// argument (--output=FILE or -o FILE).
// 
// Make sure we have the right numeric suffix from the original name
// before changing the name. If the EXP table name is "table1" and the
// user-specified REPORT file prefix is "report", make the file name
// "report1".
//
void Report::setpath(const char* path) {
    //
    // extract the original numeric suffix from report name and append
    // onto the new one.
    //
    char* oldpath = path_;
    int pos = strlen(oldpath)-1; 
    while (isdigit(oldpath[pos])) 
	pos--;
    char buf[1024];  strcpy (buf, path);  strcat (buf, &oldpath[pos+1]);
    path_ = strcpy_with_alloc (buf);
    delete[] oldpath;
}

void Report::collectresults (int iteration) {
    EXCON_LOG2(collect)
	<< "Collecting " << reporters_->size() << " results for report `"
	<< name_ << "'.\n";
    for (int i=1;i<=reporters_->size();i++) {
	Reporter *reporter = (Reporter*)reporters_->cycle();
	const Value *result = reporter->reportval(iteration);
	EXCON_LOG2(collect)
	    << "   Reporter `" << reporter->getvarname() << " returns `" 
	    << result->getstrval() << "'.\n";
	row_->enq (result); 
    }
}
void Report::printheader () {
    if (headers_->size() == 0)
	return;
    char buf[256];  buf[0] = '\0';
    FILE* fp = fopen (path_, "a");  
    if (fp == 0) { errorreportfile(path_); return; }
    for (int i=1;i<=headers_->size();i++) {
	const char* header = (char*) headers_->cycle();
	strcat (buf,header);  strcat (buf," ");
    }
    strcat (buf,"\n");
    EXCON_LOG2(collect)
	<<  "REPORT: Writing into file `" << path_ << "' the value `"
	<< buf << "'\n";
    fprintf(fp, buf);
    fclose(fp);
}

void Report::printrow () {
    char buf[256];  buf[0] = '\0';
    FILE* fp = fopen (path_, "a");  
    if (fp == 0) { errorreportfile(path_); return; }
    for (int i=1;i<=row_->size();i++) {
	const Value *val = (Value*)row_->cycle();
	strcat (buf,val->getstrval());  
	strcat (buf, separator_);
    }
    strcat (buf,"\n");
    EXCON_LOG2(collect)
	<<  "REPORT: Writing into file `" << path_ << "' the value `"
	<< buf << "'\n";
    fprintf(fp, buf);
    fclose(fp);
}
