//
// job.cc: Unix job exec manager
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <cstdio>
#include <cstdlib>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "exp.h"
#include "global.h"
#include "job.h"
#include "logger.h"
#include "utils.h"

using namespace std;

//
// Need the wait stuff for maximal portability. Looks like SunOS4.1.3 CC
// has the 'union wait' and the /usr/include/sys/wait.h there has the
// int version.
//
#if !HAVE_UNION_WAIT

#define	WAIT_T int

#ifndef	WTERMSIG
#define WTERMSIG(x) ((x) & 0x7f)
#endif
#ifndef	WCOREDUMP
#define WCOREDUMP(x) ((x) & 0x80)
#endif
#ifndef	WEXITSTATUS
#define WEXITSTATUS(x) (((x) >> 8) & 0xff)
#endif
#ifndef	WIFSIGNALED
#define WIFSIGNALED(x) (WTERMSIG (x) != 0)
#endif
#ifndef	WIFEXITED
#define WIFEXITED(x) (WTERMSIG (x) == 0)
#endif

#else	/* Have `union wait'.  */

#define WAIT_T union wait
#ifndef	WTERMSIG
#define WTERMSIG(x)	((x).w_termsig)
#endif
#ifndef	WCOREDUMP
#define WCOREDUMP(x)	((x).w_coredump)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(x)	((x).w_retcode)
#endif
#ifndef	WIFSIGNALED
#define	WIFSIGNALED(x)	(WTERMSIG(x) != 0)
#endif
#ifndef	WIFEXITED
#define	WIFEXITED(x)	(WTERMSIG(x) == 0)
#endif

#endif	/* Don't have `union wait'.  */

/************************************************************************/

Job::Job(const char* name, int jobid, pid_t pid) {
    name_ = strcpy_with_alloc(name);
    jobid_ = jobid;
    pid_ = pid;
    status_ = 0;
}

Job::~Job() {
    delete[] name_;  name_ = 0;
    pid_ = -1;
    jobid_ = status_ = -1;
}

/************************************************************************/


JobMgr::JobMgr(unsigned limit) : limit_(limit) 
{
    joblist_ = new Stq;
    donejobs_ = new Stq;
}

JobMgr::~JobMgr() { 
    limit_ = -1;
    // CHECK: these deletions.
    unsigned size = joblist_->size();
    while(size--) {
	Job* job = (Job*)joblist_->pop();
	delete job; job = 0;
    }
    delete joblist_; joblist_ = 0;
    size = donejobs_->size();
    while(size--) {
	Job* job = (Job*)donejobs_->pop();
	delete job; job = 0;
    }
    delete donejobs_; donejobs_ = 0;
}

bool JobMgr::freecpu() {
    return (joblist_->size() < limit_) ? true : false; 
}

int JobMgr::execute_job(const char* runscript, int jobid) {
    int retcode = 0;
    pid_t pid = fork();
    switch (pid) {
	case -1 : 
	    perror ("JobMgr::execute_job");
	    retcode = -1;
	    break;

	case  0 : 
	    execlp ("/bin/sh", "/bin/sh", "-c", runscript, (char*)0); 

	    EXCON_LOG_ALWAYS
		<< "JobMgr(child): execution failed on ``" << runscript
		<< "''." << endl;
	    exit(-1);

	default : 
	    //
	    // only log in the parent, so it goes here.
	    //
	    EXCON_LOG2(job) 
		<< "EXCON: Running job '" << runscript << "'..." << endl;

	    (void)addjob(runscript, jobid, pid);
	    break;
    }
    return retcode;
}

int JobMgr::njobs_running() {
    return joblist_->size();
}

#ifndef NON_BLOCKING

//
// blocking call, only returns WHEN and IF ONLY a child process gets done.
// returns: 
//    number of jobs that are waiting for client to reap.
//

int JobMgr::wait_for_completed_job() {
    int numdone = 0;
    pid_t pid;
    WAIT_T status;
    while((pid = wait(&status)) > 0) {
	Job* job = getjob(pid);
	if (job) {
	    job->setstatus(WEXITSTATUS(status));
	    donejobs_->enq(job);
	    removejob(job);
	    ++numdone;
	}
    }
    return numdone;
}

#else/*NON_BLOCKING*/

//
// non-blocking call
// returns: 
//    number of jobs that are waiting for client to reap. (0 if none)
//

int JobMgr::wait_for_completed_job() {
    int numdone = 0;
    int retcode = -1;
    for (int i = 1; i <= joblist_->size(); i++) {
	Job* job = (Job*)joblist_->cycle();
	pid_t pid = job->getpid();
	WAIT_T status;
	if ((waitpid (pid, &status, WNOHANG)) > 0) {
	    assume(job == getjob(pid));
	    job->setstatus(WEXITSTATUS(status));
	    donejobs_->enq(job);
	    ++numdone;
	}
    }

    // MUST NOT DO THIS IN THE LOOP ABOVE. joblist_ gets munged otherwise.
    for (i = 1; i <= donejobs_->size(); i++) {
	Job* job = (Job*)donejobs_->cycle();
	removejob(job);
    }
    return numdone;
}

#endif/*!NON_BLOCKING*/

int JobMgr::reap_job(int& status) {
    int jobid = -1;
    if (donejobs_->size() > 0) {
	Job* job = (Job*)donejobs_->pop();
	jobid = job->getjobid();
	status = job->getstatus();
	delete job; job = 0;
    }
    return jobid;
}

//
// add_job:
// 
// return the jobnumber if successful or -1 on error
//

Job* JobMgr::addjob(const char* jobname, int jobid, pid_t pid) {

    EXCON_LOG2(job)
	<< "JobMgr::addjob: adding job `" << jobname << "'" << endl;

    Job* job = new Job(jobname, jobid, pid);
    joblist_->enq(job);
    return job;
}

Job* JobMgr::getjob(pid_t pid) {
    Job* job = 0;
    unsigned size = joblist_->size();
    while(size--) {
	Job* tmpjob = (Job*)joblist_->cycle();
	if (tmpjob->getpid() == pid)
	    job = tmpjob;
    }
    return job;
}

void JobMgr::removejob(const Job* rmjob) {

    EXCON_LOG2(job)
	<< "JobMgr::removejob: removing job id '" << rmjob->getjobid()
	<< "'." << endl;

    Job* removed = 0;
    unsigned size = joblist_->size();
    while(size--) {
	Job* job = (Job*)joblist_->pop(); 
	if (job == rmjob)
	    removed = job;
	else
	    joblist_->enq(job);
    }
    assume(removed == rmjob);
}
