//
// exp2report_init.cc: Initialize the EXP2REPORT specific stuff
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1996 Mumit Khan
//

// ====================================================================== //

#include <string>
#include <string.h>
#include <stdio.h>
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <stdlib.h>
#include <pwd.h>		// for user info
#include <sys/types.h>		// for uid_t, pid_t etc

#include <getopt.h>		// better pick up GNU getopt!
#include <libcxrl.h>

#include <fstream.h>

#include "exp2report.h"
#include "exp.h"
#include "global.h"
#include "job.h"
#include "logger.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"

// ====================================================================== //


//
// names of the logfile EXCON uses. Defaults to "exp2report.log";
//
static string logfile = "exp2report.log";

//
// the trace string on command line line.
//
static string trace_str;

//
// the search path given on command line.
//
static string search_path;

//
// debug flag
//
static bool g_verbose = false;

//
// default flags.
//
static unsigned default_traces = Logger::minimal | Logger::load;


// ====================================================================== //

//
// command line options and such
//
struct option long_options[] = {
    { "gfile",		required_argument,	0, 'g' },
    { "search-path",	required_argument,	0, 'I' },
    { "expfile",	required_argument,	0, 'e' },
    { "toolsfile",	required_argument,	0, 't' },
    { "output",		required_argument,	0, 'o' },
    { "report",		required_argument,	0, 'o' },
    { "logfile",	required_argument,	0, 'l' },
    { "traces",		required_argument,	0, 'T' },
    { "debug",		optional_argument,	0, 'd' },
    { "verbose",	optional_argument,	0, 'v' },
    { "help",		no_argument,		0, 'h' },
    { "more-help",	no_argument,		0, 129 },
    { "print-gfile",	no_argument,		0, 130 },
    { 0,		0,			0,  0  }
};

static void usage(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "exp2report: generate a report from EXCON run (EXP and TOOLS file)\n\n"
    "Usage: exp2report [OPTIONS]\n\n"
    "Most options have reasonable defaults or are prompted for\n"
    "The following OPTIONS are recognized (in either short or long form):\n\n" 
    "  -g FILE, --gfile=FILE        Read g-parameters from FILE\n"
    "  -I PATH, --search-path=PATH  Use PATH (eg., .:tools) to search\n"
    "  -t FILE, --tool=FILE         Read TOOLS from FILE\n"
    "  -e FILE, --exp=FILE          Read EXPERIMENT from FILE\n"
    "  -o FILE, --output=FILE       Write the report to FILE\n"
    "  -d,      --debug             Turn on debugging. Not pretty!\n"
    "  -v,      --verbose           Turn on verbose mode.\n"
    "  -h,      --help              Print this info\n"
    "           --more-help         Print more help info\n"
    "           --print-gfile       Show example gfile parameters\n"
    "\n\n"
    "Example:\n"
    "  % exp2report -t excon.tools -e excon.exp -o report\n"
    "\n";

    if (die)
	exit(diecode);
}

static void more_usage(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "exp2report: generate a report from EXCON run (EXP and TOOLS file)\n\n"
    "Usage: exp2report [OPTIONS]\n\n"
    "Most options have reasonable defaults or are prompted for\n"
    "The following OPTIONS are recognized (in either short or long form):\n"
    "(you can use unique substrings for the long form; eg., you could\n"
    "say --out instead of --output since that's unique)\n\n"
    "  -g FILE, --gfile=FILE\n"
    "   You can supply write-mm all the parameters via \"gfile\" just like\n"
    "   all the other Toolset programs. Command line parameters override\n"
    "   the values in FILE, so you can use these as default values.\n"
    "   Use --print-gfile to see an example g-file.\n"
    "\n"
    "  -I PATH, --search-path=PATH\n"
    "   PATH is a colon- or comma- separated list of directories that EXCON\n"
    "   searches for TOOLS and EXPERIMENT files, when these files are named\n"
    "   within < and > (eg., <excon.tools>). Multiple -I flags are simply\n"
    "   concatenated as they are seen.\n"
    "   An example of PATH would be /usr/local/excon/tools:toolsdir\n"
    "\n"
    "  -t FILE, --tool=FILE\n"
    "   TOOLS file contains EXCON tool definitions. Usually users use the\n"
    "   Toolset provided one in $TOOLSET_DATA directory. Specify the FILE\n"
    "   within < and > to force EXCON to search for it.\n"
    "   Typically named ``excon.tools''\n"
    "\n"
    "  -e FILE, --exp=FILE\n"
    "   EXPRIMENT file contains user experiment sequence and also parameters\n"
    "   for the models and tools. Specify the FILE within < and > to force\n"
    "   EXCON to search for it.\n"
    "   Typically named ``excon.exp''\n"
    "\n"
    "  -o FILE, --output=FILE, --report=FILE\n"
    "   EXCON writes the experiment report to FILE.\n"
    "\n"
    "  -d,      --debug\n"
    "   Print out gobs of information about EXCONs run. Not pretty. This is\n"
    "   equivalent to seting all the internal traces, ie, --traces=all\n"
    "\n"
    "  -v,      --verbose           Turn on verbose mode.\n"
    "  -h,      --help              Print this info\n"
    "           --more-help         Print more help info\n"
    "   Print a bit more elaborate command line help information\n"
    "\n"
    "           --print-gfile       Show example gfile parameters\n"
    "\n"
    "Examples:\n"
    "  % exp2report -t excon.tools -e excon.exp -o report\n"
    "\n";

    if (die)
	exit(diecode);
}

static void print_gfile(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "###\n"
    "#\n"
    "# Example EXP2REPORT gfile\n"
    "#\n"
    "###\n"
    "#\n"
    "# verbose mode (y/n).\n"
    "#\n"
    "verbose = n\n"
    "#\n"
    "# EXCON file search path for INCLUDE/TOOLS/EXP files. This is a\n"
    "# colon separated list of directories that EXCON searches when\n"
    "# sees a filename of the form <FILE>, where FILE is the filename\n"
    "# to open. EXCON always searches current directory and then\n"
    "# TOOLSET_DATA directory at the end.\n"
    "#\n"
    "search_path = ~/excon/tools:~khan/excon/tools\n"
    "#\n"
    "# EXCON Tools file. This can in turn INCLUDE others. If you enter\n"
    "# the file name as <FILE> (eg., <excon.tools>), EXCON will search\n"
    "# the directories in the search path for excon.tools file.\n"
    "#\n"
    "toolsfile = <excon.tools>\n"
    "#\n"
    "# EXCON Experiment file. This can in turn INCLUDE others. If you enter\n"
    "# the file name as <FILE> (eg., <excon.exp>), EXCON will search\n"
    "# the directories in the search path for excon.exp file.\n"
    "#\n"
    "expfile = excon.exp\n"
    "#\n"
    "# EXP2REPORT Report file. This is where you get your output.\n"
    "#\n"
    "output = report.out\n"
    "\n";

    if (die)
	exit(diecode);
}

static int read_gfile(const string& gfile,
    string& toolsfile, string& expfile, string& output
) {
    //
    // now read the gfile is supplied before the rest of the command
    // is parsed, so we can override all of it via cmdline.
    //
    g_file(const_cast<char*>(gfile.c_str()));
    char buf[1024];
    if (g_peek("verbose")) {
	g_string("verbose", "Verbose mode (y/n)?", buf);
	if (buf[0] == 'y' || buf[0] == 'Y' || buf[0] == '1')
	    g_verbose = true;
    }
    if (g_peek("search_path")) {
	g_string("search_path", "Enter EXCON file search path ", buf);
	search_path = buf;
    }
    if (g_peek("toolsfile")) {
	g_string("toolsfile", "Enter EXCON TOOLS file", buf);
	toolsfile = buf;
    }
    if (g_peek("expfile")) {
	g_string("expfile", "Enter EXCON experiment file", buf);
	expfile = buf;
    }
    if (g_peek("output")) {
	g_string("output", "Enter EXP2REPORT output report file", buf);
	output = buf;
    }
    return 0;
}
    
//
// parse the command line. First see if the user supplied a g-file or
// not; if so, read the g-file first to build the default set of params
// and THEN read the command line to override the g-file parameters.
//
static int parse_args(int argc, char* argv[],
    string& toolsfile, string& expfile, string& output
) {
    int c, ind;
    string gfile;
    int errcnt = 0;

    //
    // first pass thru the command line args to read gfile and set verbose
    // flag. Also check for flags like --help, --print-examples etc and
    // handle in the first pass.
    //
    while ((c = getopt_long(
	argc, argv, "g:I:e:t:o:l:T:dvh", long_options, &ind)) != EOF
    ) {
    if (c == 0 && long_options[ind].flag == 0)
	c = long_options[ind].val;

	switch (c) {
	case 'g':
	    gfile = optarg;
	    break;

	case 'v':
	    if (optarg) {
		g_verbose = (
		    optarg[0] == 'y' || optarg[0] == 'Y' || optarg[0] == '1'
		);
	    } else {
		g_verbose = true;
	    }
	    break;

	case 'h':
	    usage(cout, true, 0);
	    break;
	
	case 129:
	    more_usage(cout, true, 0);
	    break;

	case 130:
	    print_gfile(cout, true, 0);
	    break;

	case '?':
	    ++errcnt;
	    break;
	    
	default:
	    break;
	}
    }
    if (errcnt) {
	usage(cerr, true, 1);
    }
    if (gfile.length()) {
	read_gfile(gfile, toolsfile, expfile, output);
    }

    //
    // reset optind to make a second pass. This time we override the
    // g-file params (if any) with command line params (if any).
    //
    optind = 0;
    while ((c = getopt_long(
	argc, argv, "g:I:e:t:o:l:T:dvh", long_options, &ind)) != EOF
    ) {
    if (c == 0 && long_options[ind].flag == 0)
	c = long_options[ind].val;

	switch (c) {
	case 'I':
	    search_path = optarg;
	    break;

	case 'e':
	    expfile = optarg;
	    break;

	case 't':
	    toolsfile = optarg;
	    break;

	case 'o':
	    output = optarg;
	    break;

	case 'l':
	    logfile = optarg;
	    break;
	
	case 'T':
	    trace_str = optarg;
	    break;

	case 'd':
	    Globals::debug = true;
	    break;

	case '?':
	    ++errcnt;
	    break;
	}
    }

    if (argc != optind) {
	cerr << "ERROR: bogus command line argument(s)." << endl;
	++errcnt;
    } 
    
    if (errcnt) {
	usage(cerr, true, 1);
    }

    char buf[1024];
    if (expfile.size() == 0) {
	promptforval (
	    "Enter experiment file name (default = excon.exp) > ", buf
	);
	if (strlen(buf) == 0) 
	     strcpy (buf, "excon.exp");
	expfile = buf;
    }
    if (toolsfile.size() == 0) {
	promptforval (
	    "Enter tools file name   (default = <excon.tools>) > ", buf
	);
	if (strlen(buf) == 0) 
	    strcpy (buf, "<excon.tools>");
	toolsfile = buf;
    }
    if (output.size() == 0) {
	promptforval (
	    "Enter output report file name (default = report.out) > ", buf
	);
	if (strlen(buf) == 0) 
	    strcpy (buf, "report.out");
	output = buf;
    }
    return 0;
}

static int assign_trace_flag(const char* str) {
    int trace_flag = -1;
    if (strcasecmp(str, "default") == 0)
	trace_flag = default_traces;
    else if (strcasecmp(str, "none") == 0)
	trace_flag = Logger::none;
    else if (strcasecmp(str, "minimal") == 0)
	trace_flag = Logger::minimal;
    else if (strcasecmp(str, "load") == 0)
	trace_flag = Logger::load;
    else if (strcasecmp(str, "nested_load") == 0)
	trace_flag = Logger::nested_load;
    else if (strcasecmp(str, "parse") == 0)
	trace_flag = Logger::parse;
    else if (strcasecmp(str, "run") == 0)
	trace_flag = Logger::run;
    else if (strcasecmp(str, "report") == 0)
	trace_flag = Logger::report;
    else if (strcasecmp(str, "resolve") == 0)
	trace_flag = Logger::resolve;
    else if (strcasecmp(str, "collect") == 0)
	trace_flag = Logger::collect;
    else if (strcasecmp(str, "debug") == 0)
	trace_flag = Logger::debug;
    else if (strcasecmp(str, "job") == 0)
	trace_flag = Logger::job;
    else if (strcasecmp(str, "all") == 0)
	trace_flag = Logger::all;
    
    return trace_flag;
}

static int init_logger() {
    unsigned trace_flags = default_traces;

    if (Globals::debug) {
	trace_flags = Logger::all;
    } else if (trace_str.size()) {
	char* tmpbuf = new char[trace_str.size() + 1];
	strcpy(tmpbuf, trace_str.c_str());

	char* this_trace;
	for(char* p = tmpbuf; (this_trace = strtok(p, ",:+")); p = 0) {
	    int this_trace_flag = assign_trace_flag(this_trace);
	    if (this_trace_flag == -1) {
		cerr << "exp2report (ERROR): Illegal trace flag `" 
		    << this_trace << "'. Ignored." << endl;
	    } else {
		trace_flags |= this_trace_flag;
	    }
	}
	delete[] tmpbuf;
    }
    //
    // unless user explicitly says "none", turn on minimal logging.
    //
    if (trace_flags != Logger::none)
	trace_flags |= Logger::minimal;
    Globals::logger->set_loglevel(trace_flags);

    ofstream* ofp = new ofstream(logfile.c_str());
    if (!*ofp) {
	cerr << "exp2report: Error opening log file \"" << logfile 
	    << "\". Will not log to file. Check permissions." << endl;
    } else {
	Globals::logger->add(*ofp);
    }
    return 0;
}

static int init_file_search_path() {
    //
    // initialize the search path for include files in TOOLS and EXP
    // files. The syntax is the following:
    //    excon [options] -I dir1:[dir2[:dir3]]]
    // the search path is then set to dir1 ... dirn in order, and then
    // "." and $TOOLSET_DATA (which is the default if no searchpath is 
    // specified).
    //
    if (search_path.size()) {
	char* tmpbuf = new char[search_path.size() + 1];
	strcpy(tmpbuf, search_path.c_str());
	char* include_dir;
	for(char *p = tmpbuf; (include_dir = strtok(p, ",:+")); p = 0) {
	    char tmpbuf2[1024];
	    if (expandpath(include_dir, tmpbuf2)) {
		EXCON_LOG_ALWAYS
		    << endl
		    << "EXCON (Warning): Directory `" << include_dir
		    << "' in search path is invalid. Ignored."
		    << endl;
	    } else {
		Globals::file_search_path->enq(strcpy_with_alloc(tmpbuf2));
	    }
	}
	delete[] tmpbuf;
    }

    Globals::file_search_path->enq(strcpy_with_alloc("."));
    char const* datadir = getenv("TOOLSET_DATA");
    if (datadir == 0) {
	datadir = TOOLSET_DATA;
    }
    Globals::file_search_path->enq(strcpy_with_alloc(datadir));

    return 0;
}

static int init_program_search_path() {
    //
    // initialize the search path from PATH env variable for PROGRAM 
    // files in TOOLS file.
    //
    char const* pathvar = getenv("PATH");
    if (pathvar == 0) {
	EXCON_LOG_ALWAYS
	    << "EXCON Warning: "
	    << "No PATH variable defined in login environment!"
	    << endl
	    << "    May not be able to run programs."
	    << endl;
    } else {
	char* tmpbuf = new char[strlen(pathvar) + 1];
	strcpy(tmpbuf, pathvar);
	char* program_dir;
	for(char *p = tmpbuf; (program_dir = strtok(p, ":")); p = 0) {
	    Globals::program_search_path->enq(strcpy_with_alloc(program_dir));
	}
	delete[] tmpbuf;
    }
    return 0;
}

static int init_user() {
    struct passwd* pw_entry = getpwuid(getuid());
    assume(pw_entry != 0);
    Globals::username = pw_entry->pw_name;
    Globals::homedir = pw_entry->pw_dir;
    return 0;
}

//=======================================================================//

//
// initialize as many globals here as we can. Some require that cmd
// is parsed first (eg., JobMgr needs ``njobs'' param), so we need to
// initialize these in post_init_globals().
//
static int pre_init_globals(int /*argc*/, char* argv[]) {
    //
    // setup username and such.
    //
    init_user();

    //
    // set up the logger first so others can use it.
    //
    Globals::logger      = new Logger(cerr);
    Globals::programname = getprogramname (argv);
    Globals::cleanupmgr  = new CleanupMgr ();  // CLEANUP FILES WE LEAVE AROUND
    Globals::pmu         = new Pmu        ();  // GLOBAL PATTERN MATCH UNIT
    Globals::stats       = new Stats      ();  // PROGRAM STATS COLLECTOR
    Globals::toolsmgr    = new ToolsMgr   ();  // TOOLS MANAGER
    Globals::expmgr      = new Exp2ReportMgr();  // EXPERIMENT MANAGER
    Globals::file_search_path = new Stq;
    Globals::program_search_path = new Stq;

    return 0;
}

static int post_init_globals(string& toolsfile, string& expfile) {
    init_logger();

    if (!getenv("TOOLSET_DATA")) {
	EXCON_LOG_ALWAYS
	    << "****** Exp2Report startup warning ****** " << endl
	    << "    TOOLSET_DATA variable is not defined in environment."
	    << endl
	    << "    Using " << TOOLSET_DATA << ". Some files may not be found."
	    << endl
	    << "    Please initialize Toolset environment with setup_toolset"
	    << endl
	    << "*********************************** "
	    << endl;
    }

    init_file_search_path();
    init_program_search_path();

    char buf[1024];
    switch(search_file(expfile.c_str(), Globals::file_search_path, buf)) {
	case 0:
	    expfile = buf;
	    break;
	
	case 1:
	    EXCON_LOG_ALWAYS 
		<< "EXP2REPORT Error: Cannot find EXPERIMENT file `" 
		<< expfile << "' in standard search path." 
		<< endl;
	    exit(1);
	    break;

	case 2:
	    EXCON_LOG_ALWAYS 
		<< "EXP2REPORT Error: Cannot open EXPERIMENT file `" 
		<< expfile << "'."
		<< endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    switch(search_file(toolsfile.c_str(), Globals::file_search_path, buf)) {
	case 0:
	    toolsfile = buf;
	    break;
	
	case 1:
	    EXCON_LOG_ALWAYS 
		<< "EXP2REPORT Error: Cannot find TOOLS file `" << toolsfile
		<< "' in standard search path." 
		<< endl;
	    exit(1);
	    break;

	case 2:
	    EXCON_LOG_ALWAYS 
		<< "EXP2REPORT Error: Cannot open TOOLS file `" 
		<< toolsfile << "'."
		<< endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    return 0;
}

//=======================================================================//

int exp2report_init(int argc, char* argv[],
    string& toolsfile, string& expfile, string& output
) {
    pre_init_globals(argc, argv);
    parse_args(argc, argv, toolsfile, expfile, output);
    post_init_globals(toolsfile, expfile);
    return 0;
}

//=======================================================================//
