//
// tools2gfile.cc: extract the gfile corresponding to a/set-of TOOL/s
//
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1996 Mumit Khan
//
//

// ====================================================================== //

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include <fstream.h>
#include <string>
#include <list>

#include "exp.h"
#include "global.h"
#include "logger.h"
#include "job.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

// ====================================================================== //

static void usage();
static int parse_args(int argc, char* argv[], bool& do_all,
    string& toolsfile, list<string>& toolnames
);
static int export_gfiles(const string& toolsfile, 
    bool do_all, const list<string>& toolnames
);

// ====================================================================== //

//
// global variables. Need this for the EXCON main globals. FIX/CLEAN/TODO.
//

bool g_verbose = false;
bool GLB_debug    = false;  // TURN ON DEBUGGING INFO
bool GLB_resdebug = false;  // DEBUG RESOLVE STEPS
bool GLB_repdebug = false;  // DEBUG REPORT CONSTRUCTION
bool GLB_rundebug = false;  // DEBUG EXPERIMENT EXECUTION
bool GLB_coldebug = false;  // DEBUG REPORT COLLECTION
char* GLB_programname; // NAME USED TO INVOKE THIS PROGRAM
Pmu* GLB_pmu;
ExpMgr* GLB_expmgr;
ToolsMgr* GLB_toolsmgr;
CleanupMgr* GLB_cleanupmgr;
Stats* GLB_stats;
JobMgr* GLB_jobmgr;
Logger* GLB_logger;
Stq* GLB_file_search_path = 0;


// ====================================================================== //

int main(int argc, char** argv) {
    string toolsfile;
    list<string> toolnames;
    bool do_all = false;

    parse_args(
        argc, argv, do_all, toolsfile, toolnames
    );

    return export_gfiles(toolsfile, do_all, toolnames);
}


// ====================================================================== //

static int get_all_tools(const ToolsMgr& toolsmgr, list<string>& tools) {
    const Stq* all_tools = toolsmgr.gettools();
    for (int i = 0; i < all_tools->size(); ++i) {
	const Tool* tool = (const Tool*)all_tools->cycle();
	tools.push_back(tool->getname());
    }
    return all_tools->size();
}

static void resolve_uniq(const string& toolname, const Value* value, 
    string& strval
) {
    strval = value->getstrval();
    if (strstr(strval.c_str(), "$UNIQUE")) {
	char subst[1024];
	strsub("$UNIQUE", toolname.c_str(), strval.c_str(), subst); 
	strval = subst;
    } 
} 

static int export_gfile(const ToolsMgr& toolsmgr, const string& toolname) {
    const Tool* tool = toolsmgr.gettool(toolname.c_str());
    if (!tool) {
	cerr << "tools2gfile: Tool `" << toolname << "' not found." << endl;
	return 1;
    }

    string gfile = toolname + ".g";
    ofstream ofp(gfile.c_str());
    if (!ofp) {
	cerr << "tools2gfile: cannot open file `" << gfile << "'." << endl;
	return 1;
    }

    if (g_verbose) {
	cerr << "Tool `" << toolname << "' variables ==> `" << gfile
	    << "'." << endl;
    }

    ofp << "######" << endl << "#" << endl
	<< "# TOOL : " << toolname << endl
	<< "# GFILE automatically generated from EXCON TOOLS file" << endl
	<< "#" << endl << "######" << endl << endl; 

    int nvars = tool->nvars();
    for(int i = 1; i <= nvars; ++i) {
	const Var* var = tool->getvar(i);
	if (var->hasprop("$INPUTS")) {		// only export INPUT vars.
	    const string varname = var->getname();
	    const Value* value = var->getval();
	    string strval;
	    resolve_uniq(toolname, value, strval);
	    ofp << varname << "\t=\t" << strval << endl;
	}
    }

    ofp << endl << "#" << endl << "#" << endl << "######" << endl << endl; 

    return 0;
}

//
// export_gfiles:
//
// read an EXCON toolsfile and exports the TOOL variables as gfile params.
//
static int export_gfiles(const string& toolsfile, 
    bool do_all, const list<string>& toolnames
) {
    ToolsMgr toolsmgr;
    toolsmgr.load(toolsfile.c_str());

    list<string> export_list;
    if (do_all) {
	get_all_tools(toolsmgr, export_list);
    } else {
	export_list = toolnames;
    }

    list<string>::const_iterator it = export_list.begin();
    for(; it != export_list.end(); ++it) {
	const string& toolname = *it;
	export_gfile(toolsmgr, toolname);
    }

    return 0;
}

// ====================================================================== //

static void usage () {
    cerr << endl
        << "Usage: tools2gfile [-hv] -t toolsfile [-a | TOOL_NAMES ]"
        << endl << endl
        << "    -v:     turn on verbose mode" << endl
        << "    -h:     print this info" << endl
        << endl
        << "    -t:     EXCON tools file to load" << endl
	<< "    -a:     export ALL tools in Tools file"
	<< endl
        << " TOOLS:     Name of the tools to export (if -a not specified)"
        << endl << endl
        << "Examples:\n"
        << "    tools2gfile -v -t excon.tools -a" << endl
        << "    tools2gfile -t excon.tools xprocess maskmaker" << endl
	<< endl;
    return;
}

static int parse_args(int argc, char* argv[], bool& do_all,
    string& toolsfile, list<string>& toolnames
) {
    int errcnt = 0;
    extern int optind;
    extern char *optarg;

    int c;

    do_all = false;

    while ((c = getopt (argc, argv, "hvat:")) != EOF) {
        switch (c) {
            case 't':
		toolsfile = optarg;
                break;

            case 'a':
                do_all = true;
                break;

            case 'v':
                g_verbose = true;
                break;

            case 'h':
		usage();
		exit(0);
                break;

            default:
                cerr << "tools2gfile:: Error in command line.\n";
                usage ();
                exit (1);
                break;
        }
    }

    if (argc > optind) {
	do_all = false;
	for(int i = optind; i < argc; ++i) {
	    toolnames.push_back(argv[i]);
	}
    }

    if (!do_all && toolnames.size() == 0) {
        cerr << "tools2gfile: Must specify -a or toolnames to export"
            << endl;
        usage();
        exit(1);
    }
    if (toolsfile.size() == 0) {
        cerr << "tools2gfile: Must specify a tools file to export from"
            << endl;
        usage();
        exit(1);
    }
    return (errcnt) ? -1 : 0;
}

// ====================================================================== //
