C +++
C
C gfile.F: Gfile manipulation routine in FORTRAN
C
C Source: src/lib/gfile.F
C
C Author: Mumit Khan <khan@xraylith.wisc.edu>
C
C ----------------------------------------------
C                SHADOW
C      Center for X-ray Lithography
C     University of Wisconsin-Madison
C  3731 Schneider Dr., Stoughton, WI, 53589
C ----------------------------------------------
C
C Copyright(c) 1996 Mumit Khan
C 

C
C Currently exported routines:
C
C  gfload_source: Loads a SOURCE gfile
C  gfload_oe:     Loads a OE gfile
C  gfsave_source: saves a SOURCE gfile
C  gfsave_oe:     saves a OESOURCE gfile
C
C
C Local routines:
C
C  gfgettoken : Gets 2 token separated by =
C  gfload     : loads a gfile into memory
C  gfind      : finds a gparameter in memory after loading
C  gferror    : report errors
C  gupdatex   : update value in namelist (various entry points)
C  gstorex    : save value from namelist (various entry points)
C  

#if defined(unix) || HAVE_F77_CPP
#	include		<header.txt>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:HEADER.TXT/LIST'
#endif


	subroutine gferror (line, tok1, tok2, iflag)
	implicit integer (a-z)
c++
c
c gferror -- report errors in loading/saving gfiles
c 
c returns: 
c    iflag:  0 if ok
c            1 if comment or empty line
c           -1 if bad things happenned
c
c--
	return
	end
c
c
c

	subroutine gfgettoken (line, tok1, tok2, iflag)
	implicit integer (a-z)
c++
c
c gfgettoken -- gets the 2 tokens separated by = in gfile
c 
c returns: 
c    tok1:   1st token, left justified
c    tok2:   2nd token, left justified (empty string is ' ')
c    iflag:  0 if ok
c            1 if comment or empty line
c           -1 if bad things happenned
c
c--
	character*(*) line, tok1, tok2
c
	l = len(line)
	if (l .eq. 0) then
	    iflag = 0
	    goto 99
	endif
c
	call fstrtrim_l (line, i1)
	if (i1 .eq. 0) then
	    iflag = 1
	    goto 99
	else if (line(i1:i1) .eq. '#') then
	    iflag = 1
	    goto 99
	endif
c
c	check for '=' in the left-justified portion of the string only.
c	if = is the first character (ie., pointed by i1), it's invalid. 
c
	call fstrchr (line, '=', index)
	if (index .le. i1) then
	    iflag = -1
	    goto 99
	endif
	tok1 = line(i1:index-1)
	call fstrtrim_l (line(index+1:), i2)
c
c i2 is relative to line(index+1:).
c
	if (i2 .eq. 0) then
	    tok2 = ' '
	else
	    tok2 = line(index+i2:)
	endif
	iflag = 0
	goto 99
c
 99	continue
 	return
	end
	
	subroutine gfind (name, index)
	implicit integer (a-z)
c++
c gfind  -- finds a gparam in memory
c
c inputs:
c    name -- g parameter name
c
c outputs:
c    index: index returned (0 means not found)
c
c--
	character*(*) name
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
c
	character*(max_strlen) locname
c
        i = 0
	call fstrtrim (name, i1, i2)
	if (i1 .lt. 1) then
	    goto 99
	endif
	locname = name(i1:i2)
	call fstrlocase (locname)
c
c both locname and g_name(i) area already left justified and lowercased.
c
	do 10 i = 1, ng_total
	    if (locname .eq. g_name(i)) goto 99
 10	continue
	i = 0
c
 99	continue
	index = i
	return
	end

c
	subroutine gupdatex
	implicit integer (a-z)
c++
c
c  Implements the following routines which convert the internal
c    representation to external representation and store both in
c    the namelist tables.
c
c	gupdate_a (name,string) - store alpha value
c	gupdate_i (name,number) - store integer value
c	gupdate_r (name,real)   - store real value
c--
	character*(*) name
	character*(*) string
c
	double precision dreal
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
c
c  alphanumeric
c
	entry gupdate_a (name,string)
	call gfind (name, index)
	if (index .eq. 0) then
	    string = ' '
	    goto 900
	endif
	string = g_value(index)
	return
c
c  integer
c
	entry gupdate_I (name,number)
	call gfind (name, index)
	if (index .eq. 0) then
	    number = 0
	    goto 900
	endif
	call gformat_i (g_value(index), number, iflag)
	if (iflag .ne. 0) then
	    call fstrtrim (name, i1, i2)
	    call fstrtrim (g_value(index), i3, i4)
	    write (*,1000) g_value(index)(i3:i4), name(i1:i2),
     $		gfname(1:gfnamelen)
     	    stop 1
        endif
 1000	format ('ERROR: Illegal value type "', a, '" for "', a, 
     $      '" in file "', a, '"')
	return
c
c  real double precision
c
	entry gupdate_r (name,dreal)
	call gfind (name, index)
	if (index .eq. 0) then
	    dreal = 0.0
	    goto 900
	endif
	call gformat_r (g_value(index),dreal, iflag)
	if (iflag .ne. 0) then
	    call fstrtrim (name, i1, i2)
	    call fstrtrim (g_value(index), i3, i4)
	    write (*,1010) g_value(index)(i3:i4), name(i1:i2),
     $		gfname(1:gfnamelen)
     	    stop 1
        endif
 1010	format ('ERROR: Illegal value type "', a, '" for "', a, 
     $      '" in file "', a, '"')
	return
c
 900	write (*,901) name, gfname(1:gfnamelen)
 901	format ('WARNING: Can''t locate parameter ',a, ' in gfile ', a)
	end
c
	subroutine gupdatex_v
	implicit integer (a-z)
c++
c
c  Implements the following routines which convert the internal
c    representation to external representation and store both in
c    the namelist tables.
c
c	gupdate_a (name,index,string) - store alpha value
c	gupdate_i (name,index,number) - store integer value
c	gupdate_r (name,index,real)   - store real value
c--
	character*(*) name
	character*(*) string
c
	double precision dreal
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	character*(max_strlen) local1, local2
c
c  alphanumeric
c
	entry gupdate_v_a (name,index,string)
	write (local1, 1010, err = 900) name, index
	call despace(local1, local2, klen)
	call gupdate_a (local2(1:klen), string)
 1010	format (a, '(', i4, ')')
	return
c
c  integer
c
	entry gupdate_v_i (name,index,number)
	write (local1, 1020, err = 900) name, index
	call despace(local1, local2, klen)
	call gupdate_i (local2(1:klen),number)
 1020	format (a, '(', i4, ')')
	return
c
c  real double precision
c
	entry gupdate_v_r (name,index,dreal)
	write (local1, 1030, err = 900) name, index
	call despace(local1, local2, klen)
	call gupdate_r (local2(1:klen),dreal)
 1030	format (a, '(', i4, ')')
	return
c
 900	write (*,901) name, gfname(1:gfnamelen)
 901	format ('WARNING: Can''t locate parameter ',a, ' in gfile ', a)
	end
c
	subroutine gformat_x
	implicit integer (a-z)
c++
c	subroutine gformat_i (string, value_i, iflag)
c	subroutine gformat_r (string, value_r, iflag)
c
c	value		- input/datatype x
c			  value to be converted
c
c	string		- output/string
c			  formatted value
c--
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
c  for various  data types
c
	integer			value_i
	double precision	value_r
	character*(*)		string
c
c  false entry point
c
	stop 'FORMAT_X is not an entry point'
c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
c  Integer
c
	entry gformat_i (string, value_i, iflag)
	read (string, *, err=99) value_i
 	iflag = 0
	return
c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
c  Double precision numbers
c
	entry gformat_r (string, value_r, iflag)
	read (string, *, err=99) value_r
 	iflag = 0
	return
c
 99	continue
     	iflag = -1
	return
	end

	subroutine gstorex
c++
c
c  Implements the following routines which convert the internal
c    representation to external representation and store both in
c    the namelist tables.
c
c	gstore_a (unit, name,string) - store alpha value
c	gstore_i (unit, name,number) - store integer value
c	gstore_r (unit, name,real)   - store real value
c--
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#	include 	<namelist.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
     	INCLUDE		'SHADOW$INC:NAMELIST.BLK/LIST'
#endif
c
	character*(*) name
	character*(*) string
c
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	integer unit
	double precision dreal
c
c
c  alphanumeric
c
	entry gstore_a (unit, name, string)
	write (unit, 1000, err = 900) name, string
 1000	format (a, ' = ', a)
	return
c
c  integer
c
	entry gstore_i (unit,name,number)
	write (unit, *, err = 900) name, ' = ', number
	return
c
c  real double precision
c
	entry gstore_r (unit,name,dreal)
	write (unit, *, err = 900) name, ' = ', dreal
	return
c
 900	write (*,901) name, gfname(1:gfnamelen)
 901	format ('Warning: Error writing parameter ',a, ' to gfile ', a)
	end
c

	subroutine gstorex_v
c++
c
c  Implements the following routines which convert the internal
c    representation to external representation and store both in
c    the namelist tables.
c
c	gstore_a (unit, name,index,string) - store alpha value
c	gstore_i (unit, name,index,number) - store integer value
c	gstore_r (unit, name,index,real)   - store real value
c--
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#	include 	<namelist.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
     	INCLUDE		'SHADOW$INC:NAMELIST.BLK/LIST'
#endif
c
	character*(*) name
	character*(*) string
c
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	character*256 local1, local2
	integer unit
	double precision dreal
c
c
c  alphanumeric
c
	entry gstore_v_a (unit, name,index,string)
	write (local1, 1010, err = 900) name, index
	call despace(local1, local2, klen)
	call gstore_a (unit, local2(1:klen), string)
 1010	format (a, '(', i4, ')')
	return
c
c  integer
c
	entry gstore_v_i (unit, name,index,number)
	write (local1, 1020, err = 900) name, index
	call despace(local1, local2, klen)
	call gstore_i (unit, local2(1:klen),number)
 1020	format (a, '(', i4, ')')
	return
c
c  real double precision
c
	entry gstore_v_r (unit, name,index,dreal)
	write (local1, 1030, err = 900) name, index
	call despace(local1, local2, klen)
	call gstore_r (unit, local2(1:klen),dreal)
 1030	format (a, '(', i4, ')')
	return
c
 900	write (*,901) name, gfname(1:gfnamelen)
 901	format ('WARNING: Can''t locate parameter ',a, ' in gfile ', a)
	end
c
	subroutine gfload (fname, iflag)
	implicit integer (a-z)
c++
c gfload  -- loads a gfile into memory
c
c inputs:
c    fname -- the gfile to open
c
c outputs:
c    iflag: 0 ok
c	   -1 file not found
c	   -2 format errors
c	   -3 other errors
c--
	character*(*) fname
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	character*256 line
	character*(max_strlen) tok1, tok2
c
	call fstrtrim (fname, i1, i2)
	if (i1 .lt. 1) then
	    iflag = -3
	    goto 99
	endif
c
	ng_total = 0
	line_no = 0
	gfname = fname(i1:i2)
	gfnamelen = i2-i1+1
c
	unit = 11
c
	open (unit=unit,file=fname(i1:i2), status='old', iostat=ierr)
	if (ierr .ne. 0) then
	    iflag = -1
	    goto 99
	endif
 10	continue
	read (unit,1000,end=98) line
	line_no = line_no + 1
	call gfgettoken (line, tok1, tok2, iflag)
	if (iflag .eq. 0) then
	    if (ng_total .eq. max_params) then
	    	write (6, *) 'Too many gparameters in file!'
		write (6, *) 'Fix the array ''max_params'' value'
		iflag = -3
		goto 99
	    endif
	    ng_total = ng_total + 1
	    call fstrlocase (tok1)
	    call fstrtrim (tok1, it1, it2)
	    g_name(ng_total) = tok1(it1:it2)
	    g_value(ng_total) = tok2
	else if (iflag .eq. -1) then
	    write (*,*) '(', fname(i1:i2), ':', line_no, '):' //
     $		'Illegal "name = value" format'
	    iflag = -2
	    goto 99
	endif
	goto 10
c
 98	close (unit)
 	iflag = 0
 99	continue
	return
 1000	format (a)
	end
c
c
	subroutine gfload_source (fname, iflag)
c
c inputs:
c    fname -- the gfile to open
c
c outputs:
c    iflag: 0 ok
c	   -1 file not found
c	   -2 format errors
c	   -3 other errors
c--
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#	include 	<namelist.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
     	INCLUDE		'SHADOW$INC:NAMELIST.BLK/LIST'
#endif
c
	character*(*) fname
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	call gfload (fname, iflag)
	if (iflag .ne. 0) then
	    call fstrtrim (fname, i1, i2)
	    if (iflag .eq. -1) then
		write (*,*) 'Error opening g file "',
     $		    fname(i1:i2), '"'
	    else if (iflag .eq. -2) then
		write (*,*) 'Illegal "name = value" format'
	    else if (iflag .eq. -3) then
		write (*,*) 'Error reading g file "test.g"'
	    endif
	    goto 99
	endif
c
c
	call gupdate_I ('fdistr', fdistr)
	call gupdate_I ('fgrid', fgrid)
	call gupdate_I ('fsour', fsour)
	call gupdate_I ('fsource_depth', fsource_depth)
	call gupdate_I ('f_coher', f_coher)
	call gupdate_I ('f_color', f_color)
	call gupdate_I ('f_phot', f_phot)
	call gupdate_I ('f_pol', f_pol)
	call gupdate_I ('f_polar', f_polar)
	call gupdate_I ('f_opd', f_opd)
	call gupdate_I ('f_wiggler', f_wiggler)
	call gupdate_I ('f_bound_sour', f_bound_sour)
	call gupdate_I ('f_sr_type', f_sr_type)
	call gupdate_I ('istar1', istar1)
	call gupdate_I ('npoint', npoint)
	call gupdate_I ('ncol', ncol)
	call gupdate_I ('n_circle', n_circle)
	call gupdate_I ('n_color', n_color)
	call gupdate_I ('n_cone', n_cone)
	call gupdate_I ('ido_vx', ido_vx)
	call gupdate_I ('ido_vz', ido_vz)
	call gupdate_I ('ido_x_s', ido_x_s)
	call gupdate_I ('ido_y_s', ido_y_s)
	call gupdate_I ('ido_z_s', ido_z_s)
	call gupdate_I ('ido_xl', ido_xl)
	call gupdate_I ('ido_xn', ido_xn)
	call gupdate_I ('ido_zl', ido_zl)
	call gupdate_I ('ido_zn', ido_zn)
	call gupdate_r ('sigxl1', sigxl1)
	call gupdate_r ('sigxl2', sigxl2)
	call gupdate_r ('sigxl3', sigxl3)
	call gupdate_r ('sigxl4', sigxl4)
	call gupdate_r ('sigxl5', sigxl5)
	call gupdate_r ('sigxl6', sigxl6)
	call gupdate_r ('sigxl7', sigxl7)
	call gupdate_r ('sigxl8', sigxl8)
	call gupdate_r ('sigxl9', sigxl9)
	call gupdate_r ('sigxl10', sigxl10)
	call gupdate_r ('sigzl1', sigzl1)
	call gupdate_r ('sigzl2', sigzl2)
	call gupdate_r ('sigzl3', sigzl3)
	call gupdate_r ('sigzl4', sigzl4)
	call gupdate_r ('sigzl5', sigzl5)
	call gupdate_r ('sigzl6', sigzl6)
	call gupdate_r ('sigzl7', sigzl7)
	call gupdate_r ('sigzl8', sigzl8)
	call gupdate_r ('sigzl9', sigzl9)
	call gupdate_r ('sigzl10', sigzl10)
	call gupdate_r ('conv_fact', conv_fact)
	call gupdate_r ('cone_max', cone_max)
	call gupdate_r ('cone_min', cone_min)
	call gupdate_r ('epsi_dx', epsi_dx)
	call gupdate_r ('epsi_dz', epsi_dz)
	call gupdate_r ('epsi_x', epsi_x)
	call gupdate_r ('epsi_z', epsi_z)
	call gupdate_r ('hdiv1', hdiv1)
	call gupdate_r ('hdiv2', hdiv2)
	call gupdate_r ('ph1', ph1)
	call gupdate_r ('ph2', ph2)
	call gupdate_r ('ph3', ph3)
	call gupdate_r ('ph4', ph4)
	call gupdate_r ('ph5', ph5)
	call gupdate_r ('ph6', ph6)
	call gupdate_r ('ph7', ph7)
	call gupdate_r ('ph8', ph8)
	call gupdate_r ('ph9', ph9)
	call gupdate_r ('ph10', ph10)
	call gupdate_r ('bener', bener)
	call gupdate_r ('pol_angle', pol_angle)
	call gupdate_r ('pol_deg', pol_deg)
	call gupdate_r ('r_aladdin', r_aladdin)
	call gupdate_r ('r_magnet', r_magnet)
	call gupdate_r ('sigdix', sigdix)
	call gupdate_r ('sigdiz', sigdiz)
	call gupdate_r ('sigmax', sigmax)
	call gupdate_r ('sigmay', sigmay)
	call gupdate_r ('sigmaz', sigmaz)
	call gupdate_r ('vdiv1', vdiv1)
	call gupdate_r ('vdiv2', vdiv2)
	call gupdate_r ('wxsou', wxsou)
	call gupdate_r ('wysou', wysou)
	call gupdate_r ('wzsou', wzsou)
	call gupdate_r ('plasma_angle', plasma_angle)
	call gupdate_A ('file_traj', file_traj)
	call gupdate_A ('file_source', file_source)
	call gupdate_A ('file_bound', file_bound)
	call gupdate_I ('oe_number', oe_number)
	call gupdate_I ('idummy', idummy)
	call gupdate_r ('dummy', dummy)
	call gupdate_I ('f_new', f_new)
c
	iflag = 0
c
 99	continue
	return
	end
c
c
c
	subroutine gfload_oe (fname, iflag)
c
c inputs:
c    fname -- the gfile to open
c
c outputs:
c    iflag: 0 ok
c	   -1 file not found
c	   -2 format errors
c	   -3 other errors
c--
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#	include 	<namelist.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
     	INCLUDE		'SHADOW$INC:NAMELIST.BLK/LIST'
#endif
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	character*(*) fname
c
	call gfload (fname, iflag)
	if (iflag .ne. 0) then
	    call fstrtrim (fname, i1, i2)
	    if (iflag .eq. -1) then
		write (*,*) 'Error opening g file "', 
     $		    fname(i1:i2), '"'
	    else if (iflag .eq. -2) then
		write (*,*) 'Illegal "name = value" format'
	    else if (iflag .eq. -3) then
		write (*,*) 'Error reading g file "test.g"'
	    endif
	    goto 99
	endif
c
	call gupdate_I ('fmirr', fmirr)
	call gupdate_I ('f_torus', f_torus)
	call gupdate_I ('fcyl', fcyl)
	call gupdate_I ('f_ext', f_ext)
	call gupdate_I ('fstat', fstat)
	call gupdate_I ('f_screen', f_screen)
	call gupdate_I ('f_plate', f_plate)
	call gupdate_I ('fslit', fslit)
	call gupdate_I ('fwrite', fwrite)
	call gupdate_I ('f_ripple', f_ripple)
	call gupdate_I ('f_move', f_move)
	call gupdate_I ('f_thick', f_thick)
	call gupdate_I ('f_bragg_a', f_bragg_a)
	call gupdate_I ('f_g_s', f_g_s)
	call gupdate_I ('f_r_ran', f_r_ran)
	call gupdate_I ('f_grating', f_grating)
	call gupdate_I ('f_mosaic', f_mosaic)
	call gupdate_I ('f_johansson', f_johansson)
	call gupdate_I ('f_side', f_side)
	call gupdate_I ('f_central', f_central)
	call gupdate_I ('f_convex', f_convex)
	call gupdate_I ('f_reflec', f_reflec)
	call gupdate_I ('f_rul_abs', f_rul_abs)
	call gupdate_I ('f_ruling', f_ruling)
	call gupdate_I ('f_pw', f_pw)
	call gupdate_I ('f_pw_c', f_pw_c)
	call gupdate_I ('f_virtual', f_virtual)
	call gupdate_I ('fshape', fshape)
	call gupdate_I ('fhit_c', fhit_c)
	call gupdate_I ('f_mono', f_mono)
	call gupdate_I ('f_refrac', f_refrac)
	call gupdate_I ('f_default', f_default)
	call gupdate_I ('f_refl', f_refl)
	call gupdate_I ('f_hunt', f_hunt)
	call gupdate_I ('f_crystal', f_crystal)
	call gupdate_I ('f_phot_cent', f_phot_cent)
	call gupdate_I ('f_roughness', f_roughness)
	call gupdate_I ('f_angle', f_angle)
	call gupdate_I ('npoint', npoint)
	call gupdate_I ('ncol', ncol)

#if 0
	do 10 i = 1, 10
	    call gupdate_v_I ('scr_number', i, scr_number(i))
 10	continue
#else
	call gupdate_I ('scr_number(1)', scr_number(1))
	call gupdate_I ('scr_number(2)', scr_number(2))
	call gupdate_I ('scr_number(3)', scr_number(3))
	call gupdate_I ('scr_number(4)', scr_number(4))
	call gupdate_I ('scr_number(5)', scr_number(5))
	call gupdate_I ('scr_number(6)', scr_number(6))
	call gupdate_I ('scr_number(7)', scr_number(7))
	call gupdate_I ('scr_number(8)', scr_number(8))
	call gupdate_I ('scr_number(9)', scr_number(9))
	call gupdate_I ('scr_number(10)', scr_number(10))
#endif

	call gupdate_I ('n_screen', n_screen)

#if 0
	do 20 i = 1, 10
	    call gupdate_v_I ('i_screen', i, i_screen(i))
	    call gupdate_v_I ('i_abs', i, i_abs(i))
	    call gupdate_v_I ('i_stop', i, i_stop(i))
	    call gupdate_v_I ('i_slit', i, i_slit(i))
	    call gupdate_v_I ('k_slit', i, k_slit(i))
 20	continue
#else
	call gupdate_I ('i_screen(1)', i_screen(1))
	call gupdate_I ('i_screen(2)', i_screen(2))
	call gupdate_I ('i_screen(3)', i_screen(3))
	call gupdate_I ('i_screen(4)', i_screen(4))
	call gupdate_I ('i_screen(5)', i_screen(5))
	call gupdate_I ('i_screen(6)', i_screen(6))
	call gupdate_I ('i_screen(7)', i_screen(7))
	call gupdate_I ('i_screen(8)', i_screen(8))
	call gupdate_I ('i_screen(9)', i_screen(9))
	call gupdate_I ('i_screen(10)', i_screen(10))

	call gupdate_I ('i_abs(1)', i_abs(1))
	call gupdate_I ('i_abs(2)', i_abs(2))
	call gupdate_I ('i_abs(3)', i_abs(3))
	call gupdate_I ('i_abs(4)', i_abs(4))
	call gupdate_I ('i_abs(5)', i_abs(5))
	call gupdate_I ('i_abs(6)', i_abs(6))
	call gupdate_I ('i_abs(7)', i_abs(7))
	call gupdate_I ('i_abs(8)', i_abs(8))
	call gupdate_I ('i_abs(9)', i_abs(9))
	call gupdate_I ('i_abs(10)', i_abs(10))

	call gupdate_I ('i_stop(1)', i_stop(1))
	call gupdate_I ('i_stop(2)', i_stop(2))
	call gupdate_I ('i_stop(3)', i_stop(3))
	call gupdate_I ('i_stop(4)', i_stop(4))
	call gupdate_I ('i_stop(5)', i_stop(5))
	call gupdate_I ('i_stop(6)', i_stop(6))
	call gupdate_I ('i_stop(7)', i_stop(7))
	call gupdate_I ('i_stop(8)', i_stop(8))
	call gupdate_I ('i_stop(9)', i_stop(9))
	call gupdate_I ('i_stop(10)', i_stop(10))

	call gupdate_I ('i_slit(1)', i_slit(1))
	call gupdate_I ('i_slit(2)', i_slit(2))
	call gupdate_I ('i_slit(3)', i_slit(3))
	call gupdate_I ('i_slit(4)', i_slit(4))
	call gupdate_I ('i_slit(5)', i_slit(5))
	call gupdate_I ('i_slit(6)', i_slit(6))
	call gupdate_I ('i_slit(7)', i_slit(7))
	call gupdate_I ('i_slit(8)', i_slit(8))
	call gupdate_I ('i_slit(9)', i_slit(9))
	call gupdate_I ('i_slit(10)', i_slit(10))

	call gupdate_I ('k_slit(1)', k_slit(1))
	call gupdate_I ('k_slit(2)', k_slit(2))
	call gupdate_I ('k_slit(3)', k_slit(3))
	call gupdate_I ('k_slit(4)', k_slit(4))
	call gupdate_I ('k_slit(5)', k_slit(5))
	call gupdate_I ('k_slit(6)', k_slit(6))
	call gupdate_I ('k_slit(7)', k_slit(7))
	call gupdate_I ('k_slit(8)', k_slit(8))
	call gupdate_I ('k_slit(9)', k_slit(9))
	call gupdate_I ('k_slit(10)', k_slit(10))
#endif

	call gupdate_I ('istar1', istar1)
	call gupdate_R ('cil_ang', cil_ang)
	call gupdate_R ('ell_the', ell_the)
	call gupdate_I ('n_plates', n_plates)
	call gupdate_I ('ig_seed', ig_seed)
	call gupdate_I ('mosaic_seed', mosaic_seed)
	call gupdate_R ('alpha', alpha)
	call gupdate_R ('ssour', ssour)
	call gupdate_R ('theta', theta)
	call gupdate_R ('simag', simag)
	call gupdate_R ('rdsour', rdsour)
	call gupdate_R ('rtheta', rtheta)
	call gupdate_R ('off_soux', off_soux)
	call gupdate_R ('off_souy', off_souy)
	call gupdate_R ('off_souz', off_souz)
	call gupdate_R ('alpha_s', alpha_s)
	call gupdate_R ('rlen1', rlen1)
	call gupdate_R ('rlen2', rlen2)
	call gupdate_R ('rmirr', rmirr)
	call gupdate_R ('axmaj', axmaj)
	call gupdate_R ('axmin', axmin)
	call gupdate_R ('cone_a', cone_a)
	call gupdate_R ('r_maj', r_maj)
	call gupdate_R ('r_min', r_min)
	call gupdate_R ('rwidx1', rwidx1)
	call gupdate_R ('rwidx2', rwidx2)
	call gupdate_R ('param', param)
	call gupdate_R ('hunt_h', hunt_h)
	call gupdate_R ('hunt_l', hunt_l)
	call gupdate_R ('blaze', blaze)
	call gupdate_R ('ruling', ruling)
	call gupdate_R ('order', order)
	call gupdate_R ('phot_cent', phot_cent)
	call gupdate_R ('x_rot', x_rot)
	call gupdate_R ('d_spacing', d_spacing)
	call gupdate_R ('a_bragg', a_bragg)
	call gupdate_R ('spread_mos', spread_mos)
	call gupdate_R ('thickness', thickness)
	call gupdate_R ('r_johansson', r_johansson)
	call gupdate_R ('y_rot', y_rot)
	call gupdate_R ('z_rot', z_rot)
	call gupdate_R ('offx', offx)
	call gupdate_R ('offy', offy)
	call gupdate_R ('offz', offz)
	call gupdate_R ('sllen', sllen)
	call gupdate_R ('slwid', slwid)
	call gupdate_R ('sltilt', sltilt)
	call gupdate_R ('cod_len', cod_len)
	call gupdate_R ('cod_wid', cod_wid)
	call gupdate_R ('x_sour', x_sour)
	call gupdate_R ('y_sour', y_sour)
	call gupdate_R ('z_sour', z_sour)
	call gupdate_R ('x_sour_rot', x_sour_rot)
	call gupdate_R ('y_sour_rot', y_sour_rot)
	call gupdate_R ('z_sour_rot', z_sour_rot)
	call gupdate_R ('r_lambda', r_lambda)
	call gupdate_R ('theta_i', theta_i)
	call gupdate_R ('alpha_i', alpha_i)
	call gupdate_R ('t_incidence', t_incidence)
	call gupdate_R ('t_source', t_source)
	call gupdate_R ('t_image', t_image)
	call gupdate_R ('t_reflection', t_reflection)
	call gupdate_A ('file_source', file_source)
	call gupdate_A ('file_rip', file_rip)
	call gupdate_A ('file_refl', file_refl)

	do 30 i = 1, 10
#if HP_F77_BUGGY_NAMELIST
	    call gupdate_v_A ('file_abs', i, filabs(i))
#else
	    call gupdate_v_A ('file_abs', i, file_abs(i))
#endif
 30	continue

	call gupdate_A ('file_mir', file_mir)
	call gupdate_A ('file_rough', file_rough)

	do 40 i = 1, 10
	    call gupdate_v_R ('thick', i, thick(i))
	    call gupdate_v_R ('rx_slit', i, rx_slit(i))
 40	continue

	do 50 i = 1, 5
	    call gupdate_v_R ('d_plate', i, d_plate(i))
 50	continue

	do 60 i = 1, 10
	    call gupdate_v_R ('rz_slit', i, rz_slit(i))
	    call gupdate_v_R ('sl_dis', i, sl_dis(i))
 60	continue

	call gupdate_I ('fzp', fzp)
	call gupdate_R ('holo_r1', holo_r1)
	call gupdate_R ('holo_r2', holo_r2)
	call gupdate_R ('holo_del', holo_del)
	call gupdate_R ('holo_gam', holo_gam)
	call gupdate_R ('holo_w', holo_w)
	call gupdate_R ('holo_rt1', holo_rt1)
	call gupdate_R ('holo_rt2', holo_rt2)
	call gupdate_R ('azim_fan', azim_fan)
	call gupdate_R ('dist_fan', dist_fan)
	call gupdate_R ('coma_fac', coma_fac)
	call gupdate_R ('alfa', alfa)
	call gupdate_R ('gamma', gamma)
	call gupdate_R ('r_ind_obj', r_ind_obj)
	call gupdate_R ('r_ind_ima', r_ind_ima)
	call gupdate_R ('rul_a1', rul_a1)
	call gupdate_R ('rul_a2', rul_a2)
	call gupdate_R ('rul_a3', rul_a3)
	call gupdate_R ('rul_a4', rul_a4)
	call gupdate_I ('f_polsel', f_polsel)
	call gupdate_I ('f_facet', f_facet)
	call gupdate_I ('f_fac_orient', f_fac_orient)
	call gupdate_I ('f_fac_latt', f_fac_latt)
	call gupdate_R ('rfac_lenx', rfac_lenx)
	call gupdate_R ('rfac_leny', rfac_leny)
	call gupdate_R ('rfac_phax', rfac_phax)
	call gupdate_R ('rfac_phay', rfac_phay)
	call gupdate_R ('rfac_delx1', rfac_delx1)
	call gupdate_R ('rfac_delx2', rfac_delx2)
	call gupdate_R ('rfac_dely1', rfac_dely1)
	call gupdate_R ('rfac_dely2', rfac_dely2)
	call gupdate_A ('file_fac', file_fac)
	call gupdate_I ('f_segment', f_segment)
	call gupdate_I ('iseg_xnum', iseg_xnum)
	call gupdate_I ('iseg_ynum', iseg_ynum)
	call gupdate_A ('file_segment', file_segment)
	call gupdate_A ('file_segp', file_segp)
	call gupdate_R ('seg_lenx', seg_lenx)
	call gupdate_R ('seg_leny', seg_leny)
	call gupdate_I ('f_koma', f_koma)
	call gupdate_A ('file_koma', file_koma)
	call gupdate_I ('f_exit_shape', f_exit_shape)
	call gupdate_I ('f_inc_mnor_ang', f_inc_mnor_ang)
	call gupdate_R ('zko_length', zko_length)
	call gupdate_R ('rkoma_cx', rkoma_cx)
	call gupdate_R ('rkoma_cy', rkoma_cy)
	call gupdate_I ('f_koma_ca', f_koma_ca)
	call gupdate_A ('file_koma_ca', file_koma_ca)
	call gupdate_I ('f_koma_bounce', f_koma_bounce)
	call gupdate_R ('x_rip_amp', x_rip_amp)
	call gupdate_R ('x_rip_wav', x_rip_wav)
	call gupdate_R ('x_phase', x_phase)
	call gupdate_R ('y_rip_amp', y_rip_amp)
	call gupdate_R ('y_rip_wav', y_rip_wav)
	call gupdate_R ('y_phase', y_phase)
	call gupdate_I ('n_rip', n_rip)
	call gupdate_R ('rough_x', rough_x)
	call gupdate_R ('rough_y', rough_y)
	call gupdate_I ('oe_number', oe_number)
	call gupdate_I ('idummy', idummy)
	call gupdate_R ('dummy', dummy)
c
	iflag = 0
c
 99	continue
	return
	end
c
c
	subroutine gfsave (fname, iflag)
	implicit integer (a-z)
c++
c gfsave  -- saves a gfile
c
c inputs:
c    fname -- the gfile to write
c
c outputs:
c    iflag: 0 ok
c	   -1 cannot open new file
c	   -2 other errors
c--
	character*(*) fname
c
	parameter (max_params=1000)
	parameter (max_strlen=80)
c
	common /gdata/ ng_total, g_name, g_value
	character*(max_strlen) g_name (max_params)
	character*(max_strlen) g_value (max_params)
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	character*(max_strlen) name, value
c
	call fstrtrim (fname, i1, i2)
	if (i1 .lt. 1) then
	    iflag = -2
	    goto 99
	endif
c
	line_no = 0
	gfname = fname(i1:i2)
	gfnamelen = i2-i1+1
	unit = 11
c
	open (unit=unit,file=fname(i1:i2),status='unknown',iostat=ierr)
	if (ierr .ne. 0) then
	    iflag = -1
	    goto 99
	endif
c
	do 10 i = 1, ng_total
	    name = g_name(i)
	    value = g_value(i)
	    call fstrtrim (g_name(i), i1, i2)
	    call fstrtrim (g_value(i), i3, i4)
	    write (unit, *) g_name(i)(i1:i2)//' = '// g_value(i)(1:i4)
 10	continue
 	close (unit)
c
 99	continue
	return
 1000	format (a)
	end
c
	subroutine gfsave_source (fname, iflag)
c++
c gfsave_source  -- saves a SOURCE gfile
c
c inputs:
c    fname -- the gfile to write
c
c outputs:
c    iflag: 0 ok
c	   -1 cannot open new file
c	   -2 other errors
c--
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#	include 	<namelist.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
     	INCLUDE		'SHADOW$INC:NAMELIST.BLK/LIST'
#endif
c
	character*(*) fname
	integer unit
c
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	call fstrtrim (fname, i1, i2)
	if (i1 .lt. 1) then
	    iflag = -2
	    goto 99
	endif
c
	line_no = 0
	gfname = fname(i1:i2)
	gfnamelen = i2-i1+1
	unit = 11
c
	open (unit=unit,file=fname(i1:i2),status='unknown',iostat=ierr)
	if (ierr .ne. 0) then
	    iflag = -1
	    goto 99
	endif
c
	call gstore_I (unit, 'fdistr', fdistr)
	call gstore_I (unit, 'fgrid', fgrid)
	call gstore_I (unit, 'fsour', fsour)
	call gstore_I (unit, 'fsource_depth', fsource_depth)
	call gstore_I (unit, 'f_coher', f_coher)
	call gstore_I (unit, 'f_color', f_color)
	call gstore_I (unit, 'f_phot', f_phot)
	call gstore_I (unit, 'f_pol', f_pol)
	call gstore_I (unit, 'f_polar', f_polar)
	call gstore_I (unit, 'f_opd', f_opd)
	call gstore_I (unit, 'f_wiggler', f_wiggler)
	call gstore_I (unit, 'f_bound_sour', f_bound_sour)
	call gstore_I (unit, 'f_sr_type', f_sr_type)
	call gstore_I (unit, 'istar1', istar1)
	call gstore_I (unit, 'npoint', npoint)
	call gstore_I (unit, 'ncol', ncol)
	call gstore_I (unit, 'n_circle', n_circle)
	call gstore_I (unit, 'n_color', n_color)
	call gstore_I (unit, 'n_cone', n_cone)
	call gstore_I (unit, 'ido_vx', ido_vx)
	call gstore_I (unit, 'ido_vz', ido_vz)
	call gstore_I (unit, 'ido_x_s', ido_x_s)
	call gstore_I (unit, 'ido_y_s', ido_y_s)
	call gstore_I (unit, 'ido_z_s', ido_z_s)
	call gstore_I (unit, 'ido_xl', ido_xl)
	call gstore_I (unit, 'ido_xn', ido_xn)
	call gstore_I (unit, 'ido_zl', ido_zl)
	call gstore_I (unit, 'ido_zn', ido_zn)
	call gstore_r (unit, 'sigxl1', sigxl1)
	call gstore_r (unit, 'sigxl2', sigxl2)
	call gstore_r (unit, 'sigxl3', sigxl3)
	call gstore_r (unit, 'sigxl4', sigxl4)
	call gstore_r (unit, 'sigxl5', sigxl5)
	call gstore_r (unit, 'sigxl6', sigxl6)
	call gstore_r (unit, 'sigxl7', sigxl7)
	call gstore_r (unit, 'sigxl8', sigxl8)
	call gstore_r (unit, 'sigxl9', sigxl9)
	call gstore_r (unit, 'sigxl10', sigxl10)
	call gstore_r (unit, 'sigzl1', sigzl1)
	call gstore_r (unit, 'sigzl2', sigzl2)
	call gstore_r (unit, 'sigzl3', sigzl3)
	call gstore_r (unit, 'sigzl4', sigzl4)
	call gstore_r (unit, 'sigzl5', sigzl5)
	call gstore_r (unit, 'sigzl6', sigzl6)
	call gstore_r (unit, 'sigzl7', sigzl7)
	call gstore_r (unit, 'sigzl8', sigzl8)
	call gstore_r (unit, 'sigzl9', sigzl9)
	call gstore_r (unit, 'sigzl10', sigzl10)
	call gstore_r (unit, 'conv_fact', conv_fact)
	call gstore_r (unit, 'cone_max', cone_max)
	call gstore_r (unit, 'cone_min', cone_min)
	call gstore_r (unit, 'epsi_dx', epsi_dx)
	call gstore_r (unit, 'epsi_dz', epsi_dz)
	call gstore_r (unit, 'epsi_x', epsi_x)
	call gstore_r (unit, 'epsi_z', epsi_z)
	call gstore_r (unit, 'hdiv1', hdiv1)
	call gstore_r (unit, 'hdiv2', hdiv2)
	call gstore_r (unit, 'ph1', ph1)
	call gstore_r (unit, 'ph2', ph2)
	call gstore_r (unit, 'ph3', ph3)
	call gstore_r (unit, 'ph4', ph4)
	call gstore_r (unit, 'ph5', ph5)
	call gstore_r (unit, 'ph6', ph6)
	call gstore_r (unit, 'ph7', ph7)
	call gstore_r (unit, 'ph8', ph8)
	call gstore_r (unit, 'ph9', ph9)
	call gstore_r (unit, 'ph10', ph10)
	call gstore_r (unit, 'bener', bener)
	call gstore_r (unit, 'pol_angle', pol_angle)
	call gstore_r (unit, 'pol_deg', pol_deg)
	call gstore_r (unit, 'r_aladdin', r_aladdin)
	call gstore_r (unit, 'r_magnet', r_magnet)
	call gstore_r (unit, 'sigdix', sigdix)
	call gstore_r (unit, 'sigdiz', sigdiz)
	call gstore_r (unit, 'sigmax', sigmax)
	call gstore_r (unit, 'sigmay', sigmay)
	call gstore_r (unit, 'sigmaz', sigmaz)
	call gstore_r (unit, 'vdiv1', vdiv1)
	call gstore_r (unit, 'vdiv2', vdiv2)
	call gstore_r (unit, 'wxsou', wxsou)
	call gstore_r (unit, 'wysou', wysou)
	call gstore_r (unit, 'wzsou', wzsou)
	call gstore_r (unit, 'plasma_angle', plasma_angle)
	call gstore_A (unit, 'file_traj', file_traj)
	call gstore_A (unit, 'file_source', file_source)
	call gstore_A (unit, 'file_bound', file_bound)
	call gstore_I (unit, 'oe_number', oe_number)
	call gstore_I (unit, 'idummy', idummy)
	call gstore_r (unit, 'dummy', dummy)
	call gstore_I (unit, 'f_new', f_new)
c
 	close (unit)
	iflag = 0
c
 99	continue
	return
	end
c
c
	subroutine gfsave_oe (fname, iflag)
c++
c gfsave_oe  -- saves a OE gfile
c
c inputs:
c    fname -- the gfile to write
c
c outputs:
c    iflag: 0 ok
c	   -1 cannot open new file
c	   -2 other errors
c--
#if defined(unix) || HAVE_F77_CPP
#	include		<common.blk>
#	include 	<namelist.blk>
#elif defined(vms)
     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
     	INCLUDE		'SHADOW$INC:NAMELIST.BLK/LIST'
#endif
c
	character*(*) fname
	integer unit
c
	common /gfloaddata/ line_no, gfnamelen, gfname
	character*256 gfname
	integer line_no, gfnamelen
c
	call fstrtrim (fname, i1, i2)
	if (i1 .lt. 1) then
	    iflag = -2
	    goto 99
	endif
c
	line_no = 0
	gfname = fname(i1:i2)
	gfnamelen = i2-i1+1
	unit = 11
c
	open (unit=unit,file=fname(i1:i2),status='unknown',iostat=ierr)
	if (ierr .ne. 0) then
	    iflag = -1
	    goto 99
	endif
c
	call gstore_I (unit, 'fmirr', fmirr)
	call gstore_I (unit, 'f_torus', f_torus)
	call gstore_I (unit, 'fcyl', fcyl)
	call gstore_I (unit, 'f_ext', f_ext)
	call gstore_I (unit, 'fstat', fstat)
	call gstore_I (unit, 'f_screen', f_screen)
	call gstore_I (unit, 'f_plate', f_plate)
	call gstore_I (unit, 'fslit', fslit)
	call gstore_I (unit, 'fwrite', fwrite)
	call gstore_I (unit, 'f_ripple', f_ripple)
	call gstore_I (unit, 'f_move', f_move)
	call gstore_I (unit, 'f_thick', f_thick)
	call gstore_I (unit, 'f_bragg_a', f_bragg_a)
	call gstore_I (unit, 'f_g_s', f_g_s)
	call gstore_I (unit, 'f_r_ran', f_r_ran)
	call gstore_I (unit, 'f_grating', f_grating)
	call gstore_I (unit, 'f_mosaic', f_mosaic)
	call gstore_I (unit, 'f_johansson', f_johansson)
	call gstore_I (unit, 'f_side', f_side)
	call gstore_I (unit, 'f_central', f_central)
	call gstore_I (unit, 'f_convex', f_convex)
	call gstore_I (unit, 'f_reflec', f_reflec)
	call gstore_I (unit, 'f_rul_abs', f_rul_abs)
	call gstore_I (unit, 'f_ruling', f_ruling)
	call gstore_I (unit, 'f_pw', f_pw)
	call gstore_I (unit, 'f_pw_c', f_pw_c)
	call gstore_I (unit, 'f_virtual', f_virtual)
	call gstore_I (unit, 'fshape', fshape)
	call gstore_I (unit, 'fhit_c', fhit_c)
	call gstore_I (unit, 'f_mono', f_mono)
	call gstore_I (unit, 'f_refrac', f_refrac)
	call gstore_I (unit, 'f_default', f_default)
	call gstore_I (unit, 'f_refl', f_refl)
	call gstore_I (unit, 'f_hunt', f_hunt)
	call gstore_I (unit, 'f_crystal', f_crystal)
	call gstore_I (unit, 'f_phot_cent', f_phot_cent)
	call gstore_I (unit, 'f_roughness', f_roughness)
	call gstore_I (unit, 'f_angle', f_angle)
	call gstore_I (unit, 'npoint', npoint)
	call gstore_I (unit, 'ncol', ncol)

	do 10 i = 1, 10
	    call gstore_v_I (unit, 'scr_number', i, scr_number(i))
 10	continue

	call gstore_I (unit, 'n_screen', n_screen)

	do 20 i = 1, 10
	    call gstore_v_I (unit, 'i_screen', i, i_screen(i))
 20	continue
	do 30 i = 1, 10
	    call gstore_v_I (unit, 'i_abs', i, i_abs(i))
 30	continue
	do 40 i = 1, 10
	    call gstore_v_I (unit, 'i_stop', i, i_stop(i))
 40	continue
	do 50 i = 1, 10
	    call gstore_v_I (unit, 'i_slit', i, i_slit(i))
 50	continue
	do 60 i = 1, 10
	    call gstore_v_I (unit, 'k_slit', i, k_slit(i))
 60	continue

	call gstore_I (unit, 'istar1', istar1)
	call gstore_R (unit, 'cil_ang', cil_ang)
	call gstore_R (unit, 'ell_the', ell_the)
	call gstore_I (unit, 'n_plates', n_plates)
	call gstore_I (unit, 'ig_seed', ig_seed)
	call gstore_I (unit, 'mosaic_seed', mosaic_seed)
	call gstore_R (unit, 'alpha', alpha)
	call gstore_R (unit, 'ssour', ssour)
	call gstore_R (unit, 'theta', theta)
	call gstore_R (unit, 'simag', simag)
	call gstore_R (unit, 'rdsour', rdsour)
	call gstore_R (unit, 'rtheta', rtheta)
	call gstore_R (unit, 'off_soux', off_soux)
	call gstore_R (unit, 'off_souy', off_souy)
	call gstore_R (unit, 'off_souz', off_souz)
	call gstore_R (unit, 'alpha_s', alpha_s)
	call gstore_R (unit, 'rlen1', rlen1)
	call gstore_R (unit, 'rlen2', rlen2)
	call gstore_R (unit, 'rmirr', rmirr)
	call gstore_R (unit, 'axmaj', axmaj)
	call gstore_R (unit, 'axmin', axmin)
	call gstore_R (unit, 'cone_a', cone_a)
	call gstore_R (unit, 'r_maj', r_maj)
	call gstore_R (unit, 'r_min', r_min)
	call gstore_R (unit, 'rwidx1', rwidx1)
	call gstore_R (unit, 'rwidx2', rwidx2)
	call gstore_R (unit, 'param', param)
	call gstore_R (unit, 'hunt_h', hunt_h)
	call gstore_R (unit, 'hunt_l', hunt_l)
	call gstore_R (unit, 'blaze', blaze)
	call gstore_R (unit, 'ruling', ruling)
	call gstore_R (unit, 'order', order)
	call gstore_R (unit, 'phot_cent', phot_cent)
	call gstore_R (unit, 'x_rot', x_rot)
	call gstore_R (unit, 'd_spacing', d_spacing)
	call gstore_R (unit, 'a_bragg', a_bragg)
	call gstore_R (unit, 'spread_mos', spread_mos)
	call gstore_R (unit, 'thickness', thickness)
	call gstore_R (unit, 'r_johansson', r_johansson)
	call gstore_R (unit, 'y_rot', y_rot)
	call gstore_R (unit, 'z_rot', z_rot)
	call gstore_R (unit, 'offx', offx)
	call gstore_R (unit, 'offy', offy)
	call gstore_R (unit, 'offz', offz)
	call gstore_R (unit, 'sllen', sllen)
	call gstore_R (unit, 'slwid', slwid)
	call gstore_R (unit, 'sltilt', sltilt)
	call gstore_R (unit, 'cod_len', cod_len)
	call gstore_R (unit, 'cod_wid', cod_wid)
	call gstore_R (unit, 'x_sour', x_sour)
	call gstore_R (unit, 'y_sour', y_sour)
	call gstore_R (unit, 'z_sour', z_sour)
	call gstore_R (unit, 'x_sour_rot', x_sour_rot)
	call gstore_R (unit, 'y_sour_rot', y_sour_rot)
	call gstore_R (unit, 'z_sour_rot', z_sour_rot)
	call gstore_R (unit, 'r_lambda', r_lambda)
	call gstore_R (unit, 'theta_i', theta_i)
	call gstore_R (unit, 'alpha_i', alpha_i)
	call gstore_R (unit, 't_incidence', t_incidence)
	call gstore_R (unit, 't_source', t_source)
	call gstore_R (unit, 't_image', t_image)
	call gstore_R (unit, 't_reflection', t_reflection)
	call gstore_A (unit, 'file_source', file_source)
	call gstore_A (unit, 'file_rip', file_rip)
	call gstore_A (unit, 'file_refl', file_refl)

	do 70 i = 1, 10
#if HP_F77_BUGGY_NAMELIST
	    call gstore_v_A (unit, 'file_abs', i, filabs(i))
#else
	    call gstore_v_A (unit, 'file_abs', i, file_abs(i))
#endif
 70	continue

	call gstore_A (unit, 'file_mir', file_mir)
	call gstore_A (unit, 'file_rough', file_rough)

	do 80 i = 1, 10
	    call gstore_v_R (unit, 'thick', i, thick(i))
 80	continue
	do 90 i = 1, 10
	    call gstore_v_R (unit, 'rx_slit', i, rx_slit(i))
 90	continue

	do 100 i = 1, 5
	    call gstore_v_R (unit, 'd_plate', i, d_plate(i))
 100	continue

	do 110 i = 1, 10
	    call gstore_v_R (unit, 'rz_slit', i, rz_slit(i))
 110	continue
	do 120 i = 1, 10
	    call gstore_v_R (unit, 'sl_dis', i, sl_dis(i))
 120	continue

	call gstore_I (unit, 'fzp', fzp)
	call gstore_R (unit, 'holo_r1', holo_r1)
	call gstore_R (unit, 'holo_r2', holo_r2)
	call gstore_R (unit, 'holo_del', holo_del)
	call gstore_R (unit, 'holo_gam', holo_gam)
	call gstore_R (unit, 'holo_w', holo_w)
	call gstore_R (unit, 'holo_rt1', holo_rt1)
	call gstore_R (unit, 'holo_rt2', holo_rt2)
	call gstore_R (unit, 'azim_fan', azim_fan)
	call gstore_R (unit, 'dist_fan', dist_fan)
	call gstore_R (unit, 'coma_fac', coma_fac)
	call gstore_R (unit, 'alfa', alfa)
	call gstore_R (unit, 'gamma', gamma)
	call gstore_R (unit, 'r_ind_obj', r_ind_obj)
	call gstore_R (unit, 'r_ind_ima', r_ind_ima)
	call gstore_R (unit, 'rul_a1', rul_a1)
	call gstore_R (unit, 'rul_a2', rul_a2)
	call gstore_R (unit, 'rul_a3', rul_a3)
	call gstore_R (unit, 'rul_a4', rul_a4)
	call gstore_I (unit, 'f_polsel', f_polsel)
	call gstore_I (unit, 'f_facet', f_facet)
	call gstore_I (unit, 'f_fac_orient', f_fac_orient)
	call gstore_I (unit, 'f_fac_latt', f_fac_latt)
	call gstore_R (unit, 'rfac_lenx', rfac_lenx)
	call gstore_R (unit, 'rfac_leny', rfac_leny)
	call gstore_R (unit, 'rfac_phax', rfac_phax)
	call gstore_R (unit, 'rfac_phay', rfac_phay)
	call gstore_R (unit, 'rfac_delx1', rfac_delx1)
	call gstore_R (unit, 'rfac_delx2', rfac_delx2)
	call gstore_R (unit, 'rfac_dely1', rfac_dely1)
	call gstore_R (unit, 'rfac_dely2', rfac_dely2)
	call gstore_A (unit, 'file_fac', file_fac)
	call gstore_I (unit, 'f_segment', f_segment)
	call gstore_I (unit, 'iseg_xnum', iseg_xnum)
	call gstore_I (unit, 'iseg_ynum', iseg_ynum)
	call gstore_A (unit, 'file_segment', file_segment)
	call gstore_A (unit, 'file_segp', file_segp)
	call gstore_R (unit, 'seg_lenx', seg_lenx)
	call gstore_R (unit, 'seg_leny', seg_leny)
	call gstore_I (unit, 'f_koma', f_koma)
	call gstore_A (unit, 'file_koma', file_koma)
	call gstore_I (unit, 'f_exit_shape', f_exit_shape)
	call gstore_I (unit, 'f_inc_mnor_ang', f_inc_mnor_ang)
	call gstore_R (unit, 'zko_length', zko_length)
	call gstore_R (unit, 'rkoma_cx', rkoma_cx)
	call gstore_R (unit, 'rkoma_cy', rkoma_cy)
	call gstore_I (unit, 'f_koma_ca', f_koma_ca)
	call gstore_A (unit, 'file_koma_ca', file_koma_ca)
	call gstore_I (unit, 'f_koma_bounce', f_koma_bounce)
	call gstore_R (unit, 'x_rip_amp', x_rip_amp)
	call gstore_R (unit, 'x_rip_wav', x_rip_wav)
	call gstore_R (unit, 'x_phase', x_phase)
	call gstore_R (unit, 'y_rip_amp', y_rip_amp)
	call gstore_R (unit, 'y_rip_wav', y_rip_wav)
	call gstore_R (unit, 'y_phase', y_phase)
	call gstore_I (unit, 'n_rip', n_rip)
	call gstore_R (unit, 'rough_x', rough_x)
	call gstore_R (unit, 'rough_y', rough_y)
	call gstore_I (unit, 'oe_number', oe_number)
	call gstore_I (unit, 'idummy', idummy)
	call gstore_R (unit, 'dummy', dummy)
 	close (unit)
	iflag = 0
c
 99	continue
	return
	end
c
c
#ifdef TESTING
c
	subroutine test1
c
	character*256 fname, string
c
	numarg = iargc ()
	if (numarg .eq. 0) then
	    fname = 'test.00.g'
	else
 	    call getarg (1, fname)
	endif
	call fstrtrim(fname, i1, i2)
c
	call gfload_source (fname, iflag)
	if (iflag .ne. 0) then
	    if (iflag .eq. -1) then
		write (*,*) 'test1: Error opening g file "' // 
     $		    fname(i1:i2) // '"'
	    else if (iflag .eq. -2) then
		write (*,*) 'test1: Illegal "name = value" format'
	    else if (iflag .eq. -3) then
		write (*,*) 'test1: Error reading g file "test.g"'
	    endif
	    goto 99
	endif
	call gfsave_source ('junk.00', iflag)
c
 99	continue
	return
	end
c
	subroutine test2
c
	character*256 fname, string
c
	numarg = iargc ()
	if (numarg .eq. 0) then
	    fname = 'test.01.g'
	else
 	    call getarg (1, fname)
	endif
	call fstrtrim(fname, i1, i2)
c
	call gfload_oe (fname, iflag)
	if (iflag .ne. 0) then
	    if (iflag .eq. -1) then
		write (*,*) 'test2: Error opening g file "' // 
     $		    fname(i1:i2) // '"'
	    else if (iflag .eq. -2) then
		write (*,*) 'test2: Illegal "name = value" format'
	    else if (iflag .eq. -3) then
		write (*,*) 'test2: Error reading g file "test.g"'
	    endif
	    goto 99
	endif
	call gfsave_oe ('junk.01', iflag)
c
 99	continue
	return
	end
c
c
	program gtest
	implicit integer (a-z)
c
	call test1
	call test2
c
	stop
	end
#endif
