#! /bin/sh
#
# system-to-excon: Create EXCON experiment file from SHADOW systemfile.
#
# Author: Mumit Khan <khan@xraylith.wisc.edu>
#
# Source: src/utils/excon/system-to-excon
#
# ----------------------------------------------
#	       SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: system-to-excon
# Revision 1.3  1994/01/13  21:20:53  khan
# changed default experiment file name to shadow.exp from excon.exp
#
# Revision 1.2  1993/12/16  17:15:40  khan
# Changed command line to get the SOURCE namelist file from user.
#
# Revision 1.1  1993/12/15  15:35:56  khan
# Initial revision
#
#
#

# environment variables that this shell script sets/changes:
# export VARIABLES.

# error message function
error () {
    echo ""
    echo "`basename $0`:ERROR: $@" 1>&2
    echo ""
    exit 1
}

# usage message function
usage () {
    if [ ! -z "$@" ]; then
	echo "ERROR: $@" 1>&2
    fi
    echo "Usage: `basename $0` [-debug] [-o EXPFILE] -s sourcefile [systemfile]" 1>&2
    exit 1
}

# don't bother to continue if SHADOW environmet isn't set up properly.
if [ -z "${SHADOW_ROOT}" ]; then
	error \
"SHADOW environment is not properly setup for this script
to run.  Set the environment variables via the \`\`.shadowrc'' shell script 
provided with SHADOW distribution."
fi

if [ -z "${SHADOW_BIN}" ]; then
    SHADOW_BIN="${SHADOW_ROOT}/bin"
fi

# Now all the programs that we need to run.
G_TO_NML=${SHADOW_BIN}/g-to-nml
NML_TO_G=${SHADOW_BIN}/nml-to-g

#
# check whether to use -n, \c, or newline-tab for echo
#
if (echo "testing\c"; echo 1,2,3) | grep c >/dev/null; then
  # Stardent Vistra SVR4 grep lacks -e, says ghazi@caip.rutgers.edu.
  if (echo -n testing; echo 1,2,3) | sed s/-n/xn/ | grep xn >/dev/null;
then
    ac_n= ac_c='
' ac_t='        '
  else
    ac_n=-n ac_c= ac_t=
  fi
else
  ac_n= ac_c='\c' ac_t=
fi


#
# set up the variables, including the ones from EXCON.
#
EXPFILE="shadow.exp"		# name of default EXCON experiment file.
SYSTEMFILE="systemfile.dat"	# name of default SYSTEMFILE.
SOURCEFILE=			# no default for SOURCE file. User supplied
RETVAL=0			# return codes from programs.
EXPSEQ=
NUMOE=0
STARTFILES=
GFILES=

# Parse command line args.
while [ $# -gt 0 ]; do
    case "$1" in
	-s)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a SHADOW SOURCE namelist file"
	    fi
	    shift
	    SOURCEFILE="$1"
	    ;;
	-o)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a EXCON experiment file name"
	    fi
	    shift
	    EXPFILE="$1"
	    ;;
	-debug|-d)
	    set -x
	    ;;
	-help|-h)
	    usage ""
	    ;;
	*)
	    # must be the SYSTEM file name
	    SYSTEMFILE="$1"
	    ;;
    esac
    shift
done


# check sanity.

if [ ! -r "${SYSTEMFILE}" ]; then
    usage "SYSTEMFILE \"${SYSTEMFILE}\" not readable or does not exist."
fi

if [ -z "${SOURCEFILE}" ]; then
    usage "Must supply a SOURCE namelist file (eg., start.00)."
elif [ ! -r "${SOURCEFILE}" ]; then
    usage "SOURCEFILE \"${SOURCEFILE}\" not readable or does not exist."
fi

if [ -f "${EXPFILE}" ]; then
    echo $ac_n "Saving existing experiment file ${EXPFILE} to ${EXPFILE}.bak ..." $ac_c
    /bin/cp ${EXPFILE} ${EXPFILE}.bak >/dev/null 2>&1
    if [ $? -ne 0 ]; then
	echo " Failed. (Ignored)"
    else
	echo " Done."
	/bin/rm ${EXPFILE} 2>/dev/null
    fi
fi

touch ${EXPFILE} >/dev/null 2>&1
if [ $? -ne 0 ]; then
    error "Cannot write to experiment file ${EXPFILE}. Check permissions"
fi

# 
# FUN begins. First figure out what the SOURCE and OE's are, then write
# the experiment sequence.
#
# also create the G files corresponding to these START files.
#

# FUNCTION to get the START file names
getstartfiles () {
    while [ $# -gt 0 ]
    do
	while read x
	do
	    echo $x
	done < $1
	shift
    done
}

NUMOE="`wc -l ${SYSTEMFILE} | awk '{print $1}'`"
STARTFILES="${SOURCEFILE} `getstartfiles ${SYSTEMFILE} 2>/dev/null`"

unset getstartfiles

echo "Creating GFILES if necessary ..."

EXPSEQ="source"
set - $STARTFILES

GFILE=$1
head -1 $1 | grep -i tsour > /dev/null 2>&1
if test $? -eq 0; then
    GFILE=${1}.G
    ${NML_TO_G} -o ${GFILE} -t source ${1}

    echo "    ${1} --> $GFILE"
else
    echo "    ${1} --> OK"
fi
GFILES="$GFILE"

#
# NOTE: Does it work for the case when no OEs are defined? (CHECK)
#
shift
i=1
while [ $# -gt 0 ]; do
    EXPSEQ="$EXPSEQ -> trace${i}"
    GFILE=$1
    head -1 $1 | grep -i totoe > /dev/null 2>&1
    if test $? -eq 0; then
	GFILE=${1}.G
	${NML_TO_G} -o ${GFILE} -t oe ${1}
	echo "    ${1} --> $GFILE"
    else 
	echo "    ${1} --> OK"
    fi
    GFILES="$GFILES $GFILE"

    # fix the oe_number in the G files.
    ed $GFILE >/dev/null  << __EOB__0
/oe_number
s/oe_number.*\$/oe_number = ${i}/
w
q
__EOB__0

    i=`expr $i + 1`
    shift
done
unset GFILE


#
# WRITE THE TEMPLATE EXPERIMENT FILE.
#

#
# write the sequence header.
#

cat >> "${EXPFILE}" << __EOB__1
# PROGRAM SEQUENCE 
# 
# if you have only one SOURCE, no need to tack on a number at the end, but
# must do so for the Optical elements, since EXCON has to know how to
# address each one; eg., the following is a valid experiment sequence:
# source -> trace1 -> trace2 -> trace3 -> trace4
#
${EXPSEQ}

__EOB__1


#
# now comes the INPUT section header.
#

cat >> "${EXPFILE}" << __EOB__2

# INPUTS
#
# the following G files were created from the SHADOW MENU definitions. 
# DO NOT MESS WITH THE FOLLOWING SECTION WITH THE G FILES. 
#
__EOB__2

set - $GFILES
echo "source.\$GFILE		= $1"  >> "${EXPFILE}"
shift
i=1
while [ $# -gt 0 ]; do
    echo "trace${i}.\$GFILE		= $1"  >> "${EXPFILE}"
    i=`expr $i + 1`
    shift
done

cat >> "${EXPFILE}" << __EOB__4

#
# you could override ALL the defaults here for all the tools. Any
# parameter not specified here will default to the entries in the G
# files above.
#
# source.paramName	 = value
# trace1.paramName	 = value
# trace2.paramName	 = value
# etc etc etc
#

__EOB__4


#
# now comes the EXPERIMENT section header.
#

cat >> "${EXPFILE}" << __EOB__5


# EXPERIMENT (INCLUSIVE)
# vary source.paramName from X to Y by Z
# vary trace1.paramName from 86.0 to 89.5 by 0.2


__EOB__5

#
# and finally the REPORT section header.
#

cat >> "${EXPFILE}" << __EOB__6

# OUTPUTS (table and graph are reserve words)
# table1.columns = source.paramName, trace1.paramName, trace2.paramName


__EOB__6


echo ""
echo "The EXCON experiment template is in the file \`\`$EXPFILE''".
echo ""

exit 0
