#! /bin/sh
#
# excon-trace: EXCON wrapper for SHADOW TRACE program.
#
# Author: Mumit Khan <khan@xraylith.wisc.edu>
#
# Source: src/utils/excon/excon-trace
#
# ----------------------------------------------
#	       SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: excon-trace
# Revision 1.2  1994/01/08  21:15:23  khan
# Fixed screen filename problem.
#
# Revision 1.1  1993/12/15  15:35:56  khan
# Initial revision
#
#
#

# environment variables that this shell script sets/changes:
# export VARIABLES.

# error message function
error () {
    echo "`basename $0`: $@" 1>&2
    exit 1
}

# usage message function
usage () {
    if [ ! -z "$@" ]; then
	echo "`basename $0`: $@" 1>&2
    fi
    echo "Usage: `basename $0` -g GFILE [-help] [-debug]" 1>&2
    exit 1
}


# don't bother to continue if SHADOW environmet isn't set up properly.
if [ -z "${SHADOW_ROOT}" ]; then
	error \
"SHADOW environment is not properly setup for this script
to run.  Set the environment variables via the \`\`.shadowrc'' shell script 
provided with SHADOW distribution."
fi

if [ -z "${SHADOW_BIN}" ]; then
    SHADOW_BIN=${SHADOW_ROOT}/bin
fi

# Now all the programs that we need to run.
G_TO_NML=${SHADOW_BIN}/g-to-nml
TRACE=${SHADOW_BIN}/trace

#
# set up the variables, including the ones from EXCON.
#
GFILE=				# name of gfile to read from.
OE=0				# what OE are we doing now.
LOGFILE=			# store output of program in this file
APPEND=0			# append or overwrite (default).
RETVAL=0			# return codes from programs.
PREVOE=0			# last OE run for trace.
PREVIMAGE=
STARTFILE=
ENDFILE=

# Parse command line args.
while [ $# -gt 0 ]; do
    case "$1" in
	-g)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a gfile name"
	    fi
	    shift
	    GFILE="$1"
	    ;;
	-logfile|-l)
	    #
  	    # this option creates a log file and redirects the programs'
	    # output to this file. 
	    #
	    # Note: Undocumented in usage() function.
	    #
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a log file name"
	    fi
	    shift
	    LOGFILE="$1"
	    ;;
	-append|-a)
	    #
  	    # this option is only useful when logging is turned on. When
	    # specified, the output of child programs' is appended to the
	    # logfile if it already exists.
	    #
	    # Note: Undocumented in usage() function.
	    #
	    APPEND=1
	    ;;
	-debug|-d)
	    set -x
	    ;;
	-help|-h)
	    usage ""
	    ;;
	*)
	    usage "Illegal command line option $1"
	    ;;
    esac
    shift
done


#
# check sanity
#

# FUNCTION get a parameter value from a gfile.
getgparam() {
    echo `cat "$GFILE" | grep "$1" | awk -F= '{print $2}'`
}

if [ -z "${GFILE}" ]; then
    usage "No GFILE name supplied."
fi

OE=`getgparam oe_number`

if [ ${OE} -lt 1 ]; then
    error "OE number (${OE}) must be > 0. How did you create the G file?"
fi

if [ ${APPEND} -eq 0 ]; then
    rm -f ${LOGFILE} 2>/dev/null
fi

# 
# FUN begins.
#

# make sure the extension of star/mirr files has exactly TWO digits. 
# DAMN SHADOW for these insane stupidities.

FILESUFFIX="`echo ${OE} | awk '{printf "%02d", $1}'`"
IMAGEFILE=star.$FILESUFFIX
MIRRFILE=mirr.$FILESUFFIX
ENDFILE=end.$FILESUFFIX
OPTAXFILE=optax.$FILESUFFIX

#
# SCREEN files are weird. the format is ``screen.xxyy'', where xx = OE
# and yy = screen_number. We simply scan for ALL files of the format
# ``screen.xx*'' and then make appropriate names for them.
#
SCREENFILES=screen.${FILESUFFIX}

# create a link to the EXCON generated STAR and MIRR file name.
EXCON_IMAGEFILE=`getgparam starfile`
EXCON_MIRRFILE=`getgparam mirrfile`
EXCON_ENDFILE=`getgparam endfile`
STARTFILE=`getgparam startfile`
PREVIMAGE=`getgparam file_source`
EXCON_OPTAXFILE=`getgparam optaxfile`
EXCON_SCREENFILES=`getgparam screenfiles`

unset getgparam				# not needed any more after this

#
# now run the programs. Notice that I don't really care if LOGFILE is set
# or not, since Bourne shell will not redirect to file with zero length
# name (which is the case when -logfile option is not specified by user).
#

# create the namelist file from supplied gfile if needed.
if test "$SHADOW_START_FORMAT" = "FORTRAN"; then
    ${G_TO_NML} -o ${STARTFILE} -t oe ${GFILE} 1>&2 >> ${LOGFILE}
else
    STARTFILE=$GFILE
fi

#
# now run TRACE with the created NAMELIST. If ${G_TO_NML} exits with
# an error, don't bother running TRACE.
#

if [ $? -eq 0 ]; then		# why bother going on if already screwed!
    #
    # trace needs different input for the 1st OE.
    #
    if [ ${OE} -eq 1 ]; then
	${TRACE} -m batch 1>&2 >> ${LOGFILE} << __EOB1
0
${STARTFILE}
0
EXIT
__EOB1

    else
    #
    # trace needs the number of the previous OE in this case.
    #
	PREVOE=`expr ${OE} - 1`
	${TRACE} -m batch 1>&2 >> ${LOGFILE} << __EOB2
1
${PREVOE}
${PREVIMAGE}
0
${STARTFILE}
0
EXIT
__EOB2
    fi
fi
RETVAL=$?


if [ -f $IMAGEFILE ]; then
    mv $IMAGEFILE $EXCON_IMAGEFILE
fi

if [ -f $MIRRFILE ]; then
    mv $MIRRFILE $EXCON_MIRRFILE
fi

if [ -f $ENDFILE ]; then
    mv $ENDFILE $EXCON_ENDFILE
fi

#
# don't rename it! simply copy it. TRACE for the next OE will expect
# hard coded filename ``optax.##'', where ## == OE_NUMBER. But for EXCON
# to be able to track these files, and save them, we need to make copies.
#
if [ -f $OPTAXFILE ]; then
    cp $OPTAXFILE $EXCON_OPTAXFILE
fi
#
# special case for the screen files. since there can be more than 1 file,
# we scan for screen.OE#* and then iterate over these.
#
ALLSCRFILES="`ls ${SCREENFILES}[0-9][0-9] 2>/dev/null`"
for i in $ALLSCRFILES; do
    mv $i ${i}.${EXCON_SCREENFILES}
done

exit ${RETVAL}
