#
# shlayout.tcl: Main Tcl script for the SHADOW LAYOUT GUI
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for NanoTechnology
# University of Wisconsin-Madison
# 1415 Engineering Dr., Madison, WI 53706
# ------------------------------------------------
#
# Copyright (c) 2000 Mumit Khan
#

######################### Initialize options ##############################
#
# shlayout:init:env is really executed ONLY once per interpreter. It has
# checks to avoid reinventing the wheel if we get here by "New Window"
# command.
#

proc shlayout:init:env {rwin} {
  global shlayout shadow_library
  if {[info exists shlayout(initialized)]} {
    incr shlayout(instances)
    incr shlayout(total_instances)
    shlayout:vputs "instance # $shlayout(instances)"
    return
  } else {
    set shlayout(instances) 1
    set shlayout(total_instances) 1
    shlayout:vputs "first instance"
  }
  #
  # make all the globals available here.
  #
  eval global [uplevel \#0 info vars]
  # Handle Tcl library environment variables
  if ![info exists env(SHADOW_GUI_ROOT)] {
    shlayout:vputs "Warning: env(SHADOW_GUI_ROOT) does not exist."
    set env(SHADOW_GUI_ROOT) [pwd]
  }

  if {[catch {set env(SHADOW_GUI_ROOT)} shlayout(shadow_gui_root)]} {
    set shlayout(shadow_gui_root) $shlayout(shadow_root)
  }

  if ![info exists shadow_library] {
    shlayout:vputs "Warning: shadow_library variable is not set."
    set shadow_library [pwd]
  }
  set shlayout(shadow_tcl_library) $shadow_library

  set tcl_precision 17

  # Set-up various paths
  global auto_path
  lappend auto_path $shlayout(shadow_tcl_library) 
  if {[catch {file join x y}]} {
    lappend auto_path $shlayout(shadow_tcl_library)/shlayout
  } else {
    lappend auto_path [file join $shlayout(shadow_tcl_library) shlayout]
  }
  lappend auto_path [pwd]
  lappend auto_path [pwd]/../tcl
  # add BLT library to the path.
  global auto_path blt_version
  if {! [info exists blt_version]} {
    set blt_version 2.4
  }
  set bltlib "blt$blt_version"
  lappend auto_path [file join [file dirname [info library]] $bltlib]
  lappend auto_path $blt_library

  # Check for correct Tcl version
  if {[info tclversion] < 8.0} {
    puts stderr "shlayout: Tcl version < 8.0 cannot be used with shlayout"
    exit 1
  }
}

######################### Initialize globals ##############################

# initialization that only happen once per interpreter, like the 
# shlayout:init:env, but only for shlayout specific globals.

proc shlayout:init:_once_globals {} {
  global env shlayout

  if {[info exists shlayout(initialized)]} {return}

  # CHECK/FIXME: Earlier version of SHADOW data reader had a bug that
  # caused it to read ray-by-ray incorrectly (off-by-one error), so
  # we read by cols. However, this method may actually be quicker; of
  # course, given the typically small number of rays when doing the
  # layout, this probably has very little effect, so let's make this
  # the default.
  # cf: shlayout:read_shadow_image
  set shlayout(shdata_read_by_column)	1

  # 
  # Scaling data for exporting to VTK format. Turn it off by default.
  #
  set shlayout(no_scale_3d)		1

  #
  # datasets and files. This implies datasets shared by all instances.
  #
  # Dataset directory.
  set shlayout(directory)		[pwd]
  # Number of mirrors in the system, and the number of rays in the images.
  set shlayout(num_mirrors)		0
  set shlayout(num_rays)		0
  # Set to true after data successfully loaded.
  set shlayout(data_loaded)		0

  # Pathnames of all toplevels, each being an instance of the layout tool. 
  set shlayout(toplevels)		""

  #
  # Colors for various items
  #
  set shlayout(color_good)		blue
  set shlayout(color_lost)		green
  set shlayout(color_optax)		red
  set shlayout(color_mirror)		black

  set shlayout(color_mirror_fill)	yellow

  #
  # Size of the plots
  #
  set shlayout(plot_body_width)		8i
  set shlayout(plot_body_height)	4i

  #
  # Various other parameters.
  #
  set shlayout(enlarge_mirror_size)	0
  set shlayout(enlarge_mirror_pct)	10

  # 
  # Defaults that are copied from instance to a new instance.
  #
  set shlayout(defaults_to_copy) [list num_mirrors num_rays \
                                  show_grid show_footprint show_optax \
				  show_mirror fill_mirror \
				  top_panel_view bottom_panel_view]

  set shlayout(verbose)			0
}

#
# This sets up the instance specific globals that are private to each
# new instance of SHADOW Layout.
#
proc shlayout:init:globals {rwin} {
  global env shlayout

  shlayout:init:_once_globals
  set shlayout($rwin,instance)		$shlayout(total_instances)

  #
  # plot states
  #
  # Number of mirrors and rays to display (and what type of ray). Should
  # be <= to the maximum available in the global datasets.
  set shlayout($rwin,num_mirrors)	0
  set shlayout($rwin,num_rays)		0
  set shlayout($rwin,rays)		all
  # And the particular views.
  set shlayout($rwin,top_panel_view)	"top_view"
  set shlayout($rwin,bottom_panel_view)	"side_view"
  # Display options.
  set shlayout($rwin,show_grid)		1
  set shlayout($rwin,show_footprint)	0
  set shlayout($rwin,show_optax)	1
  set shlayout($rwin,show_mirror)	1
  set shlayout($rwin,show_mirror_command_prefixes)	[list]
  set shlayout($rwin,fill_mirror)	1

  # allowed scaling types are -- auto, cart, fill and user
  # CHECK/NOTUSED
  set shlayout($rwin,scale)		auto
  set shlayout($rwin,user_limits)	""	;# {xmin, xmax, ymin, ymax}

  # interface globals (various window paths typically)
  if {$shlayout(total_instances) == 1} {
    set inst ""
  } else {
    set inst "#$shlayout(total_instances)"
  }
  set shlayout($rwin,win_title)		"SHADOW Layout$inst"
  set shlayout($rwin,plot_master_w)	""	;# parent window of BLT graphs
  set shlayout($rwin,top_panel_graph_w)	""	;# Top panel BLT graph
  set shlayout($rwin,bottom_panel_graph_w) ""	;# Bottom panel BLT graph
  set shlayout($rwin,console_w)	""
  set shlayout($rwin,console_w_stub)	""

  # Dynamic menu entries (File->Print, File->Export).
  set shlayout($rwin,print_cmd_menu)	""
  set shlayout($rwin,export_cmd_menu)	""

  if {[info exists shlayout($rwin,caller)]} {
    set caller_rwin $shlayout($rwin,caller)
    foreach what $shlayout(defaults_to_copy) {
      set shlayout($rwin,$what)	$shlayout($caller_rwin,$what)
    }
  }
}

######################### Initialize preferences, startup etc 

#
# initialize user preferences
#
proc shlayout:init:preferences {rwin} { }

#
# initialize fonts. Should be called *after* init:preferences
# We should really use "tix resetoptions" and change the scheme used by
# Tix, but Tix only has 12 and 14 point font schemes and we need smaller
# ones esp on laptops.
#
proc shlayout:init:fonts {rwin} {
  global shlayout shlayout_prefs
}

#
# initialize the image to be loaded and such. Called after all other
# intialization is done.
#
proc shlayout:init:startup {rwin} {
  global shlayout shlayout_prefs

  # Load the datasets if this is the first instance.
  if {! $shlayout(data_loaded) && $shlayout(num_mirrors) > 0} {
    set dir $shlayout(directory)
    set num_mirrors $shlayout(num_mirrors)
    shlayout:load_optical_system $rwin $dir $num_mirrors
  }
  shlayout:plot_all_views $rwin
}

######################### Utilities #######################################


#----------------------------------------------------------------------#
# Misc front-end Routines 
#----------------------------------------------------------------------#

proc shlayout:cleanup_and_exit {{status 0}} {
  global shlayout
  exit $status
}

# effects - Print usage message and exist
proc shlayout:usage {{status 1}} {
  puts stderr {
Usage: shlayout [-hv] [-n number_of_rays] [-m number_of_mirrors] [-d dir]
 
    -v:         turn on verbose mode (default: off)
    -h:         print this info
    -n:         How many rays to show (defaults to 9)
    -m:         Number of mirrors in the optical system 
    -d:         Directory with the files (default: current working dir)
 
Examples:

    % shlayout                  ;# Start up SHADOW layout GUI
    % shlayout -n 25 -m 2       ;# Start up and show layout for 2 OEs 

  }
  exit $status
}

# verbose mode puts
proc shlayout:vputs {args} {
  global shlayout
  if {[info exists shlayout(verbose)] && $shlayout(verbose)} {
    puts stderr "shlayout: [join $args]"
  }
}

# parse the command line arguments and fill up the shlayout array elements.
proc shlayout:parse_args {rwin} {
  global argv shlayout
  shlayout:vputs "parsing argv: $argv"
  while {[llength $argv] != 0} {
    set arg [lindex $argv 0]
    set argv [lrange $argv 1 end]
    shlayout:vputs "arg = $arg, argv = $argv"

    case $arg in {
      -h* {
	# don't exit with an error status - not an error
	shlayout:usage 0
      }
      -v* {
	set shlayout(verbose) 1
	shlayout:vputs "setting verbose mode to on"
      }
      -n {
	set num_rays [lindex $argv 0]
	set argv [lrange $argv 1 end]
	set shlayout($rwin,num_rays) $num_rays
	shlayout:vputs "Using $num_rays rays"
      }
      -m {
	set num_mirrors [lindex $argv 0]
	set argv [lrange $argv 1 end]
	set shlayout($rwin,num_mirrors) $num_mirrors
	set shlayout(num_mirrors) $num_mirrors
	shlayout:vputs "Using $num_mirrors optical elements"
      }
      -d {
	set directory [lindex $argv 0]
	set argv [lrange $argv 1 end]
	set shlayout(directory) $directory
	shlayout:vputs "Using $directory for all the files"
      }
      * {
	puts stderr "shlayout: bad option \"$arg\"."
	shlayout:usage
      }
    }
  }
}

#
# Apply the cmd to all the current instances. The command issued is
# `$cmd $rwin $args', where the $rwin is injected by this proc.
#
proc shlayout:apply_to_all_toplevels {cmd args} {
  global shlayout
  foreach rwin $shlayout(toplevels) {
    shlayout:vputs "Applying to toplevel $rwin"
    eval $cmd $rwin $args
  }
}

####################### Command callbacks ################################

proc shlayout:close_cmd {rwin} {
  global shlayout
  # release resources. CHECK
  destroy $rwin
}

proc shlayout:exit_cmd {rwin {status 0}} {
  global shlayout
  shlayout:cleanup_and_exit $status
}

#
# Open a new window with the same datasets, and also inherit some of
# view properties from the caller. Note that we always start with 
# showing all the mirrors available in the system, not the number being 
# displayed in the caller. Harmless, but still a bug. CHECK/BUG/FIXME.
#
proc shlayout:new_window_cmd {rwin} {
  global shlayout
  for {set cnt 1} {$cnt <= 50} {incr cnt} {
    set toplevel .shlayout#$cnt
    if {![winfo exists $toplevel]} {
      set shlayout($toplevel,caller) $rwin
      shlayout:main $toplevel $shlayout(directory) \
	$shlayout(num_mirrors) $shlayout($rwin,num_rays)
      return
    }
  }
  error "Too many (50) Shadow Layout windows open. Please close some"
}

#
# this has to happen for ALL the toplevels, so the rwin parameter (for
# specifying the "this" toplevel) is  not needed here.
# really not used.
proc shlayout:show_console_cmd {} {
  global shlayout shlayout_prefs
  foreach rwin $shlayout(toplevels) {
    shlayout:vputs "console for toplevel $rwin"
    catch {pack unpack $shlayout($rwin,console_w)}
    if {$shlayout_prefs(show_console)} {
      pack $shlayout($rwin,console_w) \
	-after $shlayout($rwin,console_w_stub) -fill x 
    }
  }
}

proc shlayout:toggle_grid_cmd {rwin} {
  global shlayout
  catch {busy hold $rwin}
  foreach w {top bottom} {
    $shlayout($rwin,${w}_panel_graph_w) grid configure \
      -hide [expr {! $shlayout($rwin,show_grid)}]
  }
  catch {busy release $rwin}
}

proc shlayout:toggle_footprint_cmd {rwin} {
  global shlayout
  shlayout:refresh_plot_cmd $rwin
}

proc shlayout:toggle_optax_cmd {rwin} {
  global shlayout
  shlayout:refresh_plot_cmd $rwin
}

proc shlayout:toggle_mirror_cmd {rwin} {
  global shlayout
  shlayout:refresh_plot_cmd $rwin
  #
  # Enable/Disable the options such as "Fill Mirror" based on 
  # "Show Mirror" value.
  #
  set state [expr {$shlayout($rwin,show_mirror) ? "normal" : "disabled"}]
  foreach cmd $shlayout($rwin,show_mirror_command_prefixes) {
    eval $cmd -state $state
  }
}

proc shlayout:show_data_cmd {rwin} {
  shlayout:show_tabular_data $rwin
}

#
# shlayout:enlarge_mirror_size_cmd: Optionally expand the mirror size by
# +/- enlarge_mirror_pct % (* 0.5 in each dimension).
#
proc shlayout:enlarge_mirror_size_cmd {rwin} {
  global shlayout

  # These are the global data maps.
  global shlayout_axis
  global shlayout_rays
  global shlayout_minmax
  global shlayout_mirror_faces

  # Since we recompute the mirror faces, we also need to recompute the
  # bounding box. This is usually done during the optical system
  # loading process (see shlayout:load_optical_system).
  shlayout:make_mirror_faces $rwin \
    shlayout_axis shlayout_minmax shlayout_mirror_faces
  shlayout:compute_bb $rwin \
    shlayout_axis shlayout_rays shlayout_minmax shlayout_mirror_faces
  
  # Now replot.
  shlayout:refresh_plot_cmd $rwin
}

####################### Menubar callbacks ################################

##################### File Menubar callbacks ############################

proc shlayout:load_optical_system_cmd {rwin} {
  global shlayout

  if {[shlayout:misc:load_dialog $rwin \
       $shlayout(directory) $shlayout(num_mirrors) \
       $shlayout($rwin,num_rays)] == 0} {
    # Cancelled by user.
    return
  }

  set num_mirrors $shlayout($rwin,tmp,num_mirrors)
  set num_rays $shlayout($rwin,tmp,num_rays)
  set directory $shlayout($rwin,tmp,directory)

  shlayout:load_optical_system $rwin $directory $num_mirrors
  if {$shlayout(data_loaded)} {
    set shlayout($rwin,num_rays) $num_rays
    set shlayout($rwin,num_mirrors) $num_mirrors
  }
  shlayout:apply_to_all_toplevels shlayout:GUI:update_window_title
  shlayout:apply_to_all_toplevels shlayout:refresh_plot_cmd
}

proc shlayout:print_cmd {rwin panel view} {
  global shlayout
  set graph_w $shlayout($rwin,${panel}_panel_graph_w)
  shlayout:postscript_dialog $graph_w ${view}.ps
}

proc shlayout:export_cmd {rwin panel view} {
  global shlayout

  if {$panel != "3d"} {
    dialog .exec {Export Data} \
      "Only 3D data export supported at this time, sorry." \
      "info" 0 Dismiss
    return
  }

  if {$shlayout($rwin,num_mirrors) == 0} {
    dialog .exec {Export Raw Data} \
      "No mirrors in the optical system!" \
      "error" 0 Dismiss
    return
  }
  set types {
    {{Data Files}		{.dat}}
    {{All Files}		{*}}
  }
  set file [tk_getSaveFile \
    -filetypes $types \
    -defaultextension ".dat" \
    -initialfile layout.dat \
    -title "Save raw layout data to file"
  ]
  if {$file != ""} {
    shlayout:export_raw3d $rwin $file
  }
}

proc shlayout:export_vtk_cmd {rwin} {
  global shlayout
  if {$shlayout($rwin,num_mirrors) == 0} {
    dialog .exec {Export 3D VTK Object} \
      "No mirrors in the optical system!" \
      "error" 0 Dismiss
    return
  }
  set types {
    {{VTK Files}		{.vtk}}
    {{All Files}		{*}}
  }
  set file [tk_getSaveFile \
    -filetypes $types \
    -defaultextension ".vtk" \
    -initialfile layout.vtk \
    -title "Save 3D VTK object to file"
  ]
  if {$file != ""} {
    shlayout:export_vtk $rwin $file
  }
}

proc shlayout:exec_tcl_command {w} {
  set cmd [$w get]
  # set_msg "Executing command \"$cmd\" ..."
  if {[catch [list uplevel #0 $cmd] msg]} {
    dialog .exec {Execute Command} \
	"Error executing XMenu(Tcl) command \"$cmd\"\n($msg)" \
	"error" 0 Dismiss
  } else {
    if {$msg != ""} {
      dialog .exec {Execute Command} "$msg" "info" 0 Dismiss
    }
  }
  $w delete 0 end
}

##################### Option Menubar callbacks ############################

proc shlayout:refresh_plot_cmd {rwin} {
  update
  shlayout:plot_all_views $rwin
}

####################### PlotOpts callbacks ###############################

#
# shlayout:choose_panel_view: 
#
# Pick the view that are plotted in either the top or bottom panels, 
# designated by `which' parameter.
#
# FIXME: For now simply replot the entire bit without any optimization
# whatsoever.
#
proc shlayout:choose_panel_view {rwin which view} {
  global shlayout

  set shlayout($rwin,${which}_panel_view) $view
  shlayout:plot_all_views $rwin
}


######################### GUI #######################################

proc shlayout:GUI:topmenu {rwin menu_name} {
  global shlayout shlayout_prefs

  # Unfortunately, these reset options play havoc with the defaults
  # when a new window opens up. Disable for now.
  # catch {tix resetoptions TK TK}

  # make menu bar itself
  menu $menu_name -tearoff 0

  # build widget $menu_name.file
  $menu_name add cascade \
    -menu "$menu_name.file" \
    -label {File} \
    -underline {0}

  # build widget $menu_name.file.m
  menu $menu_name.file \
    -tearoff 0

  $menu_name.file add command \
    -command "shlayout:load_optical_system_cmd $rwin" \
    -label {Load Optical System} \
    -underline {0}

  # build cascade menu for printing the plot views.
  $menu_name.file add cascade \
    -menu "$menu_name.file.print" \
    -label {Print} \
    -underline {-1}

  # build widget $menu_name.file.print
  menu $menu_name.file.print \
    -tearoff 0

  # We add these dynamically as the plots change, so just remember
  # the parent widget name.
  set shlayout($rwin,print_cmd_menu) $menu_name.file.print

  # ============================================== 

  # build cascade menu for exporting the data.
  $menu_name.file add cascade \
    -menu "$menu_name.file.export" \
    -label {Export} \
    -underline {-1}

  # build widget $menu_name.file.export
  menu $menu_name.file.export \
    -tearoff 0

  # We add these dynamically as the plots change, so just remember
  # the parent widget name.
  set shlayout($rwin,export_cmd_menu) $menu_name.file.export

  $menu_name.file add separator

  $menu_name.file add command \
    -command "shlayout:new_window_cmd $rwin" \
    -label {New Window} \
    -underline {0}

  $menu_name.file add command \
    -command "shlayout:close_cmd $rwin" \
    -label {Close Window} \
    -underline {0}

  # only toplevel "." gets the exit command. this makes sure that if
  # we're running under SHADOW GUI, we don't exit the entire GUI when
  # closing the Plot window.
  if ![string compare $rwin "."] {
    $menu_name.file add separator
    $menu_name.file add command \
      -command "shlayout:exit_cmd $rwin" \
      -label {Exit} \
      -underline {1} \
      -accelerator {Ctrl-x}
  }

  # build widget $menu_name.options
  $menu_name add cascade \
    -menu "$menu_name.options" \
    -label {Options} \
    -underline {0}

  # build widget $menu_name.options
  menu $menu_name.options \
    -tearoff 0
  set nitems 0

  $menu_name.options add command \
    -label {Refresh Plots} \
    -command "shlayout:refresh_plot_cmd $rwin" \
    -underline {-1}
  incr nitems

  $menu_name.options add checkbutton \
    -label {Verbose Mode} \
    -variable shlayout(verbose) \
    -underline {-1}
  incr nitems

  global tcl_platform
  if ![string compare $tcl_platform(platform) unix] {
    $menu_name.options add checkbutton \
      -label {Strict Motif} \
      -variable tk_strictMotif \
      -onvalue 1 -offvalue 0 \
      -underline {-1}
    incr nitems
  }

  $menu_name.options add checkbutton \
    -label {Show Grid} \
    -variable shlayout($rwin,show_grid) \
    -command "shlayout:toggle_grid_cmd $rwin" \
    -underline {-1}
  incr nitems

  $menu_name.options add checkbutton \
    -label {Show Footprint} \
    -variable shlayout($rwin,show_footprint) \
    -command "shlayout:toggle_footprint_cmd $rwin" \
    -underline {-1}
  incr nitems

  $menu_name.options add checkbutton \
    -label {Show OptAx} \
    -variable shlayout($rwin,show_optax) \
    -command "shlayout:toggle_optax_cmd $rwin" \
    -underline {-1}
  incr nitems

  $menu_name.options add checkbutton \
    -label {Show Mirror} \
    -variable shlayout($rwin,show_mirror) \
    -command "shlayout:toggle_mirror_cmd $rwin" \
    -underline {-1}
  incr nitems

  $menu_name.options add checkbutton \
    -label {Fill Mirror} \
    -variable shlayout($rwin,fill_mirror) \
    -command "shlayout:toggle_mirror_cmd $rwin" \
    -underline {-1}
  incr nitems

  lappend shlayout($rwin,show_mirror_command_prefixes) \
    "$menu_name.options entryconfigure [expr {$nitems - 1}]"

  $menu_name.options add checkbutton \
    -label "Enlarge Mirror by $shlayout(enlarge_mirror_pct)%" \
    -variable shlayout(enlarge_mirror_size) \
    -command "shlayout:enlarge_mirror_size_cmd $rwin" \
    -underline {-1}
  incr nitems

  lappend shlayout($rwin,show_mirror_command_prefixes) \
    "$menu_name.options entryconfigure [expr {$nitems - 1}]"

  $menu_name.options add command \
    -label {Show Data} \
    -command "shlayout:show_data_cmd $rwin" \
    -underline {-1}

  # FIXME
  if {0} {
  $menu_name.options add command \
    -label {Show Info} \
    -state disabled \
    -command "shlayout:show_info $rwin" \
    -underline {-1}
  }

  $menu_name.options add separator

  # build cascade menu for Preferences
  $menu_name.options add cascade \
    -menu "$menu_name.options.preferences" \
    -label {Preferences} \
    -underline {-1}

  # build cascaded menu for $menu_name.preferences
  menu $menu_name.options.preferences \
    -tearoff 0

  $menu_name.options.preferences add checkbutton \
    -label {Show Console Window} \
    -variable shlayout_prefs(show_console) \
    -command "shlayout:show_console_cmd" \
    -underline {-1}

  $menu_name.options.preferences add separator

  $menu_name.options.preferences add command \
    -label {Reload preferences} \
    -state disabled \
    -command "shlayout:load_prefs_cmd $rwin" \
    -underline {-1}

  $menu_name.options.preferences add command \
    -label {Save preferences} \
    -state disabled \
    -command "shlayout:save_prefs_cmd $rwin" \
    -underline {-1}

  $menu_name.options.preferences add command \
    -label {Reset to defaults} \
    -state disabled \
    -command "shlayout:reset_prefs_cmd $rwin" \
    -underline {-1}

  # build widget $menu_name.help
  $menu_name add cascade \
    -menu "$menu_name.help" \
    -label {Help} \
    -underline {0}

  # build widget $menu_name.help
  menu $menu_name.help \
    -tearoff 0

  $menu_name.help add command \
    -command "shlayout:about_cmd $rwin" \
    -label {About} \
    -state disabled \
    -underline {0}

  $menu_name.help add command \
    -command "shlayout:about_author_cmd $rwin" \
    -label {Author} \
    -state disabled \
    -underline {0}

  $menu_name.help add command \
    -command "shlayout:help_on_topmenu_cmd $rwin" \
    -label {On pulldown menu} \
    -state disabled \
    -underline {3}

  $menu_name.help add command \
    -command "shlayout:help_on_commands_cmd $rwin" \
    -label {On Tcl commands} \
    -state disabled \
    -underline {7}

  focus $menu_name

  #
  # install the global bindings (accelerators).
  #
  bind all	<Control-x>	"shlayout:exit_cmd $rwin"

  # Unfortunately, these reset options play havoc with the defaults
  # when a new window opens up. Disable for now. Note the tix configure
  # command before the GUI is built (GUI:make), which is used instead
  # of the tix resetoptions pair in this proc.
  # catch {tix resetoptions TixGray 14Point}
  return $menu_name
}

proc shlayout:GUI:make_console_window {rwin w} {
  global shlayout
  label $w.label -text "Console:: "
  entry $w.entry -relief sunken
  bind $w.entry <Return> {shlayout:exec_tcl_command %W}
  pack $w.label -side left -anchor w -expand 0 
  pack $w.entry -side left -anchor w -expand 1 -fill x
  return $w
}

proc shlayout:GUI:display_choices {rwin w} {
  global shlayout
  set name [tixOptionName $w]
  option add *$name*TixControl*label.width 7
  option add *$name*TixControl*label.anchor w
  option add *$name*TixControl*entry.width 4

  tixControl $w.num_mirrors -label "Mirrors" -disablecallback true 
  tixControl $w.num_rays -label "Rays" -disablecallback true

  $w.num_mirrors config -variable shlayout($rwin,num_mirrors)
  $w.num_rays config -variable shlayout($rwin,num_rays)

  pack $w.num_mirrors -side top -padx 5 -pady 3 -fill y
  pack $w.num_rays -side top -padx 5 -pady 3 -fill y

  bind [$w.num_mirrors subwidget entry] <Return> "shlayout:plot_all_views $rwin"
  bind [$w.num_rays subwidget entry] <Return> "shlayout:plot_all_views $rwin"

  button $w.apply -text "Apply" -command "shlayout:plot_all_views $rwin"
  pack $w.apply -padx 5 -pady 3 -fill none -side bottom

}

proc shlayout:GUI:view_choices {rwin w} {
  global shlayout
  set name [tixOptionName $w]
  option add *$name*TixOptionMenu*label.anchor w

  set label_names(top)    "Top   "
  set label_names(bottom) "Bottom"
  foreach which {top bottom} {
    tixOptionMenu $w.${which}_panel -label "$label_names($which)" \
      -disablecallback true \
      -command [list shlayout:choose_panel_view $rwin $which] \
      -options {
	label.width 7
        menubutton.width 10
      }

    $w.${which}_panel add command side_view  -label "Side View"
    $w.${which}_panel add command top_view   -label "Top View"
    $w.${which}_panel add command front_view -label "Front View"

    $w.${which}_panel config -value $shlayout($rwin,${which}_panel_view)
    $w.${which}_panel config -disablecallback false

    pack $w.${which}_panel -side top -padx 5 -pady 3 -fill y
  }
}

proc shlayout:GUI:ray_type_choices {rwin w} {
  global shlayout
  set name [tixOptionName $w]
  option add *$name*TixSelect*label.anchor c
  option add *$name*TixSelect*orientation vertical
  option add *$name*TixSelect*labelSide top

  tixSelect $w.rays -label "" -radio true \
    -disablecallback true -command [list shlayout:choose_rays $rwin]

  $w.rays add good -text Good -state disabled
  $w.rays add lost -text Lost -state disabled
  $w.rays add all  -text All  -state normal

  $w.rays config -value $shlayout($rwin,rays)
  $w.rays config -disablecallback false

  pack $w.rays -side left -padx 5 -pady 3 -fill x
}

proc shlayout:GUI:scale_choices {rwin w} {
  global shlayout
  set name [tixOptionName $w]
  option add *$name*TixSelect*label.anchor c
  option add *$name*TixSelect*orientation vertical
  option add *$name*TixSelect*labelSide top

  tixSelect $w.scale -label "" -radio true \
    -disablecallback true -command [list shlayout:choose_scale $rwin]

  $w.scale add auto  -text "Automatic"
  $w.scale add user  -text "User-defined" -state disabled

  $w.scale config -value $shlayout($rwin,scale)
  $w.scale config -disablecallback false

  pack $w.scale -side left -padx 5 -pady 3 -fill x
}

proc shlayout:GUI:view_opt_choices {rwin w} {
  global shlayout

  checkbutton $w.show_grid \
    -text {Show Grid} \
    -variable shlayout($rwin,show_grid) \
    -command "shlayout:toggle_grid_cmd $rwin"

  checkbutton $w.show_footprint \
    -text {Show Footprint} \
    -variable shlayout($rwin,show_footprint) \
    -command "shlayout:toggle_footprint_cmd $rwin" 

  checkbutton $w.show_optax \
    -text {Show OptAx} \
    -variable shlayout($rwin,show_optax) \
    -command "shlayout:toggle_optax_cmd $rwin" 

  checkbutton $w.show_mirr \
    -text {Show Mirror} \
    -variable shlayout($rwin,show_mirror) \
    -command "shlayout:toggle_mirror_cmd $rwin" 

  checkbutton $w.fill_mirr \
    -text {Fill Mirror} \
    -variable shlayout($rwin,fill_mirror) \
    -state [expr {$shlayout($rwin,show_mirror) ? "normal" : "disabled"}] \
    -command "shlayout:toggle_mirror_cmd $rwin" 
  
  lappend shlayout($rwin,show_mirror_command_prefixes) \
    "$w.fill_mirr configure"

  checkbutton $w.enlarge_mirr \
    -text {Enlarge Mirror} \
    -variable shlayout(enlarge_mirror_size) \
    -state [expr {$shlayout($rwin,show_mirror) ? "normal" : "disabled"}] \
    -command "shlayout:enlarge_mirror_size_cmd $rwin" 
  
  lappend shlayout($rwin,show_mirror_command_prefixes) \
    "$w.enlarge_mirr configure"

  # pack $w.show_grid -side top -padx 5 -pady 3 -fill y -anchor nw
  # pack $w.show_footprint -side top -padx 5 -pady 3 -fill y -anchor nw
  # pack $w.show_optax -side top -padx 5 -pady 3 -fill y -anchor nw
  # pack $w.show_mirr -side top -padx 5 -pady 3 -fill y -anchor nw
  # pack $w.fill_mirr -side top -padx 5 -pady 3 -fill y -anchor nw

  #
  # Sometimes the grid is just much easier. For example, here I'm
  # trying to indent the "Fill Mirror" a bit to show that it's
  # dependent on "Show Mirror".
  #
  set row 0
  grid $w.show_grid   -columnspan 2 -col 0 -row $row -sticky nw
  incr row
  grid $w.show_footprint   -columnspan 2 -col 0 -row $row -sticky nw
  incr row
  grid $w.show_optax  -columnspan 2 -col 0 -row $row -sticky nw
  incr row
  grid $w.show_mirr   -columnspan 2 -col 0 -row $row -sticky nw
  incr row
  grid $w.fill_mirr   -columnspan 1 -col 1 -row $row -sticky nw
  incr row
  grid $w.enlarge_mirr  -columnspan 1 -col 1 -row $row -sticky nw
  grid columnconfigure $w 0 -weight 1
  grid columnconfigure $w 1 -weight 2
}

proc shlayout:GUI:plotopts {rwin w} {

  tixLabelFrame $w.display      -label "Display"
  tixLabelFrame $w.views        -label "Plots"
  tixLabelFrame $w.rays         -label "Rays"
  tixLabelFrame $w.view_opt     -label "View Options"
  #tixLabelFrame $w.scale        -label "Scaling"

  shlayout:GUI:display_choices   $rwin [$w.display subwidget frame]
  shlayout:GUI:view_choices      $rwin [$w.views subwidget frame]
  shlayout:GUI:view_opt_choices  $rwin [$w.view_opt subwidget frame]
  shlayout:GUI:ray_type_choices  $rwin [$w.rays subwidget frame]
  #shlayout:GUI:scale_choices    $rwin [$w.scale subwidget frame]

  # The grid stuff just takes too much typing. This is also out of
  # date by now, but I'll leave it here as a reference in case I
  # want to go back to using grid instead.
  # grid $w.display -row 0 -column 0 -columnspan 1 -rowspan 1 -sticky news
  # grid $w.views -row 1 -column 0 -columnspan 1 -rowspan 1 -sticky news
  # grid $w.scale -row 3 -column 0 -columnspan 1 -rowspan 1 -sticky news
  # grid $w.rays -row 2 -column 0 -columnspan 1 -rowspan 1 -sticky news

  pack $w.display -side top -expand yes -fill both
  pack $w.views -side top -expand yes -fill y
  pack $w.view_opt -side top -expand yes -fill both
  pack $w.rays -side top -expand yes -fill both
  #pack $w.scale -side top -expand yes -fill both
}

proc shlayout:GUI:plotbody {rwin w} {
  global shlayout shlayout_prefs
  set shlayout($rwin,plot_master_w) ${w}
}

proc shlayout:GUI:update_window_title {rwin} {
  global shlayout

  if {$shlayout(data_loaded)} {
    set num_mirrors $shlayout(num_mirrors)
    set num_rays $shlayout(num_rays)
    set sub_title "(Loaded with $num_mirrors mirrors and $num_rays rays)"
  } else {
    set sub_title "(No optical system loaded)"
  }
  wm title $rwin "$shlayout($rwin,win_title) $sub_title"
}

proc shlayout:GUI:make {rwin} {
  global shlayout

  wm protocol $rwin WM_SAVE_YOURSELF "shlayout:exit_cmd $rwin"
  #
  # BUG WORKAROUND (FIXME/CHECK):
  # Don't install a wm protocol handler for WM_DELETE_WINDOW if the
  # toplevel is not "." (ie., if it's running under SHADOW GUI) to
  # avoid a bug Tix that causes core-dumps. 
  if {$rwin == "."} {
    wm protocol $rwin WM_DELETE_WINDOW "shlayout:exit_cmd $rwin"
  }

  # The tix resetoptions in GUI:topmenu doesn't work as expected, so
  # we force the scheme here for now. See comments in GUI:topmenu. Tix
  # bug?
  tix configure -scheme TixGray -fontset 14Point

  set console_w_stub [frame ${rwin}.console_w_stub]
  set body           [frame ${rwin}.body]
  set plotopts       [frame ${body}.plotopts -relief sunken -bd 3]
  set plotbody       [frame ${body}.plotbody -relief sunken -bd 3]

  set shlayout($rwin,console_w_stub) $console_w_stub
  set shlayout($rwin,console_w) \
    [shlayout:GUI:make_console_window $rwin [frame ${rwin}.console_w -bd 2]]

  if {[string compare $rwin "."]} {
    set topmenu ${rwin}.topmenu
  } else {
    set topmenu .topmenu
  }
  shlayout:GUI:topmenu $rwin $topmenu
  shlayout:GUI:plotopts $rwin $plotopts
  shlayout:GUI:plotbody $rwin $plotbody

  # The grid stuff just takes too much typing. This is also out of
  # date by now, but I'll leave it here as a reference in case I
  # want to go back to using grid instead.
  # grid $plotopts -row 0 -column 0 -sticky nw
  # grid $plotbody -row 0 -column 1 -sticky news
  # grid columnconfigure $body 0 -weight 0
  # grid columnconfigure $body 1 -weight 1
  # grid rowconfigure $body 0 -weight 1

  pack $plotopts -side left -anchor nw
  pack $plotbody -side left -expand yes -fill both

  ${rwin} configure -menu $topmenu
  pack $console_w_stub -side top -fill x ;# -expand 1
  pack $body -side top -expand yes -fill both

  lappend shlayout(toplevels) $rwin

  shlayout:show_console_cmd
}

##################### SHADOW Layout data handlers ########################

#
# The global "ray" map -- shlayout_rays(location,id,item) ==> value
#   location : 0 ... nmirror+1 (0 is the source image)
#   id : 1 ... N, N = number or rays begin plotted
#   item: one of x,y or z for coordinate or f for status flag.
#

#
# The global "minmax" map -- shlayout_minmax(location,which,item) ==> value
#   location : 0 ... nmirror+1 (0 is the source image, bb for BoundingBox)
#   which : min, max or center
#   item: one of x,y or z for coordinate
#

#
# The Optical Axis map -- shlayout_axis(location,pt) ==> value
#   location : 0 ... nmirror+1 (0 is the source image)
#   pt: 1 ... 24 for the 24 data points per mirror.
#

#
# shlayout:read_optaxis:
#
# Read in the final optaxis file and fill in the global axis map.
#
proc shlayout:read_optaxis {num_mirrors file axis} {
  upvar 1 $axis axis_map

  if {[catch {open $file "r"} fid]} {
    error "Error opening Optical Axis file \"$file\"."
  }

  if {[catch {read $fid} raw_data]} {
    error "Error reading Optical Axis file \"$file\"."
  }
  close $fid
  set expected_len [expr {$num_mirrors * 25}]
  if {[llength $raw_data] != $expected_len} {
    error "Insufficient data in Optical Axis file \"$file\"."
  }

  # FIXME/CHECK: Some f77 implementations use D instead of E in the
  # floating point outputs, and Tcl can't grok that. Since we know that
  # OPTAX file contains *ONLY* numbers, we do the substitution here
  # until we fix the format statements in SHADOW itself. Damn.
  # This may only happen on Solaris when built with g77, but need to
  # double check this point.

  set regspec {([0-9\.]*)[dD]([-+]?[0-9]*)}
  set regsubspec {\1e\2}
  regsub -all $regspec $raw_data $regsubspec raw_data

  #
  # Reformat the data. The optax file contains AXIS_DATA_LEN (=25) data 
  # points for each location -- the mirror index followed by 24 data 
  # points. We skip the index and read the 24 data points for each 
  # location.
  #
  # Note that Tcl is 0-based and we're using 1-based arrays to conform
  # to SHADOW terminology.
  #
  set AXIS_DATA_LEN 25
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    set first [expr {($mirr - 1) * $AXIS_DATA_LEN + 1}]
    set last [expr {$first + ($AXIS_DATA_LEN - 2)}]
    #
    # Instead of using a straight map such as the following:
    #    set axis_map($mirr) [lrange $raw_data $first $last]
    # use an indexed map instead that helps understanding the equations.
    #
    set pt 1
    foreach val [lrange $raw_data $first $last] {
      set axis_map($mirr,$pt) $val
      incr pt
    }
  }

  # Add the num_mirrors entry for optax. Must be done after we know 
  # that the data has been read in correctly.
  set axis_map(num_mirrors) $num_mirrors
  return
}

#
# shlayout:read_shadow_image:
#
# Read in a SHADOW image file and return an array containing
# {x,y,z,flag} 4-tuple for each ray. The `num_rays' entry contains
# the total number of rays read in from the file.
#
proc shlayout:read_shadow_image {file rays minmax} {
  global shlayout
  upvar 1 $rays ray_vec
  upvar 1 $minmax minmax_vec

  if {[catch {shdata create ray_tmp_data -load $file} msg]} {
    error "Error loading SHADOW image file \"$file\": $msg"
  }

  # Number of rays in the image.
  set num_rays [lindex [ray_tmp_data info rays] 0]

  #
  # Fill in the minmax info first -- x,y,z and center. The data 
  # returned `info column ...' looks like the following:
  #    <col#> <#rays> <[x|y|z]> min max center stddev
  #
  ray_tmp_data compute minmax

  set minmax_data_x [ray_tmp_data info column 1]
  set minmax_data_y [ray_tmp_data info column 2]
  set minmax_data_z [ray_tmp_data info column 3]

  set minmax_vec(min,x)      [lindex $minmax_data_x 3]
  set minmax_vec(max,x)      [lindex $minmax_data_x 4]
  set minmax_vec(center,x)   [lindex $minmax_data_x 5]

  set minmax_vec(min,y)      [lindex $minmax_data_y 3]
  set minmax_vec(max,y)      [lindex $minmax_data_y 4]
  set minmax_vec(center,y)   [lindex $minmax_data_y 5]

  set minmax_vec(min,z)      [lindex $minmax_data_z 3]
  set minmax_vec(max,z)      [lindex $minmax_data_z 4]
  set minmax_vec(center,z)   [lindex $minmax_data_z 5]

  if {$shlayout(shdata_read_by_column)} {
    # CHECK/FIXME: Earlier version of SHADOW data reader had a bug that
    # caused it to read ray-by-ray incorrectly (off-by-one error), so
    # we read by cols. However, this method may actually be quicker; of
    # course, given the typically small number of rays when doing the
    # layout, this probably has very little effect, so let's make this
    # the default.

    # Note that the col id is 1-based. For each column of interest, 
    # {x,y,z} and the 10th element which is the good/lost flag, we
    # fill in each reay.

    # Map the column index to its name.
    set col_names(1)	x
    set col_names(2)	y
    set col_names(3)	z
    set col_names(10)	f

    foreach col {1 2 3 10} {
      ray_tmp_data col $col col_tmp_vec
      set col_name $col_names($col)
      # Note that the ray id is 1-based, ie., 1...num_rays, not 0-based.
      # For each ray, we fill in the correct column.
      for {set ray 1} {$ray <= $num_rays} {incr ray} {
	set ray_vec($ray,$col_name) $col_tmp_vec([expr {$ray - 1}])
      }
      col_tmp_vec delete
    }
  } else {
    # Note that the ray id is 1-based, ie., 1...num_rays, not 0-based.
    # For each ray, we get the first 3 elements, {x,y,z}, and the 10th
    # element which is the good/lost flag.
    for {set ray 1} {$ray <= $num_rays} {incr ray} {
      ray_tmp_data ray $ray ray_tmp_vec
      # I used to use a single list for 4-tuple, but now have moved to
      # a fully-enumerated map.
      # set ray_vec($ray) [concat $ray_tmp_vec(0:2) $ray_tmp_vec(9)]
      #
      set ray_vec($ray,x) $ray_tmp_vec(0)
      set ray_vec($ray,y) $ray_tmp_vec(1)
      set ray_vec($ray,z) $ray_tmp_vec(2)
      set ray_vec($ray,f) $ray_tmp_vec(9)
      ray_tmp_vec delete
    }
  }
  shdata delete ray_tmp_data

  # Add the num_rays entry for this image. Must be done after we know 
  # that the data has been read in correctly.
  set ray_vec(num_rays) $num_rays

  return
}

#
# shlayout:read_all_rays:
#
# Read in all the SHADOW images necessary and collect those in a single
# map shlayout_rays. 
#
proc shlayout:read_all_rays {rwin dir num_mirrors axis rays minmax} {
  global shlayout
  upvar 1 $axis axis_map
  upvar 1 $rays ray_map
  upvar 1 $minmax minmax_map

  # Make the filenames to read.
  set optax_filename [file join $dir [format "optax.%2.2d" $num_mirrors]]
  set source_filename [file join $dir begin.dat]
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    set mirror_filename [file join $dir [format "mirr.%2.2d" $mirr]]
    lappend mirror_filenames $mirror_filename
  }
  set image_filename [file join $dir [format "star.%2.2d" $num_mirrors]]

  shlayout:read_optaxis $num_mirrors $optax_filename axis_map
  shlayout:read_shadow_image $source_filename \
    source_rays source_minmax
  lappend ray_map_names source_rays
  lappend minmax_map_names source_minmax
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    set mirror_filename [lindex $mirror_filenames [expr {$mirr - 1}]]
    shlayout:read_shadow_image $mirror_filename \
      mirror${mirr}_rays mirror${mirr}_minmax
    lappend ray_map_names mirror${mirr}_rays
    lappend minmax_map_names mirror${mirr}_minmax
  }
  shlayout:read_shadow_image $image_filename \
    image_rays image_minmax
  lappend ray_map_names image_rays
  lappend minmax_map_names image_minmax

  # 
  # Sanity check the number of rays read in each image.
  #
  set num_rays $source_rays(num_rays)
  for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
    set ray_map_name [lindex $ray_map_names $loc]
    set image_num_rays [set [set ray_map_name](num_rays)]
    if {$num_rays != $image_num_rays} {
      if {$loc == $num_mirrors + 1} {
        set where "final image ($image_filename)"
      } else {
	set mirror_filename [lindex $mirror_filenames [expr {$loc - 1}]]
        set where "mirror $loc image ($mirror_filename)"
      }
      set ref "source ($source_filename)"
      set errmsg "Ray number mismatch: $ref has $num_rays and "
      append errmsg "$where has $image_num_rays instead!"
      error $errmsg
    }
  }

  #
  # Now gather the data in the global "minmax" and "ray" maps, from 
  # source (location 0) to the final image (location $num_mirrors+1).
  #

  for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
    set minmax_map_name [lindex $minmax_map_names $loc]
    # Fill in the minmax info for this location.
    set minmax_map($loc,min,x)    [set [set minmax_map_name](min,x)]
    set minmax_map($loc,max,x)    [set [set minmax_map_name](max,x)]
    set minmax_map($loc,center,x) [set [set minmax_map_name](center,x)]
    set minmax_map($loc,min,y)    [set [set minmax_map_name](min,y)]
    set minmax_map($loc,max,y)    [set [set minmax_map_name](max,y)]
    set minmax_map($loc,center,y) [set [set minmax_map_name](center,y)]
    set minmax_map($loc,min,z)    [set [set minmax_map_name](min,z)]
    set minmax_map($loc,max,z)    [set [set minmax_map_name](max,z)]
    set minmax_map($loc,center,z) [set [set minmax_map_name](center,z)]

    # And now the actual ray coordinates.
    set ray_map_name [lindex $ray_map_names $loc]
    set ray_map(num_rays) $num_rays
    for {set ray 1} {$ray <= $num_rays} {incr ray} {
      set ray_map($loc,$ray,x) [set [set ray_map_name]($ray,x)]
      set ray_map($loc,$ray,y) [set [set ray_map_name]($ray,y)]
      set ray_map($loc,$ray,z) [set [set ray_map_name]($ray,z)]
      set ray_map($loc,$ray,f) [set [set ray_map_name]($ray,f)]
    }
  }

  return
}

#
# shlayout:convert_to_lab_coordinates:
#
# Convert all the rays to real space.
#
# FIXME/CHECK: HANDLE lost/good rays as well.
#
proc shlayout:convert_to_lab_coordinates {rwin axis rays} {
  global shlayout
  upvar 1 $axis axis_map
  upvar 1 $rays ray_map

  shlayout:vputs "convert_to_lab_coordinates: ENTER"

  set num_mirrors $axis_map(num_mirrors)
  set num_rays $ray_map(num_rays)

  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    set versor($mirr,1,1) $axis_map($mirr,10)			;# UVECT
    set versor($mirr,1,2) $axis_map($mirr,11)
    set versor($mirr,1,3) $axis_map($mirr,12)
    set versor($mirr,2,1) $axis_map($mirr,13)			;# VVEC
    set versor($mirr,2,2) $axis_map($mirr,14)
    set versor($mirr,2,3) $axis_map($mirr,15)
    set versor($mirr,3,1) $axis_map($mirr,16)			;# WVEC
    set versor($mirr,3,2) $axis_map($mirr,17)
    set versor($mirr,3,3) $axis_map($mirr,18)
  }

  # The source rays remain the same, so start location from 1st mirror.
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
      set cur_x $ray_map($mirr,$ray,x)
      set cur_y $ray_map($mirr,$ray,y)
      set cur_z $ray_map($mirr,$ray,z)

      set new_x [expr {$axis_map($mirr,4) +
      	$cur_x * $versor($mirr,1,1) +
      	$cur_y * $versor($mirr,2,1) +
      	$cur_z * $versor($mirr,3,1)}]

      set new_y [expr {$axis_map($mirr,5) +
      	$cur_x * $versor($mirr,1,2) +
      	$cur_y * $versor($mirr,2,2) +
      	$cur_z * $versor($mirr,3,2)}]

      set new_z [expr {$axis_map($mirr,6) +
      	$cur_x * $versor($mirr,1,3) +
      	$cur_y * $versor($mirr,2,3) +
      	$cur_z * $versor($mirr,3,3)}]

      set ray_map($mirr,$ray,x) $new_x
      set ray_map($mirr,$ray,y) $new_y
      set ray_map($mirr,$ray,z) $new_z
    }
  }

  # Now handle the last set of rays, from the last mirror to final image.

  set uvec(1) $axis_map($num_mirrors,10)			;# uvec >> X
  set uvec(2) $axis_map($num_mirrors,11)
  set uvec(3) $axis_map($num_mirrors,12)
  set vvec(1) $axis_map($num_mirrors,19)			;# VREF >> Y
  set vvec(2) $axis_map($num_mirrors,20)
  set vvec(3) $axis_map($num_mirrors,21)
  set wvec(1) $axis_map($num_mirrors,22)			;# WREF >> Z
  set wvec(2) $axis_map($num_mirrors,23)
  set wvec(3) $axis_map($num_mirrors,24)

  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    set mirr $num_mirrors
    set loc [expr {$num_mirrors + 1}]

    set cur_x $ray_map($loc,$ray,x)
    set cur_y $ray_map($loc,$ray,y)
    set cur_z $ray_map($loc,$ray,z)

    set new_x [expr {$axis_map($mirr,7) +
      $cur_x * $uvec(1) +
      $cur_y * $vvec(1) +
      $cur_z * $wvec(1)}]

    set new_y [expr {$axis_map($mirr,8) +
      $cur_x * $uvec(2) +
      $cur_y * $vvec(2) +
      $cur_z * $wvec(2)}]

    set new_z [expr {$axis_map($mirr,9) +
      $cur_x * $uvec(3) +
      $cur_y * $vvec(3) +
      $cur_z * $wvec(3)}]

    set ray_map($loc,$ray,x) $new_x
    set ray_map($loc,$ray,y) $new_y
    set ray_map($loc,$ray,z) $new_z
  }

  shlayout:vputs "convert_to_lab_coordinates: LEAVE"
  return
}

#
# shlayout:compute_bb: Compute the bounding box. Stuff the `bb' entry
# of the minmax map on output, with min/max/center for x/y/z items.
#
# The mirror faces *MUST* be built before this is called!
#
proc shlayout:compute_bb {rwin axis rays minmax mirror_faces} {
  global shlayout
  upvar $axis axis_map
  upvar $rays ray_map
  upvar $minmax minmax_map
  upvar $mirror_faces mirror_faces_map

  shlayout:vputs "compute_bb: ENTER"

  set num_mirrors $axis_map(num_mirrors)
  set num_rays $ray_map(num_rays)

  #
  # First the optical axis.
  #

  # The source point.
  set xmin $axis_map(1,1)
  set ymin $axis_map(1,2)
  set zmin $axis_map(1,3)

  foreach {xmax ymax zmax} [list $xmin $ymin $zmin] { }

  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    # Get the optical axis, taking care to pick out the right
    # elements in the axis map (elements 4:6 at the mirror, and
    # 7:10 at the CP; 1-based for SHADOW compatibility).
    foreach offset "4 7" {
      set x_index [expr {0 + $offset}]
      set y_index [expr {1 + $offset}]
      set z_index [expr {2 + $offset}]
      set x $axis_map($mirr,$x_index)
      set y $axis_map($mirr,$y_index)
      set z $axis_map($mirr,$z_index)
      if {$xmin >= $x} { set xmin $x }
      if {$xmax < $x}  { set xmax $x }
      if {$ymin >= $y} { set ymin $y }
      if {$ymax < $y}  { set ymax $y }
      if {$zmin >= $z} { set zmin $z }
      if {$zmax < $z}  { set zmax $z }
    }
  }

  # Now for all the rays.
  set num_rays $ray_map(num_rays)
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
      set x $ray_map($loc,$ray,x)
      set y $ray_map($loc,$ray,y)
      set z $ray_map($loc,$ray,z)
      if {$xmin >= $x} { set xmin $x }
      if {$xmax < $x}  { set xmax $x }
      if {$ymin >= $y} { set ymin $y }
      if {$ymax < $y}  { set ymax $y }
      if {$zmin >= $z} { set zmin $z }
      if {$zmax < $z}  { set zmax $z }
    }
  }

  # Also do the mirror faces, useful when plotting no rays, but just
  # the optical axis and mirrors.

  # Mirror hexahedron vertices.
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    foreach {x0 x1 x2 x3 x4 x5 x6 x7 x8} \
      $mirror_faces_map($mirr,vertices,x) { }
    foreach {y0 y1 y2 y3 y4 y5 y6 y7 y8} \
      $mirror_faces_map($mirr,vertices,y) { }
    foreach {z0 z1 z2 z3 z4 z5 z6 z7 z8} \
      $mirror_faces_map($mirr,vertices,z) { }
    for {set pt 0} {$pt < 8} {incr pt} {
      set x [set x$pt]
      set y [set y$pt]
      set z [set z$pt]
      if {$xmin >= $x} { set xmin $x }
      if {$xmax < $x}  { set xmax $x }
      if {$ymin >= $y} { set ymin $y }
      if {$ymax < $y}  { set ymax $y }
      if {$zmin >= $z} { set zmin $z }
      if {$zmax < $z}  { set zmax $z }
    }
  }

  set minmax_map(bb,min,x)    $xmin
  set minmax_map(bb,max,x)    $xmax
  set minmax_map(bb,center,x) [expr {$xmin + ($xmax - $zmin)/2.0}]
  set minmax_map(bb,min,y)    $ymin
  set minmax_map(bb,max,y)    $ymax
  set minmax_map(bb,center,y) [expr {$ymin + ($ymax - $ymin)/2.0}]
  set minmax_map(bb,min,z)    $zmin
  set minmax_map(bb,max,z)    $zmax
  set minmax_map(bb,center,z) [expr {$zmin + ($zmax - $zmin)/2.0}]

  shlayout:vputs "compute_bb: LEAVE"
}

#
# shlayout:check_files
# 
# Check if the data files are available.
#
proc shlayout:check_files {dir num_mirrors {op readable}} {
  global shlayout

  if {$num_mirrors == 0} { 
    return "" 
  }

  set missing_files [list]
  set files_to_check [list]

  set optax_filename [file join $dir [format "optax.%2.2d" $num_mirrors]]
  lappend files_to_check $optax_filename
  set source_filename [file join $dir begin.dat]
  lappend files_to_check $source_filename
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    set mirror_filename [file join $dir [format "mirr.%2.2d" $mirr]]
    lappend files_to_check $mirror_filename
  }
  set image_filename [file join $dir [format "star.%2.2d" $num_mirrors]]
  lappend files_to_check $image_filename

  foreach f $files_to_check {
    if {! [file $op $f]} {
      lappend missing_files $f
    }
  }
  return $missing_files
}

#
# shlayout:load_optical_system
# 
# Workhorse routine that reads all the files, builds the various arrays
# as well as the mirror faces.
#
# The data read in go into global data blocks; note that the data is 
# shared by all instances, and unlike some of the configuration info,
# is not indexed by the layout tool instance ``rwin''.
#
# Also update the num_mirrors and num_rays entries in shlayout map to
# denote the maximum number of mirrors and rays in the datasets.
#
proc shlayout:load_optical_system {rwin dir num_mirrors} {
  global shlayout

  # These are the global data maps.
  global shlayout_axis
  global shlayout_rays
  global shlayout_minmax
  global shlayout_mirror_faces

  shlayout:vputs "Loading optical system from $dir with $num_mirrors mirrors"

  # Steps:
  #  1. Read in the optical axis so we can do the conversion to laboratory
  #     space. Fill in UVEC, VVEC, and WVEC as well. 
  #  2. Read in the source image first and fill in the 0'th slot. 
  #  3. Read in the final image (STAR.xx), and compute the final set of
  #     rays and fill in the NMIRROR+1 slot.
  #  4. Read in all the mirrors one by one, and fill in the 1...NMIRROR
  #     slots.

  # Don't bomb out if we can't find the files or have error reading,
  # rather just make empty plots by setting num_mirrors to 0.
  if {[catch {shlayout:read_all_rays $rwin $dir $num_mirrors \
              shlayout_axis shlayout_rays shlayout_minmax} msg]} {

    # Invalidate the datasets. The rest of the code should be using these 
    # before accessing (possibly) stale data.
    set shlayout(data_loaded) 0
    set shlayout(num_mirrors) 0
    set shlayout(num_rays) 0

    # As a safety precaution, unset the global datasets as well.
    catch {unset shlayout_axis}
    catch {unset shlayout_rays}
    catch {unset shlayout_minmax}
    catch {unset shlayout_mirror_faces}

    dialog .error {Error reading files} "$msg" "info" 0 Dismiss
  } else {
    shlayout:vputs "Done loading. Now compute the various parameters."

    shlayout:convert_to_lab_coordinates $rwin shlayout_axis shlayout_rays

    # Build the 3D mirror faces in the local coordinate space. This must
    # be done before the bounding box is computed.
    shlayout:make_mirror_faces $rwin \
      shlayout_axis shlayout_minmax shlayout_mirror_faces

    # Compute the bounding box in the lab space. The minmax data computed
    # in shadow_read_image for each location is in the local coordinate
    # space, but the bounding box is in the lab reference frame, computed
    # by shlayout:convert_to_lab_coordinates.
    shlayout:compute_bb $rwin \
      shlayout_axis shlayout_rays shlayout_minmax shlayout_mirror_faces

    shlayout:vputs "Done with computations."

    set shlayout(num_mirrors) $shlayout_axis(num_mirrors)
    set shlayout(num_rays) $shlayout_rays(num_rays)
    set shlayout(data_loaded) 1
  }
}

namespace eval vector { }

proc vector::normalize {v} {
  foreach {v1 v2 v3} [lrange $v 0 end] { }
  set len [expr {sqrt ($v1 * $v1 + $v2 * $v2 + $v3 * $v3)}]
  return [list [expr {$v1/$len}] [expr {$v2/$len}] [expr {$v3/$len}]]
}

proc vector::add {v w} {
  foreach {v1 v2 v3} [lrange $v 0 end] { }
  foreach {w1 w2 w3} [lrange $w 0 end] { }
  return [list \
    [expr {$v1 + $w1}] \
    [expr {$v2 + $w2}] \
    [expr {$v3 + $w3}]]
}

proc vector::cross_product {v w} {
  foreach {v1 v2 v3} [lrange $v 0 end] { }
  foreach {w1 w2 w3} [lrange $w 0 end] { }

  set z1 [expr {$v2 * $w3 - $v3 * $w2}]
  set z2 [expr {$v3 * $w1 - $v1 * $w3}]
  set z3 [expr {$v1 * $w2 - $v2 * $w1}]
  return [list $z1 $z2 $z3]
}

# matrix.tcl - matrix math routines
# Jonas Beskow 2000

# the internal representation of a matrix is an array structured as below:
# m(rows)                    - number of rows
# m(cols)                    - number of columns
# m(0,0) m(0,1) ... m($i,$j) - elements of the matrix
#
# matrices are passed as lists (i.e. array get/set is used to 
# serialize/restore the matrix array)

namespace eval matrix { }

# mult - multiply two matrices

proc matrix::mult {m1l m2l} {
 array set m1 $m1l
 array set m2 $m2l

 if {$m1(cols) != $m2(rows)} {
  error "matrix inner dimensions must agree"
 }
 for {set i 0} {$i<$m1(rows)} {incr i} {
  for {set j 0} {$j<$m2(cols)} {incr j} {
   set iprod 0
   for {set k 0} {$k<$m1(cols)} {incr k} {
    set iprod [expr {$iprod + $m1($i,$k)*$m2($k,$j)}]
   }
   set m($i,$j) $iprod
  }
 }
 set m(rows) $m1(rows)
 set m(cols) $m2(cols)
 array get m
}

# transpose - transpose a matrix

proc matrix::transpose {ml} {
 array set m $ml
 
 for {set i 0} {$i<$m(rows)} {incr i} {
  for {set j 0} {$j<$m(cols)} {incr j} {
   set n($j,$i) $m($i,$j)
  }
 }
 set n(rows) $m(cols)
 set n(cols) $m(rows)
 array get n
}

# eye - generate a unit matrix of a given dimension

proc matrix::eye {dim} {
 set m(rows) $dim
 set m(cols) $dim
 for {set i 0} {$i<$dim} {incr i} {
  for {set j 0} {$j<$dim} {incr j} {
   if {$i==$j} {set m($i,$j) 1.0} {set m($i,$j) 0.0}
  }
 }
 array get m
}

# det - calculate the determinant of a matrix

proc matrix::det {ml} {
 array set m $ml
 
 if {$m(rows)!=$m(cols)} {error "matrix must be square"}
 set n $m(rows)
 set det 0
 for {set i 0} {$i<$n} {incr i} {lappend colperm0 $i}
 
 foreach {colperm alpha} [_permute2 $colperm0] {
  set row 0
  set prod 1
  foreach col $colperm {
   set prod [expr {$prod*$m($row,$col)}]
   incr row
  }
  if {$alpha%2==0} {set sign 1} {set sign -1}
  set det [expr {$det+$sign*$prod}]
 }
 return $det
}

# det2 - calculate the determinant of a matrix (recursive algorithm)
# algorithm:
# Choose a fixed row value i.
# The determinant can be calculated emanating from the ith row.
# |A| = Ai,1 * ai,1 + Ai,2 * ai,2 + Ai,3 * ai,3 + ... Ai,n * ai,n 
# The value of Ai,j = (-1)^(i+j) * (the determinant of the sub-matrix of A, 
#   obtained from A by crossing out the ith row and the jth column)

proc matrix::det2 {ml} {
 array set m $ml
 set self [lindex [info level 0] 0]
 if {$m(rows)!=$m(cols)} {error "matrix must be square"}
 set n $m(rows)
 if {$n==1} {return $m(0,0)}
 set i 0
 set det 0
 for {set j 0} {$j<$n} {incr j} {
  if {($i+$j)%2==0} {set sign 1} {set sign -1}
  set subm [_wipe $ml $i $j]
  set subdet [$self $subm]
  set det [expr {$det + $sign * $subdet * $m($i,$j)}]
 }
 return $det
}

# lset - convert from list to internal representation

proc matrix::lset {list} {
 
 set i 0
 foreach row $list {
  #puts row=$row
  set j 0
  foreach elem $row {
   set m($i,$j) $elem
   incr j
  }
  incr i
 }
 set m(rows) $i
 set m(cols) $j
 array get m
}

# lget - convert from internal to list representation

proc matrix::lget {ml} {
 array set m $ml
 
 set list ""
 for {set i 0} {$i<$m(rows)} {incr i} {
  set row {}
  for {set j 0} {$j<$m(cols)} {incr j} {
   lappend row $m($i,$j)
  }
  #puts row=$row
  lappend list $row
 }
 set list
}

# cset - convert from string to internal representation

proc matrix::cset {s} {
 set list ""
 regsub -all {([[:blank:]])*(\n)+([[:blank:]])*} [string trim $s] \n s
 #puts s=$s
 regsub -all {[[:blank:]]+} $s " " s
 #puts s=$s
 foreach row [split $s \n] {
  lappend list [split $row " "]
 }
 lset $list
}

# lget - convert from internal to string representation

proc matrix::cget {ml} {
 array set m $ml

 set str ""
 for {set i 0} {$i<$m(rows)} {incr i} {
  set row {}
  for {set j 0} {$j<$m(cols)} {incr j} {
   append str $m($i,$j)
   if {$j!=[expr $m(cols)-1]} {append str \t}
  }
  if {$i!=[expr $m(rows)-1]} {append str \n}
 }
 string trim $str
}

#
# utility routines to generate common 4x4 matrices for 3D transformation
#

proc matrix::translate {x y z} {
 lset [list \
 "1 0 0 $x" \
 "0 1 0 $y" \
 "0 0 1 $z" \
 "0 0 0 1"]
}

proc matrix::scale {x y z} {
 lset [list \
   "$x 0 0 0" \
   "0 $y 0 0" \
   "0 0 $z 0" \
   "0 0 0 1"]
}

proc matrix::rotX {a} {
 set ca [expr {cos($a)}]
 set sa [expr {sin($a)}]
 set nsa [expr {-$sa}]
 lset [list \
   "1    0    0    0" \
   "0    $ca  $nsa 0" \
   "0    $sa  $ca  0" \
   "0    0    0    1"]
}

proc matrix::rotY {a} {
 set ca [expr {cos($a)}]
 set sa [expr {sin($a)}]
 set nsa [expr {-$sa}]
 lset [list \
 "$ca  0    $sa  0" \
 "0    1    0    0" \
 "$nsa 0    $ca  0" \
 "0    0    0    1"]
}

proc matrix::rotZ {a} {
 set ca [expr {cos($a)}]
 set sa [expr {sin($a)}]
 set nsa [expr {-$sa}]
 lset [list \
 "$ca  $nsa 0    0" \
 "$sa  $ca  0    0" \
 "0    0    1    0" \
 "0    0    0    1"]
}

# _permute2
# - recursive help function to calculate all possible permutations of a list
# returns a list of the form {perm1 n1 perm2 n2 ... permN nN} where 
# permK is a sub-list containing a permutation and nK is the number of
# exchanges necessary to bring the original sequence to permK

proc matrix::_permute2 {list} {
 set self [lindex [info level 0] 0]
 if {[llength $list]==1} {return [list $list 0]}
 set result ""
 set first [lindex $list 0]
 set tail [lrange $list 1 end]
 foreach {p n} [$self $tail] {
  for {set i 0} {$i<=[llength $p]} {incr i} {
   lappend result [linsert $p $i $first] [expr $i+$n]
  }
 }
 return $result
}

# _wipe - help func to det2

proc matrix::_wipe {ml rr cc} {

 array set m $ml
 set m2(rows) [expr {$m(rows)-1}]
 set m2(cols) [expr {$m(cols)-1}]
 for {set r 0} {$r<$m(rows)} {incr r} {
  if {$r==$rr} continue
  if {$r>$rr} {set r2 [expr {$r-1}]} else {set r2 $r}
  for {set c 0} {$c<$m(cols)} {incr c} {
   if {$c==$cc} continue
   if {$c>$cc} {set c2 [expr {$c-1}]} else {set c2 $c}
   set m2($r2,$c2) $m($r,$c)
  }
 }
 array get m2
}

# 
# shlayout:make_mirror_faces
#
# Create the hexahedron representation of the mirrors in the layout.
# Each mirror is defined by 8 points and 6 faces.
#
proc shlayout:make_mirror_faces {rwin axis minmax mirror_faces} {
  global shlayout
  upvar 1 $axis axis_map
  upvar 1 $minmax minmax_map
  upvar 1 $mirror_faces mirror_faces_map

  set num_mirrors $axis_map(num_mirrors)

  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    set mirror_location \
      [list $axis_map($mirr,4) $axis_map($mirr,5) $axis_map($mirr,6)]
    set vector_normal \
      [list $axis_map($mirr,16) $axis_map($mirr,17) $axis_map($mirr,18)]

    # The mirror "box" at center.
    set xmin $minmax_map($mirr,min,x)
    set ymin $minmax_map($mirr,min,y)
    set zmin $minmax_map($mirr,min,z)
    set xmax $minmax_map($mirr,max,x)
    set ymax $minmax_map($mirr,max,y)
    set zmax $minmax_map($mirr,max,z)


    # Optionally expand the mirror size by something like 10% (ie., 
    # 5% in each dimension). The percentage is controlled by the
    # global parameter shlayout(enlarge_mirror_pct).
    if {$shlayout(enlarge_mirror_size)} {
      set delta [expr {$shlayout(enlarge_mirror_pct) * 0.01}]
      foreach which {x y z} {
	set ${which}delta \
	  [expr {([set ${which}max] - [set ${which}min]) * $delta / 2.0}]
        set ${which}min [expr {[set ${which}min] - [set ${which}delta]}]
        set ${which}max [expr {[set ${which}max] + [set ${which}delta]}]
      }
    }

    # Note that SHADOW defines z=0 to be on the mirror surface, so we have 
    # to shift it down by the thickness; the maximum z is then negative.
    set z_th [expr {$zmin - $zmax}]
    set zmin 0.0
    set zmax $z_th

    # Rotation matrix. Section 5.7, Foley & van Dam.
    #
    #      |                             0 |
    #      | |Y x V|  |V x (Y x V)| |V|  0 |
    # R =  |                             0 |
    #      |    0          0         0   1 |
    #

    # Compute Y X V
    set col1 [vector::cross_product [list 0 1 0] $vector_normal]
    set col2 [vector::cross_product $vector_normal $col1]
    set col3 $vector_normal

    set col1 [vector::normalize $col1]
    set col2 [vector::normalize $col2]
    set col3 [vector::normalize $col3]

    set rotation_matrix [matrix::lset [list \
      [list [lindex $col1 0] [lindex $col2 0] [lindex $col3 0] 0] \
      [list [lindex $col1 1] [lindex $col2 1] [lindex $col3 1] 0] \
      [list [lindex $col1 2] [lindex $col2 2] [lindex $col3 2] 0] \
      [list          0               0                0        1]]]

    # bottom face.
    set m_pt1 [matrix::lset \
      [list [list $xmin] [list $ymin] [list $zmin] [list 1]]]
    set m_pt2 [matrix::lset \
      [list [list $xmax] [list $ymin] [list $zmin] [list 1]]]
    set m_pt3 [matrix::lset \
      [list [list $xmax] [list $ymax] [list $zmin] [list 1]]]
    set m_pt4 [matrix::lset \
      [list [list $xmin] [list $ymax] [list $zmin] [list 1]]]

    # top face.
    set m_pt5 [matrix::lset \
      [list [list $xmin] [list $ymin] [list $zmax] [list 1]]]
    set m_pt6 [matrix::lset \
      [list [list $xmax] [list $ymin] [list $zmax] [list 1]]]
    set m_pt7 [matrix::lset \
      [list [list $xmax] [list $ymax] [list $zmax] [list 1]]]
    set m_pt8 [matrix::lset \
      [list [list $xmin] [list $ymax] [list $zmax] [list 1]]]

    for {set idx 1} {$idx <= 8} {incr idx} {
      # Rotate the point.
      set m_pt$idx [matrix::mult $rotation_matrix [set m_pt$idx]]
      # Get the {x,y,z} 3-tuple from the matrix.
      set pt$idx [lrange [matrix::lget [set m_pt$idx]] 0 2]
      # And translate the point to the mirror origin. 
      set pt$idx [vector::add [set pt$idx] $mirror_location]
    }

    set indexmap(x) 0
    set indexmap(y) 1
    set indexmap(z) 2

    for {set idx 1} {$idx <= 8} {incr idx} {
      set xx$idx [lindex [set pt$idx] $indexmap(x)]
      set yy$idx [lindex [set pt$idx] $indexmap(y)]
      set zz$idx [lindex [set pt$idx] $indexmap(z)]
    }

    set mirror_faces_map($mirr,vertices,x) [list $xx1 $xx2 $xx3 $xx4 \
      $xx5 $xx6 $xx7 $xx8]
    set mirror_faces_map($mirr,vertices,y) [list $yy1 $yy2 $yy3 $yy4 \
      $yy5 $yy6 $yy7 $yy8]
    set mirror_faces_map($mirr,vertices,z) [list $zz1 $zz2 $zz3 $zz4 \
      $zz5 $zz6 $zz7 $zz8]

    set mirror_faces_map($mirr,face1,x) [list $xx1 $xx2 $xx3 $xx4 $xx1]
    set mirror_faces_map($mirr,face1,y) [list $yy1 $yy2 $yy3 $yy4 $yy1]
    set mirror_faces_map($mirr,face1,z) [list $zz1 $zz2 $zz3 $zz4 $zz1]

    set mirror_faces_map($mirr,face2,x) [list $xx5 $xx6 $xx7 $xx8 $xx5]
    set mirror_faces_map($mirr,face2,y) [list $yy5 $yy6 $yy7 $yy8 $yy5]
    set mirror_faces_map($mirr,face2,z) [list $zz5 $zz6 $zz7 $zz8 $zz5]

    set mirror_faces_map($mirr,face3,x) [list $xx1 $xx2 $xx6 $xx5 $xx1]
    set mirror_faces_map($mirr,face3,y) [list $yy1 $yy2 $yy6 $yy5 $yy1]
    set mirror_faces_map($mirr,face3,z) [list $zz1 $zz2 $zz6 $zz5 $zz1]

    set mirror_faces_map($mirr,face4,x) [list $xx2 $xx3 $xx7 $xx6 $xx2]
    set mirror_faces_map($mirr,face4,y) [list $yy2 $yy3 $yy7 $yy6 $yy2]
    set mirror_faces_map($mirr,face4,z) [list $zz2 $zz3 $zz7 $zz6 $zz2]

    set mirror_faces_map($mirr,face5,x) [list $xx3 $xx4 $xx8 $xx7 $xx3]
    set mirror_faces_map($mirr,face5,y) [list $yy3 $yy4 $yy8 $yy7 $yy3]
    set mirror_faces_map($mirr,face5,z) [list $zz3 $zz4 $zz8 $zz7 $zz3]

    set mirror_faces_map($mirr,face6,x) [list $xx4 $xx1 $xx5 $xx8 $xx4]
    set mirror_faces_map($mirr,face6,y) [list $yy4 $yy1 $yy5 $yy8 $yy4]
    set mirror_faces_map($mirr,face6,z) [list $zz4 $zz1 $zz5 $zz8 $zz4]
  }
  return
}
#
# shlayout:make_view:
#
#
proc shlayout:make_view {rwin graph_w horiz vert title num_rays \
    num_mirrors axis rays minmax mirror_faces} {
  global shlayout
  upvar $axis axis_map
  upvar $rays ray_map
  upvar $minmax minmax_map
  upvar $mirror_faces mirror_faces_map

  set g $graph_w
  catch {destroy $g}

  blt::graph $g \
    -width $shlayout(plot_body_width) \
    -height $shlayout(plot_body_height) 

  $g legend configure -hide yes
  $g configure -title "$title"
  $g grid configure -hide [expr {! $shlayout($rwin,show_grid)}]
  $g axis configure x -title "$horiz"
  $g axis configure y -title "$vert"

  # 
  # Only access data if the number of mirrors is > 0. This could happen
  # if either (1) datasets are not loaded, or (2) the user chose 0 mirrors
  # to display.
  #
  if {$num_mirrors > 0} {
    Blt_ZoomStack $g
    Blt_Crosshairs $g
    #
    # Let the user print via the File->Print menu instead.
    #
    # Blt_PrintKey $g

    for {set ray 1} {$ray <= $num_rays} {incr ray} {
      for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
	set x $ray_map($loc,$ray,$horiz)
	set y $ray_map($loc,$ray,$vert)
	lappend xdata "$x"
	lappend ydata "$y"
      }
      $g element create ray$ray -xdata $xdata -ydata $ydata -symbol none \
	-color $shlayout(color_good)
      if {$shlayout($rwin,show_footprint)} {
        $g element configure ray$ray \
	  -symbol circle -linewidth 0 -pixels 1.5 \
	  -scalesymbols 1
      }
      unset xdata
      unset ydata
    }

    # now plot the optical axis in a different color. Each OE's optical
    # axis is given by the first 3 {x,y,z} tuples in the optical axis
    # entry for that OE. We handle the first point specially, and after
    # that take the 2nd and 3rd for each mirror and plot that. 
    # The charmap below is used to pick out the right entries from the optax
    # data. After the optical axis is drawn, we optionally plot out the 
    # mirrors as well.

    set indexmap(x) 1
    set indexmap(y) 2
    set indexmap(z) 3
    set h_index $indexmap($horiz)
    set v_index $indexmap($vert)

    # the first point we handle right away.
    lappend xdata $axis_map(1,$h_index)
    lappend ydata $axis_map(1,$v_index)

    for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
      # Draw the optical axis, taking care to pick out the right
      # elements in the axis map.
      foreach offset "3 6" {
	set h_index1 [expr {$h_index + $offset}]
	set v_index1 [expr {$v_index + $offset}]
	set x $axis_map($mirr,$h_index1)
	set y $axis_map($mirr,$v_index1)
	lappend xdata "$x"
	lappend ydata "$y"
      }
      if {$shlayout($rwin,show_mirror)} {
	set lw [expr {$shlayout($rwin,fill_mirror) ? 1 : 2}]
	for {set idx 1} {$idx <= 6} {incr idx} {
	  $g element create mirror${mirr}_face${idx} \
	    -xdata $mirror_faces_map($mirr,face$idx,$horiz) \
	    -ydata $mirror_faces_map($mirr,face$idx,$vert) \
	    -symbol none \
	    -linewidth $lw \
	    -color $shlayout(color_mirror)
	}
	if {$shlayout($rwin,fill_mirror)} {
	  for {set idx 1} {$idx <= 6} {incr idx} {
	    set coords [list]
	    for {set pt 0} {$pt < 4} {incr pt} {
	      set x [lindex $mirror_faces_map($mirr,face$idx,$horiz) $pt]
	      set y [lindex $mirror_faces_map($mirr,face$idx,$vert) $pt]
	      lappend coords "$x $y"
	    }
	    $g marker create polygon \
	      -name mirror${mirr}_marker${idx} \
	      -coords [join $coords] \
	      -under true \
	      -fill $shlayout(color_mirror_fill)
	  }
	}
      }
    }
    if {$shlayout($rwin,show_optax)} {
      set lw 2
      $g element create optax -xdata $xdata -ydata $ydata -symbol splus \
	-color $shlayout(color_optax) -dashes {2 4 2} -linewidth $lw
    }
    unset xdata
    unset ydata

    $g axis configure x -loose yes
    $g axis configure y -loose yes
  }

  return $g
}

proc shlayout:export_raw3d {rwin file} {
  global shlayout

  # These are the global data maps.
  global shlayout_axis
  global shlayout_rays
  global shlayout_minmax
  global shlayout_mirror_faces

  set num_mirrors $shlayout($rwin,num_mirrors)
  set num_rays    $shlayout($rwin,num_rays)

  # Cap the number of mirrors and rays from datasets. If the datasets
  # are not loaded, both will become 0.
  if {$num_mirrors > $shlayout(num_mirrors)} {
    set num_mirrors $shlayout(num_mirrors)
  }
  if {$num_rays > $shlayout(num_rays)} {
    set num_rays $shlayout(num_rays)
  }

  if {$num_mirrors == 0} {
    # Error reading files, or some such thing. No data present.
    return
  } else {
    # Points due to optical axis =  1 + $num_mirrors * 2
    #                               ^
    #                             SOURCE       
    set npoints_axis [expr {1 + $num_mirrors * 2}]
    # Points due to rays = (  1 + $num_mirrors +   1  ) * $num_rays
    #                         ^                    ^
    #                       SOURCE               IMAGE
    set npoints_rays  [expr {(1 + $num_mirrors + 1) * $num_rays}]
    # Points due to mirrors = $num_mirrors * 8 
    set npoints_mirrors [expr {$num_mirrors * 8}]
    set npoints [expr {$npoints_axis + $npoints_rays + $npoints_mirrors}] 
  }

  if {[catch {open $file "w"} fid]} {
    dialog .error "Error opening file $file" "$fid" "error" 0 Dismiss
    return
  }

  # Format:
  #   Header
  #     # SHADOW Layout version x.y
  #   System 
  #     <num_rays> <num_mirrors>
  #   BoundingBox
  #     <xmin> <ymin> <zmin> <xmax> <ymax> <zmax>
  #   Optical Axis polygons
  #     <np> <x1 y1 z1> <x2 y2 z2> ... <x_np y_np z_np>
  #   Rays
  #     <np> <x1 y1 z1> <x2 y2 z2> ... <x_np y_np z_np>
  #     <np> <x1 y1 z1> <x2 y2 z2> ... <x_np y_np z_np>
  #     ...
  #   Mirror faces -- 6 per mirror -- polygons with 4 points each.
  #     <np> <x1 y1 z1> ... <x4 y4 z4>
  #     <np> <x1 y1 z1> ... <x4 y4 z4>
  #     ...
  #

  puts $fid "# SHADOW Layout version 1.0"
  puts $fid "$num_rays $num_mirrors"

  set xmin $shlayout_minmax(bb,min,x)
  set xmax $shlayout_minmax(bb,max,x)
  set ymin $shlayout_minmax(bb,min,y)
  set ymax $shlayout_minmax(bb,max,y)
  set zmin $shlayout_minmax(bb,min,z)
  set zmax $shlayout_minmax(bb,max,z)

  puts $fid "$xmin $ymin $zmin $xmax $ymax $zmax"

  #
  # Now start printing out the actual points. The first set of points
  # is the optical axis.
  #

  # Optical axis.

  set data "$npoints_axis"

  # Handle the source right away.
  set x $shlayout_axis(1,1)
  set y $shlayout_axis(1,2)
  set z $shlayout_axis(1,3)
  append data " $x $y $z"

  # Now the rest of the axis.
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    # Draw the optical axis, taking care to pick out the right
    # elements in the axis map.
    foreach offset "4 7" {
      set x_index [expr {0 + $offset}]
      set y_index [expr {1 + $offset}]
      set z_index [expr {2 + $offset}]
      set x $shlayout_axis($mirr,$x_index)
      set y $shlayout_axis($mirr,$y_index)
      set z $shlayout_axis($mirr,$z_index)
      append data " $x $y $z"
    }
  }

  puts $fid $data

  # Then the actual rays.
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    set data [expr {(1 + $num_mirrors + 1)}]
    for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
      set x $shlayout_rays($loc,$ray,x)
      set y $shlayout_rays($loc,$ray,y)
      set z $shlayout_rays($loc,$ray,z)
      append data " $x $y $z"
    }
    puts $fid $data
  }

  # Lastly the mirrors faces.

  # And lastly the mirror polygons -- 6 polygons per mirror. Each
  # face has 4 points.

  set faces [list {0 1 2 3} {4 5 6 7} {0 1 5 4}  {1 2 6 5}  \
    {2 3 7 6}  {3 0 4 7}]
  set nfaces [llength $faces]
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    foreach {x0 x1 x2 x3 x4 x5 x6 x7 x8} \
      $shlayout_mirror_faces($mirr,vertices,x) { }
    foreach {y0 y1 y2 y3 y4 y5 y6 y7 y8} \
      $shlayout_mirror_faces($mirr,vertices,y) { }
    foreach {z0 z1 z2 z3 z4 z5 z6 z7 z8} \
      $shlayout_mirror_faces($mirr,vertices,z) { }
    for {set face 0} {$face < $nfaces} {incr face} {
      set data "4"
      set vertices [lindex $faces $face]
      set vlist [list]
      foreach vertex $vertices {
	set x [set x$vertex]
	set y [set y$vertex]
	set z [set z$vertex]
	append data " $x $y $z"
      }
      puts $fid $data
    }
  }

  close $fid
}

proc shlayout:export_vtk {rwin file} {
  global shlayout

  # These are the global data maps.
  global shlayout_axis
  global shlayout_rays
  global shlayout_minmax
  global shlayout_mirror_faces

  set num_mirrors $shlayout($rwin,num_mirrors)
  set num_rays    $shlayout($rwin,num_rays)

  # Cap the number of mirrors and rays from datasets. If the datasets
  # are not loaded, both will become 0.
  if {$num_mirrors > $shlayout(num_mirrors)} {
    set num_mirrors $shlayout(num_mirrors)
  }
  if {$num_rays > $shlayout(num_rays)} {
    set num_rays $shlayout(num_rays)
  }

  if {$num_mirrors == 0} {
    # Error reading files, or some such thing. No data present.
    return
  } else {
    # Points due to optical axis =  1 + $num_mirrors * 2
    #                               ^
    #                             SOURCE       
    set npoints_axis [expr {1 + $num_mirrors * 2}]
    # Points due to rays = (  1 + $num_mirrors +   1  ) * $num_rays
    #                         ^                    ^
    #                       SOURCE               IMAGE
    set npoints_rays  [expr {(1 + $num_mirrors + 1) * $num_rays}]
    # Points due to mirrors = $num_mirrors * 8 
    set npoints_mirrors [expr {$num_mirrors * 8}]
    set npoints [expr {$npoints_axis + $npoints_rays + $npoints_mirrors}] 
  }

  if {[catch {open $file "w"} fid]} {
    dialog .error "Error opening file $file" "$fid" "error" 0 Dismiss
    return
  }

  puts $fid "# vtk DataFile Version 2.1"
  puts $fid "SHADOW 3D Layout viewer ($num_mirrors mirrors, $num_rays rays)"
  puts $fid "ASCII\n"
  puts $fid "DATASET POLYDATA"
  puts $fid "POINTS $npoints float"
 
  # we'll scale the numbers so that the 3D view at least shows up. If
  # we don't scale, the 3D view can be very hard to see due to the
  # very different dimensions in the 3 directions.
  # The bounding is created by shlayout:load_optical_system and is in 
  # the minmax map already.

  set xmin $shlayout_minmax(bb,min,x)
  set xmax $shlayout_minmax(bb,max,x)
  set ymin $shlayout_minmax(bb,min,y)
  set ymax $shlayout_minmax(bb,max,y)
  set zmin $shlayout_minmax(bb,min,z)
  set zmax $shlayout_minmax(bb,max,z)

  set x_w [expr {$xmax - $xmin}]
  set y_w [expr {$ymax - $ymin}]
  set z_w [expr {$zmax - $zmin}]

  # This is the scale factor.
  set max_w $x_w

  if {$max_w < $y_w} { set max_w $y_w }
  if {$max_w < $z_w} { set max_w $z_w }

  shlayout:vputs "widths = $x_w, $y_w and $z_w ; MAX width $max_w"

  #
  # Now start printing out the actual points. The first set of points
  # is the optical axis.
  #

  # Optical axis.

  # Handle the source right away.
  set x $shlayout_axis(1,1)
  set y $shlayout_axis(1,2)
  set z $shlayout_axis(1,3)
  if {! $shlayout(no_scale_3d)} {
    set x [expr {($x - $xmin) / $x_w * $max_w}]
    set y [expr {($y - $ymin) / $y_w * $max_w}]
    set z [expr {($z - $zmin) / $z_w * $max_w}]
  }
  puts $fid "$x $y $z"

  # Now the rest of the axis.
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    # Draw the optical axis, taking care to pick out the right
    # elements in the axis map.
    foreach offset "4 7" {
      set x_index [expr {0 + $offset}]
      set y_index [expr {1 + $offset}]
      set z_index [expr {2 + $offset}]
      set x $shlayout_axis($mirr,$x_index)
      set y $shlayout_axis($mirr,$y_index)
      set z $shlayout_axis($mirr,$z_index)
      if {! $shlayout(no_scale_3d)} {
	set x [expr {($x - $xmin) / $x_w * $max_w}]
	set y [expr {($y - $ymin) / $y_w * $max_w}]
	set z [expr {($z - $zmin) / $z_w * $max_w}]
      }
      puts $fid "$x $y $z"
    }
  }

  # Then the actual rays.
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
      set x $shlayout_rays($loc,$ray,x)
      set y $shlayout_rays($loc,$ray,y)
      set z $shlayout_rays($loc,$ray,z)
      if {! $shlayout(no_scale_3d)} {
	set x [expr {($x - $xmin) / $x_w * $max_w}]
	set y [expr {($y - $ymin) / $y_w * $max_w}]
	set z [expr {($z - $zmin) / $z_w * $max_w}]
      }
      puts $fid "$x $y $z"
    }
  }

  # Lastly the mirrors faces.

  # Mirror hexahedron vertices.
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    foreach {x0 x1 x2 x3 x4 x5 x6 x7 x8} \
      $shlayout_mirror_faces($mirr,vertices,x) { }
    foreach {y0 y1 y2 y3 y4 y5 y6 y7 y8} \
      $shlayout_mirror_faces($mirr,vertices,y) { }
    foreach {z0 z1 z2 z3 z4 z5 z6 z7 z8} \
      $shlayout_mirror_faces($mirr,vertices,z) { }
    for {set pt 0} {$pt < 8} {incr pt} {
      set x [set x$pt]
      set y [set y$pt]
      set z [set z$pt]
      if {! $shlayout(no_scale_3d)} {
	set x [expr {($x - $xmin) / $x_w * $max_w}]
	set y [expr {($y - $ymin) / $y_w * $max_w}]
	set z [expr {($z - $zmin) / $z_w * $max_w}]
      }
      puts $fid "$x $y $z"
    }
  }

  # Now write out the connectivity/topology.

  # Total lines: Number of rays + one for the optax.
  # Total points: 1 + 2*$num_mirrors for optax, and
  # $num_mirrors + 2 for each ray. Add 1 per line for the point
  # count in the polyline.

  set nlines [expr {1 + $num_rays}]
  set line_list_size [expr {(1 + (1 + $num_mirrors * 2)) + \
    ((1 + ($num_mirrors + 2)) * $num_rays)}]
  puts $fid "LINES $nlines $line_list_size"

  # Start with the optical axis.
  # Then the optical axis -- a total of 1 + $num_mirrors * 2 points.
  set offset 0
  set pts [list]

  # The source point.
  lappend pts $offset
  incr offset

  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    lappend pts $offset [expr {$offset + 1}]
    incr offset 2
  }
  puts $fid "[expr {1 + $num_mirrors * 2}] $pts"

  # Then the rays.
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    set pts [list]
    for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
      lappend pts [expr {$offset + $loc}]
    }
    incr offset [expr {$num_mirrors + 2}]
    puts $fid "[expr {$num_mirrors + 2}] $pts"
  }

  # And lastly the mirror polygons -- 6 polygons per mirror. Each
  # face has 4 points, and +1 for the point count, hence the 5.
  puts $fid "POLYGONS [expr {$num_mirrors * 6}] \
    [expr {($num_mirrors * 6) * 5}]"

  set faces [list {0 1 2 3} {4 5 6 7} {0 1 5 4}  {1 2 6 5}  \
    {2 3 7 6}  {3 0 4 7}]
  set nfaces [llength $faces]
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    for {set face 0} {$face < $nfaces} {incr face} {
      set vertices [lindex $faces $face]
      set vlist [list]
      foreach vertex $vertices {
	lappend vlist [expr {$offset + $vertex}]
      }
      puts $fid "4 $vlist"
    }
    incr offset 8
  }

  # Now write out the attributes (no scalars, just color for now).

  puts $fid "POINT_DATA $npoints"
  puts $fid "COLOR_SCALARS colors 4"
  for {set pt 0} {$pt < $npoints_axis} {incr pt} {
    puts $fid "1.0 0.0 0.0 1.0"
  }
  for {set pt 0} {$pt < $npoints_rays} {incr pt} {
    puts $fid "0.0 0.0 0.0 1.0"
  }
  # pick a good color for the mirrors. FIXME: Use preferences???
  set bisque "1.0000 0.8941 0.7686"
  set azure "0.9412 1.0000 1.0000"
  set color $bisque
  set color "[expr 89/255.0] 1.0 1.0"
  set color "1.0 [expr 182/255.0] [expr 162/255.0]"
  for {set pt 0} {$pt < $npoints_mirrors} {incr pt} {
    puts $fid "$color 1.0"
  }

  close $fid
}

proc shlayout:plot_all_views {rwin} {
  global shlayout

  # These are the global data maps.
  global shlayout_axis
  global shlayout_rays
  global shlayout_minmax
  global shlayout_mirror_faces

  catch {blt::busy hold $rwin}

  # The number of mirrors and rays to display in this tool instance.
  set num_mirrors $shlayout($rwin,num_mirrors)
  set num_rays    $shlayout($rwin,num_rays)

  # Cap the number of mirrors and rays from datasets. If the datasets
  # are not loaded, both will become 0.
  if {$num_mirrors > $shlayout(num_mirrors)} {
    set num_mirrors $shlayout(num_mirrors)
  }
  if {$num_rays > $shlayout(num_rays)} {
    set num_rays $shlayout(num_rays)
  }

  # Update the values so that these show up in the user interface.
  set shlayout($rwin,num_mirrors) $num_mirrors
  set shlayout($rwin,num_rays) $num_rays

  set view_opts(top_view,title)   "Top View"
  set view_opts(top_view,horiz)   "y"
  set view_opts(top_view,vert)    "x"

  set view_opts(front_view,title) "Front View"
  set view_opts(front_view,horiz) "x"
  set view_opts(front_view,vert)  "z"

  set view_opts(side_view,title)  "Side View"
  set view_opts(side_view,horiz)  "y"
  set view_opts(side_view,vert)   "z"

  set t_view $shlayout($rwin,top_panel_view)
  set b_view $shlayout($rwin,bottom_panel_view)

  set g_top_panel $shlayout($rwin,plot_master_w).g_top
  shlayout:make_view $rwin $g_top_panel \
    $view_opts($t_view,horiz) $view_opts($t_view,vert) \
    $view_opts($t_view,title) \
    $num_rays $num_mirrors \
    shlayout_axis shlayout_rays shlayout_minmax shlayout_mirror_faces

  set g_bottom_panel $shlayout($rwin,plot_master_w).g_bottom
  shlayout:make_view $rwin $g_bottom_panel \
    $view_opts($b_view,horiz) $view_opts($b_view,vert) \
    $view_opts($b_view,title) \
    $num_rays $num_mirrors \
    shlayout_axis shlayout_rays shlayout_minmax shlayout_mirror_faces

  # The grid stuff just takes too much typing. This is also out of
  # date by now, but I'll leave it here as a reference in case I
  # want to go back to using grid instead.
  # grid $g_top_panel  -row 0 -column 0 -sticky news
  # grid $g_bottom_panel -row 1 -column 0 -sticky news
  # set g_parent $shlayout($rwin,plot_master_w)
  # grid columnconfigure $g_parent 0 -weight 1 

  set shlayout($rwin,top_panel_graph_w) $g_top_panel
  set shlayout($rwin,bottom_panel_graph_w) $g_bottom_panel

  pack $g_top_panel -side top -expand yes -fill both
  pack $g_bottom_panel -side top -expand yes -fill both 

  shlayout:misc:add_plots_to_menu $rwin

  catch {blt::busy release $rwin}
}

proc shlayout:show_tabular_data {rwin} {
  global shlayout tcl_platform

  # These are the global data maps.
  global shlayout_axis
  global shlayout_rays
  global shlayout_minmax
  global shlayout_mirror_faces

  set num_mirrors $shlayout($rwin,num_mirrors)
  set num_rays    $shlayout($rwin,num_rays)

  # Cap the number of mirrors and rays from datasets. If the datasets
  # are not loaded, both will become 0.
  if {$num_mirrors > $shlayout(num_mirrors)} {
    set num_mirrors $shlayout(num_mirrors)
  }
  if {$num_rays > $shlayout(num_rays)} {
    set num_rays $shlayout(num_rays)
  }

  if {$num_mirrors == 0} {
    # Error reading files, or some such thing. No data present.
    return
  }

  if {[catch {package require Tktable}]} {
    tk_dialog .shview3d {Package error} \
      "Tktable package is required for tabular data." \
      "error" 0 Dismiss
    return
  }

  if {$rwin == "."} { set w .tabular_data } { set w ${rwin}.tabular_data }
  set title "Tabular data"
  catch {destroy $w}
  toplevel $w
  wm title $w $title
  wm iconname $w $title

  set num_points_per_ray [expr {$num_mirrors + 2}]
  set num_points_per_optax [expr {1 + 2 * $num_mirrors}]

  # Table header first.
  set table $w.tab
  set info $w.info

  # The array displayed in the table.
  global g_tabdata${rwin}
  catch {unset g_tabdata${rwin}}

  set g_tabdata${rwin}(1,0) OptAxis
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    set g_tabdata${rwin}([expr {$ray + 1}],0) "Ray $ray"
  }
  set g_tabdata${rwin}(0,1) "NP"
  for {set pt 1} {$pt <= $num_points_per_optax} {incr pt} {
    set g_tabdata${rwin}(0,[expr {$pt + 1}]) "Point $pt"
  }

  # Handle the source right away.
  set x $shlayout_axis(1,1)
  set y $shlayout_axis(1,2)
  set z $shlayout_axis(1,3)

  set row 1
  set col 1
  set g_tabdata${rwin}($row,$col) $num_points_per_optax
  incr col
  set g_tabdata${rwin}($row,$col) [list $x $y $z]
  incr col

  # Now the rest of the axis.
  for {set mirr 1} {$mirr <= $num_mirrors} {incr mirr} {
    # Draw the optical axis, taking care to pick out the right
    # elements in the axis map.
    foreach offset "4 7" {
      set x_index [expr {0 + $offset}]
      set y_index [expr {1 + $offset}]
      set z_index [expr {2 + $offset}]
      set x $shlayout_axis($mirr,$x_index)
      set y $shlayout_axis($mirr,$y_index)
      set z $shlayout_axis($mirr,$z_index)
      set g_tabdata${rwin}($row,$col) [list $x $y $z]
      incr col
    }
  }

  # Then the actual rays.
  for {set ray 1} {$ray <= $num_rays} {incr ray} {
    incr row
    set col 1
    set g_tabdata${rwin}($row,$col) $num_points_per_ray
    incr col
    for {set loc 0} {$loc <= $num_mirrors + 1} {incr loc} {
      set x $shlayout_rays($loc,$ray,x)
      set y $shlayout_rays($loc,$ray,y)
      set z $shlayout_rays($loc,$ray,z)
      set g_tabdata${rwin}($row,$col) [list $x $y $z]
      incr col
    }
  }

  frame $info -relief sunken -bd 2
  label $info.active_label -text "Value: "
  entry $info.active_value -textvariable g_tabdata${rwin}(active) \
    -state disabled
  pack $info.active_label -side left 
  pack $info.active_value -side left -expand yes -fill x

  table $table \
    -variable g_tabdata${rwin} \
    -titlerows 1 -titlecols 1 \
    -cols [expr {$num_points_per_optax + 2}] \
    -rows [expr {$num_rays + 1 + 1}] \
    -state disabled \
    -anchor nw

  pack $info -side top -fill x -expand yes -padx 5 -pady 5
  pack $table -side top -fill both -expand yes -padx 5 -pady 5

  # Withdraw the window, then update all the geometry information
  # so we know how big it wants to be, then center the window in the
  # display and de-iconify it.

  wm withdraw $w
  update idletasks
  set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	  - [winfo vrootx [winfo parent $w]]]
  set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	  - [winfo vrooty [winfo parent $w]]]
  wm geom $w +$x+$y
  update idle
  wm deiconify $w

  raise $w
  focus $w
}

################################ Misc #####################################

# 
# This is a helper dialog called by the Layout command to get the right
# parameters.
#
proc shlayout:misc:load_dialog {rwin dir num_mirrors num_rays} {
  global shlayout

  global g_dialog_button
  set g_dialog_button 0

  set w .layout_dialog
  catch {destroy $w}
  toplevel $w -class Dialog

  set title "Load Optical System"

  wm title $w $title
  wm iconname $w "Dialog"
  wm transient $w [winfo toplevel [winfo parent $w]]

  set text {
  The Load Optical System command loads a new dataset from the disk and
  displays the layout. You need to specify the following parameters:

  (1) Which directory the data is to be loaded from,
  (2) How many mirrors in the layout, and
  (3) How many rays do you want displayed initially (choose a small number).

  Given the number of mirrors, the Layout tool will load the appropriate
  data files (source image, optax, mirror images and the final image) and
  display the layout with all the mirrors in the system and the number of
  rays you specify here. You can always change the number of rays to display
  using the display options.
  }

  frame ${w}.label
  label ${w}.label.title -text "Load Optical System"
  pack ${w}.label.title -anchor center -padx 5 -pady 5

  frame ${w}.msg -relief sunken -bd 1
  label ${w}.msg.text -justify left -text "$text"
  pack ${w}.msg.text -anchor center -padx 5 -pady 5

  frame ${w}.values -relief sunken

  set label_width [string length "Number of Mirrors in the optical system"]

  tixControl $w.values.num_mirrors \
    -label "Number of Mirrors in the optical system" \
    -disablecallback true \
    -options "
      label.width $label_width
      label.anchor e
      entry.width 4
    "
  tixFileEntry $w.values.directory \
    -label "Select the directory to load data from" \
    -disablecallback true \
    -options "
      label.width $label_width
      label.anchor e
    "
  tixControl $w.values.num_rays \
    -label "Number of Rays to display in the layout" \
    -disablecallback true \
    -options "
      label.width $label_width
      label.anchor e
      entry.width 4
    "
  
  set shlayout($rwin,tmp,num_mirrors) $num_mirrors
  set shlayout($rwin,tmp,num_rays) $num_rays
  set shlayout($rwin,tmp,directory) $dir

  $w.values.num_mirrors config -variable shlayout($rwin,tmp,num_mirrors)
  $w.values.directory config -variable shlayout($rwin,tmp,directory)
  $w.values.num_rays config -variable shlayout($rwin,tmp,num_rays)
  
  pack $w.values.num_mirrors -side top -fill x -expand yes
  pack $w.values.directory -side top -fill x -expand yes
  pack $w.values.num_rays -side top -fill x -expand yes

  frame ${w}.border -bd 1 -relief groove

  frame ${w}.confirm
  button ${w}.confirm.accept -text "Accept" \
      -command "set g_dialog_button 1"
  bind ${w}.confirm.accept <Return>	{set g_dialog_button 1}
  button ${w}.confirm.cancel -text "Cancel" \
      -command "set g_dialog_button 0"
  bind ${w}.confirm.cancel <Return>	{set g_dialog_button 0}
  pack ${w}.confirm.accept -side left -padx 25 -pady 5
  pack ${w}.confirm.cancel -side right -padx 25 -pady 5

  pack ${w}.label -side top -padx 5 -pady 5 -expand yes -fill x
  pack ${w}.msg -side top -padx 5 -pady 5 -ipadx 5 -ipady 5 -anchor w \
      -expand yes -fill both
  pack ${w}.values -side top -padx 5 -pady 5 -expand yes -fill x
  pack ${w}.border -side top -ipady 1 -expand yes -fill x
  pack ${w}.confirm -side top -padx 5 -pady 5 -expand yes -fill x

  set old_focus [focus]
  catch {grab $w}
  focus $w.values

  #
  # now position at the center of root window.
  #

  # 4. Withdraw the window, then update all the geometry information
  # so we know how big it wants to be, then center the window in the
  # display and de-iconify it.

  wm withdraw $w
  update idletasks
  set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
      - [winfo vrootx [winfo parent $w]]]
  set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
      - [winfo vrooty [winfo parent $w]]]
  wm geom $w +$x+$y
  wm deiconify $w

  tkwait variable g_dialog_button

  grab release $w
  destroy $w
  catch {focus -force $old_focus}
  return $g_dialog_button
}

# Add the current plots to the File->Print and File->Export menus. 
proc shlayout:misc:add_plots_to_menu {rwin} {
  global shlayout

  set view_name(top_view)   "Top View"
  set view_name(front_view) "Front View"
  set view_name(side_view)  "Side View"

  foreach m "print export" {
    set cmd_menu $shlayout($rwin,${m}_cmd_menu)
    $cmd_menu delete 0 end
    foreach p "top bottom" {
      set view $shlayout($rwin,${p}_panel_view)
      set name $view_name($view)
      $cmd_menu add command \
        -label "$name" \
        -command "shlayout:${m}_cmd $rwin $p $view" \
	-underline {-1}
    }
  }

  set cmd_menu $shlayout($rwin,export_cmd_menu)
  $cmd_menu add command \
    -command "shlayout:export_cmd $rwin 3d 3d" \
    -label {3D raw data} \
    -underline {-1}

  $cmd_menu add command \
    -command "shlayout:export_vtk_cmd $rwin" \
    -label {3D VTK Object} \
    -underline {-1}
}

#################### Printing Utilities ###################################

# These are taken from BLT 2.4 library, and quite possibly needs to be
# updated by now.

proc shlayout:postscript_dialog {graph psfile} {
  set top $graph.top
  catch {destroy $top}

  toplevel $top -class Dialog
  wm title $top "Print"
  wm iconname $top "Dialog"
  wm transient $top [winfo toplevel [winfo parent $top]]

  foreach var { center landscape maxpect preview decorations padx 
    pady paperwidth paperheight width height colormode } {
    global $graph.$var
    set $graph.$var [$graph postscript cget -$var]
  }
  set row 1
  set col 0
  label $top.title -text "PostScript Options"
  blt::table $top $top.title -cspan 7
  foreach bool { center landscape maxpect preview decorations } {
    set w $top.$bool-label
    label $w -text "-$bool" -font *courier*-r-*12* 
    blt::table $top $row,$col $w -anchor e -pady { 2 0 } -padx { 0 4 }
    set w $top.$bool-yes
    global $graph.$bool
    radiobutton $w -text "yes" -variable $graph.$bool -value 1
    blt::table $top $row,$col+1 $w -anchor w
    set w $top.$bool-no
    radiobutton $w -text "no" -variable $graph.$bool -value 0
    blt::table $top $row,$col+2 $w -anchor w
    incr row
  }
  label $top.modes -text "-colormode" -font *courier*-r-*12* 
  blt::table $top $row,0 $top.modes -anchor e  -pady { 2 0 } -padx { 0 4 }
  set col 1
  foreach m { color greyscale } {
    set w $top.$m
    radiobutton $w -text $m -variable $graph.colormode -value $m
    blt::table $top $row,$col $w -anchor w
    incr col
  }
  set row 1
  frame $top.sep -width 2 -bd 1 -relief sunken
  blt::table $top $row,3 $top.sep -fill y -rspan 6
  set col 4
  foreach value { padx pady paperwidth paperheight width height } {
    set w $top.$value-label
    label $w -text "-$value" -font *courier*-r-*12* 
    blt::table $top $row,$col $w -anchor e  -pady { 2 0 } -padx { 0 4 }
    set w $top.$value-entry
    global $graph.$value
    entry $w -textvariable $graph.$value -width 8
    blt::table $top $row,$col+1 $w -cspan 2 -anchor w -padx 8
    incr row
  }
  set w $top.filename_label
  label $w -text "Filename: " -font *courier*-r-*12*
  blt::table $top $row,0 $w -anchor w  -pady { 2 0 } -padx { 0 4 }

  set w $top.filename_entry
  global $graph.filename
  set $graph.filename $psfile
  entry $w -text "$psfile" -font *courier*-r-*12* -textvariable $graph.filename
  blt::table $top $row,1 $w -cspan 5 -anchor w -padx 8
  incr row

  blt::table configure $top c3 -width .125i
  button $top.print -text "Print" -command "shlayout:print_postscript $graph"
  blt::table $top $row,0 $top.print  -width 1i -pady 2 -cspan 2
  button $top.cancel -text "Cancel" -command "destroy $top"
  blt::table $top $row,4 $top.cancel  -width 1i -pady 2 -cspan 3

  focus $top.print

  wm withdraw $top
  update idletasks
  set x [expr {[winfo screenwidth $top]/2 - [winfo reqwidth $top]/2 \
    - [winfo vrootx [winfo parent $top]]}]
  set y [expr {[winfo screenheight $top]/2 - [winfo reqheight $top]/2 \
    - [winfo vrooty [winfo parent $top]]}]
  wm geom $top +$x+$y
  update idle
  wm deiconify $top

}

proc shlayout:print_postscript {graph} {
  foreach var { center landscape maxpect preview decorations padx 
    pady paperwidth paperheight width height colormode } {
    global $graph.$var
    set old [$graph postscript cget -$var]
    if { [catch {$graph postscript configure -$var [set $graph.$var]}] != 0 } {
      $graph postscript configure -$var $old
      set $graph.$var $old
    }
  }
  global $graph.filename
  $graph postscript output [set $graph.filename]
  destroy $graph.top
}

################################ Main #####################################

#
# parse the command line and initialize the interface etc
#
proc shlayout:main {{rwin .} {dir .} {num_mirrors 0} {num_rays 0}} {
  global shlayout
  if {[string compare $rwin "."]} {		;# need toplevel
    toplevel $rwin -class ShadowLayout
  }
  wm withdraw $rwin
  update idletasks

  shlayout:init:env $rwin
  shlayout:init:globals $rwin

  set shlayout(directory) $dir
  set shlayout(num_mirrors) $num_mirrors
  set shlayout($rwin,num_mirrors) $num_mirrors
  set shlayout($rwin,num_rays) $num_rays

  shlayout:parse_args $rwin
  shlayout:init:preferences $rwin
  shlayout:init:fonts $rwin
  shlayout:GUI:make $rwin
  shlayout:init:startup $rwin
  shlayout:GUI:update_window_title $rwin
  update idletasks
  wm deiconify $rwin
  set shlayout(initialized) 1
}

######################## Start program ####################################
#

#
# DO NOT call shlayout:main if running under SHADOW GUI!
#
if {![info exists gvars]} {
  shlayout:main
}

