/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

#ifndef uLapack_included
#define uLapack_included

#include "nMatrix.h"

/*
   compute eigenvalues of a general real matrix. Note that eigenvalues
   are not ordered for obvious reasons
*/
//@ManMemo: Compute eigenvalues for general matrices
/** This function is overloaded for float, double, fComplex and Complex.
The columns of the matrix eVect are the eigenvectors.
The diagonal elements of the eVal matrix are the eigenvalues.
eVect and eVal are returned as fComplex for float and fComplex X.
eVect and eVal are returned as Complex for double and Complex X.

The function uses the expert drivers in Lapack - sgeevx, dgeevx,
cgeevx and zgeevx for float, double,fComplex and Complex matrices respectively.

Balancing can be done by giving the balance input parameter one of the
following values:
\begin{itemize}
\item 'N' - no permutation or scaling of matrix rows/columns(default)
\item 'P' - perform permutations to make the matrix more upper triangular. Do not diagonally scale
\item 'S' - Diagonally scale the matrix to make the rows and columns of the matrix more equal in norm. Do not permute
\item 'B' - both diagonally scale and permute the matrix
\end{itemize}
In these routines, only the right eigenvectors are computed.  For further details on these routines refer to the \URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.

Example:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <uMatrix.h>
#include <uLapack.h>

int main()
{
   // Declare required matrices
   uMatrix<fComplex> A(2,2);
   uMatrix<fComplex> B(2,2);
   uMatrix<fComplex> C(2,2);

   // Assign dummy data to A
   A(0,0) = 4.0; A(0,1) = 2.0;
   A(1,0) = fComplex(-2.0,3.0); A(1,1) = 4.0;
   // Find eigenvectors of A Balancing also done   
   // Matrix B has the eigenvectors, C has the eigenvalues 
   uGenEig(A,B,C,'B'); // Balancing done
   // The definition of eigen vectors tested here
   cout << "The following matrix:\n";

   cout << C << endl;
   cout << "must be almost equal to the following matrix:\n";
   cout << uInv(B)*A*B<< endl;
}
\end{verbatim}
*/
void uGenEig( const uMatrix<fComplex>& X, uMatrix<fComplex>& eVect,
	      uMatrix<fComplex>& eVal, char balance= 'N');
void uGenEig( const uMatrix<float>& X, uMatrix<fComplex>& eVect,
	      uMatrix<fComplex>& eVal, char balance= 'N');
void uGenEig( const uMatrix<double>& X, uMatrix<Complex>& eVect,
	      uMatrix<Complex>& eVal, char balance = 'N' );
void uGenEig( const uMatrix<Complex>& X, uMatrix<Complex>& eVect,
	      uMatrix<Complex>& eVal, char balance= 'N' );


/*
   compute eigenvalues of a symmetrical  real matrix.
   eigenvalues are ordered from smallest to largest.
*/
//@ManMemo: Compute eigenvalues for symetric and hermitian symmetric matrices.
/** This function is overloaded for float, double, fComplex and Complex.
The columns of the matrix eVect are the eigenvectors.
The diagonal elements of the eVal matrix are the eigenvalues.
The eigenvalues are ordered from smallest to the largest.
eVect is returned as fComplex for fComplex X.
eVect is returned as float for float X.
eVect is returned as double for double X.
eVect is returned as Complex for Complex X.
eVal is returned as float for float and fComplex X.
eVal is returned as double for double and Complex X.

The function uses the routines - ssyev, dsyev, cheev and zheev for
float, double, fComplex and Complex matrices repectively. For further
details on these routines refer to the \URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.

Example:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <uMatrix.h>
#include <uLapack.h>

int main()
{

   uMatrix<Complex> A(2,2);
   uMatrix<Complex> B(2,2);
   uMatrix<double> C(2,2);

   // A hermitian symmetric matrix defined here
   A(0,0) = 4.0; A(0,1) = Complex(-2.0,3.0);
   A(1,0) = conj(A(0,1)); A(1,1) = A(0,0);
   // Eigenvectors are computed here. B contains the eigenvectors and
   // C contains the eigenvalues 
   uSymEig(A,B,C);
   //Definition of eigenvector tested here
   cout << "The following matrix:\n";

   cout << C << endl;
   cout << "must be almost equal to the following matrix:\n";
   cout << uInv(B)*A*B<< endl;
}
\end{verbatim}

*/
void uSymEig( const uMatrix<fComplex>& X, uMatrix<fComplex>& eVect,
	      uMatrix<float>& eVal );
void uSymEig( const uMatrix<float>& X, uMatrix<float>& eVect,
	      uMatrix<float>& eVal );
void uSymEig( const uMatrix<double>& X, uMatrix<double>& eVect,
	      uMatrix<double>& eVal );
void uSymEig( const uMatrix<Complex>& X, uMatrix<Complex>& eVect,
	      uMatrix<double>& eVal );


//@ManMemo: Compute the inverse of a general matrix.
/** This function is overloaded for float, double, fComplex and Complex.
Returns the inverse of the same type as A.
 The function uses the routines sgetri, dgetri, cgetri and zgetri for float,
 double, fComplex and Complex matrices respectively. For further
details on these routines refer to the \URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.

Example:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <uMatrix.h>
#include <uLapack.h>

int main()
{

   uMatrix<float> A(2,2);
   uMatrix<float> B(2,2);
   
   // The  matrix defined here
   A(0,0) = 4.0; A(0,1) = 1;
   A(1,0) = -10 ; A(1,1) = A(0,0);
   cout << "The following matrix must be almost identity\n";
  // Inverse computed here
   B= uInv(A);
   cout <<  A*B << endl;
}
\end{verbatim}

*/
uMatrix<float> uInv( const uMatrix<float>& B );
uMatrix<double> uInv( const uMatrix<double>& B );
uMatrix<fComplex> uInv( const uMatrix<fComplex>& B );
uMatrix<Complex> uInv( const uMatrix<Complex>& B );
double uDet( const uMatrix<double>& B);




/* Obtain linear least squares/minimum norm solutions for matrix equations
of the form Ax=b using SVD.  
*/
//@ManMemo: Compute the LLS/minimum norm solutions for matrix equations.
/** This function is overloaded for float, double, fComplex and Complex.
The function computes x in the matrix equation Ax = B.
A and b has to be of the same type.
x is returned of the same type as A.
If A is square and full rank, the exact solution is obtained. Otherwise,
a least square solution or minimum norm solution is obtained.

The functions use the routines sgelss, dgelss, cgelss and zgelss for float, double, fComplex and Complex matrices. These routines use the SVD to compute the
Least squares/minimum norm solution. Singular values less than the machine precision are considered zero. For further
details on these routines refer to the \URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.

Example:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <uMatrix.h>
#include <uLapack.h>

int main()
{

   uMatrix<float> A(2,2);
   uMatrix<float> x(2,1);
   uMatrix<float> b(2,1);
   
   // The  matrix defined here
   A(0,0) = 4.0; A(0,1) = -1;
   A(1,0) = 10; A(1,1) = A(0,0);
   b(0,0) = 1;
   b(1,0) = 1;
   
   // Solution to Ax = b computed here
   x = uLlsSVD(A,b);
   // Solution checked
   cout << "The following matrix:\n";
  
   cout << b << endl;
   cout << "must be almost equal to the following matrix:\n";
   cout << A*x << endl;
}
\end{verbatim}

*/
uMatrix<float> uLlsSVD( const uMatrix<float>& A, const uMatrix<float>& b);
uMatrix<double> uLlsSVD( const uMatrix<double>& A, const uMatrix<double>& b);
uMatrix<fComplex> uLlsSVD( const uMatrix<fComplex>& A, const uMatrix<fComplex>& b);
uMatrix<Complex> uLlsSVD( const uMatrix<Complex>& A, const uMatrix<Complex>& b);

/* SVD of general matrices 
                                 H
Returns U, S and V such that U*S*V = A. S is returned as
a row vector with the singular values in descending order of magnitude 
*/
//@ManMemo: Compute the SVD of general matrices
/** This function is overloaded for float, double, fComplex and Complex.
Returns U, S and V such that U*diag(S)*V^{H}. S is returned as
a row vector with the singular values in descending order of magnitude.
U and V are returned of the same type as A.
S is returned float for float and fComplex A; S is returned double for double 
and Complex A.

The functions use the routines sgesvd, dgesvd, cgesvd and zgesvd for float, double, fComplex and Complex matrices. For details on these routines refer to the \URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.

Example:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <uMatrix.h>
#include <uLapack.h>

int main()
{

   uMatrix<float> A(2,2);
   uMatrix<float> U(2,2);
   uMatrix<float> S(1,2);
   uMatrix<float> V(2,2);
   
   // The  matrix defined here
   A(0,0) = 4.0; A(0,1) = -1;
   A(1,0) = 10; A(1,1) = A(0,0);
   
   // The SVD done here
   uSVD(A,U,S,V);
   // The singular values written as a matrix 
   uMatrix<float> S_diag(2,2);
   S_diag(0,0)=S(0,0);
   S_diag(0,1)=0;
   S_diag(1,0)=0;
   S_diag(1,1)=S(0,1);
   
   // Solution checked - definition of SVD
   cout << "The following matrix:\n";
  
   cout << A << endl;
   cout << "must be almost equal to the following matrix:\n";
   cout << U*S_diag*hTranspose(V) << endl;
}
\end{verbatim}

*/

void uSVD( const uMatrix<float>& A, uMatrix<float>& U,
	      uMatrix<float>& S, uMatrix<float>& V );
void uSVD( const uMatrix<fComplex>& A, uMatrix<fComplex>& U,
	      uMatrix<float>& S, uMatrix<fComplex>& V );
void uSVD( const uMatrix<double>& A, uMatrix<double>& U,
	      uMatrix<double>& S, uMatrix<double>& V );
void uSVD( const uMatrix<Complex>& A, uMatrix<Complex>& U,
	      uMatrix<double>& S, uMatrix<Complex>& V );


/* Estimation of 2-norm condition numbers for any general matrix using SVD */

//@ManMemo: Compute the condition number of general matrices using SVD
/** This function is overloaded for float, double, fComplex and Complex.
Returns float for float and fComplex A;returns double for double and Complex A.
The function uses the singular values output using the uSVD computation. The condition number is the ratio of the lowest singular value to the largest singular value.

Example:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <uMatrix.h>
#include <uLapack.h>

int main()
{

   uMatrix<float> A(2,2);
   
   // The  matrix defined here
   A(0,0) = 1.0; A(0,1) = 0;
   A(1,0) = 0; A(1,1) = 1000;
   // The condition number of this matrix is ratio of A(1,1)/A(0,0)
   cout << "The following number must be almost equal to 1000:\n";
    
   cout << uCond(A) << endl;
}
\end{verbatim}


*/
float uCond(const uMatrix<float>&A);
float uCond(const uMatrix<fComplex>&A);
double uCond(const uMatrix<double>&A);
double uCond(const uMatrix<Complex>&A);




/*
   Add one matrix to an other.
   It call the blas xAXPY 
*/
//@ManMemo: Add one matrix to an other calling the blas  xAXPY: y=y+a*x
/**  Add one matrix to an other calling the blas  xAXPY: y=y+a*x
    This function is overloaded for float, double, fComplex and Complex.
 For further details on these routines refer to the 
\URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.
*/

void uAXPY(fComplex a,  const uMatrix<fComplex>& X, uMatrix<fComplex>& Y );
void uAXPY(Complex a,  const uMatrix<Complex>& X, uMatrix<Complex>& Y );
void uAXPY(double  a,  const uMatrix<double>& X, uMatrix<double>& Y );
void uAXPY(float a,  const uMatrix<float>& X, uMatrix<float>& Y );


/*
   Add one matrix to an other.
   It call the blas xAXPY 
*/
//@ManMemo: Add one matrix to an other calling the blas  xAXPY: z=y+a*x
/**  Add one matrix to an other calling the blas  xAXPY: z=y+a*x
    This function is overloaded for float, double, fComplex and Complex.
 For further details on these routines refer to the 
\URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.
*/

void uAXPY(fComplex a,  const uMatrix<fComplex>& X, const uMatrix<fComplex>& Y, uMatrix<fComplex>& Z);
void uAXPY(Complex a,  const uMatrix<Complex>& X, const uMatrix<Complex>& Y, uMatrix<Complex>& Z );
void uAXPY(double  a,  const uMatrix<double>& X, const uMatrix<double>& Y, uMatrix<double>& Z);
void uAXPY(float a,  const uMatrix<float>& X, const uMatrix<float>& Y, uMatrix<float>& Z);



/*
  Moltiplicazione
*/
//@ManMemo: Multiplies A times B and put the result in C. 
/**  Multiplies A times B and put the result in C. Calls
 blas  xGEMM: C=beta*C+alpha*A*B.
 This function is overloaded for float, double, fComplex and Complex.
 For further details on these routines refer to the 
 \URL[Lapack Users Guide]{http://www.netlib.org/lapack/lug/lapack_lug.html}.
*/

void uGemm(fComplex alpha, const uMatrix<fComplex>& A,
	                   const uMatrix<fComplex>& B,fComplex beta,
                           uMatrix<fComplex>&        C );

void uGemm(Complex alpha, const uMatrix<Complex>& A,
	                   const uMatrix<Complex>& B,Complex beta,
                           uMatrix<Complex>&        C );

void uGemm(double alpha, const uMatrix<double>& A,
	                   const uMatrix<double>& B,double  beta,
                           uMatrix<double>&        C );

void uGemm(float alpha, const uMatrix<float>& A,
	                   const uMatrix<float>& B,float   beta,
                           uMatrix<float>&        C );



void  JoinV( const  uMatrix<Complex> &  A , const uMatrix<Complex>  &  B,  uMatrix<Complex>  &  C);

void  JoinH( const  uMatrix<Complex> &  A, const uMatrix<Complex>  &  B,  uMatrix<Complex>  &  C);

void  JoinH( const  uMatrix<double> &  A, const uMatrix<double>  &  B,  uMatrix<double>  &  C);

void  JoinV( const  uMatrix<double> &  A , const uMatrix<double>  &  B,  uMatrix<double>  &  C);


#endif







