/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

/*
$Id: uLapack.cc,v 1.8 1996/08/12 22:16:10 shenoy Exp $
$Author: shenoy $
$Source: /home/weber/work/uMatrix/RCS/uLapack.cc,v $
$Revision: 1.8 $
$Log: uLapack.cc,v $
// Revision 1.8  1996/08/12  22:16:10  shenoy
// 1. Overloaded uSymEig for Complex matrices
// 2. Documented all matrix functions using Lapack to include the routine
//    used and a brief description and examples.
//
// Revision 1.7  1996/08/11  01:43:50  shenoy
// Changed code in uGenEig routines to use the expert drivers in Lapack.
// Now the user has the option of doing balancing prior to computing the
// eigenvectors. Also added code to overload uGenEig for Complex
// matrices.
//
// Revision 1.6  1996/06/17  15:36:20  weber
// Added Gnu library public license header
//
// Revision 1.5  1996/04/23  17:51:26  weber
// Removed the transposes in the uInv function because the transpose of
// the inverse is the inverse of the transpose! This should save quite a
// bit of time.
//
// Revision 1.4  1996/04/17  16:55:09  weber
// Removed declarations of loop counters so that the code compiles
// independently of the -no-for-scope flag or gcc version
//
// Revision 1.3  95/11/06  18:15:00  weber
// More gcc 2.7.0 and ANSI scoping related fixups
// 
// Revision 1.2  1995/11/06  17:27:00  weber
// g++ 2.7.0 enforces ANSI scoping rules for declarations so a bunch of
// int's had to be added to for statements that relied on ints declared
// in the previous block.
//
*/

#include "lapack.h"
#include "nLapack.h"

/* Single precision complex routines */

void uGenEig( const uMatrix<fComplex>& X, uMatrix<fComplex>& eVect, 
	     uMatrix<fComplex>& eVal, char balance = 'N' )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void uGenEig(fComplex) error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobvl;
   char jobvr;
   char balanc;
   char sense;
   
   int N;
   int lda;
   fComplex vl; 
   int ldvl; 
   int ldvr; 
   int ilo;
   int ihi;
   
   fComplex *work;
   float *rwork;

   float abnrm;
   
   int lwork; 
   int info;

   // Other temporaries
   uMatrix<fComplex> A(X);
   uMatrix<fComplex> vr(X.rows(),X.columns());
   uMatrix<fComplex> w(X.rows(),1);

   // Set up call to cgeevx
   balanc = balance;
   
   jobvl = 'N';
   jobvr = 'V';
   sense = 'N';
   lwork = 4*A.rows();
   work = new fComplex[lwork];
   float scale[A.rows()];
   float rconde[A.rows()];
   float rcondv[A.rows()];

   N=A.rows();
   rwork = new float[2*N];
   lda = N;
   vr = fComplex(0,0);
   ldvl = 1;
   ldvr = A.rows();


   // A.transpose();

   cgeevx_(&balanc, &jobvl, &jobvr,&sense, &N, A.address(), &lda, w.address(), 
	   &vl, &ldvl, vr.address(), &ldvr, &ilo, &ihi, scale, &abnrm, rconde,
	   rcondv, work, &lwork, rwork, &info );
   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uGenEig(fComplex) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uGenEig(fComplex) error in parameter " << -1*info << endl;
	 abort();
      }
   }

   eVal.resize(X.rows(),X.rows());
   eVect.resize(X.rows(),X.rows());

   // Rescue eigenvalues
   eVal = fComplex(0,0);

   for ( int i = 0; i < X.rows(); i++ )
   {
      eVal(i,i) = w(i,0);
   } 

   // Extract eigenvalues
   eVect=vr;
   // eVect = transpose(vr);
   delete[] work;
   delete[] rwork;

}
void uGenEig( const uMatrix<Complex>& X, uMatrix<Complex>& eVect, 
	     uMatrix<Complex>& eVal, char balance = 'N' )
{
   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void uGenEig(Complex) error: Input matrix must be square\n";
      abort();
   }
   // Temporaries required by lapack
   char jobvl;
   char jobvr;
   char balanc;
   char sense;
   
   int N;
   int lda;
   Complex vl; 
   int ldvl; 
   int ldvr; 
   int ilo;
   int ihi;
   
   Complex *work;
   double *rwork;

   double abnrm;
   
   int lwork; 
   int info;

   // Other temporaries
   uMatrix<Complex> A(X);
   uMatrix<Complex> vr(X.rows(),X.columns());
   uMatrix<Complex> w(X.rows(),1);



   // Set up call to zgeev
   balanc = balance;
   
   jobvl = 'N';
   jobvr = 'V';
   sense = 'N';
   lwork = 4*A.rows();
   work = new Complex[lwork];
   double scale[A.rows()];
   double rconde[A.rows()];
   double rcondv[A.rows()];

   N=A.rows();
   rwork = new double[2*N];
   lda = N;
   vr = Complex(0,0);
   ldvl = 1;
   ldvr = A.rows();


   // A.transpose();
   zgeevx_(&balanc, &jobvl, &jobvr,&sense, &N, A.address(), &lda, w.address(), 
	   &vl, &ldvl, vr.address(), &ldvr, &ilo, &ihi, scale, &abnrm, rconde,
	   rcondv, work, &lwork, rwork, &info );
   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uGenEig(Complex) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uGenEig(Complex) error in parameter " << -1*info << endl;
	 abort();
      }
   }

   eVal.resize(X.rows(),X.rows());
   eVect.resize(X.rows(),X.rows());

   // Rescue eigenvalues
   eVal = Complex(0,0);

   for ( int i = 0; i < X.rows(); i++ )
   {
      eVal(i,i) = w(i,0);
   } 

   // Extract eigenvalues
   eVect=vr;
   // eVect = transpose(vr);
   delete[] work;
   delete[] rwork;

}

void uGenEig( const uMatrix<float>& X, uMatrix<fComplex>& eVect, 
	     uMatrix<fComplex>& eVal, char balance = 'N' )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void uGenEig() error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobvl;
   char jobvr;
   char balanc;
   char sense;
   
   int N;
   int lda;
   float vl; 
   int ldvl; 
   int ldvr;
   int ilo;
   int ihi;
   
   float *work; 
   int *iwork;
   float abnrm;
   
   int lwork; 
   int info;

   // Other temporaries
   uMatrix<float> A(X);
   uMatrix<float> vr(X.rows(),X.columns());
   uMatrix<float> wi(X.rows(),1);
   uMatrix<float> wr(X.rows(),1);

   // Set up call to sgeevx
   balanc = balance;
   
   jobvl = 'N';
   jobvr = 'V';
   sense = 'N';
   
   N = A.rows();
   lda = N;
   vr = 0.0;
   ldvl = 1;
   ldvr = A.rows();
   lwork = 4*A.rows();
   work = new float[lwork];
   iwork = new int[2*N-2];
   float scale[A.rows()];
   float rconde[A.rows()];
   float rcondv[A.rows()];
   

   // A.transpose();
   sgeevx_(&balanc, &jobvl, &jobvr,&sense, &N, A.address(), &lda, wr.address(),
	   wi.address(),  &vl, &ldvl, vr.address(), &ldvr, &ilo, &ihi, scale,
	   &abnrm, rconde, rcondv, work, &lwork, iwork, &info );

   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uSymEig(float) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uSymEig(float) error in parameter " << -1*info << endl;
	 abort();
      }
   }



   eVal.resize(X.rows(),X.rows());
   eVect.resize(X.rows(),X.rows());

   // Rescue eigenvalues
   eVal = fComplex(0,0);
   
   int i;
   for ( i = 0; i < X.rows(); i++ )
   {
      eVal(i,i) = fComplex( wr(i,0), wi(i,0) );
   } 

   // Extract eigenvalues

   for ( i = 0; i < X.rows(); i++ ) // Index along eigenvectors
   {
      if ( fabs( wi(i,0) ) == 0.0 ) // See if eigenvalue is real
      {
	 // Just copy it
	 for ( int j = 0; j < X.rows(); j++ )
	 {  
	    eVect(j,i) = fComplex( vr(j,i), 0.0 );
	 }
      }
      else // Ok, it was a complex eigenvalue
      {
	 for ( int j = 0; j < X.rows(); j++ )
	 {  
	    eVect(j,i) = fComplex( vr(j,i), vr(j,i+1) );
	    eVect(j,i+1) = fComplex( vr(j,i), -vr(j,i+1) );
	 }
	 i++; // This is naughty but...
      }
   }
   delete[] work;
   delete[] iwork;
   
}
void uGenEig( const uMatrix<double>& X, uMatrix<Complex>& eVect, 
	     uMatrix<Complex>& eVal, char balance = 'N' )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void uGenEig() error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobvl;
   char jobvr;
   char balanc;
   char sense;
   
   int N;
   int lda;
   double vl; 
   int ldvl; 
   int ldvr;
   int ilo;
   int ihi;
   
   double *work; 
   int *iwork;
   double abnrm;
   
   int lwork; 
   int info;

   // Other temporaries
   uMatrix<double> A(X);
   uMatrix<double> vr(X.rows(),X.columns());
   uMatrix<double> wi(X.rows(),1);
   uMatrix<double> wr(X.rows(),1);

   // Set up call to dgeev
   balanc = balance;
   
   jobvl = 'N';
   jobvr = 'V';
   sense = 'N';
   
   N = A.rows();
   lda = N;
   vr = 0.0;
   ldvl = 1;
   ldvr = A.rows();
   lwork = 4*A.rows();
   work = new double[lwork];
   iwork = new int[2*N-2];
   double scale[A.rows()];
   double rconde[A.rows()];
   double rcondv[A.rows()];
   

   // A.transpose();
   dgeevx_(&balanc, &jobvl, &jobvr,&sense, &N, A.address(), &lda, wr.address(),
	   wi.address(),  &vl, &ldvl, vr.address(), &ldvr, &ilo, &ihi, scale,
	   &abnrm, rconde, rcondv, work, &lwork, iwork, &info );

   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uSymEig(double) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uSymEig(double) error in parameter " << -1*info << endl;
	 abort();
      }
   }



   eVal.resize(X.rows(),X.rows());
   eVect.resize(X.rows(),X.rows());

   // Rescue eigenvalues
   eVal = Complex(0,0);
   
   int i;
   for ( i = 0; i < X.rows(); i++ )
   {
      eVal(i,i) = Complex( wr(i,0), wi(i,0) );
   } 

   // Extract eigenvalues

   for ( i = 0; i < X.rows(); i++ ) // Index along eigenvectors
   {
      if ( fabs( wi(i,0) ) == 0.0 ) // See if eigenvalue is real
      {
	 // Just copy it
	 for ( int j = 0; j < X.rows(); j++ )
	 {  
	    eVect(j,i) = Complex( vr(j,i), 0.0 );
	 }
      }
      else // Ok, it was a complex eigenvalue
      {
	 for ( int j = 0; j < X.rows(); j++ )
	 {  
	    eVect(j,i) = Complex( vr(j,i), vr(j,i+1) );
	    eVect(j,i+1) = Complex( vr(j,i), -vr(j,i+1) );
	 }
	 i++; // This is naughty but...
      }
   }
   delete[] work;
   delete[] iwork;
   
}


void uSymEig( const uMatrix<fComplex>& X, uMatrix<fComplex>& eVect, 
	     uMatrix<float>& eVal )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void simEig(fComplex) error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobz = 'V';
   char uplo = 'U';
   int N = X.rows();
   int lda = N;
   int lwork = 3*N-1;
   fComplex *work = new fComplex[lwork];
   float *rwork = new float[3*N];
   int info;

   uMatrix<float> w(N,1);
   uMatrix<fComplex> A = X; // transpose(X);

   cheev_( &jobz, &uplo, &N, A.address(), &lda, w.address(),
           work, &lwork, rwork, &info );

   eVal.resize(N,N); eVal = 0.0;
   eVect.resize(N,N); eVect = fComplex(0.0,0.0);
   for (int i = 0; i < N; i++ )
   {
      eVal(i,i) = w(i,0);
   }
   eVect = A ; // transpose(A);
   delete[] work;
   delete[] rwork;
}

/* Double precision eigenfunctions */
void uSymEig( const uMatrix<Complex>& X, uMatrix<Complex>& eVect, 
	     uMatrix<double>& eVal )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void symEig(Complex) error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobz = 'V';
   char uplo = 'U';
   int N = X.rows();
   int lda = N;
   int lwork = 3*N-1;
   Complex *work = new Complex[lwork];
   double *rwork = new double[3*N];
   int info;

   uMatrix<double> w(N,1);
   uMatrix<Complex> A = X; // transpose(X);

   zheev_( &jobz, &uplo, &N, A.address(), &lda, w.address(),
           work, &lwork, rwork, &info );

   eVal.resize(N,N); eVal = 0.0;
   eVect.resize(N,N); eVect = Complex(0.0,0.0);
   for (int i = 0; i < N; i++ )
   {
      eVal(i,i) = w(i,0);
   }
   eVect = A; // transpose(A);
   delete[] work;
   delete[] rwork;
}


void uSymEig( const uMatrix<double>& X, uMatrix<double>& eVect, 
	     uMatrix<double>& eVal )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void simEig(double) error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobz = 'V';
   char uplo = 'U';
   int N = X.rows();
   int lda = N;
   int lwork = 3*N-1;
   double *work = new double[lwork];
   int info;

   uMatrix<double> w(N,1);
   uMatrix<double> A = X; // transpose(X);

   dsyev_( &jobz, &uplo, &N, A.address(), &lda, w.address(),
           work, &lwork, &info );

   eVal.resize(N,N); eVal = 0.0;
   eVect.resize(N,N); eVect = 0.0;
   for (int i = 0; i < N; i++ )
   {
      eVal(i,i) = w(i,0);
   }
   eVect = A; // transpose(A);
   delete[] work;
}

/* Single precision eigen routines */


void uSymEig( const uMatrix<float>& X, uMatrix<float>& eVect, 
	     uMatrix<float>& eVal )
{

   // Check that everything is kosher with the supplied matrices
   if ( X.rows() != X.columns() )
   {
      cout << "void simEig() error: Input matrix must be square\n";
      abort();
   }

   // Temporaries required by lapack
   char jobz = 'V';
   char uplo = 'U';
   int N = X.rows();
   int lda = N;
   int lwork = 3*N-1;
   float *work = new float[lwork];
   int info;

   uMatrix<float> w(N,1);
   uMatrix<float> A = X; // transpose(X);

   ssyev_( &jobz, &uplo, &N, A.address(), &lda, w.address(),
           work, &lwork, &info );

   eVal.resize(N,N); eVal = 0.0;
   eVect.resize(N,N); eVect = 0.0;
   for (int i = 0; i < N; i++ )
   {
      eVal(i,i) = w(i,0);
   }
   eVect = A; // transpose(A);
   delete[] work;
}

uMatrix<float> uInv( const uMatrix<float>& B )
{
   int *ipiv;
   int info,size;
   float *work;
   uMatrix<float> A(B);
   if ( A.rows() != A.columns() )
   {
      cerr << "Matrix must be square in procedure uInv( uMatrix<float> A )\n";
      abort();
   }
   size = A.rows();
   ipiv = new int[size];
   work = new float[size];
   sgetrf_( &size, &size, A.address(), &size, ipiv, &info );
   if ( info != 0 )
   {
      cerr << "Error while factorizing matrix in uInv( uMatrix<float> A )\n";
      abort();
   }
   sgetri_( &size, A.address(), &size, ipiv, work, &size, &info );
   if ( info > 0 )
   {
      cerr << "Warning: Matrix is singular in uInv( uMatrix<float> A )\n";
   }
   if ( info < 0 )
   {
      cerr << "Error while inverting matrix in uInv( uMatrix<float> A )\n";
      abort();
   }
   delete[] ipiv;
   delete[] work;
   A.markAsTemporary();
   return A; // Put into row major format and return
}

uMatrix<double> uInv( const uMatrix<double>& B )
{
   int *ipiv;
   int info,size;
   double *work;
   uMatrix<double> A(B);
   if ( A.rows() != A.columns() )
   {
      cerr << "Matrix must be square in procedure uInv( uMatrix<double> A )\n";
      abort();
   }
   size = A.rows();
   ipiv = new int[size];
   work = new double[size];
   dgetrf_( &size, &size, A.address(), &size, ipiv, &info );
   if ( info != 0 )
   {
      cerr << "Error while factorizing matrix in uInv( uMatrix<double> A )\n";
      abort();
   }
   dgetri_( &size, A.address(), &size, ipiv, work, &size, &info );
   if ( info > 0 )
   {
      cerr << "Warning: Matrix is singular in uInv( uMatrix<double> A )\n";
   }
   if ( info < 0 )
   {
      cerr << "Error while inverting matrix in uInv( uMatrix<double> A )\n";
      abort();
   }
   delete[] ipiv;
   delete[] work;
   A.markAsTemporary();
   return A; // Put into row major format and return
}

double uDet( const uMatrix<double>& B )
{
   int *ipiv;
   int info,size;
   uMatrix<double> A(B);
   if ( A.rows() != A.columns() )
   {
      cerr << "Matrix must be square in procedure uInv( uMatrix<double> A )\n";
      abort();
   }
   size = A.rows();
   ipiv = new int[size];
   dgetrf_( &size, &size, A.address(), &size, ipiv, &info );
   if ( info != 0 )
   {
     return 0;
      cerr << "Error while factorizing matrix in uInv( uMatrix<double> A )\n";
      abort();
   }
   double res=1;
   for(int i=0; i<A.rows();i++)
     {
       res*=A(i,i);
     }

   delete ipiv;
   return res;
}

// Complex inversions
// Commented out because of gcc internal error stuff.
uMatrix<fComplex> uInv( const uMatrix<fComplex>& B )
{
   int *ipiv;
   int info,size;
   fComplex *work;
   uMatrix<fComplex> A(B);
   if ( A.rows() != A.columns() )
   {
      cerr << "Matrix must be square in procedure uInv( uMatrix<fComplex> A )\n";
      abort();
   }
   size = A.rows();
   ipiv = new int[size];
   work = new fComplex[size];
   cgetrf_( &size, &size, A.address(), &size, ipiv, &info );
   if ( info != 0 )
   {
      cerr << "Error while factorizing matrix in uInv( uMatrix<fComplex> A )\n";
      abort();
   }
   cgetri_( &size, A.address(), &size, ipiv, work, &size, &info );
   if ( info > 0 )
   {
      cerr << "Warning: Matrix is singular in uInv( uMatrix<fComplex> A )\n";
   }
   if ( info < 0 )
   {
      cerr << "Error while inverting matrix in uInv( uMatrix<fComplex> A )\n";
      abort();
   }
   delete[] ipiv;
   delete[] work;
   A.markAsTemporary();
   return A; // Put into row major format and return
}


uMatrix<Complex> uInv( const uMatrix<Complex>& B )
{
   int *ipiv;
   int info,size;
   Complex *work;

   uMatrix<Complex> A(B);

   if ( A.rows() != A.columns() )
   {
      cerr << "Matrix must be square in procedure uInv( uMatrix<Complex> A )\n";
      abort();
   }

   size = A.rows();
   ipiv = new int[size];
   work = new Complex[size];
   // cout << " chiamo Lapack \n";
   zgetrf_( &size, &size, A.address(), &size, ipiv, &info );
   if ( info != 0 )
   {
      cerr << "Error while factorizing matrix in uInv( uMatrix<Complex> A )\n";
      abort();
   }
   zgetri_( &size, A.address(), &size, ipiv, work, &size, &info );
   if ( info > 0 )
   {
      cerr << "Warning: Matrix is singular in uInv( uMatrix<Complex> A )\n";
   }
   if ( info < 0 )
   {
      cerr << "Error while inverting matrix in uInv( uMatrix<Complex> A )\n";
      abort();
   }
   // cout << " Lapack OK \n";
   delete[] ipiv;
   delete[] work;
   A.markAsTemporary();
   return A; // Put into row major format and return
}

/* Single precision linear least squares - for float and fComplex types */

uMatrix<float> uLlsSVD( const uMatrix<float>& A, const uMatrix<float>& b)
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);
   // Check supplied matrices for compatiability
   if (A.rows() != b.rows())
   {
      cerr<< "Matrix sizes do not match in uLlsSVD( const uMatrix<float>& A, const uMatrix<float>& b) \n ";
      abort();
   }
   // temporaries required by lapack
   int info, lda,ldb,lwork,m,n,nrhs,rank;
   float rcond;
   float *s, *work;
   
   //set up to call sgelss

   m=A.rows();
   n=A.columns();
   nrhs=b.columns();

  // cute arithmetic-if expressions for finding maxs and mins

   lda=(m > 1) ? m : 1; // max(1,m)
   
   ldb= (lda > n) ? lda: n; // max(1,m,n)

   rcond = -1.0;   // machine precision used
   
   int maxmn = ( m > n) ? m : n; // max(m,n)
   int minmn = (m > n) ? n : m; // min(m,n)
   s=new float[ minmn]; 
      
   lwork=2*(3*minmn + ( (2*minmn>nrhs) ? ( (2*minmn > maxmn? 2*minmn:maxmn)) : (nrhs> maxmn ?  nrhs :maxmn))); 

  /* both m > n and n> m conditions satisfied for lwork. lwork is made twice 
 the  minimum needed lwork 
*/
   // Other temporaries to store A and b and store the solution x
   uMatrix<float> A1(A);
   uMatrix<float> x(n,nrhs);
   uMatrix<float> b1(ldb,nrhs);
   uIndex j(0,1,m-1);
   uIndex k(0,1,nrhs-1);
   
   b1.insert( j, k, b);
   
   work= new float[lwork];
   A1.transpose(); // Fortran reads in column format
   b1.transpose();    
   sgelss_(&m,&n,&nrhs,A1.address(),&lda,b1.address(),&ldb,s,&rcond,&rank,work, &lwork,&info);
   if (info !=0)
   {
      if (info > 0)
	  {
	     cout << " uLlsSVD( const uMatrix<float>& A, const uMatrix<float>& b) failed to converge \n";
	     abort();
	  }
      else 
      {
	 cout << "uLlsSVD( const uMatrix<float>& A, const uMatrix<float>& b) error in parameter" << -1*info << endl;
	 abort();
      }
   }
   
   delete [] s;
   delete [] work;     
// Extract the solution from b1
   b1.transpose(); //Change format for C
   
   uIndex j1(0,1,n-1);
   uIndex k1(0,1,nrhs-1);
   x=b1(j1,k1);
   x.markAsTemporary();
   return x;
}   

uMatrix<fComplex> uLlsSVD( const uMatrix<fComplex>& A, const uMatrix<fComplex>& b)
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);

   // Check supplied matrices for compatiability
   if (A.rows() != b.rows())
   {
      cerr<< "Matrix sizes do not match in uLlsSVD( const uMatrix<fComplex>& A, const uMatrix<fComplex>& b) \n ";
      abort();
   }
   // temporaries required by lapack
   int info, lda,ldb,lwork,m,n,nrhs,rank;
   float rcond;
   float *s, *rwork;
   fComplex *work;
   //set up to call cgelss
   m=A.rows();
   n=A.columns();
   nrhs=b.columns();

  // cute arithmetic-if expressions for finding maxs and mins

   lda=(m > 1) ? m : 1; // max(1,m)
   
   ldb= (lda > n) ? lda: n; // max(1,m,n)

   rcond = -1.0;   // machine precision used
   
   int maxmn = ( m > n) ? m : n; // max(m,n)
   int minmn = (m > n) ? n : m; // min(m,n)
   s=new float[ minmn]; 
   rwork = new float[ (5*minmn-4 > 1)? 5*minmn-4 :1];    
   lwork=2*(2*minmn + ( nrhs > maxmn ?  nrhs :maxmn)); 
  
  /*
  both m > n and n> m conditions satisfied for lwork. lwork is made twice 
  the  minimum needed lwork 
  */
   // Other temporaries to store A and b and store the solution x

   uMatrix<fComplex> A1(A);
   uMatrix<fComplex> x(n,nrhs);
   uMatrix<fComplex> b1(ldb,nrhs);
   

   uIndex j(0,1,m-1);
   uIndex k(0,1,nrhs-1);
   
   
   b1.insert( j, k, b);
    
   work= new fComplex[lwork];
   A1.transpose(); // Fortran reads in column format
   b1.transpose();    
   cgelss_(&m,&n,&nrhs,A1.address(),&lda,b1.address(),&ldb,s,&rcond,&rank,work, &lwork,rwork,&info);
   if (info !=0)
   {
      if (info > 0)
	  {
	     cout << " uLlsSVD( const uMatrix<fComplex>& A, const uMatrix<fComplex>& b) failed to converge \n";
	     abort();
	  }
      else 
      {
	 cout << "uLlsSVD( const uMatrix<fComplex>& A, const uMatrix<fComplex>& b) error in parameter" << -1*info << endl;
	 abort();
      }
   }
   
   delete [] s;
   delete [] work;     
   delete [] rwork;
   
// Extract the solution from b1
   b1.transpose(); //Change format for C
   
   uIndex j1(0,1,n-1);
   uIndex k1(0,1,nrhs-1);
   x=b1(j1,k1);
   x.markAsTemporary();
   return x;
}   

/* Double precision linear least squares - for double and Complex types */

uMatrix<double> uLlsSVD( const uMatrix<double>& A, const uMatrix<double>& b)
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);
   // Check supplied matrices for compatiability
   if (A.rows() != b.rows())
   {
      cerr<< "Matrix sizes do not match in uLlsSVD( const uMatrix<double>& A, const uMatrix<double>& b) \n ";
      abort();
   }
   // temporaries required by lapack
   int info, lda,ldb,lwork,m,n,nrhs,rank;
   double rcond;
   double *s, *work;
   


   //set up to call dgelss

   m=A.rows();
   n=A.columns();
   nrhs=b.columns();

  // cute arithmetic-if expressions for finding maxs and mins

   lda=(m > 1) ? m : 1; // max(1,m)
   
   ldb= (lda > n) ? lda: n; // max(1,m,n)

   rcond = -1.0;   // machine precision used
   
   int maxmn = ( m > n) ? m : n; // max(m,n)
   int minmn = (m > n) ? n : m; // min(m,n)
   s=new double[ minmn]; 
      
   lwork=2*(3*minmn + ( (2*minmn>nrhs) ? ( (2*minmn > maxmn? 2*minmn:maxmn)) : (nrhs> maxmn ?  nrhs :maxmn))); 

  /* both m > n and n> m conditions satisfied for lwork. lwork is made twice 
 the  minimum needed lwork 
*/
   // Other temporaries to store A and b and store the solution x
   uMatrix<double> A1(A);
   uMatrix<double> x(n,nrhs);
   uMatrix<double> b1(ldb,nrhs);
   uIndex j(0,1,m-1);
   uIndex k(0,1,nrhs-1);
   
   b1.insert( j, k, b);
   
   work= new double[lwork];
   A1.transpose(); // Fortran reads in column format
   b1.transpose();    
   dgelss_(&m,&n,&nrhs,A1.address(),&lda,b1.address(),&ldb,s,&rcond,&rank,work, &lwork,&info);
   if (info !=0)
   {
      if (info > 0)
	  {
	     cout << " uLlsSVD( const uMatrix<double>& A, const uMatrix<double>& b) failed to converge \n";
	     abort();
	  }
      else 
      {
	 cout << "uLlsSVD( const uMatrix<double>& A, const uMatrix<double>& b) error in parameter" << -1*info << endl;
	 abort();
      }
   }
   
   delete [] s;
   delete [] work;     
// Extract the solution from b1
   b1.transpose(); //Change format for C
   
   uIndex j1(0,1,n-1);
   uIndex k1(0,1,nrhs-1);
   x=b1(j1,k1);
   x.markAsTemporary();
   return x;
}   

uMatrix<Complex> uLlsSVD( const uMatrix<Complex>& A, const uMatrix<Complex>& b)
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);
   // Check supplied matrices for compatiability
   if (A.rows() != b.rows())
   {
      cerr<< "Matrix sizes do not match in uLlsSVD( const uMatrix<Complex>& A, const uMatrix<Complex>& b) \n ";
      abort();
   }
   // temporaries required by lapack
   int info, lda,ldb,lwork,m,n,nrhs,rank;
   double rcond;
   double *s, *rwork;
   Complex *work;
   


   //set up to call zgelss

   m=A.rows();
   n=A.columns();
   nrhs=b.columns();

  // cute arithmetic-if expressions for finding maxs and mins

   lda=(m > 1) ? m : 1; // max(1,m)
   
   ldb= (lda > n) ? lda: n; // max(1,m,n)

   rcond = -1.0;   // machine precision used
   
   int maxmn = ( m > n) ? m : n; // max(m,n)
   int minmn = (m > n) ? n : m; // min(m,n)
   s=new double[ minmn]; 
   rwork = new double[ (5*minmn-4 > 1)? 5*minmn-4 :1];    
   lwork=2*(2*minmn + ( nrhs > maxmn ?  nrhs :maxmn)); 
  
  /* both m > n and n> m conditions satisfied for lwork. lwork is made twice 
 the  minimum needed lwork 
*/
   // Other temporaries to store A and b and store the solution x
   uMatrix<Complex> A1(A);
   uMatrix<Complex> x(n,nrhs);
   uMatrix<Complex> b1(ldb,nrhs);
   

   uIndex j(0,1,m-1);
   uIndex k(0,1,nrhs-1);
   
   
   b1.insert( j, k, b);
    
   
   
   work= new Complex[lwork];
   A1.transpose(); // Fortran reads in column format
   b1.transpose();    
   zgelss_(&m,&n,&nrhs,A1.address(),&lda,b1.address(),&ldb,s,&rcond,&rank,work, &lwork,rwork,&info);
   if (info !=0)
   {
      if (info > 0)
	  {
	     cout << " uLlsSVD( const uMatrix<Complex>& A, const uMatrix<Complex>& b) failed to converge \n";
	     abort();
	  }
      else 
      {
	 cout << "uLlsSVD( const uMatrix<Complex>& A, const uMatrix<Complex>& b) error in parameter" << -1*info << endl;
	 abort();
      }
   }
   
   delete [] s;
   delete [] work;     
   delete [] rwork;
   
// Extract the solution from b1
   b1.transpose(); //Change format for C
   
   uIndex j1(0,1,n-1);
   uIndex k1(0,1,nrhs-1);
   x=b1(j1,k1);
   x.markAsTemporary();
   return x;
}   

/* SVD for general matrices */

void uSVD( const uMatrix<float>& A, uMatrix<float>& U,
	  uMatrix<float>& S, uMatrix<float>& V ) 
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);


   // temporaries required by lapack
   char jobu,jobvt;
   int info,lda,ldu,ldvt,lwork,m,n;
   float *work;
   uMatrix<float> A1(A);
   uMatrix<float> u(A.rows(),A.rows());
   uMatrix<float> v(A.columns(),A.columns());
   uMatrix<float> s (1,((A.rows() > A.columns()) ? A.columns(): A.rows())); 

   // Set up call to sgesvd
   jobu = 'A';
   jobvt = 'A';
   m = A.rows();
   n = A.columns();   
   lda = m > 1? m :1;
   ldu = m;
   ldvt = n;

   int maxmn = m >n ? m :n;
   int minmn = m > n ? n: m;
   lwork = 2*((3*minmn+maxmn)  > (5*minmn -4) ? (3*minmn+maxmn): (5*minmn -4));
   work = new float[lwork];   
   A1.transpose();
   u.transpose();
   v.transpose();
   
   sgesvd_( &jobu, &jobvt, &m, &n, A1.address(), &lda, s.address(),
	   u.address(), &ldu, v.address(), &ldvt, work, &lwork, &info);
   
   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uSVD(float) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uSVD(float) error in parameter " << -1*info << endl;
	 abort();
      }
   }

   U=transpose(u);
   V=v;
   S=s;
   
   
   delete[] work;

}
void uSVD( const uMatrix<fComplex>& A, uMatrix<fComplex>& U,
	  uMatrix<float>& S, uMatrix<fComplex>& V ) 
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);

   // temporaries required by lapack
   char jobu,jobvt;
   int info,lda,ldu,ldvt,lwork,m,n;
   fComplex *work;
   float *rwork;
   uMatrix<fComplex> A1(A);
   uMatrix<fComplex> u(A.rows(),A.rows());
   uMatrix<fComplex> v(A.columns(),A.columns());
   uMatrix<float> s (1,((A.rows() > A.columns()) ? A.columns(): A.rows())); 

   // Set up call to cgesvd
   jobu = 'A';
   jobvt = 'A';
   m = A.rows();
   n = A.columns();   
   lda = m > 1? m :1;
   ldu = m;
   ldvt = n;

   int maxmn = m >n ? m :n;
   int minmn = m > n ? n: m;
   lwork = 2*(2*minmn+maxmn);
   
   work = new fComplex[lwork];   

   rwork = new float[5*maxmn];
   
   A1.transpose();
   u.transpose();
   v.transpose();
   
   cgesvd_( &jobu, &jobvt, &m, &n, A1.address(), &lda, s.address(),
	   u.address(), &ldu, v.address(), &ldvt, work, &lwork, rwork, &info);
   
   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uSVD(fComplex) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uSVD(fComplex) error in parameter " << -1*info << endl;
	 abort();
      }
   }

   U = transpose(u);
   V = conj(v);
   S = s;
      
   delete[] work;
   delete[] rwork;
}

void uSVD( const uMatrix<double>& A, uMatrix<double>& U,
	  uMatrix<double>& S, uMatrix<double>& V ) 
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);

   // temporaries required by lapack
   char jobu,jobvt;
   int info,lda,ldu,ldvt,lwork,m,n;
   double *work;
   uMatrix<double> A1(A);
   uMatrix<double> u(A.rows(),A.rows());
   uMatrix<double> v(A.columns(),A.columns());
   uMatrix<double> s (1,((A.rows() > A.columns()) ? A.columns(): A.rows())); 

   // Set up call to dgesvd
   jobu = 'A';
   jobvt = 'A';
   m = A.rows();
   n = A.columns();   
   lda = m > 1? m :1;
   ldu = m;
   ldvt = n;

   int maxmn = m >n ? m :n;
   int minmn = m > n ? n: m;
   lwork = 2*((3*minmn+maxmn)  > (5*minmn -4) ? (3*minmn+maxmn): (5*minmn -4));
   work = new double[lwork];   
   A1.transpose();
   u.transpose();
   v.transpose();
   
   dgesvd_( &jobu, &jobvt, &m, &n, A1.address(), &lda, s.address(),
	   u.address(), &ldu, v.address(), &ldvt, work, &lwork, &info);
   
   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uSVD(double) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uSVD(double) error in parameter " << -1*info << endl;
	 abort();
      }
   }

   U=transpose(u);
   V=v;
   S=s;

   delete[] work;

}
void uSVD( const uMatrix<Complex>& A, uMatrix<Complex>& U,
	  uMatrix<double>& S, uMatrix<Complex>& V ) 
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);

   // temporaries required by lapack
   char jobu,jobvt;
   int info,lda,ldu,ldvt,lwork,m,n;
   Complex *work;
   double *rwork;
   uMatrix<Complex> A1(A);
   uMatrix<Complex> u(A.rows(),A.rows());
   uMatrix<Complex> v(A.columns(),A.columns());
   uMatrix<double> s (1,((A.rows() > A.columns()) ? A.columns(): A.rows())); 

   // Set up call to zgesvd
   jobu = 'A';
   jobvt = 'A';
   m = A.rows();
   n = A.columns();   
   lda = m > 1? m :1;
   ldu = m;
   ldvt = n;

   int maxmn = m >n ? m :n;
   int minmn = m > n ? n: m;
   lwork = 2*(2*minmn+maxmn);
   
   work = new Complex[lwork];   
   rwork = new double[5*maxmn];
   
   A1.transpose();
   u.transpose();
   v.transpose();
   
   zgesvd_( &jobu, &jobvt, &m, &n, A1.address(), &lda, s.address(),
	   u.address(), &ldu, v.address(), &ldvt, work, &lwork, rwork, &info);
   
   if ( info != 0 )
   {
      if ( info > 0 )
      {
	 cout << "void uSVD(Complex) failed to converge\n";
	 abort();
      }
      else
      {
	 cout << "void uSVD(Complex) error in parameter " << -1*info << endl;
	 abort();
      }
   }

   U=transpose(u);
   V=conj(v);
   S=s;
   
   
   delete[] work;
   delete[] rwork;

}

/* Condition number using SVD routine */

float uCond(const uMatrix<float> &A)  
{
  cout<< "uLlsSVD non ancora cambiata in nMatrix\n";
  exit(0);

   uMatrix<float> U,S,V;
   uSVD(A,U,S,V);
   int no_sing=S.columns();
   if (S(0,no_sing-1) == 0)
   {
      cerr << "Condition number infinite \n";
      abort();
   }
   
   float cond_no= S(0,0)/S(0,no_sing-1);
   return cond_no;
}

float uCond(const uMatrix<fComplex> &A)  
{
   uMatrix<fComplex> U,V;
   uMatrix<float> S;
   
   uSVD(A,U,S,V);
   int no_sing=S.columns();
   if (S(0,no_sing-1) == 0)
   {
      cerr << "Condition number infinite\n";
      abort();
   }
   
   float cond_no= S(0,0)/S(0,no_sing-1);
   return cond_no;
}
double uCond(const uMatrix<double> &A)  
{
   uMatrix<double> U,S,V;
   uSVD(A,U,S,V);
   int no_sing=S.columns();
   if (S(0,no_sing-1) == 0)
   {
      cerr << "Condition number infinite\n";
      abort();
   }
   
   double cond_no= S(0,0)/S(0,no_sing-1);
   return cond_no;
}

double uCond(const uMatrix<Complex> &A)  
{
   uMatrix<Complex> U,V;
   uMatrix<double> S;
   
   uSVD(A,U,S,V);
   int no_sing=S.columns();
   if (S(0,no_sing-1) == 0)
   {
      cerr << "Condition number infinite\n";
      abort();
   }
   
   double cond_no= S(0,0)/S(0,no_sing-1);
   return cond_no;
}


void uAXPY(fComplex a,  const uMatrix<fComplex>& X, const uMatrix<fComplex>& Y, uMatrix<fComplex>& Z)
{
  Z=Y;
  uAXPY(a,X,Z);
};
void uAXPY(Complex a,  const uMatrix<Complex>& X, const uMatrix<Complex>& Y, uMatrix<Complex>& Z )
{
  Z=Y;
  uAXPY(a,X,Z);
};
void uAXPY(double  a,  const uMatrix<double>& X, const uMatrix<double>& Y, uMatrix<double>& Z)
{
  Z=Y;
  uAXPY(a,X,Z);
};
void uAXPY(float a,  const uMatrix<float>& X, const uMatrix<float>& Y, uMatrix<float>& Z)
{
  Z=Y;
  uAXPY(a,X,Z);
};


void uAXPY(fComplex a,  const uMatrix<fComplex>& X, uMatrix<fComplex>& Y )
{
  static int n,incx=1,incy=1;
   if ( (n=X.rows()*X.columns()) != Y.rows()*Y.columns()   )
   {
      cerr << "uAXPY(fComplex.. : Arguments have incompatible sizes\n";
      abort();
   }
   caxpy_(&n,&a,X.address(),&incx,Y.address(),&incy);
};

void uAXPY(Complex a,  const uMatrix<Complex>& X, uMatrix<Complex>& Y )
{
  static int n,incx=1,incy=1;
   if ( (n=X.rows()*X.columns()) != Y.rows()*Y.columns()   )
   {
      cerr << "uAXPY(Complex.. : Arguments have incompatible sizes\n";
      abort();
   }
   zaxpy_(&n,&a,X.address(),&incx,Y.address(),&incy);
};


void uAXPY(double a,  const uMatrix<double>& X, uMatrix<double>& Y )
{
  static int n,incx=1,incy=1;
  if ( (n=X.rows()*X.columns()) != Y.rows()*Y.columns()   )
    {
      cerr << "uAXPY(double.. : Arguments have incompatible sizes\n";
      abort();
    }
  daxpy_(&n,&a,X.address(),&incx,Y.address(),&incy);
};

void uAXPY(float a,  const uMatrix<float>& X, uMatrix<float>& Y )
{
  static int n,incx=1,incy=1;
  if ( (n=X.rows()*X.columns()) != Y.rows()*Y.columns()   )
    {
      cerr << "uAXPY(float.. : Arguments have incompatible sizes\n";
      abort();
    }
  saxpy_(&n,&a,X.address(),&incx,Y.address(),&incy);
};



void uGemm(fComplex alpha, const uMatrix<fComplex>& A,
	                   const uMatrix<fComplex>& B,fComplex &beta,
                           uMatrix<fComplex>&        C )
{
  if ( A.columns() != B.rows())
    {
      cerr << "uGemm(<fComplex> : incompatible sizes\n";
      abort();
    }
  C.resize(A.rows(), B.columns());
   static int lda, ldb, ldc, m, n,   k ;

   lda = A.rows();
   ldb = B.rows(); 
   ldc = C.rows();
   m = A.rows();
   n = B.columns();
   k = B.rows();
   
   char transa='N', transb='N';

   cgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
};

void uGemm(Complex alpha, const uMatrix<Complex>& A,
	                   const uMatrix<Complex>& B,Complex beta,
                           uMatrix<Complex>&        C )
{
  if ( A.columns() != B.rows() )
    {
      cerr << "uGemm(<Complex> : incompatible sizes\n";
      abort();
    }
  C.resize(A.rows(), B.columns());
  static int lda, ldb, ldc, m, n,   k ;
  
  lda = A.rows();
  ldb = B.rows(); 
  ldc = C.rows();
  m = A.rows();
  n = B.columns();
  k = B.rows();
  
   char transa='N', transb='N';
   
   zgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
}
;

void uGemm(double alpha, const uMatrix<double>& A,
	                   const uMatrix<double>& B,double  beta,
                           uMatrix<double>&        C )
{
  if ( A.columns() != B.rows() )
    {
      cerr << "uGemm(<double> : incompatible sizes\n";
      abort();
    }
  C.resize(A.rows(), B.columns());
   static int lda, ldb, ldc, m, n,   k ;

   lda = A.rows();
   ldb = B.rows(); 
   ldc = C.rows();
   m = A.rows();
   n = B.columns();
   k = B.rows();
   
   char transa='N', transb='N';

   dgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
}
;

void uGemm(float alpha, const uMatrix<float>& A,
	                   const uMatrix<float>& B,float   beta,
                           uMatrix<float>&        C )
{
  if ( A.columns() != B.rows())
    {
      cerr << "uGemm(<float> : incompatible sizes\n";
      abort();
    }
  C.resize(A.rows(), B.columns());
   static int lda, ldb, ldc, m, n,   k ;

   lda = A.rows();
   ldb = B.rows(); 
   ldc = C.rows();
   m = A.rows();
   n = B.columns();
   k = B.rows();
   
   char transa='N', transb='N';

   sgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
};




void  JoinV( const  uMatrix<Complex> &  A, const uMatrix<Complex>  &  B,  uMatrix<Complex>  &  C)
{
  static int one=1;
  int nc;
  nc = A.columns();
  if(nc != B.columns() )
    {
      printf(" matrices incompatibles dans JoinV \n");
      exit(0);
    }

  int nr1,nr2,nr12,nr1D,nr2D;

  nr1= A.rows() ;
  nr2= B.rows() ;
  nr12=nr1+nr2;

  nr1D=2*nr1;
  nr2D=2*nr2;

  C.resize(nr1+nr2,nc);

  Complex *a,*b,*c,*c_offset;
  a=A.address();
  b=B.address();
  c=C.address();
  
  int i;
  for( i=0; i< nc; i++)
    {
      c_offset=c;
      dcopy_(&nr1D , (double*) a, &one,  (double*) c_offset,&one ); // blas
      c_offset+=nr1;
      dcopy_(&nr2D , (double*) b, &one,  (double*) c_offset,&one ); // blas
      (a+=nr1);
      (b+=nr2);
      (c+=nr12);
    }
}

void  JoinH( const  uMatrix<Complex> &  A , const uMatrix<Complex>  &  B,  uMatrix<Complex>  &  C)
{
  static int one=1;
  if(A.rows() != B.rows() )
    {
      printf(" matrices incompatibles dans JoinH \n");
      exit(0);
    }


  C.resize(A.rows(),A.columns()+B.columns());

  Complex *a,*b,*c,*c_offset;

  a=A.address();
  b=B.address();
  c=C.address();

  int sizea,sizeb,sizea2,sizeb2;

  sizea= A.columns()*A.rows();
  sizeb= B.columns()*B.rows();

  sizea2=2*sizea;
  sizeb2=2*sizeb;

      
  c_offset=c;
  
  dcopy_(&sizea2 , (double*) a, &one,  (double*) c_offset,&one ); // blas
  
  c_offset+=sizea;
  
  dcopy_(&sizeb2 , (double*) b, &one,  (double*) c_offset,&one ); // blas

}







void  JoinV( const  uMatrix<double> &  A, const uMatrix<double>  &  B,  uMatrix<double>  &  C)
{
  static int one=1;
  int nc;
  nc = A.columns();
  if(nc != B.columns() )
    {
      printf(" matrices incompatibles dans JoinV \n");
      exit(0);
    }

  int nr1,nr2,nr12,nr1D,nr2D;

  nr1= A.rows() ;
  nr2= B.rows() ;
  nr12=nr1+nr2;

  nr1D=nr1;
  nr2D=nr2;

  C.resize(nr1+nr2,nc);

  double *a,*b,*c,*c_offset;
  a=A.address();
  b=B.address();
  c=C.address();
  
  int i;
  for( i=0; i< nc; i++)
    {
      c_offset=c;
      dcopy_(&nr1D , (double*) a, &one,  (double*) c_offset,&one ); // blas
      c_offset+=nr1;
      dcopy_(&nr2D , (double*) b, &one,  (double*) c_offset,&one ); // blas
      (a+=nr1);
      (b+=nr2);
      (c+=nr12);
    }
}

void  JoinH( const  uMatrix<double> &  A , const uMatrix<double>  &  B,  uMatrix<double>  &  C)
{
  static int one=1;
  if(A.rows() != B.rows() )
    {
      printf(" matrices incompatibles dans JoinH \n");
      exit(0);
    }


  C.resize(A.rows(),A.columns()+B.columns());

  double  *a,*b,*c,*c_offset;

  a=A.address();
  b=B.address();
  c=C.address();

  int sizea,sizeb,sizea2,sizeb2;

  sizea= A.columns()*A.rows();
  sizeb= B.columns()*B.rows();

  sizea2=sizea;
  sizeb2=sizeb;

      
  c_offset=c;
  
  dcopy_(&sizea2 , (double*) a, &one,  (double*) c_offset,&one ); // blas
  
  c_offset+=sizea;
  
  dcopy_(&sizeb2 , (double*) b, &one,  (double*) c_offset,&one ); // blas

}









