/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */
# define SXPARAMS_VERSION      "sxparams : V1.02 Peter Boesecke 2011-06-07"
/*---------------------------------------------------------------------------
NAME

   sxparams.c --- Recalculates sx parameters for different orientations

USAGE 

   sxparams [<options>] <output orientation>
     <pro> <ori> <axis1> <axis2> <dim1> <dim2> <off1> <off2> <bis1> <bis2>
     <ras1> <ras2> <pix1> <pix2> <cen1> <cen2> <dis> <rot1> <rot2> <rot3> <wvl>
     [<bcen1> <bcen2> <bdis> <tilt1> <tilt2> <tilt3>]

OPTIONS
   
   -h                 : help
   debug=<debug-code> : verbose=0x1,level=0x6,data=0x8,temp=0x10,sx=0x20
   rot=0|1
   input=<input file name>
   head=<number of head lines to write>

DESCRIPTION

   Rasparams recalculates sx parameters for different orientations.
   All arguments following the <output orientation> are are successively 
   read from argv[0], argv[1], ... until all possible parameters have 
   been read or to the end of the argument list. If no arguments follow
   <output orientation> the parameters are read from stdin.
   If input=<input file name> is used the parameters are first tried to
   read from the argument list and then read line by line from the
   specified input file. Everything following a hash (#) sign is ignored.

   The output orientation must be specified as the first argument on the
   command line:

     <output orientation>: 1-16

   The parameters must be written in a single line in the following order,
   either as arguments on the command line or in single lines of a file.
   To omit parameters they can be marked with ..., - or a single dot.

     <pro> <ori> <axis1> <axis2> <dim1> <dim2> <off1> <off2> <bis1> <bis2>
     <ras1> <ras2> <pix1> <pix2> <cen1> <cen2> <dis> <rot1> <rot2> <rot3> <wvl>
     [<bcen1> <bcen2> <bdis> <tilt1> <tilt2> <tilt3>]

    pro               projection (1,2), ("saxs"|"waxs"), ("no"|"ewald")
    ori               orientation number (1-16)
    axis1 axis2       axis types (1,2,3), ("distance"|"angle"|"numerator")
    dim1 dim2         dimensions of 2d array 
    off1 off2         offsets of array coordinates
    bis1 bis2         binning sizes
    ras1 ras2         raster region of 2d array 
    pix1 pix2         pixel sizes [m]
    cen1 cen2         PONI (point of normal incidence)
    dis               distance sample-PONI [m]
    rot1 rot2 rot3    detector rotations [rad]
    wvl               wavelength [m]
    bcen1 bcen2       beam center (alt. cen1, cen2)
    bdis              distance sample-bcen [m] (alt. dis) 
    tilt1 tilt2 tilt3 detector tilts [rad]

EXAMPLE

  Convert parameters from orientation 4 to orientation 1:

++++++++++++++++++

  a)

  sxparams 1 1 4 1 2 500 400 52 41 1 1 - - 5e-05 4e-05 200 300 0.1 0 0.4 0 0.1_nm

  # pro ori axis1 axis2 dim1 dim2 off1 off2 bis1 bis2 ras1 ras2 pix1 pix2 \
    cen1 cen2 dis rot1 rot2 rot3 wvl bcen1 bcen2 bdis tilt1 tilt2 tilt3
  1 1 1 2 500 400 52 41 1 1 - - 5e-05 4e-05 \
    404 182 0.1 0 -0.4 0 1e-10 -441.586 182 0.10857 -1.5708 0.4 0

  b) from pipe

  sxparams 2 1 4 1 1 500 400 52 41 1 1 - - 5e-05 4e-05 \
    200 300 0.1 0 0.4 0 0.1_nm | sxparams 1

  # pro ori axis1 axis2 dim1 dim2 off1 off2 bis1 bis2 ras1 ras2 pix1 pix2 \
    cen1 cen2 dis rot1 rot2 rot3 wvl bcen1 bcen2 bdis tilt1 tilt2 tilt3
  1 1 1 1 500 400 52 41 1 1 - - 5e-05 4e-05 \
    404 182 0.1 0 -0.4 0 1e-10 -441.586 182 0.10857 -1.5708 0.4 0 

  c) from file

  sxparams input=sxparams.dat 1
  ...

  The beam parameters (bcen1, bcen2, bdis) and tilts (tilt1, tilt2, tilt3) 
  are calculated alternatively to cen1, cen2, dis, rot1, rot2 and rot3 which 
  have priority during the calculation. To use only beam parameter and tilts
  the parameters cen1, cen2, dis, rot1, rot2 and rot3 must not be set.

  In this case all parameters cen1, cen2, dis, rot1, rot2 and rot3 are
  calculated from bcen1, bcen2, bdis, tilt1, tilt2 and tilt3. Internally,
  the calculations are stil done with cen1, ..., rot3. At the end bcen1, 
  ..., tilt3 are recalculated.

RETURN VALUE
    
    0 in case of succes, otherwise or error value different from 0

HISTORY
    
---------------------------------------------------------------------------*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/
# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <stdarg.h>
# include <string.h>
# include <ctype.h>
# include <limits.h>
# include <errno.h>
# include <fcntl.h>
# include <math.h>
# include <float.h>
  
# include "strlib.h"
# include "numio.h" 
# include "raster.h" 
# include "sx.h" 

/****************************************************************************
*  Define                                                                   *
****************************************************************************/
# define Usage "sxparams [<options>] <output orientation>\n"\
"  <pro> <ori> <axis1> <axis2> <dim1> <dim2> <off1> <off2> <bis1> <bis2>\n"\
"  <ras1> <ras2> <pix1> <pix2> <cen1> <cen2> <dis> <rot1> <rot2> <rot3> <wvl>\n"\
" [<bcen1> <bcen2> <bdis> <tilt1> <tilt2> <tilt3>]\n"\
"\n "SXPARAMS_VERSION"\n\n"\
" Change of raster orientation and calculation of PoNI, rotations, \n"\
" beam center and tilts. The orientation can be given as a number (1-16)\n"\
" or a 3-tupel: first, second, third direction, e.g. 1,-2,3.\n"\
" The parameters must be given in the specified order.\n" \
" Parameters can be omitted by giving a minus sign \"-\".\n" \
" cen, dis and rots have precedence over bcen, bdis and tilts.\n" \
"\n"\
" -h help\n"\
" -v print version"

# define STRINGVSIZE 128 

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_arc
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{ static const char * Main_Error="ERROR: sxparams";

  char *nul = (char *) NULL;
  char **pargv;
  char *ps;
  int debug=0;
  int rot=0;

  char *finput=NULL;
  FILE * input=NULL;

  int ic;

  int errval=-1;
  long value=-1l;
  int cnt=0;
  long N=3;

  SXL oori;
  SXParams params_in, params_out;

  char buffer[SXBUFLEN];
  size_t buflen=SXBUFLEN;

  char errbuf[SXBUFLEN];

  char *string;

  char *stringv[STRINGVSIZE];
  int  stringc;

  int head=1;

  oori.V = 0l; oori.I = 0;

  if (argc<=1) { 
    printf("%s\n",Usage); 
    return(-1); 
  }

  pargv=&argv[1];
  while ( *pargv!=nul ) {

    if ( strncmp( *pargv,"debug=", 6 ) == 0 ) {
      // set debug level
      ps=*pargv+6;
      debug = (int) num_str2double ( ps, NULL, &errval);
      sx_debug_set(debug);
    } else  if ( strncmp( *pargv,"rot=", 4 ) == 0 ) {
      ps=*pargv+4;
      rot = (int) num_str2double ( ps, NULL, &errval);
    } else  if (( strncmp( *pargv,"-h", 2 ) == 0 )||
                ( strncmp( *pargv,"--h", 3 ) == 0 )) {
      // write help
      ps=*pargv+2;
      printf("%s\n",Usage); 
      printf("\n options:\n");
      printf("  input=<input file name>\n");
      printf("   head=<number of head lines to write>\n");
      printf("  debug=<%s>\n",sx_usage2str());
      return(-1);
    } else  if (( strncmp( *pargv,"-v", 2 ) == 0 )||
                ( strncmp( *pargv,"--v", 3 ) == 0 )) {
      // write version
      printf("  %s\n",SXPARAMS_VERSION);
      printf("    %s\n",REFERENCE_VERSION);
      printf("    %s\n",strlib_version ());
      printf("    %s\n",numio_version ());
      printf("    %s\n",raster_version ());
      printf("    %s\n",rot3d_version ());
      printf("    %s\n",r2t_version ());
      printf("    %s\n\n",sx_version ());
      return(-1);
    } else if ( strncmp( *pargv,"input=", 6 ) == 0 ) {
      // set input file
      finput=*pargv+6;
    } else if ( strncmp( *pargv,"head=", 5 ) == 0 ) {
      // set number of head lines 
      head=(int) num_str2long ( *pargv+5, NULL, &errval );
      if (errval) goto main_error;
    } else if ( cnt == 0 ) { // oori
      if (sx_debug()&SX_DEBUG) fprintf( stdout, "reading output orientation from >>%s<<\n",*pargv );
      if (strlib_is_no_skip(*pargv)) {
        oori.V = raster_str2number( N, *pargv, NULL, &errval );
        if (errval) goto main_error;
        oori.I = 1;
      }
      cnt++;
    } else if ( cnt == 1 ) { // read parameters
      if (sx_debug()&SX_DEBUG) fprintf( stdout, "reading parameters\n" );
      if (!sx_rd_params ( &params_in, pargv, &errval ))
        goto main_error;
      cnt++;

      if (!sx_tf_params ( &params_out, &params_in, oori.V, rot, &errval ))  {
        printf ("sx_tf_params: %s\n" ,sx_errval2str ( buffer, buflen, errval ));
        goto main_error;
      }

      sx_pr_params_line( stdout, &params_out, head-- ); fprintf(stdout,"\n");

    }

    if (*pargv) pargv++;

  } // while

  // read input file if no parameters have been read so far, or input is explicitely given
  if ( (cnt <= 1)||(finput) ) {
    if (finput) {
      if (sx_debug()&SX_DEBUG) fprintf( stdout, "opening >>%s<< for reading\n", finput );
      if ( !( input=fopen(finput,"r") ) ) {
        sprintf(errbuf,"fopen(%s,\"r\")",finput);
        perror(errbuf); goto main_error;
      }
    } else input=stdin;

    while ( !feof( input ) ) {

      string = fgets ( buffer, SXBUFLEN, input );

      if ( string ) {

        if (sx_debug()&SX_DEBUG) fprintf( stdout, "input line >>%s<<\n", string );

        // quit if string is empty and read from stdin
        if ((strlib_is_empty( string ))&&(strlib_is_empty(finput)))
          break; 

        // remove comments, leading and trailing white spaces, collapse rest
        strlib_collapse( strlib_trim( strlib_uncomment( string ) ) );

        if (sx_debug()&SX_DEBUG) fprintf( stdout, "clean line >>%s<<\n", string );

        if ( (!strncmp(string,"exit",4)) || (!strncmp(string,"quit",4)) )
          break;

        if (!strlib_is_empty( string )) {

          // break string at white spaces
          stringc = strlib_split ( stringv, STRINGVSIZE, string );

          if (!sx_rd_params ( &params_in, stringv, &errval ))
            goto main_error;

          if (!sx_tf_params ( &params_out, &params_in, oori.V, rot, &errval ))  {
            printf ("sx_tf_params: %s\n" ,sx_errval2str ( buffer, buflen, errval ));
            goto main_error;
          }

          sx_pr_params_line( stdout, &params_out, head-- ); fprintf(stdout,"\n");
        } // strlib_is_empty

      } else break;

    } // while

    // fclose( input )
    if (finput) { // close input only if input is not stdin
      if (sx_debug()&SX_DEBUG) fprintf( stdout, "closing >>%s<<\n", finput );
      fclose( input );
    }

  } // if (cnt<=1)

  errval = 0;

  return( errval );

main_error:

  // fclose( input )
  if (input) {
    if (sx_debug()&SX_DEBUG) fprintf( stdout, "closing >>%s<<\n", finput );
    fclose( input );
  }

  printf("%s : errval=%d\n",Main_Error, errval);

  return( errval );

} /* MAIN */
