/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_uniax
  Transformation of an isotropic waxs-projection to a rotational symmetric 
  waxs-projection. The input projection type of the pattern must be Waxs.
  To force an input image to Waxs the option -i1pro Waxs can be used. 
  Depending on the option styp the symmetry axis is either axis 1 
  (-styp 1) or axis 2 (-styp 2, default). The option -styp 0 (isotropic 
  scattering) copies the input to the output image without projection.

  The input pattern is first back-projected to the Ewald-sphere and then
  reprojected into the system of the symmetry axis by rotating it around 
  the axis of rotational symmetry. In case of -styp 0 the two projections are
  inverse to each other and the input image will only be copied to the 
  output image. 

  The scattering vectors are calculated with saxs-coordinates (-rsys saxs).

2 PURPOSE

  Arguments:
  saxs_uniax [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <styp> <sro1> <sro2> <sro3>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  Axis of rotational symmetry

  All rotations and directions are defined for orientation 1.
  The waxs projection in the s_1 | s_2 plane is transformed to a waxs
  projection corresponding to an axis of rotational symmetry around axis 1
  or axis 2 depending on the option styp:

  <styp>            : 0 == isotropic scattering,
                      1 == rotational symmetry around axis 1,
                      2 == rotational symmetry around axis 2 (default)

  In analogy to the detector rotations three rotation axis of rotation can 
  be used to specify the axis of symmetry.

  <sro1>            : <ccw rotation around direction 1> (default 0_deg)
  <sro2>            : <ccw rotation around direction 2> (default 0_deg)
  <sro3>            : <ccw rotation around direction 3> (default 0_deg)

  The input pattern is transformed into the coordinate system of the symmetry
  axis, i.e. the intensity along axis 1 is projected to axis s_1 and the 
  intensity along axis 2 is projected to axis s_2.

  To undo this projection the option -do can be given (default is +do).

2 HISTORY
  2001-12-11  PB V4.0 Peter Boesecke from saxs_waxs V4.15 
  2005-07-15  PB V4.01 ReportSaxsStatus parameter list updated
                       SetSaxsErrorExternal added
  2005-09-17  PB V4.02 VarDat
  2007-04-23  PB V4.03 function concat corrected (to avoid buffer ovfl)
  2007-06-11  PB V4.04 ImageLoop parameters updated
  2010-03-08  PB V4.05 Transform, Inverse removed (obviously unused)
  2010-05-27  PB V4.06 saxs_uniax: I0Projection, I1Projection defined (was not done)
  2010-05-31  PB V4.07 update for waxs.c V1.10
  2011-06-07  PB V4.08 strncat->strncpy
  2011-06-30  PB V4.09 update for waxs.c V1.11,
                       concat->strlib_concat 
  2012-01-31  PB V4.10 ImageLoop parameters updated

---*/
# define Version  "saxs_uniax V4.10 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <string.h>

# include "SaxsPrograms.h"
# include "waxs.h"
# include "strlib.h"

# define FREE( a) if ( a) free( a)

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc>\n\
                <odum> <odim1> <odim2> <styp> <sro1> <sro2> <sro3>\n\n\
                Rotational symmetric projection of a WAXS pattern\n"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SXyname "xyna"  /* line */  // xy-file name
# define SXy   "xy"      /* flag */  // xy-file flag
# define SXypix "xypi"   /* flag */
# define SSymType "styp" /* long */
# define SSymRot1 "sro1" /* float */
# define SSymRot2 "sro2" /* float */
# define SSymRot3 "sro3" /* float */
# define SSymDo   "do"   /* flag */  // +do (default) or undo (-do)

/*---------------------------------------------------------------------------
1 saxs_uniax

2 PURPOSE
  Projection of an inclined detector surface
---------------------------------------------------------------------------*/
void saxs_uniax (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ // unused const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  const char * RoutineName = "saxs_uniax";
  int   i, imax; 
  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float Sum, Weight, Value;
  float VarSum, VarWeight, E0Value;

  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float *pI1Data;
//unused float *pE1Data;
  float I1Dummy, I1DDummy;
//unused  float VarDDummy=DDSET(VarDummy);

  float rot1, rot2, rot3;
  long  I0Projection, I1Projection;

  int i_1, i_2;

  float f0_1, f0_2;
  float f1_1, f1_2;
//unused  float W1_1, W1_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  WaxsCoord W0, W1;
  double K;                                                       

  WParams I1wparams, I0wparams;

  /*++++++++++ int Inverse, Transform; apparently not used */

 // xyfile parameters
  char xypath[IO_len], xyname[IO_len];
  char xy_pattern[IO_len], xyfull[IO_len];
  char xfile[IO_len], yfile[IO_len];
  int xstream, ystream;
  float *xdata, *ydata;
  float *pxdata, *pydata;
  int idata;
  long xynum;
  size_t xysize;
  int ErrorValue;

  //unused IO_long Inv;
  IO_long Xy, Xypix;
  IO_line Xyname;

  IO_long  SymType, SymDo;
  IO_float SymRot1, SymRot2, SymRot3;

  char key[IO_len];

  float I0Off_1, I0Ps_1;
  float I0Off_2, I0Ps_2;
  float I1Off_1, I1Ps_1;
  float I1Off_2, I1Ps_2;
  //unused float xybis_1, xybis_2;
  float xypix_1, xypix_2;
  float xycen_1, xycen_2;
  float xydis, xywvl;
  float xyrot_1, xyrot_2, xyrot_3;

  SetSaxsError( Version, RoutineName, NULL );
 
  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,%ld] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,%ld] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- Get option values */ 
  Xy = option_flag( SXy, pcb, pstatus);
  if (*pstatus!=Success) return;
  Xypix = option_flag( SXypix, pcb, pstatus);
  if (*pstatus!=Success) return;
  Xyname = option_line ( SXyname, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* check the reference system: Saxs required */
  if (pcb->RSys.V != IO_Saxs) {
    printf("ERROR: Reference system %s required\n",reftostr(IO_Saxs));
    printf("       Start program with option -rsys %s\n",reftostr(IO_Saxs)); 
    }

  // open xy-files
  xyname[0] = xypath[0] = '\0';
  xfile[0]  = yfile[0]  = '\0';
  xstream   = ystream   = -1;
  xdata     = ydata     = (float *) NULL;
  if (Xy.V) {
    filename_parameter (xy_pattern,IO_len,Xyname.V,0);
    filename_pattern (xy_pattern,IO_len,xy_pattern,pcb->ib[0].FileNumber);
    filename_full ( xyfull, IO_len, xy_pattern );
    filename_path ( xypath, IO_len, xyfull );
    filename_name ( xyname, IO_len, xyfull );
    xysize = edf_dim_product ( ihb[1].Dim ); // identical to input array
    // open x-file
    sprintf(xfile,"%s%s%s",xypath,"x",xyname);
    xstream = edf_search_stream( xfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return; 
    if (xstream>=0) { *pstatus = FileAlreadyOpened; return; }
    xstream = edf_open_data_file ( xfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
    if (xysize>0) {
      if (!(xdata = (float *) malloc(sizeof(float)*xysize))) {
        *pstatus = NotEnoughMemoryAvailable; return; }
      pxdata = xdata;
      for (idata=0;idata<xysize;idata++) *(pxdata++)=0.0;
    }
    // open y-file
    sprintf(yfile,"%s%s%s",xypath,"y",xyname);
    ystream = edf_search_stream( yfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (ystream>=0) { FREE(xdata); *pstatus = FileAlreadyOpened; return; }
    ystream = edf_open_data_file ( yfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (xysize>0) {
      if (!(ydata = (float *) malloc(sizeof(float)*xysize))) {
        FREE(xdata); *pstatus = NotEnoughMemoryAvailable; return; }
      pydata = ydata;
      for (idata=0;idata<xysize;idata++) *(pydata++)=0.0;
    }
  }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;
 
  K    = (double) WAVENUMBER(ihb[1].WaveLength.V);
  rot1 = (double) ihb[1].DetRot[1].V;
  rot2 = (double) ihb[1].DetRot[2].V;
  rot3 = (double) ihb[1].DetRot[3].V;
 
  if (ihb[0].Projection.V !=IO_ProWaxs) {*pstatus = NotWaxsProjection; return;}
  if (ihb[1].Projection.V !=IO_ProWaxs) {*pstatus = NotWaxsProjection; return;} 

  I1Projection = ihb[1].Projection.V;
  I0Projection = ihb[0].Projection.V;
  
  waxs_Init ( &I1wparams, K, rot1, rot2, rot3 );

  waxs_Init ( &I0wparams,
              (double) WAVENUMBER(ihb[0].WaveLength.V),
              (double) ihb[0].DetRot[1].V,
              (double) ihb[0].DetRot[2].V,
              (double) ihb[0].DetRot[3].V );

  SymDo   = option_flag( SSymDo, pcb, pstatus);
  if (*pstatus!=Success) return;
  SymType = option_long  ( SSymType, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  SymRot1 = option_float ( SSymRot1, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  SymRot2 = option_float ( SSymRot2, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  SymRot3 = option_float ( SSymRot3, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* Symmetry operations are used for waxs projection coordinates. Because 
     input and output projections are both Waxs projections, both parameter 
     sets must be initialized */
  waxs_SymInit ( &I1wparams, (int) SymType.V, 
                 (double) SymRot1.V, (double) SymRot2.V, (double) SymRot3.V );
  waxs_SymInit ( &I0wparams, (int) SymType.V, 
                 (double) SymRot1.V, (double) SymRot2.V, (double) SymRot3.V );
 
  SetSaxsError( Version, RoutineName, NULL );

  if (pcb->TestBit) {
    waxs_PrintParams ( stdout , I1wparams );
    printf("\n"); }

  /* loop over the output array  */
  for (i_2=0;i_2<I0Dim_2;i_2++) {
    W0.s_2 =  (double) WORLD( i_2, Off_2[0], Ps_2[0] );
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    for (i_1=0;i_1<I0Dim_1;i_1++) {
      W0.s_1 =  (double) WORLD( i_1, Off_1[0], Ps_1[0] ); 

      if (SymDo.V)
        W1 = waxs_Uni2Iso ( &I0wparams, &I1wparams, W0 );
      else W1 = waxs_Iso2Uni ( &I0wparams, &I1wparams, W0 );

      if (!W1.status) { 

        f1_1 = INDEX(W1.s_1, Off_1[1], Ps_1[1]);
        f1_2 = INDEX(W1.s_2, Off_2[1], Ps_2[1]);

        if ( E0Data ) {
          if ( Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                 f1_1-0.5, f1_2-0.5, f1_1+0.5, f1_2+0.5, 
                 &Sum, &Weight, &VarSum, &VarWeight) ) {
            /* then do something with the data */
            Value = Sum/Weight;
            UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
            if (VarSum>=0.0) {
              E0Value = VarSum/(VarWeight*VarWeight);
              UPDATE( *pE0Data, E0Value, I0Dummy, I0DDummy );
            }
          } /* if Isum ... */
        } else {
          if (Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f1_1, f1_2, &Value)) {
            /* then do something with the data */
            UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

          } /* if (Ipol2d ... */
        } /* if E0Data */

      } /* if (!W1.status) */

      pI0Data++;
      pE0Data++;
    } /* for i_1 ... */
  } /* for i_2 ... */

  /* loop over the input array  */
  if (Xy.V) {
//    /* The displacements are expressed in image coordinates */ 
//    /* calculate I0 and I1 image coordinates */
//    IMAGEREF(I0Off_1,I0Ps_1,ihb[0].Offset[1].V);
//    IMAGEREF(I0Off_2,I0Ps_2,ihb[0].Offset[2].V);
//    IMAGEREF(I1Off_1,I1Ps_1,ihb[1].Offset[1].V);
//    IMAGEREF(I1Off_2,I1Ps_2,ihb[1].Offset[2].V);

    /* The displacements are expressed in array coordinates */ 
    /* calculate I0 and I1 array coordinates */
    ARRAYREF(I0Off_1,I0Ps_1);
    ARRAYREF(I0Off_2,I0Ps_2);
    ARRAYREF(I1Off_1,I1Ps_1);
    ARRAYREF(I1Off_2,I1Ps_2);

    for (i_2=0;i_2<I1Dim_2;i_2++) {
      W1.s_2 =  (double) WORLD( i_2, Off_2[1], Ps_2[1] );
      pI1Data = ABSPTR(I1Data,I1Dim_1,I1Dim_2,0,i_2);
      for (i_1=0;i_1<I1Dim_1;i_1++) {
        W1.s_1 =  (double) WORLD( i_1, Off_1[1], Ps_1[1] );
  
        if (SymDo.V)
          W0 = waxs_Uni2Iso ( &I1wparams, &I0wparams, W1 );
        else W0 = waxs_Iso2Uni ( &I1wparams, &I0wparams, W1 );

        if (!W0.status) {
  
          /* calculate I0 indices */
          f0_1 = INDEX(W0.s_1, Off_1[0], Ps_1[0]);
          f0_2 = INDEX(W0.s_2, Off_2[0], Ps_2[0]);

          pxdata = xdata + (pI1Data - I1Data);
          *pxdata = WORLD(f0_1, I0Off_1, I0Ps_1) - WORLD(i_1, I1Off_1, I1Ps_1);
          pydata = ydata + (pI1Data - I1Data);
          *pydata = WORLD(f0_2, I0Off_2, I0Ps_2) - WORLD(i_2, I1Off_2, I1Ps_2);
  
        } /* if (!W0.status) */

        pI1Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */
  } /* if (Xy.V) */

  // write and close xy-files
  if (Xy.V) {
    // xfile
    xynum = ihb[0].ImNum;
    printf("Writing : %s%s, Image :  %ld\n","x",xyname,xynum);
    edf_write_header_history ( xstream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    /* The xy-displacement files cover the same region as the input image, 
       i.e. dimensions, offsets and binning sizes are the same. */
    edf_write_header_float(xstream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(xstream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
      edf_write_header_float(xstream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }
    /* Normally, the geometrical parameters of the input image and of the
       xy-displacement output images should be identical. If Xypix is set
       PixSiz, Center and SampleDistance of the output image are written 
       to the xy-files. The output binning size and the geometrical
       parameters of the output image are always written to keys starting
       with the prefix KDisplaced followed by the corresponding keyword, e.g.  
       Psize_1 -> OutputPsize_1. */

    if (Xypix.V) {
      xypix_1 = ihb[0].PixSiz[1].V; xypix_2 = ihb[0].PixSiz[2].V;
      xycen_1 = ihb[0].Center[1].V; xycen_2 = ihb[0].Center[2].V;
      xydis   = ihb[0].SampleDistance.V;
      xywvl   = ihb[0].WaveLength.V;
      xyrot_1 = ihb[0].DetRot[1].V; xyrot_2 = ihb[0].DetRot[2].V; 
      xyrot_3 = ihb[0].DetRot[3].V;
    } else {
      xypix_1 = ihb[1].PixSiz[1].V; xypix_2 = ihb[1].PixSiz[2].V;
      xycen_1 = ihb[1].Center[1].V; xycen_2 = ihb[1].Center[2].V;
      xydis   = ihb[1].SampleDistance.V;
      xywvl   = ihb[1].WaveLength.V;
      xyrot_1 = ihb[1].DetRot[1].V; xyrot_2 = ihb[1].DetRot[2].V; 
      xyrot_3 = ihb[1].DetRot[3].V;
    }
    edf_write_header_float(xstream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_line(xstream,xynum,1l,
                          KProjection,protostr(I1Projection),
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    /* Write parameters of transformed image to x-displacement file */
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
      strlib_concat(key,IO_len,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
      strlib_concat(key,IO_len,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_line(xstream,xynum,1l,
           strlib_concat(key,IO_len,KDisplaced,KProjection),protostr(I0Projection),
           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    edf_write_data( xstream,xynum,1l,ihb[1].Dim,xdata,MFloat, 
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    FREE(xdata);
    edf_close_data_file ( xstream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    // yfile
    printf("Writing : %s%s, Image :  %ld\n","y",xyname,xynum);
    edf_write_header_history ( ystream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(ystream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
      edf_write_header_float(ystream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_line(ystream,xynum,1l,
                           KProjection,protostr(I1Projection),
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    /* Write parameters of transformed image to y-displacement file */
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
      strlib_concat(key,IO_len,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
      strlib_concat(key,IO_len,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_line(ystream,xynum,1l,
           strlib_concat(key,IO_len,KDisplaced,KProjection),protostr(I0Projection),
           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    edf_write_data( ystream, xynum, 1l, ihb[1].Dim, ydata, MFloat, 
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_close_data_file ( ystream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    FREE(ydata);
  }

} /* saxs_uniax*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_len];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  //unused int errval;
  IO_lvexp *pSymType;
  IO_flexp *pSymRot1, *pSymRot2, *pSymRot3; // rotation of symmetry axis

  /*--- Determine program name without directory ---*/
  filename_name ( progfn, IO_len,  pcb->argv[0] );

  /*--- Show usage if no arguments are given ---*/
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*---       specification of the axis of rotational symmetry */

  /*--- Argument  : symmetry axis */
  pSymType = (IO_lvexp*) option_parameter_search( SSymType, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_lvexp(pcb," rotational symmetry axis: 1 (axis 1), 2 (axis 2)",
             pSymType,pSymType->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation of symmetry axis around axis 1 */
  pSymRot1 = (IO_flexp*) option_parameter_search( SSymRot1, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb," first rotation: ccw around axis 1 (rad or \'_\' and unit)",
             pSymRot1,pSymRot1->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation of symmetry axis around axis 2 */
  pSymRot2 = (IO_flexp*) option_parameter_search( SSymRot2, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb," first rotation: ccw around axis 2 (rad or \'_\' and unit)",
             pSymRot2,pSymRot2->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation of symmetry axis around axis 3 */
  pSymRot3 = (IO_flexp*) option_parameter_search( SSymRot3, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb," first rotation: ccw around axis 2 (rad or \'_\' and unit)",
             pSymRot3,pSymRot3->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)      printf("i/p file            : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)      printf("o/p file            : %s\n",ib[0].Name.V);
  if (ib[1].First.I)     printf("first image         : %ld\n",ib[1].First.V);
  if (ib[1].Last.I)      printf("last image          : %ld\n",ib[1].Last.V);
  if (ib[1].Inc.I)       printf("increment           : %ld\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)     printf("output dummy        : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)    printf("output dimension 1  : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)    printf("output dimension 2  : %s\n",ib[0].Dim[2].V);
  if (pSymType->I)       printf("symmetry axis       : %s\n",pSymType->V);
  if (pSymRot1->I)       printf("symmetry rotation 1 : %s\n",pSymRot1->V);
  if (pSymRot2->I)       printf("symmetry rotation 2 : %s\n",pSymRot2->V);
  if (pSymRot3->I)       printf("symmetry rotation 3 : %s\n",pSymRot3->V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_uniax
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

  IO_long Xy;
  IO_line Xyname;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Default reference system is SAXS */
  cb.RSys.V = IO_Saxs; /* default */

 /* Define special options */
  option_define ( IO_tpflag,  SSymDo,    "TRUE",      &cb );
  option_define ( IO_tplvexp, SSymType,     "2",      &cb );
  option_define ( IO_tpflexp, SSymRot1,   "0.0",      &cb );
  option_define ( IO_tpflexp, SSymRot2,   "0.0",      &cb );
  option_define ( IO_tpflexp, SSymRot3,   "0.0",      &cb );

  option_define ( IO_tpline,  SXyname, "file.edf", &cb );
  option_define ( IO_tpflag,  SXy,     "FALSE",    &cb );
  // Write pixel size of output image to xy-files (pixel size is used by spd)
  option_define ( IO_tpflag,  SXypix,  "TRUE",     &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Set xy-flag if xy-file name is given */
  if (status==Success)
    Xy = option_flag(SXy,&cb,&status );
  if (status==Success)
    if (!Xy.I) Xyname = option_line(SXyname,&cb,NULL,NULL,0,&status);
  if (status==Success)
    if (!Xy.I) option_flag_update(Xyname.I?TRUE:FALSE,Xy.I,SXy,&cb,&status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_uniax, NULL, NULL, TRUE, &status );

  if (status == NotWaxsProjection) {
    printf("\
The program \'saxs_uniax\' can only be applied to scattering patterns in WAXS\n\
projection. One of the input or output images does not seem to be a WAXS\n\
projection. You must use the program \'saxs_waxs\' before using \'saxs_uniax\'. \n\
If you are sure that all images are WAXS projections call \'saxs_uniax\' with\n\
the options \"-i1pro Waxs -opro Waxs\".\n");
  }

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
