/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */ 
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */ 
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*************************************************************************** 
NAME
  col2array

AUTHOR
  1998        PB   V1.0 Peter Boesecke, 30-Aug-1998

HISTORY
  1999-11-24  PB cc on dec alpha
                 - print_table( char * table[0] changed to 
                   print_table( char * table[] because 0 is interpreted
                   as a dimension 
                 - some misplaced /n/ corrected
  2004-10-30  PB exit -> return
  2007-04-23  PB print_table: return value void
  2008-12-05  PB fopen
  2008-12-05  PB V1.1 open output text file as binary file to avoid
                      doubling of \r (\r\r\n) at the end of the line when
                      compiled on windows.
  2012-08-30  PB V1.2 BMode: FloatIEEE64 and FloatIEEE128, VAX types removed

***************************************************************************/
# include <stdlib.h>
# include <errno.h>
# include <stdio.h>
# include <unistd.h>

# include <ctype.h>
# include <string.h>
# include <time.h>

/***************************************************************************
* General Definitions                                                      *
***************************************************************************/
# ifndef PRIVATE
#  define PRIVATE       static /* used to declare variables of private type */
#  define PUBLIC                /* used to declare variables of public type */
# endif

# ifndef TRUE
#  define TRUE          1
#  define FALSE         0
# endif

# undef DEBUG
# ifndef FREE
#  define FREE(p) if (p!=NULL) { free(p); p=NULL; }
# endif
# ifndef MAX2
#  define MAX2( x1, x2) ( ( x1)>( x2) ? ( x1) : ( x2) )
# endif
# ifndef MIN2
#  define MIN2( n1, n2) ( ( n1)<( n2) ? ( n1) : ( n2) )
# endif

# define BufferSize           4096              /* maximum lenght of a line */
/****************************************************************************/

PRIVATE char signchar[3] = { '-', '+', '\0' };
PRIVATE char end_of_line[3] = { '\r', '\n', '\0' };

/***************************************************************************
* Binary output modes                                                      *
***************************************************************************/
enum BMode { TextMode,
             Unsigned8=1, Signed8,      Unsigned16,    Signed16,
             Unsigned32,  Signed32,     Unsigned64,    Signed64,
             FloatIEEE32, FloatIEEE64,  FloatIEEE128,  Unused12,
             EndBMode };

/***************************************************************************
* Functions                                                                *
***************************************************************************/

/*+++------------------------------------------------------------------------
NAME
    num2str_digits - splits the string at mark

SYNOPSIS

    char *num2str_digits( char **pstr, long num, long digits );

DESCRIPTION
Converts num to a string with a minimum of digits characters. Spaces 
are replaced with 0. Allocates memory for the output string *pstr. It must 
be released externally.

RETURN VALUE
Pointer to the created string or NULL in case of an error (no memory allocated).

EXAMPLE
num = 12, digits=3 --> num2str_digits = 012;
num = -12, digits=3 --> num2str_digits = -12;
num = 12, digits=1 --> num2str_digits = 12;

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
char *num2str_digits( char **pstr, long num, long digits )
{ char buffer[512];
  long num_len, str_len;
  char *pc, *pn;
 
  (void) sprintf(buffer,"%ld",num); 
  num_len = strlen(buffer);

  str_len = MAX2(digits, num_len);
  *pstr = (char *) malloc((str_len+1)*sizeof(char));
  if (*pstr==NULL) return(NULL);
  memset(*pstr,'0',str_len); (*pstr)[str_len]='\0';
 
  pc = *pstr+str_len-num_len; pn=buffer; 
  if (strchr( signchar, (int) buffer[0] )) { (*pstr)[0]=*pn++; pc++; }

  while (*pn) {
     if (*pc!=' ') *pc=*pn;
     pn++;pc++;
     }

# ifdef DEBUG
  printf(" --- num2str_digits\n");
  printf(" num    = %ld\n",num);
  printf(" digits = %ld\n",digits);
  printf(" buffer = %s\n",buffer);
  printf(" str    = %s\n",*pstr);  
  printf("\n");
# endif

  return( *pstr );
 
} /* num2str_digits */

/*+++------------------------------------------------------------------------
NAME
    string_and_digits - splits the string at mark 

SYNOPSIS

    int string_and_digits(const char *input, char mark, 
                          char **pbefore, long *pdigits, char **pafter);

DESCRIPTION
Splits the string at the first character mark, returns the number of mark
characters in digits and returns the substrings before and after mark.
The memory for before and after is allocated. If the substring is empty, an 
empty string '\0' is allocated and is returned. The memory for before and after
must be released before the next call to this routine. If mark is not found
input is copied to before.

RETURN VALUE
0 in case of success

EXAMPLE
mark=#, input=abc###def.txt is split into before=abc, digits=3, after=def.txt

AUTHOR
PB 29-Aug-1998
  --------------------------------------------------------------------------*/
int string_and_digits(const char *input, char mark, 
                       char **pbefore, long *pdigits, char **pafter)
{ const char *pmark1,*pmark2; 
  size_t before_len=(size_t) NULL, after_len=(size_t) NULL;

  pmark1 = strchr(input, (int) mark);
  pmark2 = strrchr(input, (int) mark); if (pmark2) pmark2++;

  if (pmark1) *pdigits = pmark2-pmark1; else *pdigits = (long) 0;

  if (pmark1) before_len = pmark1-input; 
    else { pmark1 = input; pmark2 = input+strlen(input);
           before_len=strlen(input); }
  if (pmark2) after_len = strlen(input) - (pmark2-input); 

  *pbefore = (char *) malloc ( (before_len+1)*sizeof(char) );
  if (*pbefore==NULL) return(-1);
  *pafter = (char *) malloc ( (after_len+1)*sizeof(char) );
  if (*pafter==NULL) { FREE(*pbefore); return(-1); }

  strncpy( *pbefore, input, before_len ); (*pbefore)[before_len]='\0';
  strncpy( *pafter, pmark2, after_len ); (*pafter)[after_len]='\0';

# ifdef DEBUG
  printf(" --- string_and_digits\n");
  printf(" input  = %s\n",input);
  printf(" mark   = %c\n",mark);
  printf(" before = %s\n",*pbefore);
  printf(" digits = %d\n",*pdigits);
  printf(" after  = %s\n",*pafter);
# endif

  return(0);

} /* string_and_digits */

/*+++------------------------------------------------------------------------
NAME
    numbered_filename - create numbered filename 

SYNOPSIS

    char *numbered_filename(char **pfilename, long num, const char *before,
                            long digits, const char *after);

DESCRIPTION
Creates a numbered filename consisting of beforenumafter. before
and after are character strings, num is a number. The pointer to the
filename is returned in pfilename. It must be released externally.  

RETURN VALUE
In case of success pointer to name_in (memory allocated)
NULL in case of an error, no additional memory is allocated.

EXAMPLE
num=45, before=input_, digits=3, after=.txt
--> *pfilename=input_045.txt

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
char *numbered_filename(char **pfilename, long num, const char *before, 
                        long digits, const char *after)
{ long filename_len;
  char *number=NULL;
 
  if ( num2str_digits( &number, num, digits )==NULL) return(NULL); 

  filename_len = strlen(before)+strlen(number)+strlen(after);

  *pfilename = (char*) malloc((filename_len+1)*sizeof(char));
  if (*pfilename==NULL) { FREE(number); return(NULL);}

  (*pfilename)[0]='\0';
  strcat(*pfilename, before);
  strcat(*pfilename, number);
  strcat(*pfilename, after);

  FREE(number);

  return(*pfilename);

} /* numbered_filename */

/*+++------------------------------------------------------------------------
NAME
    free_spreadsheet_table - releases all elements of a table 

SYNOPSIS

    void free_spreadsheet_table ( char * table[] ); 

DESCRIPTION
The memory of each character strings is released. The table must end with
NULL.

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
void free_spreadsheet_table ( char **pt )
{  while (*pt) { FREE(*pt); pt++; } 
} /* free_spreadsheet_table */

/*---------------------------------------------------------------------------
NAME

   newstr --- allocate memory and copy a character string into it

SYNOPSIS

   char * newstr( const char * string );

DESCRIPTION
  Allocates strlen(string)+1 bytes of memory and copies string into it.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.

RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * newstr( const char * string )
{ char * newstring;

  if (!(newstring = (char *) malloc(strlen(string)+1))) return((char *) NULL);
  (void) strcpy(newstring,string);

  return( newstring );

} /* newstr */

/*---------------------------------------------------------------------------
NAME

   revstr --- allocated memory and copy the reversed string into it 

SYNOPSIS

   char * revstr( const char * string );

DESCRIPTION
  Allocates memory with newstr and reverses string.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.

RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * revstr( const char * string )
{ char *str;
  long i,j;
  long str_len;

  str=newstr(string);
  if (str) {
    str_len = strlen(string);
    j=str_len;
    for (i=0;i<str_len;i++) 
      str[i]=string[--j];
    }

  return(str);
   
} /* revstr */

/*---------------------------------------------------------------------------
NAME

   rmeoln --- removes end of line characters from the end of the line

SYNOPSIS

   char * rmeoln ( char * line );

DESCRIPTION
   Removes end of line characters from line and replaces them with '\0'.
   Nothing is done when line is the NULL pointer.

ARGUMENTS
   char * line (input and output)

RETURN VALUE
   char * line (pointer to output)
---------------------------------------------------------------------------*/
char * rmeoln ( char * line )
{ char * pc = line;


  if (pc)
    while (*pc) {
      if (strchr( end_of_line, (int) *pc )) *pc='\0';
      pc++;
      }
  return( line );

} /* rmeoln */

/*+++------------------------------------------------------------------------
NAME
    read_line - read a line from a file 

SYNOPSIS

  char * read_line( FILE *channel, char **pline );

DESCRIPTION

RETURN VALUE
Pointer to line in case of success
NULL in case of an eof or error (no memory allocated)

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
char * read_line( FILE *channel, char **pline )
{ const int buflen = BufferSize;
  static char buffer[BufferSize];

  fgets (buffer, buflen, channel);

  if (!((feof( channel ))||(ferror( channel )))) 
    *pline = newstr(buffer); else return(NULL); 

  return(rmeoln(*pline));

} /* read_line */

/*+++------------------------------------------------------------------------
NAME
    list_element - return sub-string of a list

SYNOPSIS

    char *list_element( const char *line, 
                        const long ele, const char separator );

DESCRIPTION
Scans the character string line for the sub-string ele. Each substring is
separated by separator. If ele is negative str is scanned from the end.

RETURN VALUE
Pointer to extracted element string (allocated memory, must be released);
NULL pointer in case of an error, No memory is allocated in case of an error

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
char *list_element( const char *line,               
                    const long ele, const char separator )
{ char *ps, *p1=NULL, *p2=NULL;
  long cnt;
  char *str, *sub, *sub1;
  long sub_len;

# ifdef DEBUG
   printf(" --- list_element\n");
   printf("line=\"%s\",ele=%ld,separator=%c\n",line,ele,separator);
# endif

  if (ele<0) str=revstr(line); else str=newstr(line);

  ps=str;
  cnt=1;
  p1=ps;

  while ( (cnt<abs(ele))&&(ps) ) {
    ps=strchr(p1,(int) separator);
    if (ps) p1=ps+1; else p1=NULL; 
    cnt++;
    }
  if (p1) { 
    ps=strchr(p1,(int) separator); 
    if (ps) p2=ps; else p2=str+strlen(str);
    } else p2=NULL; 

  sub_len = p2-p1;
  sub = (char *) malloc((sub_len+1) * sizeof(char)); 
  if (sub==NULL) { FREE(str); return(NULL);}
  strncpy(sub, p1, sub_len);sub[sub_len]='\0';

  if (ele<0) { sub1=revstr(sub); FREE(sub); sub=sub1; }
   
  FREE(str);

# ifdef DEBUG
   printf("element=\"%s\"\n",sub);
# endif

  return(sub);

} /* list_element */

/*+++------------------------------------------------------------------------
NAME
    read_spreadsheet_column - returns the rows of a column in a spreadsheet file

SYNOPSIS

    int read_spreadsheet_column(char ***prow, const char * name_in,
                                const long col, const char separator,
                                const long skip_lines, const long skip_chars);

DESCRIPTION
Skips skip_lines and skip_chars at the beginning of the file name_in and
reads all rows of column number col. The extracted strings are returned in
the table char * row[]

RETURN VALUE
0 in case of success
No memory is allocated in case of an error

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
int read_spreadsheet_column( char ***prow, const char * name_in, 
                             const long col, const char separator, 
                             const long skip_lines, const long skip_chars )
{ const char * routine_name="read_spreadsheet_column"; 
  FILE *channel;
  char *line=NULL;
  long lcnt,ccnt;
  int  cc;
  long start_pos;
  long row_number,number_of_rows;

  /* open file */
  channel = fopen( name_in, "rb" ); /* read-only */
  if (channel == NULL) {
    perror("ERROR (read_spreadsheet_column#1)");return(-1);}

  /* find start position */
  /* skip lines */
  for (lcnt=0;lcnt<skip_lines;lcnt++) {
    if (read_line( channel, &line )==NULL) {
        perror("ERROR (read_spreadsheet_column#2)"); return(-1); } 
    FREE(line);
    }
  /* skip characters */
  for (ccnt=0;ccnt<skip_chars;ccnt++) {
    if ((fgetc( channel ))==EOF) {
        perror("ERROR (read_spreadsheet_column#3)"); return(-1); }
    } 
  start_pos = ftell( channel ); 

# ifdef DEBUG
  printf("File name      = %s\n",name_in);
  printf("Start position = %ld\n",start_pos);
# endif

  printf("  reading file %s\n",name_in);

  /* determine number of rows */
  number_of_rows=0l;
  while (read_line( channel, &line )) {
    number_of_rows++;
    FREE(line);
    }
  if (!feof( channel )) {
    perror("ERROR (read_spreadsheet_column#4)"); return(-1); }

# ifdef DEBUG
  printf("Number of rows = %ld\n",number_of_rows); 
# endif

  /* allocate memory for column */
  *prow = (char**) malloc((number_of_rows+1)*sizeof(char**));  
  if (*prow==NULL) return(-1);

  /* initialize with zeros */
  memset(*prow,'\0',(number_of_rows+1)*sizeof(char**));

  /* read from start position and allocate memory for each string */
  if (fseek( channel, start_pos, SEEK_SET)) {
    FREE(*prow); perror("ERROR (read_spreadsheet_column#5)"); return(-1); } 

  for (row_number=0;row_number<number_of_rows;row_number++) {
#    ifdef DEBUG
       printf("read_line # %ld\n",row_number+1);
#    endif
    if (read_line( channel, &line )) {
      (*prow)[row_number]=list_element( line, col, separator ); 
      FREE(line);
      } else { 
      free_spreadsheet_table( *prow ); FREE(*prow);     
      perror("ERROR (read_spreadsheet_column#6)"); return(-1); }
    }

  /* close file */ 
  fclose( channel );

  return(0);

} /* read_spreadsheet_column */

void print_table( char * table[] )
{ char ** pt;
  long index;
  
  pt = table;
  index = 1;
  while (*pt) printf("%ld: %s\n",index++,*pt++);
 
} /* print_table */

long tablelen( char * table[] )
{ char **pt;
  long len=0;
  pt=table;
  while (*pt) { pt++; len++; }
  return(len);
} /* tablelen */

/*+++------------------------------------------------------------------------
NAME
    new_spreadsheet - creates an empty spreadsheet (empty file) 

SYNOPSIS

   int new_spreadsheet( const char *output, const int bin_mode );

DESCRIPTION
bin_mode > 0: opens binary file
bin_mode = 0: opens text file

RETURN VALUE
0 in case of success
No memory is allocated in case of an error

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
int new_spreadsheet( const char *output, const int bin_mode )
{ FILE *channel;

  if (bin_mode) channel = fopen(output, "wb"); /* truncate to zero length */
    else channel = fopen(output, "wb"); /* truncate to zero length */
  if (!channel) { perror("ERROR (new_spreadsheet)"); return(-1); }
  fclose( channel );

  return(0);

} /* new_spreadsheet */

/*+++------------------------------------------------------------------------
NAME
    append_spreadsheet_row - appends a row to a spreadsheet 

SYNOPSIS

    int append_spreadsheet_row( const char **row, const char *output, 
                                const char separator, const int bin_mode )

DESCRIPTION
bin_mode = 0: Writes the elements of row[] unconverted as text to
              the end of the output file.
              The elements are separated by separator and lines
              end with \r\n.
bin_mode > 1: Converts the elements of row to binaries and writes them
              to a binary file in local endian order 

RETURN VALUE
0 in case of success
No memory is allocated in case of an error

AUTHOR
PB 30-Aug-1998
  --------------------------------------------------------------------------*/
int append_spreadsheet_row( char **row, const char *output, 
                            const char separator, const int bin_mode )
{ char **pr;
  FILE *channel;
  long int lv;
  float fv;

  if (bin_mode)
    channel = fopen(output, "ab"); /* go to end of file */
   else channel = fopen(output, "ab"); /* go to end of binary file */
  if (!channel) { perror("ERROR (append_spreadsheet_row)"); return(-1); }


  switch (bin_mode) {
    case Signed32: 
             pr = row;
             while (*pr) {
               sscanf(*pr,"%ld",&lv); /* no checks! */
               if (fwrite(&lv,sizeof(lv),1,channel)<1) {
                 perror("ERROR (append_spreadsheet_row)"); return(-1); }
               pr++;
               }
             break;
    case FloatIEEE32:
             pr = row;
             while (*pr) {
               sscanf(*pr,"%g",&fv); /* no checks! */
               if (fwrite(&fv,sizeof(fv),1,channel)<1) {
                 perror("ERROR (append_spreadsheet_row)"); return(-1); }
               pr++;
               }
             break;
    default: /* TextMode */
             pr = row;
             while (*pr) {
               if (fprintf(channel,"%s%c",*pr,separator)<1) { 
                 perror("ERROR (append_spreadsheet_row)"); return(-1); }
               pr++;
               }
             fprintf(channel, "%s", end_of_line);
             break;
    }

  fclose( channel );

  return(0);

} /* append_spreadsheet_row */

/*+++------------------------------------------------------------------------
NAME
    col2array - transformation of ascii columns to an array

SYNOPSIS

int col2array(const char * output, const char * input,
              const long first, const long last, const long inc,
              const long col, const long skip_lines, const long skip_chars,
              const char separator, const int bin_mode,
              long *pcolumns, long *plines )

DESCRIPTION
See help to main 

RETURN VALUE
0 in case of success

AUTHOR
PB 29-Aug-1998 
  --------------------------------------------------------------------------*/
int col2array(const char * output, const char * input, 
              const long first, const long last, const long inc, 
              const long col, const long skip_lines, const long skip_chars,
              const char separator, const int bin_mode,
              long *pcolumns, long *plines )
{ char *before=NULL, *after=NULL;
  long digits;
  long loop, loopmax; 
  long num;
  char *name_in=NULL;
  char **row=NULL;

  *pcolumns=0l;
  *plines=0l;
 
  if ( string_and_digits(input,'#',&before,&digits,&after) ) return(-1);

  if (inc!=0l)
    loopmax = MAX2(1l,1l + (last-first)/inc); 
   else loopmax = 1l;

  if ( new_spreadsheet( output, bin_mode ) ) { 
    FREE(before);FREE(after); return(-1); }

  num = first;
  for ( loop=1l; loop<=loopmax; loop++ ) {
    numbered_filename(&name_in, num, before, digits, after);

    if ( read_spreadsheet_column( &row, name_in, col, separator, 
                                  skip_lines, skip_chars ) ) {
      FREE(name_in);FREE(before);FREE(after); return(-1); }

    *pcolumns=MAX2(*pcolumns,tablelen(row));

    if ( append_spreadsheet_row( row, output, '\t', bin_mode ) ) {
      free_spreadsheet_table( row ); FREE(row);
      FREE(name_in);FREE(before);FREE(after); return(-1); }

    free_spreadsheet_table( row ); FREE(row); FREE(name_in);
    (*plines)++;
    num+=inc; 
    }
 
  FREE(before);FREE(after);
  return(0);

} /* col2array */

/*+++------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

# define Usage "\
col2array [-h] [-b <mode[0]>] <output> <input> <first[1]> <last[1]> <inc[1]>\n\
          <col[1]> <skip_lines[0]> <skip_chars[0]> <separator['\\t']>"
# define Help "\
Author: V1.1 Peter Boesecke, 2008-12-05\n\
A single column is extracted from a set of numbered ascii-input files and are\n\
rearranged to rows of a single ascii-output file. The columns must be\n\
separated by tabs \\t. The columns are numbered with 1, 2, 3 etc. from the\n\
start and -1, -2, -3 etc. from the end. The parameters must be entered in\n\
the described order. For omitted parameters default values are used.\n\
\n\
output : output filename, e.g. output.txt\n\
input  : input filename with optional #-character specifying the position of \n\
         the number, e.g. input_###.txt. Leading positions are filled with \n\
         zeros.\n\
first  : number of the first file (default: 1)\n\
last   : number of the last file (default: first)\n\
inc    : increment (default: 1)\n\
col    : column number (default: 1)\n\
skip_lines : number of lines to skip at the top (default: 0)\n\
skip_chars : number of characters to skip (default: 0).\n\
-b <mode>: 0->text (default), 6->Signed32; 9->FloatIEEE32\n\
The command\n\
>> col2array output.txt input_###.txt 105 095 -3 -1 2 0\n\
will read the last column of the files input_105.txt, input_102.txt, \n\
input_099.txt and input_096.txt and write it to output.txt. It will skip \n\
2 lines and 0 characters at the start of each file."

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_col2saxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{ long argn=1;
  char *output, * input;
  long first=1, last=1, inc=1;
  long col=1;
  long skip_lines=0, skip_chars=0;
  char separator = '\t';
  int  bin_mode=0;
  long columns, lines;

  if (argc>1) if (!strcmp(argv[1],"-h")) { 
    printf("NAME\n%s\nUSAGE\n%s\nHELP\n%s\n",argv[0],Usage,Help); return(-1); }

  /* options */
  if (argc>argn) if (!strcmp(argv[argn],"-b")) { argn++; 
    if (argc>argn) sscanf(argv[argn++],"%d",&bin_mode); 
      else { printf("Error reading binary mode."); return(-1); } }

  /* parameters */
  if (argc<argn+2) { printf("%s\n",Usage); return(-1); }

  if (argc>argn) output = argv[argn++];
  if (argc>argn) input  = argv[argn++];
  if (argc>argn) if ( sscanf(argv[argn++],"%d",&first) < 1 ) {
     printf("ERROR: first = %s\n",argv[--argn]); return(-1); }
  last = first;
  if (argc>argn) if ( sscanf(argv[argn++],"%d",&last) < 1 ) {
     printf("ERROR: last = %s\n",argv[--argn]); return(-1); }
  if (argc>argn) if ( sscanf(argv[argn++],"%d",&inc) < 1 ) {
     printf("ERROR: inc = %s\n",argv[--argn]); return(-1); }
  if (argc>argn) if ( sscanf(argv[argn++],"%d",&col) < 1 ) {
     printf("ERROR: col = %s\n",argv[--argn]); return(-1); }
  if (argc>argn) if ( sscanf(argv[argn++],"%d",&skip_lines) < 1 ) {
     printf("ERROR: skip_lines = %s\n",argv[--argn]); return(-1); }
  if (argc>argn) if ( sscanf(argv[argn++],"%d",&skip_chars) < 1 ) {
     printf("ERROR: skip_chars = %s\n",argv[--argn]); return(-1); }
  if (argc>argn) separator=argv[argn++][0];

#ifdef DEBUG
  if (bin_mode) printf("output     = %s (bin_mode %d file)\n",output,bin_mode);
   else printf("output     = %s (text file)\n",output);
  printf("input      = %s\n",input);
  printf("first      = %d\n",first);
  printf("last       = %d\n",last);
  printf("inc        = %d\n",inc);
  printf("col        = %d\n",col);
  printf("skip_lines = %d\n",skip_lines);
  printf("skip_chars = %d\n",skip_chars);
  printf("separator  = (char) %u\n",separator);
  printf("\n");
#endif

  if ( col2array(output, input, first, last, inc, col, 
                 skip_lines, skip_chars, separator, bin_mode,
                 &columns, &lines ) ) {
    printf("  Maximum number of columns %ld, number of lines %ld\n",
           columns,lines);
    return(-1); }

  printf("  Maximum number of columns %ld, number of lines %ld\n",
         columns,lines);
  printf("  %s end\n",argv[0]);

  return( 0 );

} /* main */
