/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/* @(#) $Source: SaxsImage.h  V2.43 by Peter Boesecke 2012-01-29        $  */
/****************************************************************************
* SaxsImage.h                                                               *
****************************************************************************/

#ifndef SAXS_IMAGE
# define SAXS_IMAGE

# include "SaxsDefinition.h"
# include "SaxsOption.h"
# include "ipol.h"

# define DefaultExtension  ".edf"
# define DefaultInput      "input"DefaultExtension
# define DefaultOutput     "output"DefaultExtension

# define IMAGELOOP(PCB,IB,FC,IFC,TFC,A,PST) \
         ImageLoop((PCB),(IB),(FC),(IFC),(TFC),(A),(PST))

typedef struct ImageHeaderBlock {
        int             blkno;          /* block number */
	/* control variables */
	int		I;		/* flag: TRUE = valid header */
	long int	ImNum;		/* (i) image number */
        int		Stream;         /* (i) stream */
	long int	StreamOpenMode;	/* (i) file open mode */
                                        /* LoopCount and ActualLoopCount are
                                           used inside SaxsImageLoop and count
                                           the number of calculation cycles.
                                           They start with 0 and are 
                                           incremented AFTER at the end of 
                                           SaxsImageLoop */
        long int        LoopCount;      /* (i/o) nominal number of loops after
                                           allocation of memory, incremented */
        long int        ActualLoopCount;/* (i/o) actual number of loops after
                                           allocation of memory, incremented */
        size_t      	ImgByteLen;     /* (o) total length of read/wr data */
	long int	Dim[4];		/* (i/o) image array dimensions */
	float         * Data;		/* (i/o) pointer to data array
					   Data[Dim[1],Dim[2]] */
	float         * VarDat;		/* (i/o) pointer to variance array
					   VarDat[Dim[1],Dim[2]] */
        int             DoNotScale;     /* (i/o) do not use this header to
                                           calculate image overlap 
                                           (default: FALSE) */
        IO_line         Header;         /* key under which the full header 
                                           is available when I-flag is TRUE */

        /* values of image transformation parameters */
        long            rori;           /* relative orientation change of input
                                           and output image (default 1, 
                                           calculated from Orientation */
        IO_float        MinClip;
        IO_float        MaxClip;
        IO_float        Const;
        IO_float        Factor;
        IO_prog         Val;
        IO_prog         ErrVal;
        IO_long         Bin[3];
        IO_long         Gnaw[3];
        IO_float        Shift[3];

        /* standard SAXS keyword values */
	IO_float	Dummy;		/* dummy value */
 	IO_float	DDummy;		/* dummy difference */
 	IO_float	Offset[3];	/* image offset in pixels */
	IO_float	Center[3];	/* image center in pixels */
	IO_float	BinSiz[3];	/* pixel size relative to orig. image */
	/* optional SAXS keyword values */
	IO_float	PixSiz[3];	/* pixel size in meters */
	IO_float	SampleDistance; /* sample distance in meters */
	IO_float	WaveLength;	/* wavelength in meters */
        IO_float        DetRot[4];      /* detector rotation */
        IO_line         Title;          /* title */
        IO_line         Time;           /* time of first creation, 
                                           e.g. start of exposure */
        IO_long         Orientation;    /* Orientation of the image. Input
                                           images are read with Orientation.V
                                           and are converted to orientation 1, 
                                           output images are saved with 
                                           Orientation.V. Internally, all 
                                           images have orientation 1. */
        IO_long         Projection;     /* Projection type of the image. 
                                           The projection type must be known 
                                           to calculate scattering 
                                           vectors correctly */ 
        IO_long         AxisType[3];    /* Axis types */
	} ImgHeadBlk;

/***************************************************************************
* Routines                                                                 *
***************************************************************************/

   extern char
        *SaxsImageVersion       ( void );
   extern void
	OpenImageFile		( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  const char *fname, long int FileNumber,
                                  long int OpenMode, int *pstatus),
        NewImage                ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  long num[], ImgHeadBlk ihb[],
                                  ImgHeadBlk * template, 
                                  void (*InitFunction) (CmdBlk *,long [],
                                  ImgHeadBlk[],int *), int *pstatus ),
        NewImageHeader          ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  long num[], ImgHeadBlk ihb[],
                                  ImgHeadBlk * template, 
                                  void (*InitFunction) (CmdBlk *,long [],
                                  ImgHeadBlk[],int *), int *pstatus ),
        BalanceHeaderBlock      ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  long num[], ImgHeadBlk ihb[], int *pstatus ),
        WriteHeaderLine         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, char * Value, int * pstatus ),
        WriteHeaderFloat        ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, float Value, int * pstatus ),
        WriteHeaderLong         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, long int Value, 
                                  int * pstatus ),
        WriteHeaderUnit         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  const char * keyword, float Value, 
                                  const char * unit, int * pstatus ),
        WriteHeader             ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  const char * header_key, int * pstatus ),
        WriteImage              ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  ImgHeadBlk ihb[], int *pstatus),
        FreeImage               ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[] ),
        CloseImageFile          ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  int * pstatus),
        GeoHeaderCopy           ( ImgHeadBlk ihb[], int blkin, int blkout ),
        DoNotScaleWith          ( ImgHeadBlk ihb[], int blk ),
	PrintImageHeaderBlock 	( ImgHeadBlk * pihb ),
	ImageLoop ( CmdBlk * pcb, ImgBlk ib[],
                    void (*Function) (CmdBlk *, long [], ImgHeadBlk [], int *),
                    void (*InitFunction) (CmdBlk *,long [],ImgHeadBlk[],int *),
                    void (*TermFunction) (CmdBlk *,long [],ImgHeadBlk[],int *),
                    int allneeded, int * pstatus ) ;
   extern int
	ReadImage 		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long num[], ImgHeadBlk ihb[],
				  int *pstatus),
        ReadImageHeader         ( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long num[], ImgHeadBlk ihb[],
				  int *pstatus),
	SearchMinMaxImage 	( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  long int * pMinImage, long int *pMaxImage,
                                  int *pstatus),
        ReadImageDimension      ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[], 
                                  long int * pDim_1, long int *pDim_2,
                                  int * pstatus ),
        ReadHeaderLine		( CmdBlk *pcb, int blkno,
                                  ImgHeadBlk ihb[], char * keyword,
				  char * Value, int * pstatus ),
        ReadHeaderFloat		( CmdBlk *pcb, int blkno,
                                  ImgHeadBlk ihb[], char * keyword,
                                  float * Value, int * pstatus ),
        ReadHeaderLong		( CmdBlk *pcb, int blkno,
                                  ImgHeadBlk ihb[], char * keyword,
                                  long int * Value, int * pstatus ),
        ImageIsNew              ( int blkno, ImgHeadBlk ihb[] ),
        ImageIsFirst            ( CmdBlk * pcb, int blkno, ImgHeadBlk ihb[] ),
        ImageIsLast             ( CmdBlk * pcb, int blkno, ImgHeadBlk ihb[] ),
        Single2Multi            ( void ),
        Multi2Single            ( void );
   extern char *
        ReportSaxsImageError    ( int ExternalError );


/***************************************************************************
* Utilities                                                                *
***************************************************************************/

   extern void
        condition_2d            ( ImgHeadBlk * pihb,
                                  IO_float MinClip, IO_float MaxClip,
                                  IO_long  Gnaw_1,  IO_long  Gnaw_2,
                                  IO_long  Bin_1,   IO_long  Bin_2,
                                  IO_prog  Val,     IO_prog  ErrVal,
                                  IO_float Factor,  IO_float Const,
                                  IO_long  Ave,     long Orientation,
                                  int Simulate, int TestBit, int *pstatus );

#endif /* SAXS_IMAGE */

/****************************************************************************
*                                                                           *
****************************************************************************/
