/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsExpression.c

2 PURPOSE
  Reading of float and long integer expressions from strings.
  See PUBLIC functions for detail.

2 CALL
  long int longexpr( const char * s, int * pstatus);
  float floatexpr( const char * s, int * pstatus);
  double doubleexpr( const char * s, int * pstatus);

2 AUTHOR
  1995 Peter Boesecke (PB)

2 HISTORY
  2004-03-16 PB All active routines moved to new module numio. 
                SaxsExpression is only an interface between numio and
                SaxsInput
  2004-04-02 PB doubleexpr: num_str2double has new parameter tail
                longexpr:   num_str2long has new parameter tail
  2005-07-11 PB errval2status
  2007-03-30 PB sizeof is always first operand
  2007-04-19 PB -Wall compiler warnings resolved
  2009-03-20 PB num_str2...: char *tail -> const char *tail

---*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

# include "SaxsExpression.h"
# include "numio.h"

/****************************************************************************
*  Functions                                                                *
****************************************************************************/

/*--------------------------------------------------------------------------  
   doubleexpr : reads a double expression
  
   	return double value (o) : value of expression
        const char * s      (i) : input string
        int * status        (o) : output status
                                  Success        : successful conversion
                                  !0, otherwise
  --------------------------------------------------------------------------*/
PUBLIC double doubleexpr( const char * s, int * pstatus)
{ double value;
  char * buffer, * pb;
  const char * tail;
  int errval;
 
  pb = buffer = (char * ) malloc(sizeof(char)*(strlen(s)+1));
  if (!buffer) { *pstatus = NotEnoughMemoryAvailable; return( 0 ); }
  strcpy(pb,s);

//  EditLine(LowerCase | Collapse | UnComment,pb);
  EditLine(Collapse | UnComment,pb);
  value = num_str2double( pb, &tail, &errval );
  *pstatus = errval2status( errval );

  if ((*pstatus == Success)&&(strlen(tail)>0))
    switch (tail[0]) {
      case ',': break;
      default : *pstatus = Failed; break;
    }

  free(buffer);
 
  return( value );
 
} /* doubleexpr */

/*--------------------------------------------------------------------------  
   floatexpr : like doubleexpr, but values are returned as float 
  --------------------------------------------------------------------------*/
PUBLIC float floatexpr( const char * s, int * pstatus)
{
     return( (float) doubleexpr( s, pstatus) );

} /* floatexpr */

/*--------------------------------------------------------------------------  
   longexpr : reads an long integer expression
  
   	return long value  (o) : value of expression
        const char * s     (i) : input string
        int * status       (o) : output status
                                 Success         : successful conversion
                                 !0, otherwise
  --------------------------------------------------------------------------*/
PUBLIC long int longexpr( const char * s, int * pstatus)
{ long int value;
  char * buffer, * pb;
  const char * tail;
  int errval;

  pb = buffer = (char * ) malloc(sizeof(char)*(strlen(s)+1));
  if (!buffer) { *pstatus = NotEnoughMemoryAvailable; return( 0l ); }
  strcpy(pb,s);
 
//  EditLine(LowerCase | Collapse | UnComment,pb);
  EditLine(Collapse | UnComment,pb);
  value = num_str2long( pb, &tail, &errval);
  *pstatus = errval2status( errval );

  if ((*pstatus == Success)&&(strlen(tail)>0))
    switch (tail[0]) {
      case ',': break;
      default : *pstatus = Failed; break;
    }

  free(buffer);

  return( value );

} /* longexpr */

/*--------------------------------------------------------------------------  
   errval2status : converts errval to status 
  --------------------------------------------------------------------------*/
int errval2status( int errval )
{ int status;
  // conversion of errval to status, necessary until SaxsError is improved
  switch (errval) {
      case NumSuccess              : status = Success; break;
      case NumMemoryAllocationError: status = NotEnoughMemoryAvailable; break;
      case NumScanError            : status = ScanError; break;
      case NumCommaExpected        : status = CommaExpected; break;
      case NumBadParenthesis       : status = BadParenthesis; break;
      case NumNoFloatNumber        : status = NoFloatNumber; break;
      case NumNoFloatFunction      : status = UnknownFloatFunction; break;
      case NumDomainError          : status = DomainError; break;
      case NumNoIntegerNumber      : status = NoIntegerNumber; break;
      case NumIntegerOverflow      : status = IntegerOverflow; break;
      case NumDivByZero            : status = DivByZero; break; break;
      case NumWriteError           : status = ExpressionWriteError;break;
      case NumNoVariable           : status = NoVariable;break;
      case NumVariableError        : status = VariableError;break;
      case NumProgramError         : status = ProgramError;break;
      case NumNoInstruction        : status = NoInstruction;break;
      case NumNoAccumulator        : status = NoAccumulator;break;
                            default: status = Failed;
  }
  return(status);
} // errval2status

/****************************************************************************/
