/****************************************************************************
*  Include                                                                  *
****************************************************************************/
# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <stdarg.h>
# include <string.h>
# include <ctype.h>
# include <limits.h>
# include <errno.h>
# include <fcntl.h>
# include <math.h>
# include <float.h>
  
# include "strlib.h"
# include "numio.h" 
# include "median.h"

/******************************************************************************
* Type Defs                                                                   *
******************************************************************************/

typedef struct median_DataElement { 
  double X;
  struct median_DataElement * Previous, * Next;
} MedianDataElement;

typedef struct median_DataList {
  MedianDataElement * Element;
  MedianDataElement * Last; // pointer to last element
  long nelements;           // number of data elements
} MedianDataList;

MedianDataElement * append_median_element( MedianDataList * list, double value )
{ MedianDataElement * newelement=(MedianDataElement *) NULL;

  if (!list)
    goto append_median_element_error;

  if ( newelement = (MedianDataElement*) malloc(sizeof(MedianDataElement)) ) {
    MedianDataElement * current;

    newelement->X = value;

    current = list->Last; 

    // add newelement after current
    if (current) {
      newelement->Previous = current;
      newelement->Next = current->Next;
      current->Next = newelement; 
    } else {
      newelement->Previous = NULL;
      newelement->Next = NULL;
      list->Element = newelement;
    }

  }

  list->Last = newelement;
  list->nelements += 1;

  return( newelement );

append_median_element_error:

  if (newelement) free(newelement);

  return( NULL );

} // append_median_element

/*--------------------------------------------------------------------------
NAME

  new_median_list --- returns a pointer to a new MedianDataList

SYNOPSIS

  MedianDataList * new_median_list( void )

ARGUMENTS

  return MedianDataList * (o) : pointer to an empty median list or
                                NULL in case of error.

DESCRPTION

 Returns a pointer to the root of a new empty list.
--------------------------------------------------------------------------*/
MedianDataList * new_median_list( void )
{ MedianDataList * newlist = (MedianDataList*) NULL;

  if (newlist = (MedianDataList *) malloc ( sizeof(MedianDataList) )) {
    newlist->Element = (MedianDataElement *) NULL; 
    newlist->Last = (MedianDataElement *) NULL;
    newlist->nelements=0;
  }

  return( newlist );

} // new_median_list

/*--------------------------------------------------------------------------
NAME

  free_median_list --- frees the full list

SYNOPSIS

  int free_median_list( MedianDataList * list ):

ARGUMENTS

  returns 0

DESCRPTION

  Deallocates all elements of list and list itself. The list pointer is
  no longer valid and must be set to NULL. If list is NULL, nothing is
  done.
--------------------------------------------------------------------------*/
int free_median_list( MedianDataList * list ) 
{ if ( list ) {
    MedianDataElement *current, *next;
    // free all list elements
    next = list->Element;
    while ( next ) {
      current = next; 
      next = current->Next;
      free( current );
    }
    // free list root
    free( list );
  }
  
  return( 0 );

} // free_median_list

/*--------------------------------------------------------------------------
NAME

  print_median_list --- print list elements into a file 

SYNOPSIS

  int print_median_list( FILE * out, MedianDataList * list, int mode )

ARGUMENTS

  FILE * out            (i) : output
  MedianDataList * list (i) : list
  int mode              (i) : 0: 1 element per line
                              1: all elements into a single line

  returns 0

DESCRPTION

  Prints all list elements to out.
--------------------------------------------------------------------------*/
int print_median_list( FILE * out, MedianDataList * list, int mode )
{
 if ( list ) {
    MedianDataElement *current, *next;
    long index;


    switch (mode) {
      case 1:
        // print list information 
        fprintf(out,"%d: ",list->nelements);

        index=1;
        next = list->Element;
        while ( next ) {
          current = next;
          next = current->Next;
    
          fprintf(out,"%g ",index,current->X);
          index++;
        }
        fprintf(out,"\n");
        break;
      default:
        // print list information 
        fprintf(out,"n = %d\n",list->nelements);

        index=1;
        next = list->Element;
        while ( next ) {
          current = next;
          next = current->Next;

          fprintf(out,"%d : %g\n",index,current->X);
          index++;
        }
    }
  }

  return( 0 );

} // print_median_list

/*--------------------------------------------------------------------------
NAME

  median_list_length --- returns the number of elements in the list

SYNOPSIS

  long median_list_length(MedianDataList * list)

ARGUMENTS

  returns 0

DESCRPTION

  Returns the number of elements in the list.

--------------------------------------------------------------------------*/
long median_list_length(MedianDataList * list)
{ 
  return( list->nelements );
} // median_list_length

/*--------------------------------------------------------------------------
NAME

  median_table --- copies all list elements to buffer

SYNOPSIS

  double * median_table( double *buf, size_t buflen, MedianDataList * list );

ARGUMENTS

  Returns a pointer to the updated buffer or NULL in case of an error.

DESCRPTION

  Copies all list elements to buffer and returns a pointer to it. If list 
  is NULL or if it is empty NULL is returned.

--------------------------------------------------------------------------*/
double * median_table( double *buf, size_t buflen, MedianDataList * list )
{ MedianDataElement * pe;
  double *pb=buf;
  size_t u=(size_t) 0;

  if (!list) 
    goto median_table_error;

  if (!(pe = list->Element) )
    goto median_table_error;

  while (pe) {
    if (u>=buflen)
      goto median_table_error;
    *pb = pe->X;
    u++; pb++; pe = pe->Next;
  }

  return( buf );

median_table_error:

  return( NULL );

} // median_table

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_median
#else
# define MAIN main
#endif

# define BUFLEN 2048
int MAIN (int argc, char *argv[])
{ static const char * Main_Error="ERROR: median";
  char buf[BUFLEN];                       // buffer for input command
  const char *pb;
  int errval;
  int debug=0;

  char *nul = (char *) NULL;
  char **pargv;
  char *ps;

  double value;
  MedianDataList *list=NULL;
  long n;
  double *a=NULL;

  double p=0.5;

  pargv=&argv[1];
  while ( *pargv!=nul ) {

    if ( strncmp( *pargv,"debug=", 6 ) == 0 ) {
      // set debug level
      ps=*pargv+6;
      debug = (int) num_str2double ( ps, NULL, &errval);
      /* .... */
    } else  if ( strncmp( *pargv,"p=", 2 ) == 0 ) {
      ps=*pargv+2;
      p = num_str2double ( ps, NULL, &errval);
    }
    if (*pargv) pargv++;

  } // while

  fgets(buf,BUFLEN, stdin);
  while(!feof(stdin)) {
    if (!(strlib_is_empty(buf))) {

      printf("-------------------------------------------------------\n");

      if (! (list = new_median_list()) )
        goto main_error;

      pb = buf;

      // num_str2double ( const char *str, const char **tail, int *perrval);
      while (!strlib_is_empty( pb )) {
        value = num_str2double ( pb, &pb, &errval);
        if (errval) {
          printf("ERROR: num_str2double rest=%s\n",pb);
          break;
        }
        append_median_element( list, value );
      }

      print_median_list( stdout, list, 1 );

      value = 0.0;
      n = median_list_length( list );
      if (n>0) {
        if ( a = (double *) malloc ( sizeof(double)*n ) ) {
          if ( median_table( a, n, list ) ) {
            value = dmedian ( a, n );
            printf("median = %g\n",value);
            value = dquantil( a, n, p );
            printf("%lg-quantil = %g\n",p,value);
          }
          free(a);
        }
      }

      free_median_list( list );

    } // if not empty

    fgets(buf,BUFLEN, stdin);
  } /* while */


  return(0);

main_error:

  free_median_list( list );

  return(-1);

} /* MAIN */
