/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_refract
  Refraction correction of 2d-surface scattering patterns

2 PURPOSE
  Refraction correction of 2d-surface scattering patterns. The calculation
  is valid for grazing incidence small angle scattering (GISAXS) and
  higher angle scattering. The incoming beam is refracted at the surface 
  scattered inside the sample and and again refracted at the surface before
  it hits the detector. This routine transforms the observed (external) 
  scattering pattern into the internal scattering pattern that would be 
  observed inside the sample. The calculation is done using Snell's law.

  1-delta : real part of the refractive index (0<=delta<1, default 0.0)
  eta     : inclination of sample surface (corresponds to angle eta on id01)
            (0<=eta<90_deg, default 0.0_deg)
  chi     : rotation of the inclined sample around the primary beam
            (corresponds to angle chi on id01)
            (0<=chi<=180_deg, default 90_deg). Chi is 90_deg when 
            the sample surface is horizontal

  Arguments:
  saxs_refract [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <delta[0.0]> <the[0.0_deg]> <chi[90.0_deg]>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  <delta>             : refraction index n = 1-delta (default 0)
  <eta>               : sample inclination (default 0_deg)
  <chi>               : sample tilt (default: 90_deg)

2 HISTORY
  2001-02-21  from saxs_mac V3.12
  2001-03-01 
  2001-03-27  refract.h
  2001-03-30  V3.0
  2001-04-08  V3.1 -psi -> -the for theta angle 
  2001-05-31  V3.11 corrected: Delta not multiplied by deg2rad
              V3.12 TwoTheta must be smaller than 90_deg
  2001-07-09  PB V4.00 new loop, no repetition
  2003-02-09  PB V4.01 line args for Dummy, DDummy, Dim
  2003-05-30  PB V4.02 ImageLoop: Function call changed
  2003-08-14  PB V4.03 option_define, all option numbers removed 
                       theta -> eta
  2003-11-30  PB V4.04 BinSiz
  2004-10-14  PB V4.05 input angles eta and chi in rad
  2004-10-31  PB V4.06 MAKE_FUNCTION
  2005-07-15  PB V4.07 ReportSaxsStatus parameter list updated
  2005-09-17  PB V4.08 VarDat
  2007-06-11  PB V4.09 ImageLoop parameters updated
  2012-01-31  PB V4.10 ImageLoop parameters updated

---*/
# define Version  "saxs_refract V4.10 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "refract.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> \n\
                <delta[0.0]> <eta[0.0_deg]> <chi[90.0_deg]>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SDelta "del"   /* float */
# define SEta   "eta"   /* float */
# define SChi   "chi"   /* float */

/*---------------------------------------------------------------------------
1 saxs_refract

2 PURPOSE
  GISAXS refraction correction
---------------------------------------------------------------------------*/
void saxs_refract (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{  const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  int   i, imax; 
  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float Sum, Weight, Value;
  float VarSum, VarWeight, E0Value;

  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float VarDDummy=DDSET(VarDummy);

  int i_1, i_2;
  float W, WL, W_1, W_2;

  float f1_1, f1_2;
  float W1, W1L, W1_1, W1_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  double delta, eta, chi;                                                       
  RefractDir Beam, Beam1;
  float L, L1;

  float tmp;
  IO_float Delta, Eta, Chi;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* check the reference system: Normal required */
  if (pcb->RSys.V != IO_Normal) {
    printf("ERROR: Reference system %s required\n",reftostr(IO_Normal));
    printf("       Start program with option -rsys %s\n",reftostr(IO_Normal)); 
    }

  if (!((ihb[0].SampleDistance.I)&&(ihb[0].SampleDistance.I))) {
    printf("ERROR: SampleDistance missing\n");
    printf("       Use option -i1dis <SampleDistance[m]>\n");
    *pstatus=Failed; return;
    }
  L = ihb[0].SampleDistance.V;
  L1 = ihb[1].SampleDistance.V;                                   

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;                                             

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  Delta = option_float ( SDelta, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Eta = option_float ( SEta, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Chi = option_float ( SChi, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  delta = (double) Delta.V;
  eta   = (double) Eta.V;
  chi   = (double) Chi.V;    
  
  refract_Init ( stdout, delta, eta, chi );

  if (pcb->TestBit) {
    refract_PrintParams ( stdout );
    printf("\n"); }                                                             

  /* loop over the output array  */
  for (i_2=0;i_2<I0Dim_2;i_2++) {
    W_2 =  WORLD( i_2, Off_2[0], Ps_2[0] );
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    for (i_1=0;i_1<I0Dim_1;i_1++) {
      W_1 =  WORLD( i_1, Off_1[0], Ps_1[0] ); 

      tmp = W_1*W_1 + W_2*W_2; 
      W   = sqrt(tmp);
      WL  = sqrt(tmp+L*L);

      Beam.cosAlpha    = W_1/W;
      Beam.sinAlpha    = W_2/W;
      Beam.cosTwoTheta = L/WL;
      Beam.sinTwoTheta = W/WL;

      if (pcb->TestBit>3) {
        printf("Incident beam\n");
        refract_PrintBeam ( stdout, Beam );
        printf("\n"); }                                                                                                                           
      Beam1 = refract_Angles ( Beam );

      if (!Beam1.status) { 

        if (pcb->TestBit>3) {
          printf("Incident beam\n");
          refract_PrintBeam ( stdout, Beam );
          printf("\n"); }                                                                                                                       
        if (Beam1.cosTwoTheta>0.0) {

          W1   = L1 * Beam1.sinTwoTheta/Beam1.cosTwoTheta;
          W1_1 = W1 * Beam1.cosAlpha;
          W1_2 = W1 * Beam1.sinAlpha;

          f1_1 = INDEX(W1_1, Off_1[1], Ps_1[1]);
          f1_2 = INDEX(W1_2, Off_2[1], Ps_2[1]);

          if ( E0Data ) {
            if ( Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                   f1_1-0.5, f1_2-0.5, f1_1+0.5, f1_2+0.5,
                   &Sum, &Weight, &VarSum, &VarWeight) ) {
              /* then do something with the data */
              Value = Sum/Weight;
              UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
              if (VarSum>=0.0) {
                E0Value = VarSum/(VarWeight*VarWeight);
                UPDATE( *pE0Data, E0Value, I0Dummy, I0DDummy );
              }
            } /* if Isum ... */
          } else {
            if (Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
              f1_1, f1_2, &Value)) {
              /* then do something with the data */

              UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

            } /* if (Ipol2d ... */
          }

        } /* if Beam1.cosTwoTheta ... */
      } /* !Beam1.status */

      pI0Data++;
      pE0Data++;
    } /* for i_1 ... */
  } /* for i_2 ... */

} /* saxs_refract*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_flexp *pDelta, *pEta, *pChi;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : difference from refraction index */
  pDelta = (IO_flexp*) option_parameter_search( SDelta, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"delta (refraction index n = 1-delta)",
             pDelta,pDelta->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sample inclination */
  pEta = (IO_flexp*) option_parameter_search( SEta, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"sample inclination",
             pEta,pEta->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sample tilt */
  pChi = (IO_flexp*) option_parameter_search( SChi, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"sample tilt",
             pChi,pChi->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (pDelta->I)       printf("delta = 1 - n      : %s\n",pDelta->V);
  if (pEta->I)         printf("sample inclination : %s\n", pEta->V);
  if (pChi->I)         printf("sample tilt        : %s\n",pChi->V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_refract
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  cb.RSys.V = IO_Normal; /* default */

 /* Define special options */
  option_define ( IO_tpflexp, SDelta, "0.0", &cb );
  option_define ( IO_tpflexp, SEta, "0.0_deg", &cb );
  option_define ( IO_tpflexp, SChi,  "90.0_deg", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_refract, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

