/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
NAME

    ascii2saxs 

PURPOSE
  Conversion of ascii tables to float

ARGUMENTS
  Usage: ascii2saxs [-h] [options] <ascii file> {<ascii file>}

OPTIONS
  -h/+h        help
  verbose    = 0|1|2|... verbose
  bskp       = <number of bytes to skip> before start of the ascii data
  lskp       = <number of lines to skip> in the ascii data
  cskp       = <number of characters to skip> after line skips
  skipcol    = <number of columns to skip>
  skiprow    = <number of rows to skip>
  dim        = <dim1>x<dim2>: array dimensions
  ori        = <rasterorientation>: raster orientation
  dum        = <dummy>: output dummy value
  commentset = <commentset>: comment character set
  delimiter  = <delimiterset>: delimiter character set
  pipe       = 0|1: read file names from stdin (stop with <ctrl-d>)
  ext        = <extension>: output file extension (default .edf)

HISTORY
  2007-12-05 V1.1 PB maxloop calculation corrected, maxloop is at least 1

---*/
# define Version  "ascii2saxs V1.1 2007-12-05 Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "edfio.h"
# include "readascii.h"

void ascii2saxshelp( FILE * out, const char * progname, int verbose )
{
  fprintf(out,"Version: %s\n",Version);
  fprintf(out,"         %s\n",RA_version());
  fprintf(out,"Usage: %s [-h] [options] <ascii file> {<ascii file>}\n",progname);

  if (!verbose) return;
  fprintf(out,"   verbose=0|1|2|... verbose\n");
  fprintf(out,"   bskp=<number of bytes to skip> before start of the ascii data\n");
  fprintf(out,"   lskp=<number of lines to skip> in the ascii data\n");
  fprintf(out,"   cskp=<number of characters to skip> after line skips\n");
  fprintf(out,"   skipcol=<number of columns to skip>\n");
  fprintf(out,"   skiprow=<number of rows to skip>\n");
  fprintf(out,"   dim=<dim1>x<dim2>: array dimensions\n");
  fprintf(out,"   ori=<rasterorientation>: raster orientation\n");
  fprintf(out,"   dum=<dummy>: output dummy value\n");
  fprintf(out,"   commentset=<commentset>: comment character set\n");
  fprintf(out,"   delimiter=<delimiterset>: delimiter character set\n");
  fprintf(out,"   +h  : this help\n");
  fprintf(out,"   pipe=0|1: read file names from stdin (stop with <ctrl-d>)\n");
  fprintf(out,"   ext=<extension>: output file extension (default .edf)\n");

} // ascii2saxshelp

/* removes trailing line feed \n */
char * trimlf ( char * s )
{ size_t slen;
  if ( s ) {
    slen = strlen( s );
    if ( slen > 0 )
      if (s[slen-1]=='\n') s[slen-1]='\0';
  }
  return ( s );
} // trimlf

/* returns 1 if s is an empty string (only white space) otherwise 0. */
int isempty( const char *s )
{ const char *ps;
  int empty=1;

  if (s) {
    ps = s;
    while (*ps) { empty = empty && isspace(*ps); ps++; }
  }
  return( empty );

} //isempty

/*+++------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_ascii2saxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{ char *nul = (char *) NULL;
  char **pargv;
  int status=-1;
  const char *ps;

  long pipe=0, verbose=1; 
  long dim1=-1, dim2=-1, ori=-1, tmp, tmp2;
  long bskp=-1, lskp=-1, cskp=-1, skipcol=-1, skiprow=-1;
  float dummy=0.0, ftmp;
  const char *delimiterset=(char *) NULL;
  const char *commentset=(char *) NULL;

  long nread, cnt, i1, i2;

  float *data;

  RADescr * descr;

  char *pstring[2];

  char *argument="";


  #define BUFLEN 1024
  char buffer[BUFLEN];

  int  errval;
  long fst, lst, inc, num;
  long loop, maxloop;
  char number[BUFLEN];
  char fname_in[BUFLEN];
  char input[BUFLEN];
  char fname_body[BUFLEN];
  char fname_outext[BUFLEN]={ 'e', 'd', 'f' ,'\0' }; // edf
  char fname_out[BUFLEN];
  char output[BUFLEN];

  int stream;
  long dim[3];

  // count options or help
  pargv = &argv[1];

  for (;*pargv!=nul;pargv++) {
    argument = *pargv;

    if (( strncmp( argument,"-h",2 )==0 )||( strncmp( argument,"--help",6 )==0 )) {
      // help
      ascii2saxshelp( stdout, argv[0], 0 );
      return( -2 );
    } else if ( strncmp( argument,"+h",2 )==0 ) {
      // help
      ascii2saxshelp( stdout, argv[0], 1 );
      return( -2 );
    } else if ( strncmp( argument,"-v",2 )==0 ) {
      // set verbose level to 0
      verbose=0;
    } else if ( strncmp( argument,"+v",2 )==0 ) {
      // set verbose level to 1
      verbose=1;
    } else if ( strncmp( argument,"verbose=", 8 ) == 0 ) {
      // set verbose level
      ps=argument+8;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) verbose=tmp; else return(status);
    } else if ( strncmp( argument,"dim=", 4 ) == 0 ) {
      // set dimension : format dim=1x2 to set dim1 and dim2, 
      //                        dim=x2 to set only dim2, 
      //                        dim=1x to set only dim1
      ps=argument+4;
      nread=sscanf(ps,"%ldx%ld",&tmp,&tmp2);
      if (nread>0) {
        dim1=tmp;
        if (nread>1) dim2=tmp2;
      } else {
        nread=sscanf(ps,"%ldx",&tmp);
        if (nread>0) {
          dim1=tmp; 
        } else {
          nread=sscanf(ps,"x%ld",&tmp2);
          if (nread>0) dim2=tmp2;
          else return(status);
        }
      }
    } else if ( strncmp( argument,"ori=", 4 ) == 0 ) {
      // set raster orientation
      ps=argument+4;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) ori=tmp; else return(status);
    } else if ( strncmp( argument,"dum=", 4 ) == 0 ) {
      // set dummy value
      ps=argument+4;
      nread=sscanf(ps,"%g",&ftmp);
      if (nread>0) dummy=ftmp; else return(status);
    } else if ( strncmp( argument,"delimiter=", 10 ) == 0 ) {
      // set delimiterset
      ps=argument+4;
      delimiterset=ps;
    } else if ( strncmp( argument,"pipe=", 5 ) == 0 ) {
      // set pipe 
      ps=argument+5;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) pipe=tmp; else return(status);
    } else if ( strncmp( argument,"bskp=", 5 ) == 0 ) {
      // set number of bytes to skip before the start of the ascii data
      ps=argument+5;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) bskp=tmp; else return(status);
    } else if ( strncmp( argument,"lskp=", 5 ) == 0 ) {
      // set number of lines to skip in the ascii data
      ps=argument+5;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) lskp=tmp; else return(status);
    } else if ( strncmp( argument,"cskp=", 5 ) == 0 ) {
      // set number of bytes to skip  after the lines have been skipped
      ps=argument+5;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) cskp=tmp; else return(status);
    } else if ( strncmp( argument,"skipcol=", 8 ) == 0 ) {
      // number of columns to skip
      ps=argument+8;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) skipcol=tmp; else return(status);
    } else if ( strncmp( argument,"skiprow=", 8 ) == 0 ) {
      // number of rows to skip
      ps=argument+8;
      nread=sscanf(ps,"%ld",&tmp);
      if (nread>0) skiprow=tmp; else return(status);
    } else if ( strncmp( argument,"comment=", 8 ) == 0 ) {
      // read comment characters 
      ps=argument+8;
      commentset=ps;
    } else if ( strncmp( argument,"ext=", 4 ) == 0 ) {
      // read comment characters 
      ps=argument+4;
      strncpy(fname_outext,ps,BUFLEN-1);fname_outext[BUFLEN-1]='\0';

    } else {

      /* File names BEGIN*/
        // extract loop parameters from file name pattern
        fst=0;
        ftmp = num_str2long (filename_parameter(number,BUFLEN,argument,1),NULL,&errval);
        if (!errval) fst=ftmp;
        lst=fst;
        ftmp = num_str2long (filename_parameter(number,BUFLEN,argument,2),NULL,&errval);
        if (!errval) lst=ftmp;
        inc=1;
        ftmp = num_str2long (filename_parameter(number,BUFLEN,argument,3),NULL,&errval);
        if (!errval) inc=ftmp;

        maxloop=0;
        if (inc!=0) maxloop = (lst-fst)/inc+1; 
        if (maxloop<=0) maxloop=1;

        // isolate file name body and append extension fname_outext for output file 
        // fname_in
        filename_parameter(fname_in,BUFLEN,argument,0);

        // fname_body
        filename_body ( fname_body, BUFLEN, filename_name ( fname_body, BUFLEN, fname_in) );

        // fname_out
        strncpy(fname_out,fname_body,BUFLEN-2-strlen(fname_outext));
        strcat(fname_out, ".");
        strcat(fname_out, fname_outext);
        fname_out[BUFLEN-1]='\0';

        if (verbose>1) {
          printf("   in=\'%s\', out=\'%s\'\n",fname_in,fname_out); 
          printf("   fst=%ld, lst=%ld, inc=%ld, maxloop=%ld\n",
            fst,lst,inc,maxloop);
        }

      /* File names END*/

      for (loop=0,num=fst;loop<maxloop;loop++,num+=inc) {

        // input
        filename_pattern ( input, BUFLEN, fname_in, num );
        // output
        filename_pattern ( output, BUFLEN, fname_out, num );

        // convert ascii file
        descr=RA_openfile( input, verbose, &status );
        if (status) {
          fprintf(stderr,"Error opening \'%s\'\n",input);
          return(status);
        }

        RA_setbskp( descr, bskp );
        RA_setlskp( descr, lskp );
        RA_setcskp( descr, cskp );
        RA_setskipcol( descr, skipcol );
        RA_setskiprow( descr, skiprow );
        RA_setdim1( descr, dim1 );
        RA_setdim2( descr, dim2 );
        RA_setori( descr, ori );

        RA_setdummy( descr, dummy );
        RA_setdelimiterset( descr, delimiterset );
        RA_setcommentset( descr, commentset );
      
        nread=RA_readfile( descr, &data, &status );
        if (status) {
          fprintf(stderr,"Error reading \'%s\'\n",input);
          RA_closefile( descr, NULL );
          return(status);
        }

        if (verbose>1) printf("%ld elements read\n",nread);

        if (verbose>4) {
          cnt=0;
          for (i2=0;i2<descr->dim2;i2++)
            for (i1=0;i1<descr->dim1;i1++)
              printf("array[%ld,%ld]=%g\n",i1+1,i2+1,data[cnt++]);
        }

        /* EDF file output BEGIN */

        if (verbose>0) printf("    Writing file \'%s\'\n", output);

        // open edf file
        stream=edf_open_data_file  ( output, "new", &errval, &status );
        if (status) {
          fprintf(stderr,"Error opening \'%s\'\n",output);
          return(status);
        }

        // write dummy value into header 
        edf_write_header_float ( stream, 1,1, "Dummy", descr->dummy, 
                                 &errval, &status );
        if (status) {
          fprintf(stderr,"Error writing keyword Dummy to \'%s\'\n",output);
          return(status);
        }

        // write orientation into header
        edf_write_header_float ( stream, 1,1, "RasterOrientation", descr->ori,
                                 &errval, &status );
        if (status) {
          fprintf(stderr,"Error writing keyword RasterOrientation to \'%s\'\n",output);
          return(status);
        }

        // write buffer to edf file
        dim[0]=2;dim[1]=descr->dim1;dim[2]=descr->dim2;
        edf_write_data ( stream, 1, 1, dim, descr->buffer, MFloat, 
                         &errval, &status );
        if (status) {
          fprintf(stderr,"Error writing data to \'%s\'\n",output);
          return(status);
        }

        // close edf file
        edf_close_data_file ( stream, &errval, &status );
        if (status) {
          fprintf(stderr,"Error closing \'%s\'\n",output);
          return(status);
        }

        /* EDF file output END */

        RA_closefile( descr, &status );
        if (status) {
          fprintf(stderr,"Error closing \'%s\'\n",input);
          return(status);
        }
      } // loop
    }
  }

  if (pipe) {
    // read from standard input
    pstring[1] = (char *) NULL;

    while ( !feof( stdin ) ) {
      pstring[0] = fgets ( buffer, BUFLEN, stdin );
      pstring[0] = trimlf ( pstring[0] );

      argument = pstring[0];

      if ( argument ) {
        if ((!strncmp(argument,"exit",4)) || (!strncmp(argument,"quit",4)))
          break;

        if (!isempty (argument)) {

          /* File names BEGIN*/
            fst=0;
            ftmp = num_str2long (filename_parameter(number,BUFLEN,argument,1),NULL,&errval);
            if (!errval) fst=ftmp;
            lst=fst;
            ftmp = num_str2long (filename_parameter(number,BUFLEN,argument,2),NULL,&errval);
            if (!errval) lst=ftmp;
            inc=1;
            ftmp = num_str2long (filename_parameter(number,BUFLEN,argument,3),NULL,&errval);
            if (!errval) inc=ftmp;
  
            maxloop=0;
            if (inc!=0) maxloop = (lst-fst+1)/inc;
            if (maxloop==0) maxloop=1;

            // fname_in
            filename_parameter(fname_in,BUFLEN,argument,0);

            // fname_body
            filename_body ( fname_body, BUFLEN, filename_name ( fname_body, BUFLEN, fname_in) );

            // fname_out
            strncpy(fname_out,fname_body,BUFLEN-2-strlen(fname_outext));
            strcat(fname_out, ".");
            strcat(fname_out, fname_outext);
            fname_out[BUFLEN-1]='\0';

            if (verbose>1) {
              printf("   in=\'%s\', out=\'%s\'\n",fname_in,fname_out);
              printf("   fst=%ld, lst=%ld, inc=%ld, maxloop=%ld\n",
                fst,lst,inc,maxloop);
            }

          /* File names END*/

          for (loop=0,num=fst;loop<maxloop;loop++,num+=inc) {

            // input
            filename_pattern ( input, BUFLEN, fname_in, num );
            // output
            filename_pattern ( output, BUFLEN, fname_out, num );

            // convert ascii file
            descr=RA_openfile( input, verbose, &status );
            if (status) {
              fprintf(stderr,"Error opening \'%s\' (status=%d)\n",input,status);
              continue;
            }

            RA_setbskp( descr, bskp );
            RA_setlskp( descr, lskp );
            RA_setcskp( descr, cskp );
            RA_setskipcol( descr, skipcol );
            RA_setskiprow( descr, skiprow );
            RA_setdim1( descr, dim1 );
            RA_setdim2( descr, dim2 );
            RA_setori( descr, ori );
            RA_setdummy( descr, dummy );
            RA_setdelimiterset( descr, delimiterset );
            RA_setcommentset( descr, commentset );

            nread=RA_readfile( descr, &data, &status );
            if (status) {
              RA_closefile( descr, NULL );
              fprintf(stderr,"Error reading \'%s\' (status=%d)\n",input,status);
              continue;
            }

            if (verbose>1) printf("%ld elements read\n",nread);

            if (verbose>4) {
              cnt=0;
              for (i2=0;i2<descr->dim2;i2++)
                for (i1=0;i1<descr->dim1;i1++)
                  printf("array[%ld,%ld]=%g\n",i1+1,i2+1,data[cnt++]);
            }

            /* EDF file output BEGIN */

            if (verbose>0) printf("    Writing file \'%s\'\n", output);

            // open edf file
            stream=edf_open_data_file  ( output, "new", &errval, &status );
            if (status) {
              fprintf(stderr,"Error opening \'%s\'\n",output);
              return(status);
            }

            // write dummy value into header
            edf_write_header_float ( stream, 1,1, "Dummy", descr->dummy,
                                     &errval, &status );
            if (status) {
              fprintf(stderr,"Error writing keyword to \'%s\'\n",output);
              return(status);
            }

            // write orientation into header
            edf_write_header_float ( stream, 1,1, "RasterOrientation", descr->ori,
                                     &errval, &status );
            if (status) {
              fprintf(stderr,"Error writing keyword RasterOrientation to \'%s\'\n",output);
              return(status);
            }

            // write buffer to edf file
            dim[0]=2;dim[1]=descr->dim1;dim[2]=descr->dim2;
            edf_write_data ( stream, 1, 1, dim, descr->buffer, MFloat,
                             &errval, &status );
            if (status) {
              fprintf(stderr,"Error writing data to \'%s\'\n",output);
              return(status);
            }

            // close edf file
            edf_close_data_file ( stream, &errval, &status );
            if (status) {
              fprintf(stderr,"Error closing \'%s\'\n",output);
              return(status);
            }

            /* EDF file output END */

            RA_closefile( descr, &status );
            if (status) {
              fprintf(stderr,"Error closing \'%s\' (status=%d)\n",input,status);
            }

          } // loop
        } else fprintf( stdout,"\n");
      }
    }
  }

  if (argument=="") ascii2saxshelp( stdout, argv[0], 0 );

  return( status );

} /* MAIN */
