/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsImageLoop.inc

2 DESCRIPTION
  Private include file of "SaxsImage.c".
  See PUBLIC function for details.

3 CALL
  void ImageLoop( CmdBlk * pcb, ImgBlk ib[],
                  void (*Function) (CmdBlk *, long [], ImgHeadBlk [], int *),
                  void (*InitFunction) (CmdBlk*, long [], ImgHeadBlk [], int*),
                  int allneeded, int * pstatus );

2 HISTORY
  12-Oct-1996 Peter Boesecke extracted from SaxsImage.c
  14-Nov-1996 PB mark_...
  21-Dec-1999 PB DummyFileName, IO_Dummy
  18-Mar-2000 PB default for output memory (saxs_image_loop) 
  30-Dec-2000 PB FreeImage removed for all input images 
  03-Jul-2001 PB FullFileName(in,out) -> !filename_full( out, IO_LEN, in )
  06-Jul-2001 PB filename_full now in OpenImageFile
  08-Jul-2001 PB OpenImageFile(...,FileNumber,...)
  09-Jul-2001 PB loop over file numbers, repetitions removed
  09-Jul-2001 PB single2multi, multi2single
  10-Jul-2001 PB saxs_new_all_images, saxs_open_all_images, 
                 saxs_read_all_images: only required images are read
  11-Jul-2001 PB single2multi, multi2single
  11-Jul-2001 PB NewImage only when output image header empty
                 control parameter Add
  12-Jul-2001 PB saxs_open_all_images opens only new output file when
                 image block ihb[0] is empty (!ihb[0].I).
  19-Aug-2001 PB WriteImage argument list has changed
  30-Aug-2001 PB single2multi -> SaxsImageLoop_single2multi
                 multi2single -> SaxsImageLoop_multi2single
                 ImageIs..., ImageToWrite
  2002-06-09  PB LoopNumber
  2002-09-01  PB saxs_image_loop modified to save existing images
                 at the end of a loop
  2003-02-02  PB new parameter lists of ReadImage, NewImage
  2003-04-06  PB saxs_read_all_images: memory info added  
  2003-04-06  PB default: FLast[1] = FFirst[1] instead of FLast[1] = 1l
  2003-04-13  PB ImageLoop: 
                   single2multi: searching range of input images,
                                 using ImageNumbers of all input images
                                 as defaults for FileNumbers
  2003-04-13  PB ImageLoop: 
                   single2multi: increments of all input images set to 0
  2003-04-15  PB functions: single2multi, multi2single
  2003-05-30  PB ImageLoop: parameter long Num[] added in Function
  2004-04-10  PB ImageLoop: OpenImageFile(...,ib[1].OpenMode.V|...)
                            ib[1].OpenMode.V added (for binary2saxs)
  2005-07-14  PB SetSaxsError without seb
  2005-08-04  PB PrintImageHeaderBlock parameters updated 
  2005-08-05  PB saxs_new_all_images: VarDat
  2005-08-06  PB saxs_image_loop: default of CreateVariance is set
  2005-09-14  PB saxs_image_loop: set CreateVariance if any ErrVal.I flag
                 is set.
  2005-10-13  PB saxs_image_loop: unnecessary semicolon removed (for cc)
  2007-04-19  PB -Wall compiler warnings resolved
  2007-04-20  PB ImageLoop, saxs_image_loop: malloc corrected (items were
                 defined as int, but used as long)
  2007-06-10  PB ImageLoop, saxs_image_loop: InitFunction added to parameter
                 list
  2007-07-17  PB saxs_image_loop: pcb->Cycles
  2010-11-08  PB saxs_open_all_images, saxs_read_all_images:
                 replace header keys in file names,
                 to change image block and parameter numbers
                 filename.c needs to be adapted. In order to read the
                 filename parameters it must ignore everything between 
                 square brackets []
  2011-06-06  PB CommonOrientation added
  2011-07-16  PB sx_tf_img: rot=1
  2012-01-29  PB If the control parameter Add.V is Null all images
                 are added, e.g. -add 0.
                 ImageToWrite replaced by write_now

---*/

/****************************************************************************
*  External Functions                                                          *
****************************************************************************/

void free_image_memory ( ImgHeadBlk *pihb );

/****************************************************************************
*  Defines                                                                  *
****************************************************************************/

#ifndef PRIVATE
#  define PRIVATE static
#endif

#ifndef PUBLIC
#  define PUBLIC
#endif

/*---------------------------------------------------------------------------
 Static variables 
----------------------------------------------------------------------------*/
PRIVATE int SaxsImageLoop_single2multi = 0, SaxsImageLoop_multi2single = 0;

/*---------------------------------------------------------------------------
1 single2multi
 
2 DESCRIPTION
  Returns TRUE if no input image has a pattern and if the output image has
  a pattern
 
  CmdBlk *pcb : pointer to control block
  ImgBlk ib[] : image block array

----------------------------------------------------------------------------*/
int single2multi( CmdBlk *pcb, ImgBlk ib[] )
{ int value;
  int f, fmax;
  int fhpi = 0, fhpo = 0;

  fmax = pcb->ImgBlkLen;

  if (fmax>1) {
    for (f=1;f<fmax;f++) {
      fhpi += filename_has_pattern(ib[f].Name.V);
    }
    fhpo = filename_has_pattern(ib[0].Name.V);
  }

  value = (!fhpi)&&fhpo;

  return( value );

} // single2multi

/*---------------------------------------------------------------------------
1 multi2single
 
2 DESCRIPTION
  Returns TRUE if at least one input image has a pattern and if the output 
  image has not a pattern
 
  CmdBlk *pcb : pointer to control block
  ImgBlk ib[] : image block array
 
----------------------------------------------------------------------------*/
int multi2single( CmdBlk *pcb, ImgBlk ib[] )
{ int value;
  int f, fmax;
  int fhpi = 0, fhpo = 0;
 
  fmax = pcb->ImgBlkLen;
 
  if (fmax>1) {
    for (f=1;f<fmax;f++) {
      fhpi += filename_has_pattern(ib[f].Name.V);
    }
    fhpo = filename_has_pattern(ib[0].Name.V);
  }
 
  value = fhpi&&(!fhpo);
 
  return( value );
 
} // multi2single

int Single2Multi ( void )
{ return( SaxsImageLoop_single2multi );
} // Single2Multi

int Multi2Single ( void )
{ return( SaxsImageLoop_multi2single );
} // Multi2Single

/*---------------------------------------------------------------------------
1 ImageIsNew
 
2 DESCRIPTION
  Return TRUE if ActualLoopCount <= 0, otherwise FALSE 
  ActualLoopCount is only incremented in loops in which an image was really used. 
  LoopCount is incremented in every loop. 
  If ActualLoopCount is Zero the image is in use the first time.
 
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/
int ImageIsNew( int blkno, ImgHeadBlk ihb[] )
{ return((ihb[blkno].ActualLoopCount>0)?FALSE:TRUE);
} // ImageIsNew

/*---------------------------------------------------------------------------
1 image_to_write 
 
2 DESCRIPTION
  Decides whether an output image (blkno 0) must be written next time. If not 
  in addition mode (!Add.I) and if oinc!=0 all existing images are written.
  If oinc==0 (ib[0].Inc.V) and ofinc!=0 (ib[0].FInc.V) existing images are 
  only written once at the end of an image loop (loop=loopmax).
  If oinc==0 and ofinc==0 existing images are only written once at
  the end of a file loop ((loop==loopmax) and (floop==floopmax)).
  In addition mode (Add.I), every Add.Vs image number and the very 
  last image is written (loop==loopmax)&&(floop==floopmax). Image numbers 
  are counted with LoopCount (ihb[0].LoopCount).
  larger than the number
 
  CmdBlk *pcb           (i) : command block
  ImgBlk ib[]           (i) : image block
  int blkno             (i) : blkno
  ImgHeadBlk ihb[]      (i) : image header block
  long floop            (i) : file loop counter
  long floopmax         (i) : file loop counter maximum
  long loop             (i) : loop counter
  long loopmax          (i) : loop counter maximum

2 RETURN VALUE
  int    FALSE : do not write
         TRUE  : write next time

----------------------------------------------------------------------------*/
int image_to_write ( CmdBlk *pcb, ImgBlk ib[], int blkno, ImgHeadBlk ihb[], 
                     long floop, long floopmax, long loop, long loopmax )
{ int value=FALSE;
  if ( blkno==0 ) {
    if (pcb->Add.I) {
      if ( ( ( pcb->Add.V != 0 ) &&
             ( ihb[blkno].LoopCount >= pcb->Add.V ) ) ||
           ( (loop==loopmax) && (floop==floopmax) ) )
        value = TRUE;
    } else {
      if ( (ib[blkno].Inc.V !=0l) ||
         ( (loop==loopmax) && ((ib[0].FInc.V!=0l)||(floop==floopmax)) ) )
        value = TRUE;
    }
  }
  return( value );
} // image_to_write

/*---------------------------------------------------------------------------
1 ImageIsFirst 
 
2 DESCRIPTION
  Return TRUE, when image is first image in input or output file 
 
  CmdBlk * pcb          (i)   : control block
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/
int ImageIsFirst( CmdBlk * pcb, int blkno, ImgHeadBlk ihb[] )
{ int value;
  value = (ihb[blkno].ImNum == pcb->ib[blkno].First.V)?1:0;
  if ((SaxsImageLoop_multi2single)&&(blkno==0)) { // if blkno output block no
    value = (pcb->ib[0].FileNumber == pcb->ib[0].FFirst.V)?value:0; 
    }
  return( value );
} // ImageIsFirst 

/*---------------------------------------------------------------------------
1 ImageIsLast
 
2 DESCRIPTION
  Return TRUE, when image is last image in input or output file
 
  CmdBlk * pcb          (i)   : control block
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/
int ImageIsLast( CmdBlk * pcb, int blkno, ImgHeadBlk ihb[] )
{ int value;
  value = ((ihb[blkno].ImNum==pcb->ib[blkno].Last.V)||
           (pcb->ib[blkno].Inc.V==0))?1:0;
  if ((SaxsImageLoop_multi2single)&&(blkno==0)) { // if blkno output block no
    value = ((pcb->ib[0].FileNumber==pcb->ib[0].FLast.V)||
             (pcb->ib[0].FInc.V==0))?value:0;
    }
  return( value );
} // ImageIsLast

/*---------------------------------------------------------------------------
1 saxs_free_all_images

2 DESCRIPTION
  Releases memories of all initialized images

  ImgHeadBlk *pihb      (i/o)   : image header block

2 HISTORY
  24-Nov-1995   Peter Boesecke creation
  10-Jul-2001   PB release only images of initialized blocks

----------------------------------------------------------------------------*/
void saxs_free_all_images ( CmdBlk * pcb, ImgHeadBlk ihb[] )
/* Release all image memory */
{ register int i;
  for (i=0;i<pcb->ImgBlkLen;i++) {
    free_image_memory( &ihb[i] );
  }
} /* saxs_free_all_images */

/*---------------------------------------------------------------------------
1 saxs_new_all_images

2 DESCRIPTION
  Allocates an array with imax image header blocks and sets the
  indicators ihb[i].I to FALSE.

  int *pstatus (o)              : SAXS status

2 RETURN VALUE
  Pointer to created ImgHeadBlk array or NULL in case of an error;

2 HISTORY
  10-Jul-2001 PB 
----------------------------------------------------------------------------*/
ImgHeadBlk * saxs_new_all_images( int imax )
{ int i;
  ImgHeadBlk *ihb;

  ihb = (ImgHeadBlk*) malloc(sizeof(ImgHeadBlk)*imax);
  if (ihb==NULL) return(ihb);

  // Init header block (.I-flag and pointer to Data)
  for (i=0;i<imax;i++) init_header_block( ihb, i ); 

  return( ihb );

} // saxs_new_all_images

/*---------------------------------------------------------------------------
1 saxs_open_all_images

2 DESCRIPTION
  Opens all data files according to the command block cb and the image
  blocks ib[i] (all defined data files from 0 to ImgBlkLen-1). 

  Checks, whether a data file is already opened and closes and reopens it
  only if necessary. 
  If a file cannot be opened the function tries either to open the remaining 
  files (allneeded==FALSE) or stops and returns 0. In case that all 
  required files could be opened the function returns 1. The stream 
  ib[].Stream of an unopened file is -1.  

  return( int )               : 1 for all needed files are opened
                                When 0 is returned saxs_image_loop
                                does not need not be called.

  CmdBlk *pcb           (i)   : ImgBlkLen
  ImgBlk ib[]           (i)   : OpenMode, Name
                        (o)   : Stream
  long int FNum[]       (i)   : FileNumber table (one value for each ib)
  ImgHeadBlk ihb[]      (o)   : image memory is released if file was closed
  int allneeded         (i)   : FALSE : The function tries to open all files 
                                        and skippes files that cannot be 
                                        opened due to a FileNotFound status. 
                                        If no other error has occured the 
                                        function returns 1, otherwise 0.
                                 TRUE : If a file is missing, the function
                                        stops without error and returns 0.
                                        If all files could be opened it 
                                        returns 1.

  int *pstatus (o)              : SAXS status

2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  24-Nov-1995 PB
  09-Jul-2001 PB checks, whether a file is already opened
  10-Jul-2001 PB allneeded,
                 A data array of an image header block of a closed and
                 reopened file is released and the header block flag .I 
                 is set to FALSE.
  11-Jul-2001 PB output image 0 opened at the end
  12-Jul-2001 PB output image file is only opened when image block is empty
  08-Nov-2010 PB open all input images sequentially starting with 1,
                 open output image afterwards, replace header keys
                 in filename (the corresponding block must be already opened),
                 FNum added
----------------------------------------------------------------------------*/
int saxs_open_all_images( CmdBlk *pcb, ImgBlk ib[], long FNum[], 
                          ImgHeadBlk ihb[], int allneeded, int * pstatus )
{ char *fname, buffer[IO_len];
  int imax = pcb->ImgBlkLen;
  register int i;
  int blkno;

  // Open all input images sequentially, starting with 1,
  // afterwards open output image to avoid an empty output file
  for (i=1;i<=imax;i++) {

    if (i<imax) blkno=i; 
    else blkno=0;

    if (!(ib[blkno].OpenMode.I && (IO_DontOpen & ib[blkno].OpenMode.V))) {
      if (!ib[blkno].OpenMode.I)
        printf("WARNING (ImgBlk[% d]) : opening mode missing\n",blkno);
      if (pcb->TestBit) {
        if (!ib[blkno].Name.I)
          printf("INFO (ImgBlk[% d]) : using default file name >>%s<<\n",
            blkno, ib[blkno].Name.V);
      }
    }

    // default: use first parameter (npar=1) of header value in block 1
    // image number 1 (Num==NULL)
    fname = linekeyvalue ( buffer, IO_len, ib[blkno].Name.V, 1,
                           pcb, ib, NULL, 1, pstatus );
    if (*pstatus!=Success) {
      printf("ERROR (ImgBlk[% d]): key replacement failed in >>%s<<.\n",
             blkno,ib[blkno].Name.V);
      return(0);
    }

    if (!AlreadyOpened( pcb,ib,blkno,fname,FNum[blkno],pstatus )) {
      // open only a new output file when image block is empty (!ihb[0])
      if ( (blkno!=0)||(!ihb[blkno].I) ) { 
        // close and open
        CloseImageFile ( pcb, ib, blkno, pstatus);
        if (*pstatus!=Success) return(0);
        // release allocated image data and mark header block as empty
        FreeImage( pcb, blkno, ihb ); 
        OpenImageFile(pcb,ib,blkno,fname,FNum[blkno],
                      ib[blkno].OpenMode.V,pstatus);
        if (*pstatus==FileNotFound) {
          *pstatus=Success;
          if (allneeded) return(0); // some files not found, stop
        }
      if (*pstatus!=Success) return(0);
      }
    }

    if (pcb->TestBit) {
      printf("ib[%d].FileName = %s (stream = %d) opened\n",
             blkno,ib[blkno].FileName,ib[blkno].Stream);
    }

  } // for i=1, ..., i<imax

  *pstatus = Success;

  return(1);

} /* saxs_open_all_images */

/*---------------------------------------------------------------------------
NAME

  saxs_read_all_images --- read all required images 

SYNOPSIS

  int saxs_read_all_images( CmdBlk *pcb, ImgBlk ib[], long FNum[], long Num[],
                            ImgHeadBlk * ihb, int allneeded, int * pstatus )

DESCRIPTION

  Reads all images of ib[i] with image number Num[i] if they are not already
  in ihb[i]. The images must have been opened with saxs_open_all_images.

  Checks, whether an image needs to be read. If ihb[i] does not contain 
  the requested image the old image is released with FreeImage and the
  requested image is read from file.
  If an image cannot be read because the file is closed (stream<0) or
  the image number is not found in the file the image header block .I flag
  is set to false. Depending on the parameter allneeded the routine 
  stops (allneeded==TRUE) or tries to load the resting images 
  (allneeded==FALSE).

  CmdBlk *pcb           (i)   : ImgBlkLen
  ImgBlk ib[]           (i)   : OpenMode, Name
                        (o)   : Stream
  long int FNum[]       (i)   : FileNumber table (one value for each ib)
  long int NUM[]        (i)   : ImageNumber table (one value for each ib)
  ImgHeadBlk ihb[]      (o)   : image memory is released if file was closed
  int allneeded         (i)   : FALSE : The function tries to read all images 
                                        and skippes images that cannot be
                                        found. If no other error has occured 
                                        the function returns 1, otherwise 0.
                                 TRUE : If an image is missing, the function
                                        stops without error and returns 0.
                                        If all files could be opened it
                                        returns 1.

  int *pstatus (o)              : SAXS status

RETURN VALUE

  return( int )               : 1 all needed images have been read or
                                  severe error, status must be checked
                                0 some required images are missing,
                                  no severe error, status does not need
                                  to be checked but function should not 
                                  be called


HISTORY

  10-Jul-2001 Peter Boesecke 
  08-Nov-2010 PB check, whether the correct file is opened (for linekeyvalue),
                 FNum added
----------------------------------------------------------------------------*/
int saxs_read_all_images( CmdBlk *pcb, ImgBlk ib[], long FNum[], long Num[],
                          ImgHeadBlk * ihb, int allneeded, int * pstatus )
{ char *fname, buffer[IO_len];
  int imax = pcb->ImgBlkLen;
  register int i;
  int retval=1;
  int blkno;

  /* read input image(s) */
  for (i=1;i<imax;i++) {
    blkno=i;

    // check whether the correct file is opened, if not, reopen

    // default: use first parameter (npar=1) of header value in block 1
    fname = linekeyvalue ( buffer, IO_len, ib[blkno].Name.V, 1,
                           pcb, ib, Num, 1, pstatus );
    if (*pstatus!=Success) {
      printf("ERROR (ImgBlk[% d]): key replacement failed in >>%s<<.\n",
             blkno,ib[blkno].Name.V);
      return(0);
    }

    if (!AlreadyOpened( pcb,ib,blkno,fname,FNum[blkno],pstatus )) {
      // close and open
      CloseImageFile ( pcb, ib, blkno, pstatus);
      if (*pstatus!=Success) return(0);
      // release allocated image data and mark header block as empty
      FreeImage( pcb, blkno, ihb );
      OpenImageFile(pcb,ib,blkno,fname,FNum[blkno],
                    ib[blkno].OpenMode.V,pstatus);
      if (*pstatus==FileNotFound) {
        *pstatus=Success;
        if (allneeded) return(0); // some files not found, stop
      }
      if (*pstatus!=Success) return(0);
    }

    if (pcb->TestBit) {
      printf("ib[%d].FileName = %s (stream = %d) opened\n",
             blkno,ib[blkno].FileName,ib[blkno].Stream);
    }
    // the correct file is opened now

    // test, whether new image must be loaded
    if ( (ihb[blkno].I)&&(ihb[blkno].ImNum!=Num[blkno]) ) {
      // free image header
      FreeImage( pcb, blkno, ihb );
    }

    if  (!ihb[blkno].I) {
     // comment
      if (!((IO_Dummy | IO_DontReadWrite) & ib[blkno].StreamOpenMode)) {
        if (IO_DontReadData & ib[blkno].StreamOpenMode) {
          if (ib[blkno].Memory.V==1) {
            printf("Reading header: %s, Image : %ld",
                   ib[blkno].FileName, Num[blkno]);
          } else {
            printf("Reading header: %s, Image : %ld, Memory : %ld",
                   ib[blkno].FileName, Num[blkno], ib[blkno].Memory.V);
          }
        } else {
          if (ib[blkno].Memory.V==1) {
            printf("Reading image: %s, Image : %ld",
                   ib[blkno].FileName, Num[blkno]);
          } else {
            printf("Reading image: %s, Image : %ld, Memory : %ld",
                   ib[blkno].FileName, Num[blkno], ib[blkno].Memory.V);
          }
        } // comment
      }

      /* +++ read image ihb[blkno] with image number Num[blkno] */
      if (!(retval=ReadImage(pcb,ib,blkno,Num,ihb,pstatus))) {
        printf(" missing\n");
      } else {
        if (*pstatus!=Success) { printf("\n"); return(1); } // error, check st.
        if (!((IO_Dummy | IO_DontReadWrite ) & ib[blkno].StreamOpenMode))
          printf(" done\n");
      }
      if (allneeded && (!retval)) break; else retval=1;
    }
  } /* for (i=1;... */

  *pstatus = Success;

  return(retval);

} // saxs_read_all_images

/*---------------------------------------------------------------------------
NAME

  saxs_close_all_images

DESCRIPTION

  Closes all data files according to the command block cb and the image
  blocks ib[i].

  close all data files
  return( void )                :

  CmdBlk *pcb           (i)     : ImgBlkLen
  ImgBlk ib[]           (i)     : Stream
                        (o)     : Stream
  int *pstatus		(o)	: SAXS status

HISTORY

  12-Feb-1995   Peter Boesecke creation
  24-Nov-1995 PB
----------------------------------------------------------------------------*/
void saxs_close_all_images( CmdBlk *pcb, ImgBlk ib[], int * pstatus )
{ register int i;

  for (i=0;i<pcb->ImgBlkLen;i++) {
    CloseImageFile(pcb,ib,i,pstatus);
    if (*pstatus!=Success) return;
    }
  *pstatus = Success;

} /* saxs_close_all_images */

/*---------------------------------------------------------------------------
NAME

  saxs_header2sxparams --- header to sx parameters

DESCRIPTION

  const ImgHeadBlk *pihb (i)   : pointer to image header block
  SXParams *params       (o)   : pointer to sx params structure

  int *pstatus           (o)   : SAXS status

HISTORY

  2011-06-06 PB
----------------------------------------------------------------------------*/
void saxs_header2params(const ImgHeadBlk *pihb, SXParams *params, int *pstatus)
{
  if ((!pihb)||(!params)||(!pstatus)) {
    *pstatus=Failed; // all pointers must be given
    return;
  }

  sx_new( params ); // initialize

  params->pro.I = pihb->Projection.I;
    params->pro.V = pihb->Projection.V;
  params->ori.I = pihb->Orientation.I;
    params->ori.V = pihb->Orientation.V;
  params->axis1.I = pihb->AxisType[1].I;
    params->axis1.V = pihb->AxisType[1].V;
  params->axis2.I = pihb->AxisType[2].I; // axis types
    params->axis2.V = pihb->AxisType[2].V;
  params->dim1.I = 1; // dimensions of 2d array 
    params->dim1.V = pihb->Dim[1];
  params->dim2.I = 1;
    params->dim2.V = pihb->Dim[2];
  params->off1.I = pihb->Offset[1].I;  // offsets of array coordinates
    params->off1.V = pihb->Offset[1].V;
  params->off2.I = pihb->Offset[2].I;
    params->off2.V = pihb->Offset[2].V;
  params->bis1.I = pihb->BinSiz[1].I; // binning sizes
    params->bis1.V = pihb->BinSiz[1].V;
  params->bis2.I = pihb->BinSiz[2].I;
    params->bis2.V = pihb->BinSiz[2].V;
  params->ras1.I = 0; // raster region of 2d array
    params->ras1.V = 0l;
  params->ras2.I = 0;
    params->ras2.V = 0l;
  params->pix1.I = pihb->PixSiz[1].I;
    params->pix1.V = pihb->PixSiz[1].V;
  params->pix2.I = pihb->PixSiz[2].I; // pixel sizes [m]
    params->pix2.V = pihb->PixSiz[2].V;
  params->cen1.I = pihb->Center[1].I; // PONI (point of normal incidence)
    params->cen1.V = pihb->Center[1].V;
  params->cen2.I = pihb->Center[2].I;
    params->cen2.V = pihb->Center[2].V;
  params->dis.I = pihb->SampleDistance.I; // distance sample-PONI [m]
    params->dis.V = pihb->SampleDistance.V;
  params->rot1.I = pihb->DetRot[1].I; // detector rotations [rad]
    params->rot1.V = pihb->DetRot[1].V;
  params->rot2.I = pihb->DetRot[2].I;
    params->rot2.V = pihb->DetRot[2].V;
  params->rot3.I = pihb->DetRot[3].I;
    params->rot3.V = pihb->DetRot[3].V;
  params->wvl.I = pihb->WaveLength.I; // wavelength [m]
    params->wvl.V = pihb->WaveLength.V;
  // alternative parameters
  // bcen1, bcen2 : beam center (alt. cen1, cen2)
  // bdis : distance sample-bcen [m] (alt. dis)
  // tilt1, tilt2, tilt3: detector tilts [rad]

  *pstatus = Success;
  
} // saxs_header2params

/*---------------------------------------------------------------------------
NAME

  saxs_params2header --- sx parameters ot header

DESCRIPTION

  ImgHeadBlk *pihb       (o)   : pointer to image header block
  const SXParams *params (i)   : pointer to sx params structure

  int *pstatus           (o)   : SAXS status

HISTORY

  2011-06-06 PB
----------------------------------------------------------------------------*/
void saxs_params2header(ImgHeadBlk *pihb, const SXParams *params, int *pstatus)
{
  if ((!pihb)||(!params)||(!pstatus)) {
    *pstatus=Failed; // all pointers must be given
    return;
  }

  pihb->Projection.I = params->pro.I;
    pihb->Projection.V = params->pro.V;
  pihb->Orientation.I = params->ori.I;
    pihb->Orientation.V = params->ori.V;
  pihb->AxisType[1].I = params->axis1.I; // axis types
    pihb->AxisType[1].V = params->axis1.V;
  pihb->AxisType[2].I = params->axis2.I;
    pihb->AxisType[2].V = params->axis2.V;
  if (params->dim1.I) pihb->Dim[1] = params->dim1.V; // dimensions of 2d array 
  if (params->dim2.I) pihb->Dim[2] = params->dim2.V; 
  pihb->Offset[1].I = params->off1.I;  // offsets of array coordinates
    pihb->Offset[1].V = params->off1.V;
  pihb->Offset[2].I = params->off2.I;
    pihb->Offset[2].V = params->off2.V;
  pihb->BinSiz[1].I = params->bis1.I; // binning sizes
    pihb->BinSiz[1].V = params->bis1.V;
  pihb->BinSiz[2].I = params->bis2.I;
    pihb->BinSiz[2].V = params->bis2.V;
  // ras1, ras2 raster region of 2d array
  pihb->PixSiz[1].I = params->pix1.I; // pixel sizes [m]
    pihb->PixSiz[1].V = params->pix1.V;
  pihb->PixSiz[2].I = params->pix2.I;
    pihb->PixSiz[2].V = params->pix2.V;
  pihb->Center[1].I = params->cen1.I; // PONI (point of normal incidence)
    pihb->Center[1].V = params->cen1.V;
  pihb->Center[2].I = params->cen2.I;
    pihb->Center[2].V = params->cen2.V;
  pihb->SampleDistance.I = params->dis.I; // distance sample-PONI [m]
    pihb->SampleDistance.V = params->dis.V;
  pihb->DetRot[1].I = params->rot1.I; // detector rotations [rad]
    pihb->DetRot[1].V = params->rot1.V;
  pihb->DetRot[2].I = params->rot2.I;
    pihb->DetRot[2].V = params->rot2.V;
  pihb->DetRot[3].I = params->rot3.I;
    pihb->DetRot[3].V = params->rot3.V;
  pihb->WaveLength.I = params->wvl.I; // wavelength [m]
    pihb->WaveLength.V = params->wvl.V;
  // alternative parameters
  // bcen1, bcen2 : beam center (alt. cen1, cen2)
  // bdis : distance sample-bcen [m] (alt. dis)
  // tilt1, tilt2, tilt3: detector tilts [rad]

  *pstatus = Success;

} // saxs_params2header

/*---------------------------------------------------------------------------
NAME

  saxs_tf_all_images --- adjust orientation of all available images

SYNOPSIS

  void saxs_tf_all_images( CmdBlk *pcb, ImgBlk ib[], ImgHeadBlk * ihb, 
                           long ori, int * pstatus )

DESCRIPTION

  Updates headers, data arrays and variance arrays of all available
  image blocks.

  CmdBlk *pcb           (i)   : ImgBlkLen
  ImgBlk ib[]           (i)   : image blocks 
  ImgHeadBlk ihb[]    (i/o)   : image header blocks

  long ori(i)                 : 0 no change of orientation,
                                otherwise common output orientation

  int *pstatus (o)            : SAXS status

RETURN VALUE

  none

HISTORY

  2011-06-06 PB creation
----------------------------------------------------------------------------*/
void saxs_tf_all_images( CmdBlk *pcb, ImgBlk ib[], ImgHeadBlk * ihb, 
                         long ori, int * pstatus )
{ ImgHeadBlk *pihb=NULL;
  SXParams params;
  size_t item_number, item_size;
  int errval;
  int blkno;
  int rot=1;

  if (pcb->TestBit) printf("saxs_tf_all_images BEGIN\n");

  for (blkno=0;blkno<pcb->ImgBlkLen;blkno++) {
    pihb = &(ihb[blkno]);
    /* use only valid blocks */
    if (pihb->I) {
       if (pcb->TestBit) printf("  saxs_header2params\n");
       saxs_header2params( pihb, &params, pstatus );
       if (*pstatus) {
         sprintf(SaxsImageMessage,"saxs_header2params: error\n");
         goto saxs_tf_all_images_error;
       }

       item_number=edf_dim_product(pihb->Dim);
       item_size=sizeof( float );
       if (pcb->TestBit) printf("  sx_tf_img\n");
       sx_tf_img ( &params,
              pihb->Data, pihb->VarDat, item_number, &params,
              pihb->Data, pihb->VarDat, item_size, ori, rot, &errval );
       if (errval) {
         // print errval
         sx_errval2str( SaxsImageMessage, IO_len, errval );
         SetSaxsErrorMessage( SaxsImageMessage );

         *pstatus=Failed;
         goto saxs_tf_all_images_error;
       }

       if (pcb->TestBit) printf("  saxs_params2header\n");
       saxs_params2header(pihb, &params, pstatus);
       if (*pstatus) {
         sprintf(SaxsImageMessage,"saxs_params2header: error\n");
         goto saxs_tf_all_images_error;
       }

    }
  }

  *pstatus = Success;

  if (pcb->TestBit) printf("saxs_tf_all_images END\n");

  return;

saxs_tf_all_images_error:

  if (pcb->TestBit) printf("saxs_tf_all_images END (status=%d)\n",*pstatus);

  return;

} // saxs_tf_all_images

/*---------------------------------------------------------------------------
1 saxs_image_loop

2 PURPOSE
  N input sequences, 1 output sequence.
  Perform sequence analysis: one output image per calculation.
  Output[i0] = Function(image1[i1], image2[i2], image3[i3], ..., imageN[iN])
  In the case that the increment is zero, the corresponding image is only
  one times read at the beginning or one times written at the end:
  The loop is defined as follows,  t also negative increments are possible,
  if First and Last are set respectively:

  ImageLoop(...) {
  for (O=OFirst,I1=I1First,I2=I2First,...;
       O<OLast,I1<I1Last,I2<I2Last,...;O+=OInc,I1+=I1Inc,I2+=I2Inc) {
    Read(I1,I2,...);
    Function((CmdBlk * ) pcb, long num[], (ImgHeadBlk *) ihb,
       (int *) pstatus );
    Write(O); }
  }

  Not existing images are skipped.

2 PARAMETERS
  return value                  : void
  CmdBlk        * pcb           :  ommand block
  ImgBlk        ib[]            : image block
  void (*Function) (CmdBlk *, long [], ImgHeadBlk [], int * ) : image loop
  void (*InitFunction) (CmdBlk *,long [],ImgHeadBlk [],int * ) : initialization
  void (*TermFunction) (CmdBlk *,long [],ImgHeadBlk [],int * ) : termination
  int           allneeded (i)   : FALSE : at least one image must be found,
                                  TRUE : if one image is missing, Function
                                         is not called and data not written
  ImgHeadBlk    ihb[]           : image header block
  long          floop           : (i) current floop number
  long          floopmax        : (i) maximum floop number
                                  If floop is equal to floopmax all existing 
                                  images are written
  int           *pstatus        : exit status

The following parameters of *pcb and ib[] are used:

pcb->ImgBlkLen  Number of image blocks>=1

ib[].First, ib[].Last, ib[].Inc
- default of sequence 1 is: 1, 1, 1
- sequence 1 is default for all following sequences

2 HISTORY
  27-Feb-1995 PB in case of ib[0].Inc==0, the output image is only allocated
                 once (for loop=1) and written only once for (loop=loopmax)
  07-Jun-1995 PB Template only, if at least one input image exist
                 (if (imax>1) Template ...)
  14-Nov-1995 PB For modified saxs_image routines.
  24-Nov-1995 PB small modifications for new routines
  25-Dec-1995 PB loop, loopmax, maxloop, Num, First, Last, Inc : int --> long
  26-Dec-1995 PB long --> long int
  18-Mar-2000 PB ib[0].Memory (default of output memory is input memory) 
  10-Jul-2001 PB saxs_free_all_images removed
  2002-06-09  PB LoopNumber
  2002-09-01  PB Writing output image at the end of the loop, if oinc==0 and
                 if it was created (ActualLoopCount>0). 
                 SearchMinMaxImage added, default of First[1] and 
                 Last[1] is the full range of images (instead of 1l).
                 If all input images are needed (allneeded==TRUE) the 
                 last input image number of block 1 is restricted to 
                 an existing image (LastImage).
                 Additional parameters floop and floopmax
  2007-06-10  PB InitFunction
  2010-11-08  PB FNum added
---------------------------------------------------------------------------*/
void saxs_image_loop( CmdBlk * pcb, ImgBlk ib[], long FNum[],
                 void (*Function) (CmdBlk *, long [], ImgHeadBlk [], int * ), 
                 void (*InitFunction) (CmdBlk *,long [],ImgHeadBlk [],int * ), 
                 void (*TermFunction) (CmdBlk *,long [],ImgHeadBlk [],int * ), 
                 int allneeded, ImgHeadBlk ihb[], 
                 long floop, long floopmax, int * pstatus )
{ char *fname, buffer[IO_len];
  int imax;
  int i;
  int allfound;
  int firstblock;
  long int loop,loopmax;
  long MinImage, MaxImage;
  long FirstImage, LastImage;
  int  errval_is_set;

  long int tmpori=0l;

  int write_now=0; // set to 1 if output image should be written

  /* dynamic allocation of memory */
  long int *maxloop, *Num, *First, *Last, *Inc;

  int ErrorValue;
  *pstatus = NotEnoughMemoryAvailable;

  imax = pcb->ImgBlkLen;
  maxloop = (long int *) malloc(sizeof(long int)*imax); if (maxloop==NULL) return;
  Num     = (long int *) malloc(sizeof(long int)*imax); if (Num==NULL) return;
  First   = (long int *) malloc(sizeof(long int)*imax); if (First==NULL) return;
  Last    = (long int *) malloc(sizeof(long int)*imax); if (Last==NULL) return;
  Inc     = (long int *) malloc(sizeof(long int)*imax); if (Inc==NULL) return;

  *pstatus = Success;

  // sx debug (show data+raster debug)
  if (pcb->TestBit > 1) sx_debug_set( 0x8 );  // +data
  if (pcb->TestBit > 2) sx_debug_set( 0x48 ); // +raster
  if (pcb->TestBit > 3) sx_debug_set( 0x68 ); // +debug

  /* Set errval_is_set if ErrVal.I-flag of any block is set */
  errval_is_set = 0;
  for (i=0;i<imax;i++) {
    if ( ib[i].ErrVal.I ) { errval_is_set = 1; break; }
  }

  /* Choose loop parameters */
  if (imax>1) {

    if (ib[1].Stream>=0) {
      SearchMinMaxImage ( pcb, ib, 1, &MinImage, &MaxImage, pstatus);
      if (*pstatus!=Success) return;
    } else {
      MinImage = 1l; MaxImage = 0l;
    }

    if (ib[1].Inc.I) Inc[1] = ib[1].Inc.V; else Inc[1] = 1l;
    if (Inc[1] < 0l) {
      FirstImage = MaxImage; LastImage = MinImage;
    } else {
      FirstImage = MinImage;  LastImage = MaxImage;
    } 
    if (ib[1].First.I) First[1] = ib[1].First.V; 
      else ib[1].First.V = First[1] = FirstImage;
    if (ib[1].Last.I) Last[1] = ib[1].Last.V; 
      else ib[1].Last.V = Last[1] = LastImage;

    if (allneeded) { // restrict Last[1] to existing image numbers
      if (Inc[1]>0) Last[1] = MIN2(Last[1],LastImage);
        else Last[1] = MAX2(Last[1],LastImage);
    }
    if (!ib[1].Last.I) ib[1].Last.V = Last[1];

    if (ib[0].First.I) First[0] = ib[0].First.V; 
       else { ib[0].First.V = First[0] = First[1]; }
    if (ib[0].Last.I) Last[0] = ib[0].Last.V; 
       else { ib[0].Last.V = Last[0] = Last[1]; }
    if (ib[0].Inc.I) Inc[0] = ib[0].Inc.V; 
       else { ib[0].Inc.V = Inc[0] = Inc[1]; }
    if (!(ib[0].Memory.I)) ib[0].Memory.V = ib[1].Memory.V;
  } else {
    if (ib[0].First.I) First[0] = ib[0].First.V; 
       else ib[0].First.V = First[0] = 1l;
    if (ib[0].Last.I) Last[0] = ib[0].Last.V; 
       else ib[0].Last.V = Last[0] = 1l;
    if (ib[0].Inc.I) Inc[0] = ib[0].Inc.V; 
       else ib[0].Inc.V = Inc[0] = 1l;
  } /* if (imax>1) */

  /* All other sequences */
  for (i=2;i<imax;i++) {
    if (ib[i].First.I) First[i] = ib[i].First.V; else First[i] = First[1];
    if (ib[i].Last.I) Last[i] = ib[i].Last.V; else Last[i] = Last[1];
    if (ib[i].Inc.I) Inc[i] = ib[i].Inc.V; 
      else if (ib[i].Last.I) Inc[i] = Inc[1]; else Inc[i] = 0l;
  } /* for */

  /* Avoid infinite loops in case of Inc[i]==0 by setting maxloop[0]=1 in
     case that all Inc[i]==0 */

  for (i=0;i<imax;i++) {
    if (Inc[i]!=0l)
      maxloop[i] = MAX2(1l,1l + ((Last[i])-(First[i]))/(Inc[i]));
    else maxloop[i] = -1l; /* infinite */
  } /* i */

  loopmax=-1l; /* infinite */
  for (i=0;i<imax;i++)
    if (loopmax>0l) {
      if (maxloop[i]>0) {loopmax=MIN2(loopmax,maxloop[i]);}
    } else {
      if (maxloop[i]>0l) {loopmax=maxloop[i];}
    } /* if (loopmax>0l) */
  loopmax = MAX2(1l,loopmax);

  /* start values */
  for(i=0;i<imax;i++) { Num[i]=First[i]; }
  /* loop */
  for (loop=1l; loop<=loopmax; loop++) {

    /* +++ check input block 1 for variance array */
    if (!pcb->CreateVariance.I) {
      if ( errval_is_set ) pcb->CreateVariance.V = 1;
      else if (imax>=1) {
        /* Use variance arrays, if input block 1 has variance data */
        if (ib[1].Memory.V>0) {
          pcb->CreateVariance.V =
            edf_test_header( ib[1].Stream, Num[1], -ib[1].Memory.V,
                             &ErrorValue, pstatus );
          // In case of an error there was no variance array found
          if (*pstatus!=Success) pcb->CreateVariance.V = 0;
        }
      }
    }

    // +++ read all required images
    saxs_read_all_images( pcb, ib, FNum, Num, ihb, allneeded, pstatus );
    if (*pstatus!=Success) return;

    /* +++ test, whether all required images are available */
    allfound = TRUE; firstblock = 0;
    for (i=imax-1;i>0;i--)
      if (ihb[i].I) firstblock = i; else allfound=FALSE;

    /* +++ tranform all images to common orientation */
    if ( pcb->CommonOrientation.I > 0 ) { // +cori was set
      if ( (pcb->CommonOrientation.V==0l) ) {
        // get orientation of output image
        tmpori = longkeyvalue( ib[0].Orientation.V,
                      pcb, ib, Num, 0, pstatus );
        if (*pstatus!=Success) return;
        if ( (ib[0].Orientation.I) && ( tmpori!=0l ) ) {
          pcb->CommonOrientation.V = tmpori;
        } else if (firstblock) {
          pcb->CommonOrientation.V = ihb[firstblock].Orientation.V;
        }
      }
      if ( (pcb->CommonOrientation.V==0l) )
        pcb->CommonOrientation.V=1l;
    } else pcb->CommonOrientation.V=0l;

    if (pcb->TestBit) 
      printf("Common orientation is %ld\n",pcb->CommonOrientation.V);
     
    // saxs_tf_all_images (no transformation if pcb->CommonOrientation.V is 0)
    saxs_tf_all_images( pcb, ib, ihb, pcb->CommonOrientation.V, pstatus );
    if (*pstatus!=Success) return;

    /* +++ set LoopNumber */
    if (!(SaxsImageLoop_single2multi||SaxsImageLoop_multi2single))
    for(i=0;i<imax;i++) ib[i].LoopNumber = Num[i];
    if (SaxsImageLoop_single2multi) 
      for(i=1;i<imax;i++) ib[i].LoopNumber = Num[i];
    if (SaxsImageLoop_multi2single) ib[0].LoopNumber = Num[0];

    for (i=1;i<imax;i++) {
      if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);
    } /* for (i=1;... */
    /*----------------------------------------------------------------
     Create output image if all needed image headers have been read :
       allneeded=TRUE  : all headers are needed,
       allneeded=FALSE : at least one image must have been found
                         (firstblock != 0).
     allfound is always TRUE if no input images are required (imax=1).
     In the case that ib[0].Inc.V==0 the output header is only created
     once (when loop=1) and written once (when loop=loopmax).
     allfound || (   (!allneeded) && (firstblock!=0) )
     TRUE            FALSE           *               TRUE
     FALSE           FALSE           *               FALSE
     FALSE           TRUE            TRUE            TRUE
     FALSE           TRUE            FALSE           FALSE
    ------------------------------------------------------------------*/
    if ( (allfound) || ( (!allneeded) && (firstblock!=0) ) ) {
      /* Create output image only when ihb is empty (ihb[0].I=0) */

      if ( (!ihb[0].I)&&(!ihb[0].Data) ) {
        // check whether the correct output file is opened, if not, reopen

        // default: use first parameter (npar=1) of header value in block 1
        fname = linekeyvalue ( buffer, IO_len, ib[0].Name.V, 1,
                               pcb, ib, Num, 1, pstatus );
        if (*pstatus!=Success) {
          printf("ERROR (ImgBlk[% d]): key replacement failed in >>%s<<.\n",
                 0,ib[0].Name.V);
          return;
        }

        if (!AlreadyOpened( pcb,ib,0,fname,FNum[0],pstatus )) {
          // close and open
          if (pcb->TestBit) {
            printf("Output file (%s, stream = %d) will be reopened\n",
                   ib[0].FileName,ib[0].Stream);
            printf("All unsaved data will be lost.\n");
          }
          CloseImageFile ( pcb, ib, 0, pstatus);
          if (*pstatus!=Success) return;
          // release allocated image data and mark header block as empty
          FreeImage( pcb, 0, ihb );
          OpenImageFile(pcb,ib,0,fname,FNum[0],
                        ib[0].OpenMode.V,pstatus);
          if (*pstatus==FileNotFound) {
            *pstatus=Success;
            if (allneeded) return; // some files not found, stop
          }
          if (*pstatus!=Success) return;
        }

        if (pcb->TestBit) {
          printf("ib[%d].FileName = %s (stream = %d) opened\n",
                 0,ib[0].FileName,ib[0].Stream);
        }
        // the correct file is opened now

        /* ++++++ ------- */

        /* write number of repetition cycles into command block */
        if (ib[0].Inc.V==0) pcb->Cycles = loopmax; else pcb->Cycles=1l;

        NewImage(pcb,ib,0,Num,ihb,&ihb[firstblock],InitFunction,pstatus);
        if (*pstatus!=Success) return;
      } /* ((ib[0]. ... */

      if (pcb->TestBit) PrintImageHeaderBlock(&ihb[0]);

     /********************* Calculation ********************************/

         SetSaxsError( pcb->MainName, "Function", NULL );
         Function(pcb,Num,ihb,pstatus);
         if (*pstatus!=Success) {return;}

     /******************************************************************/

      if (pcb->TestBit) PrintImageHeaderBlock(&ihb[0]);

      /* incrementation of ActualLoopCount */
      for(i=0;i<imax;i++) ihb[i].ActualLoopCount++;

    } /* if ( (allfound) ... */

    /* incrementation of LoopCount */
    for(i=0;i<imax;i++) ihb[i].LoopCount++;

    // decide whether output image should be written
    if (!ImageIsNew(0,ihb)) { // i.e. do not write an unused image
      write_now = image_to_write ( pcb, ib, 0, ihb, 
                  floop, floopmax, loop, loopmax );
    } else write_now = FALSE;

    if ( write_now ) {

      if (TermFunction) {
        TermFunction(pcb,Num,ihb,pstatus);
        if (*pstatus!=Success) { return; }
      }

      if (!((IO_Dummy | IO_DontReadWrite) & ib[0].StreamOpenMode)) {
        printf("Writing : %s, Image : %ld\n",
                ib[0].FileName, ihb[0].ImNum); }

      WriteImage(pcb,ib,0,ihb,pstatus);
      if (pcb->TestBit) PrintImageHeaderBlock(&ihb[0]);
      if (*pstatus!=Success) { return; }
      printf("\n");

      /* reset common orientation */
      pcb->CommonOrientation.V = 0l;

      /* Free always the output image after it has been written,
         images are always written at the end */
      FreeImage( pcb, 0, ihb ); /* if ((ib[0]. ... */
      pcb->Cycles=0l; /* reset pcb->Cycles */

      write_now = FALSE; /* reset write_now */

    } // if write_now 

    /* incrementation of loop variables */
    for(i=0;i<imax;i++) Num[i]+=Inc[i];

  } /* for (loop ... */

  free(maxloop); free(Num); free(First); free(Last); free(Inc); 

  *pstatus = Success;

} /* saxs_image_loop */

/*---------------------------------------------------------------------------
1 ImageLoop

2 PURPOSE
  Loop over file numbers (FFirst, FLast, FInc)
  open all files (saxs_open_all_images),
  do sequence analysis (see saxs_image_loop),
  close all files (saxs_close_all_images).

  ib[i].LoopNumber is equal to ihb[i].ImNum, except in the case of single2multi
  or multi2single

  single2multi : If the input filename does not contain percent characters
                 as placeholders for numbers and if the output filename
                 contains placeholders for numbers all images of the input
                 file are written to numbered output files. 

                 Input image number parameters (first, last, increment) are 
                 overwritten by input file number (if set). 

                 The default output image number is 1. 
                 
                 The default of the output file numbers are the input
                 image numbers, e.g.

                 "saxs_mac input.edf,2,10 output%%%.edf" writes images 
                 2 to 10 toe output002.edf to output010.edf with image 
                 number 1.

                 Output image loop number is set to file number:
                 ib[0].LoopNumber = FNum[0]

  multi2single:  If the input filename contains percent characters
                 as placeholders for file numbers and if the output 
                 filename does not contain placeholders all images 
                 i1first of the numbered input files are written 
                 into a single output file. 
                 
                 Output image number parameters (first, last, increment)
                 are overwritten by output file numbers. Default
                 output file numbers are input file numbers, e.g.

                 "saxs_mac -i1fst 1 input%%%.edf,2,10 output.edf,3,11" 
                 writes all images with image number 1 in input002.edf
                 to input010.edf to the output.edf with output image
                 numbers 3 to 11 (default would be 2 to 10).

                 All input image loop numbers are set to file numbers:
                 ib[i].LoopNumber = FNum[i]   (i>0)

2 CALL
  void ImageLoop ( CmdBlk * pcb, ImgBlk ib[],
                   void (*Function) (CmdBlk *, long [], ImgHeadBlk [], int *),
                   void (*InitFunction) (CmdBlk *, long [], ImgHeadBlk [], int *),
                   void (*TermFunction) (CmdBlk *, long [], ImgHeadBlk [], int *),
                   int allneeded, int * pstatus );
2 HISTORY
  1995-11-27 PB
  2001-07-09 PB for file numbers, single2multi, multi2single
  2002-06-09 PB LoopNumber
  2003-04-06 PB default: FLast[1] = FFirst[1] instead of FLast[1] = 1l
  2003-04-13 PB single2multi: searching range of input images
                              using ImageNumbers of all input images
                              as defaults for FileNumbers
  2007-06-10 PB InitFunction added to parameter list
  2012-01-31 PB TermFunction added to parameter list
---------------------------------------------------------------------------*/
PUBLIC void ImageLoop ( CmdBlk * pcb, ImgBlk ib[],
               void (*Function) (CmdBlk *, long [], ImgHeadBlk [], int *),
               void (*InitFunction) (CmdBlk *, long [], ImgHeadBlk [], int *),
               void (*TermFunction) (CmdBlk *, long [], ImgHeadBlk [], int *),
               int allneeded, int * pstatus )

{ const char * RoutineName = "ImageLoop";
  int f,fmax;
  long floop,floopmax;
  long MinImage, MaxImage;

  /* dynamic allocation of memory */
  long int *fmaxloop, *FNum, *FFirst, *FLast, *FInc;
  ImgHeadBlk *ihb;

  if (pcb->TestBit) printf(" %s\n",RoutineName);
  SetSaxsError( ModuleName, RoutineName, NULL );

  *pstatus = NotEnoughMemoryAvailable;

  fmax = pcb->ImgBlkLen;
  fmaxloop = (long int *) malloc(sizeof(long int)*fmax); if (fmaxloop==NULL) return;
  FNum     = (long int *) malloc(sizeof(long int)*fmax); if (FNum==NULL) return;
  FFirst   = (long int *) malloc(sizeof(long int)*fmax); if (FFirst==NULL) return;
  FLast    = (long int *) malloc(sizeof(long int)*fmax); if (FLast==NULL) return;
  FInc     = (long int *) malloc(sizeof(long int)*fmax); if (FInc==NULL) return;
  ihb      = (ImgHeadBlk*) saxs_new_all_images( fmax );if (ihb==NULL) return;

  *pstatus = Success;

  /* file number and image number conversion */
  SaxsImageLoop_single2multi = single2multi(pcb,ib);
  SaxsImageLoop_multi2single = multi2single(pcb,ib);

  /* file number and image number conversion */
  if (SaxsImageLoop_single2multi) {
    /* Search first and last image in input file (its only one file) */
    if (!((ib[1].First.I)&&(ib[1].Last.I))) {
      // Open input file 1
      //++++++++++++++ ib[1].OpenMode.V| (added 2004-04-10)
      OpenImageFile(pcb,ib,1,ib[1].Name.V,ib[1].FFirst.V,
                    ib[1].OpenMode.V|IO_Old|IO_FileProtect,pstatus);
      if (*pstatus!=Success) return;
      // Search range of image numbers 
      if (ib[1].Stream>=0) {
        SearchMinMaxImage ( pcb, ib, 1, &MinImage, &MaxImage, pstatus);
        if (*pstatus!=Success) return;
      } else {
        MinImage = 1l; MaxImage = 0l;
      }
      if (!ib[1].First.I) {
        ib[1].First.V = MinImage;
        ib[1].First.I = TRUE;
      }
      if (!ib[1].Last.I) {
        ib[1].Last.V = MaxImage; 
        ib[1].Last.I = TRUE; 
      }
      // Close input file 1
      CloseImageFile ( pcb, ib, 1, pstatus);
      if (*pstatus!=Success) return;
    }

    /* Input image numbers (first, last, increment) are
       overwritten by input file numbers (if set). */
    // loop added 2003-04-13 (instead of image 1 only)
    for (f=1;f<fmax;f++) {
      if (ib[f].FFirst.I) {
        ib[f].First.V = ib[f].FFirst.V; ib[f].First.I = TRUE; 
      } 
      if (ib[f].FLast.I) {
        ib[f].Last.V = ib[f].FLast.V; ib[f].Last.I = TRUE;
      }
      if (ib[f].FInc.I) {
        ib[f].Inc.V = ib[f].FInc.V; ib[f].Inc.I = TRUE;
      }
  
      ib[f].FFirst.V = ib[f].First.V; ib[f].FFirst.I = ib[f].First.I;
      ib[f].FLast.V  = ib[f].Last.V;  ib[f].FLast.I  = ib[f].Last.I;
      ib[f].FInc.V   = ib[f].Inc.V;   ib[f].FInc.I   = ib[f].Inc.I;
    }

    // single image per file with image number 1 or ofst if set
    if (!ib[0].First.I) { ib[0].First.V = 1l; ib[0].First.I = TRUE; }
  } // SaxsImageLoop_single2multi

  /* Choose file loop parameters */
  if (fmax>1) {
    if (ib[1].FFirst.I) FFirst[1] = ib[1].FFirst.V; else FFirst[1] = 1l;
    if (ib[1].FLast.I) FLast[1] = ib[1].FLast.V; else FLast[1] = FFirst[1];
    if (ib[1].FInc.I) FInc[1] = ib[1].FInc.V; else FInc[1] = 1l;
    if (ib[0].FFirst.I) FFirst[0] = ib[0].FFirst.V; 
       else ib[0].FFirst.V = FFirst[0] = FFirst[1];
    if (ib[0].FLast.I) FLast[0] = ib[0].FLast.V; 
       else ib[0].FLast.V = FLast[0] = FLast[1];
    if (ib[0].FInc.I) FInc[0] = ib[0].FInc.V; 
       else ib[0].FInc.V = FInc[0] = FInc[1];
    } else {
    if (ib[0].FFirst.I) FFirst[0] = ib[0].FFirst.V; 
      else ib[0].FFirst.V = FFirst[0] = 1l;
    if (ib[0].FLast.I) FLast[0] = ib[0].FLast.V; 
      else ib[0].FLast.V = FLast[0] = 1l;
    if (ib[0].FInc.I) FInc[0] = ib[0].FInc.V; 
      else ib[0].FInc.V = FInc[0] = 1l;
    } /* if (fmax>1) */

  /* All other sequences */
  for (f=2;f<fmax;f++) {
    if (ib[f].FFirst.I) FFirst[f] = ib[f].FFirst.V; else FFirst[f] = FFirst[1];
    if (ib[f].FLast.I) FLast[f] = ib[f].FLast.V; else FLast[f] = FLast[1];
    if (ib[f].FInc.I) FInc[f] = ib[f].FInc.V;
      else if (ib[f].FLast.I) FInc[f] = FInc[1]; else FInc[f] = 0l;
    } /* for */

  /* Avoid infinite loops in case of FInc[f]==0 by setting fmaxloop[0]=1 in
     case that all FInc[f]==0 */

  for (f=0;f<fmax;f++) {
    if (FInc[f]!=0l)
      fmaxloop[f] = MAX2(1l,1l + ((FLast[f])-(FFirst[f]))/(FInc[f]));
      else fmaxloop[f] = -1l; /* infinite */
    } /* f */

  floopmax=-1l; /* infinite */
  for (f=0;f<fmax;f++)
    if (floopmax>0l) {
      if (fmaxloop[f]>0) {floopmax=MIN2(floopmax,fmaxloop[f]);}
      } else {
      if (fmaxloop[f]>0l) {floopmax=fmaxloop[f];}
      } /* if (floopmax>0l) */
  floopmax = MAX2(1l,floopmax);

  /* start values */
  for(f=0;f<fmax;f++) { FNum[f]=FFirst[f]; }

  /* preset common orientation flag
     -oori <ori> activates always common orientation
     if not explicitely switched off with -cori */
  if (!pcb->CommonOrientation.I) {
    if (fmax>0) pcb->CommonOrientation.I=ib[0].Orientation.I?TRUE:FALSE;
  }
  /* reset common orientation */
  pcb->CommonOrientation.V = 0l;

  /* loop */
  for (floop=1l; floop<=floopmax; floop++) {

    /* file loop numbers */
    for (f=0;f<fmax;f++) ib[f].LoopNumber = FNum[f];

    /* file number and image number conversion */
    if (SaxsImageLoop_multi2single) { 
     // overwrite output image numbers with output file numbers
      ib[0].First.V = FNum[0]; ib[0].First.I = TRUE;
      ib[0].Last.V  = FNum[0]; ib[0].Last.I = TRUE;
    }
    if (SaxsImageLoop_single2multi) {
      ib[1].First.V = FNum[1]; ib[1].First.I = TRUE;
      // only 1 loop in saxs_image_loop
      ib[1].Inc.V   = 0l;      ib[1].Inc.I = TRUE;
      /* All other sequences */
      for (f=2;f<fmax;f++) {
        ib[f].First.V = FNum[f]; ib[f].First.I = TRUE;
        ib[f].Inc.V   = 0l;      ib[f].Inc.I = TRUE;
      }
    }

    // call saxs_image_loop always (check for all images is done inside)
    saxs_open_all_images( pcb, ib, FNum, ihb, allneeded, pstatus );
    if (*pstatus!=Success) { saxs_free_all_images(pcb,ihb); return; }
    saxs_image_loop(pcb,ib, FNum, Function,InitFunction,TermFunction,
                    allneeded,ihb,floop,floopmax,pstatus);
    if (*pstatus!=Success) { saxs_free_all_images(pcb,ihb); return; }

    /* incrementation of file loop variables */
    for(f=0;f<fmax;f++) FNum[f]+=FInc[f];

  } // for (floop ...

  saxs_close_all_images( pcb, ib, pstatus );
  if (*pstatus!=Success) return;

  saxs_free_all_images(pcb,ihb);

  free(fmaxloop); free(FNum); free(FFirst); free(FLast); free(FInc); free(ihb);

  *pstatus = Success;

} /* ImageLoop */

/*---------------------------------------------------------------------------
 mark_float_2d
---------------------------------------------------------------------------*/
# define MARK00( L, E )     L += ( E)
# define MARK06( L, E, V )  L[1] = L[2] = ( V); L += ( E)
# define MARK09( L, E, V )  L[0] = L[3] = ( V); L += ( E)
# define MARK0A( L, E, V )  L[1] = L[3] = ( V); L += ( E)
# define MARK0E( L, E, V )  L[1] = L[2] = L[3] = ( V); L += ( E)
# define MARK10( L, E, V )  L[4] = ( V); L += ( E)
# define MARK14( L, E, V )  L[2] = L[4] = ( V); L += ( E)
# define MARK1C( L, E, V )  L[2] = L[3] = L[4] = ( V); L += ( E)
# define MARK20( L, E, V )  L[5] = ( V); L += ( E)
# define MARK22( L, E, V )  L[1] = L[5] = ( V); L += ( E)
# define MARK24( L, E, V )  L[2] = L[5] = ( V); L += ( E)
# define MARK28( L, E, V )  L[3] = L[5] = ( V); L += ( E)
# define MARK2A( L, E, V )  L[1] = L[3] = L[5] = ( V); L += ( E)
# define MARK2E( L, E, V )  L[1] = L[2] = L[3] = L[5] = ( V); L += ( E)
# define MARK36( L, E, V )  L[1] = L[2] = L[4] = L[5] = ( V); L += ( E)
# define MARK38( L, E, V )  L[3] = L[4] = L[5] = ( V); L += ( E)
# define MARK3A( L, E, V )  L[1] = L[3] = L[4] = L[5] = ( V); L += ( E)
# define MARK3E( L, E, V )  L[1] = L[2] = L[3] = L[4] = L[5] = ( V); L += ( E)
# define MARK67( L, E, V )  L[0] = L[1] = L[2] = L[5] = L[6] = ( V); L += ( E)
# define MARK69( L, E, V )  L[0] = L[3] = L[5] = L[6] = ( V); L += ( E)
# define MARK99( L, E, V )  L[0] = L[3] = L[4] = L[7] = ( V); L += ( E)
# define MARKBA( L, E, V )  L[1]=L[2]=L[3]=L[4]=L[5]=L[7]=( V); L += ( E)

/*---------------------------------------------------------------------------
1 mark_float_2d

2 HISTORY
  14-NOV-1996 Peter Boesecke
---------------------------------------------------------------------------*/
void mark_float_2d ( float *array, long d_1, long d_2, float d, float dd )
/* Marks a region with value dummy + 0.5*ddummy. */
{ const long m_1 = 8, m_2 = 80;
  short *farray, *f;
  float *a, value;
  long i, i_1, i_2;

  if (!(f = farray = (short *) malloc(m_1*m_2*sizeof(short)))) return;
  for (i=0;i<m_1*m_2;i++) *f++ = 0;

  f = farray;
  MARK00(f,m_1);MARK00(f,m_1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK14(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK22(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK2E(f,m_1,1);MARK2A(f,m_1,1);MARK3A(f,m_1,1);MARK00(f,m_1);
  MARKBA(f,m_1,1);MARK22(f,m_1,1);MARKBA(f,m_1,1);MARK00(f,m_1);
  MARK36(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK00(f,m_1);MARK00(f,m_1);
  MARK3A(f,m_1,1);MARK24(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK20(f,m_1,1);MARK3E(f,m_1,1);MARK20(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK38(f,m_1,1);MARK28(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK00(f,m_1),MARK00(f,m_1);
  MARK0E(f,m_1,1);MARK0A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK3E(f,m_1,1);MARK2A(f,m_1,1);MARK3A(f,m_1,1);MARK00(f,m_1);
  MARK3E(f,m_1,1);MARK2A(f,m_1,1);MARK3A(f,m_1,1);MARK00(f,m_1);
  MARK3E(f,m_1,1);MARK10(f,m_1,1);MARK00(f,m_1);MARK00(f,m_1);
  MARK06(f,m_1,1);MARK69(f,m_1,1);MARK99(f,m_1,1);
  MARK67(f,m_1,1);MARK09(f,m_1,1);MARK09(f,m_1,1);
  MARK06(f,m_1,1);

  /* copy farray to array */
  value = d+0.5*dd;
  f = farray + m_1*(m_2-MIN2(m_2,d_2)); a = array;
  for (i_2 = 0;i_2<MIN2(m_2,d_2);i_2++) {
    for (i_1 = 0;i_1<MIN2(m_1,d_1);i_1++)
      if ((f[i_1]) && (DUMMY(a[i_1],d,dd)) ) a[i_1] = value;
    a+=d_1; f+= m_1;
    }

  free(farray);

} /* mark */

/*---------------------------------------------------------------------------
 mark_float_2d end
---------------------------------------------------------------------------*/

/***************************************************************************/

