/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 gel2saxs 

2 PURPOSE
  convert images from an image plate detector (with .gel extension) 
  to images written in the standard esrf data format

3 HISTORY
  10-Apr-1995 Pierre Jourde creation from gel2saxs version 1.3 of Peter Boesecke
  19-Jun-1995 PB SampleName -> ExperimentInfo
  21-Sep-1995 PB All scalers removed (for SIG). 
  28-Nov-1995 PB V3.0
  16-May-1998 PB V4.0 changed for edfio
  18-Apr-1999 PB V4.01 new calls to edf routines
  30-Apr-1999 PB V4.02 status set to Success before calling edf routines
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  23-Jul-1999 PB V4.03 call to edf_write_data_2d_raw updated for edfio 
                 version 1.07, SaxsDataVersion = 2.0
  26-Nov-1999 PB V4.04 DetectorPosition = SampleDistance;
  22-Dec-1999 PB V4.05 SubTitle -> ExperimentInfo
  2000-08-04  PB %f -> %g
  2000-11-10  PB V4.10 changes for edfio V1.30
  2000-11-12  PB V4.11 changes for edfio V1.32
                 V4.12 changes for edfio V1.34
  2000-12-31  PB V4.13 Dim[3] not used any more
  2001-07-03  PB V4.14 filename_full
  2004-10-31  PB V4.15 MAKE_FUNCTION, user_io: status->*pstatus,
                       ReportStatus removed
  2005-02-24  PB V4.16 INTERNAL_BYTE_ORDER replaced by edf_byteorder()
  2005-07-15  PB V4.17 ReportSaxsStatus parameter updated
  2008-12-05  PB V1.18 open all input files as binary file to avoid
                       translation of \r\n to \n when compiled on windows.

---*/
# define Version  " V4.18 2008-12-05, Peter Boesecke/Pierre Jourde"

# include <errno.h>
# include <unistd.h>
# include <stdio.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "GetGelFrame.h"
# include "edfio.h"

# define LineLen InputLineLength
# define Wild         "="
# define Usage "<i/p file> <dummy> <o/p file>"

# define Default_ipfn      "input.gel"
# define Default_extension DefaultExtension 
# define Default_dummy     -1

/* Error value definition, provisorically */
# ifndef GelFileNotFound
#  define GelFileNotFound CouldNotOpenFile
# endif
# ifndef GelImageProtected
#  define GelImageProtected WriteDataError
# endif
# ifndef GelFileProtected
#  define GelFileProtected WriteDataError
# endif

# ifndef TRUE
#   define TRUE 1
# endif
# ifndef FALSE
#   define FALSE 0
# endif

static int GEL2S_debug = 0;

/*+++------------------------------------------------------------------------
NAME

   ReportStatus

SYNOPSIS


DESCRIPTION

   replacement for ExpgReportStatus

HISTORY
15-Mar-1998 Peter Boesecke
----------------------------------------------------------------------------*/
void ReportStatus(int status, int ErrorValue, int PrintFlag)
{ if ((status)&&(PrintFlag)) printf("%s\n",edf_report_data_error(ErrorValue));
} /* ReportStatus */
/*+++------------------------------------------------------------------------
1 ImageNumberExists

2 DESCRIPTION
  Tests, whether an image exists already.

  return( int ) : (o)   TRUE    : image number found
                        FALSE   : error or image number not found

  int ImageNumber (i)           : image number of the header
  int *pErrorValue (o)          : error value

2 HISTORY
  17-Feb-1995   Peter Boesecke

  ---------------------------------------------------------------------------*/
int ImageNumberExists      ( int stream, int ImageNumber, int * pErrorValue )
{ int found = 0;
  int DataChain = 1;
  long *Dim, Tmp[4];
  size_t DataArraySize;
  int status = Success;


  if (GEL2S_debug) printf("ImageNumberExists BEGIN\n");

  Dim=Tmp; Dim[0] = 2l;
  found = edf_read_data_dimension ( stream, ImageNumber, DataChain,
                                    &Dim, &DataArraySize,
                                    pErrorValue, &status );

  if (GEL2S_debug) printf("ImageNumberExists END\n");

  return ( found );

} /* ImageNumberExists */

/*---------------------------------------------------------------------------
  write_data_raw
  Writes raw data into the block "<keyword> = <ImNum>"
  A raw data header is created.
  return value TRUE if no Error
  SaxsDataVersion 2.0
  ---------------------------------------------------------------------------*/

void write_data_raw ( int stream, char *HmGelVersion, int ImNum,
                   char *HmFile, int HmFrame, int DataLength, int DataFloatI,
                   int Dim_1, int Dim_2,
                   int Off_1, int Off_2,
                   void *Start,
                   char HmStartTime[], float HmDeltaTime,
                   float Dummy,
                   char * Title, char * ExperimentInfo,
                   float Center_1, float Center_2, int ICenter,
                   float PixSize_1, float PixSize_2, int IPixSize,
                   float WaveLength, int IWaveLength,
                   float SampleDistance, int ISampleDistance,
                   float DetectorPosition, int IDetectorPosition,
                   char * DetectorType, char * MachineInfo, char * OpticsInfo,
                   char * StationInfo, char * ProposalInfo,
                   int * pErrorValue )

  {
  long  Dim[4];
  long  DataNumber = (long) ImNum;
  long  DataChain = 1;
  int   status = status;
  int   cnt;
  int   DataType;
  const float eps = 1e-32;

  const char * SaxsDataVersion = "2.0";

  register int i;

  if (GEL2S_debug) printf("write_data_raw BEGIN\n");

        *pErrorValue = RoutineSucceeded;


        /* SaxsDataVersion */
        edf_write_header_line ( stream, DataNumber, DataChain,
                            KSaxsDataVersion, SaxsDataVersion,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* HmFrame */
        edf_write_header_long ( stream, DataNumber, DataChain,
                            KHFrame, HmFrame,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* HmFile */
        if (strlen(HmFile)) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KHFile, HmFile,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* Title */
        if ((Title!=(char *) NULL) && (strlen(Title))) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KTitle, Title,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* ExperimentInfo */
        if ((ExperimentInfo!=(char *) NULL) && (strlen(ExperimentInfo))) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KExperimentInfo, ExperimentInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* HmStartTime */
        if (strlen(HmStartTime)>0) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KHStartTime, HmStartTime,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* HmDeltaTime */
        if (ABS(HmDeltaTime)>eps) {
          edf_write_header_float ( stream, DataNumber, DataChain,
                              KHDeltaTime, HmDeltaTime,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* Dummy */
        edf_write_header_float( stream, DataNumber, DataChain,
                            KDummy, Dummy,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* Offsets */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KOffset_1, (long) Off_1,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        edf_write_header_long( stream, DataNumber, DataChain,
                            KOffset_2, (long) Off_2,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;
 
        /* Centers */
        if (ICenter) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KCenter_1, Center_1,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;

          edf_write_header_float( stream, DataNumber, DataChain,
                              KCenter_2, Center_2,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (IPixSize) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KPixSiz_1, PixSize_1,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;

          edf_write_header_float( stream, DataNumber, DataChain,
                              KPixSiz_2, PixSize_2,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (IWaveLength) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KWaveLength, WaveLength,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (ISampleDistance) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KSampleDistance, SampleDistance,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (IDetectorPosition) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KDetectorPosition, DetectorPosition,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((DetectorType!=(char *) NULL) && (strlen(DetectorType))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KDetectorType, DetectorType,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((MachineInfo!=(char *) NULL) && (strlen(MachineInfo))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KMachineInfo, MachineInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((OpticsInfo!=(char *) NULL) && (strlen(OpticsInfo))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KOpticsInfo, OpticsInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((StationInfo!=(char *) NULL) && (strlen(StationInfo))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KStationInfo, StationInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* Keywords for Version >=1.0 */

        switch (DataLength) {
          case 2: /* 2 bytes */
            DataType = Unsigned16;
            break;
          case 4: /* 4 bytes */
            if (DataFloatI)
              DataType = FloatIEEE32;
            else
              DataType = Unsigned32;
            break;
          default: /* conversion error */
             *pErrorValue=BadDataBlock;
             return;
          } /* switch */

        /* Dim */
        Dim[0] = 2l; Dim[1] = (long) Dim_1; Dim[2] = (long) Dim_2;

        edf_write_data_raw ( stream, DataNumber, DataChain, Dim,
                             Start, DataType, 0l, edf_byteorder(), 4l,
                             pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        *pErrorValue = RoutineSucceeded;

  if (GEL2S_debug) printf("write_data_raw END\n");

  return;
} /* write_data_raw */

/*+++------------------------------------------------------------------------
Gel2Saxs
----------------------------------------------------------------------------*/
void Gel2Saxs(CmdBlk * pcb, FILE * InFstream, int DataLength, int DataFloatI, 
              int Dim_1, int Dim_2, int Off_1, int Off_2,
              int OutStream, int Out, 
              char InName[], float Dummy, 
              int * pErrorValue) 
{ 
  int      status = Success;
  void   * Start = (void *) NULL;

  float    HmDeltaTime = 0.0;
  char     HmStartTime[InputLineLength] = "";
  char     HmGelVersion[InputLineLength] = "1.0";

  float    Center_1, Center_2, PixSize_1, PixSize_2;
  float    WaveLength, SampleDistance, DetectorPosition; 
  int      ICenter=FALSE, IPixSize=FALSE, IWaveLength=FALSE, 
           ISampleDistance=FALSE, IDetectorPosition=FALSE;
  char   * Title = (char *) NULL;
  char   * ExperimentInfo = (char *) NULL;
  char   * DetectorType = (char *) NULL;
  char   * MachineInfo = (char *) NULL;
  char   * OpticsInfo = (char *) NULL;
  char   * StationInfo = (char *) NULL;
  char   * ProposalInfo = (char *) NULL;
  char   * HmFraming = (char *) NULL;

  register int i;

  *pErrorValue = EndOfFileDetected;

  if (GEL2S_debug) printf("Gel2Saxs BEGIN\n");

  /* reading gel file */
  /* --- read gel data */
  if (GEL2S_debug) printf("  Start = %p\n",Start);
  if (status=ReadGelData (InFstream, 1, &Start)) {
     printf("ReadGelData return status = % d\n",status);return; }
  if (GEL2S_debug) printf("  Start = %p\n",Start);

  /* --- read additional experiment information */
  if (GEL2S_debug) printf("  ReadGelExperiment\n");
  if (status=ReadGelExperiment(InFstream, &Title, 
         &ExperimentInfo,  &HmFraming, HmStartTime,
         &Center_1, &Center_2, &ICenter,
         &PixSize_1, &PixSize_2, &IPixSize,
         &WaveLength, &IWaveLength,
         &SampleDistance, &ISampleDistance) ) {
    printf("ReadGelExperiment return status = % d\n",status);return; }   

  if (pcb->TestBit) {
   if (Title!=(char *) NULL) printf("Title = % s\n",Title);
   if (ExperimentInfo!=(char *) NULL) 
      printf("ExperimentInfo = % s\n",ExperimentInfo);
   if (HmFraming!=(char *) NULL) printf("HmFraming = % s\n",HmFraming);
   if (strlen(HmStartTime)>0) printf("HmStartTime = % s\n",HmStartTime);
   if (ICenter) printf("Center_1 = %g, Center_2 = %g\n",Center_1,Center_2);
   if (IPixSize) printf("PixSize_1 = %g, PixSize_2 = %g\n",PixSize_1,PixSize_2);
   if (IWaveLength) printf("WaveLength = %g\n",WaveLength);
   if (ISampleDistance) printf("SampleDistance = %g\n",SampleDistance);
   } /* if (pcb->TestBit) ...*/

  if (GEL2S_debug) printf("  ReadGelInfo\n");
  if (status=ReadGelInfo(InFstream, &DetectorType, 
        &MachineInfo, &OpticsInfo, &StationInfo, &ProposalInfo)) {
    printf("ReadGelInfo return status = % d\n",status);return; }
  if (pcb->TestBit) {
   if (DetectorType!=(char *) NULL) printf("DetectorType = % s\n",DetectorType);
   if (MachineInfo!=(char *) NULL) printf("MachineInfo = % s\n",MachineInfo);
   if (OpticsInfo!=(char *) NULL) printf("OpticsInfo = % s\n",OpticsInfo);
   if (StationInfo!=(char *) NULL) printf("StationInfo = % s\n",StationInfo);
   if (ProposalInfo!=(char *) NULL) printf("ProposalInfo = % s\n",ProposalInfo);
   } /* if (pcb->TestBit) ...*/

  /* test whether output image exists already, else error */
  if (ImageNumberExists( OutStream, Out, pErrorValue ))
     {*pErrorValue=GelImageProtected; return; }


  /* write to esrf data file */
  DetectorPosition = SampleDistance;
  write_data_raw ( OutStream, HmGelVersion, Out,
                   InName,  1, DataLength, DataFloatI,
                   Dim_1, Dim_2, Off_1, Off_2, Start,

                   HmStartTime, HmDeltaTime,

                   Dummy,
                   Title, ExperimentInfo,
                   Center_1, Center_2, ICenter,
                   PixSize_1, PixSize_2, IPixSize,
                   WaveLength, IWaveLength,
                   SampleDistance, ISampleDistance,
                   DetectorPosition, IDetectorPosition,
                   DetectorType, MachineInfo, OpticsInfo,
                   StationInfo, ProposalInfo,
                   pErrorValue );
  if (*pErrorValue!=RoutineSucceeded) return;

  /* free allocated memory */
  if (Title)          free( Title );
  if (ExperimentInfo) free( ExperimentInfo );
  if (DetectorType)   free( DetectorType );
  if (MachineInfo)    free( MachineInfo );
  if (OpticsInfo)     free( OpticsInfo );
  if (StationInfo)    free( StationInfo );
  if (ProposalInfo)   free( ProposalInfo );
  if (HmFraming)      free( HmFraming );

  if (GEL2S_debug) printf("Start = %p\n",Start);

  if (Start);          free( Start );

  if (GEL2S_debug) printf("Dld2Saxs END\n");

} /* Gel2Saxs */

/*---------------------------------------------------------------------------
user_io
Do simple keyboard io 
---------------------------------------------------------------------------*/

void user_io( int argc, char *argv[], int * parg_no, CmdBlk *pcb,
              char InName[], char OutName[], int *pOutFirst, float * pDummy, 
	      int * pDataLength, int * pDataFloatI, int * pDim_1, int * pDim_2,
              int * pOff_1, int * pOff_2, int * pstatus)

{
  char  progfn[InputLineLength];
  char  InNamePre[InputLineLength];

  FILE * InFstream;
  int prompt;
  int opt_no;
  int status;

  if (GEL2S_debug) printf("user_io BEGIN\n");

 /* Determine program name without directory name */
   (void) RightFR((char *) argv[0],DirSeparator,progfn,InputLineLength);

  /* switch to prompt mode if input file name is missing and no 
     instruction has been given */
  if (argc<(*parg_no+2)) prompt=TRUE; else prompt=FALSE;
  if (pcb->Prompt.I) prompt=(int) pcb->Prompt.V;
 
  /* show usage if no arguments are given */
  if (argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");
  
  /*--- Argument : input file name */
  *parg_no += 1;
  (void) strcpy(InName,Default_ipfn);
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     (void) strcpy(InName,(char *) argv[*parg_no]);
     }
  if (prompt) {
     *pstatus = GetLine(stdin,InName,1,"Input file name",InName);
   /* Exit in all cases after an abort from the first argument */
   if (*pstatus!=Success) return;
    } 

  /* Expand InName to the full file name */
  if (!filename_full(InName,IO_len,InName)) {
     printf("% s: Input file name or path not allowed\n",progfn);
     *pstatus=FileNotFound; return;}

  /* Determine prefix of input file name (InName) */
  (void) RightFR((char *) InName,DevSeparator,InNamePre,InputLineLength);
  (void) RightFR((char *) InNamePre,DirSeparator,InNamePre,InputLineLength);
  (void) LeftFL((char *) InNamePre,ExtSeparator,InNamePre,InputLineLength);

 /*--- Read header information of input file --- */

  if ((InFstream = fopen(InName, "rb")) == (FILE *) NULL) {
    *pstatus = FileOpeningError; return; }

  /* DataLength, Dim_1, Dim_2 */
  if (status=ReadGelType(InFstream,pDim_1,pDim_2,pDataLength,
                         pDataFloatI,pOff_1,pOff_2)) {
    *pstatus = CannotFindHeader; return; }
  
  (void) fclose(InFstream); 

  /*--- Argument : output dummy */
  *parg_no += 1;
  *pDummy = Default_dummy;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%g",pDummy)!=1) {
        *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetReal(stdin,pDummy,1,"output dummy",*pDummy);
     if (*pstatus!=Success) return;
     }

  /*--- Argument : output file name */
  *parg_no += 1;
  (void) sprintf(OutName,"%s%s",InNamePre,Default_extension);
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     (void) strcpy(OutName, (char *) argv[*parg_no]);
     }
  if (prompt) {
     *pstatus = GetLine(stdin,OutName,1,"Output file name",OutName);
     if (*pstatus!=Success) return;
     }

  /* Expand OutName to the full file name */
  if (!filename_full(OutName,IO_len,OutName)) {
     printf("% s: Output file name or path not allowed\n",progfn);
     *pstatus = FileNameError; return; }

  /*--- Argument : number of first output image */
  *parg_no += 1;
  *pOutFirst = 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pOutFirst)!=1) {
        *pstatus = ScanError; return; }
       }
  if (prompt) {
     *pstatus = GetInteger(stdin,pOutFirst,1,"First output image",*pOutFirst);
     if (*pstatus!=Success) return;
     }

  /*--- Argument : data length */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pDataLength)!=1) {
       *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = 
       GetInteger(stdin,pDataLength,1,"Data length in bytes",*pDataLength);
     if (*pstatus!=Success) return;
     }

  /*--- Argument : number of columns */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pDim_1)!=1) {
         *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pDim_1,1,"Number of columns",*pDim_1);
     if (*pstatus!=Success) return;
     }

  /*--- Argument : number of rows */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pDim_2)!=1) {
         *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pDim_2,1,"Number of rows",*pDim_2);
     if (*pstatus!=Success) return;
     }

  /*--- End of argument list */
  *parg_no += 1;
  if (argc>*parg_no) { /* too many arguments */
     printf("%s : Too many arguments\n",progfn);
     printf("Usage: %s %s\n",progfn,Usage);
     if (!prompt) { *pstatus = StoppedAfterHelp; return; }
     }

  printf("\n");
  printf("i/p file           : %s\n",InName);
  printf("output dummy       : %g\n",*pDummy);
  printf("o/p file           : %s\n",OutName);
  printf("first output image : %d\n",*pOutFirst);
  printf("data length        : %d bytes\n",*pDataLength); 
  printf("dimension 1        : %d\n",*pDim_1);
  printf("dimension 2        : %d\n",*pDim_2);   
  printf("\n");

  if (GEL2S_debug) printf("user_io END\n");

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main

RETURN VALUE:            0: SUCCESS
              other values: ERROR
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_PPP
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  int status = Success, ErrorValue;
  int arg_no = 0;
  CmdBlk cb;  /* command block  */

  FILE  * InFstream;
  char  InName[InputLineLength];
  int   OutStream;
  char  OutName[InputLineLength];
  int   Out; 
  float Dummy;
  int   DataLength, DataFloatI, Dim_1, Dim_2, Off_1, Off_2;

    /* Init options, control block and image blocks */
     InitOptions( Usage, Version, NULL, TRUE, &cb, NULL, 0 );

    /* Read options from argument list */
     ReadOptions( argv, &arg_no, &cb, NULL, &status);
     if (status!=Success) return( ReportSaxsStatus( status, 0 ) );

     user_io( argc, argv, &arg_no, &cb,
              InName, OutName, &Out, &Dummy,
              &DataLength, &DataFloatI, &Dim_1, &Dim_2, &Off_1, &Off_2,
              &status);  
     if (status!=Success) return( ReportSaxsStatus( status, 0 ) );

     /* open header file */
     if (GEL2S_debug) printf("open header file\n");
     if ((InFstream = fopen(InName, "rb")) == (FILE *) NULL)
       return( ReportSaxsStatus( FileOpeningError, 0 ) );

     /* open esrf data file */
     if (GEL2S_debug) printf("open esrf data file\n");
     OutStream = edf_open_data_file ( OutName, "any", &ErrorValue, &status );
     if (ErrorValue!=RoutineSucceeded)
       return( ReportSaxsStatus( FileOpeningError, 0 ) );


     printf(" Converting \n");

     Gel2Saxs(&cb, InFstream, 
	      DataLength, DataFloatI, Dim_1, Dim_2, Off_1, Off_2,
              OutStream, Out, InName, Dummy, &ErrorValue);
     if (ErrorValue!=RoutineSucceeded) {
        return(ReportSaxsStatus(RoutineFailed,0));}

   /* close edf file */
   status = Success;
   edf_close_data_file        ( OutStream, &ErrorValue, &status );
   if (ErrorValue!=RoutineSucceeded)
      return(ReportSaxsStatus(FileClosingError,0));

   /* close file */
   (void) fclose(InFstream);

   printf("\n");
   printf("Input  file : % s\n",InName);
   printf("Output file : % s\n",OutName);

   printf("\nEnd of % s\n",argv[0]);

   return( 0 );

} /* MAIN */
