/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define Version  "saxstest 2007-04-20, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define BlockNum 7       /* 6 input sequences + 1 output sequence */

/* This program shows how to use the routines of SaxsImage and SaxsOption 
   to read and write saxs data files */
/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/
int main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */
  ImgHeadBlk ihb[BlockNum]; /* image header blocks */
  long FNum[BlockNum];      /* file numbers */ 

  size_t Size;

  int status;
  int arg_no = 0;
  int tmp_status;

  long verbose;

  long i,j;

  /* Init options, control block and image blocks */
  InitOptions( "test program only", "20070420", TRUE, &cb, ib, BlockNum );

  for (i=0;i<BlockNum;i++) {
    if (i==0) sprintf(ib[i].Name.V,"out.edf");
    else sprintf(ib[i].Name.V,"file%ld.edf",i);
    ib[i].Name.I = TRUE;
    FNum[i]=1; // file number 1 for all
  }

  /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  if (status) goto error_exit;

  verbose = cb.TestBit;

  /* Open Images */
  for (i=0;i<BlockNum;i++) {
    printf("Opening file %s for image block %ld\n",ib[i].Name.V,i);
    if (verbose>0) printf("OpenImageFile(...,%ld,%s,%ld,...);\n",
                     i,ib[i].Name.V,FNum[i]);
    OpenImageFile(&cb,ib,i,ib[i].Name.V,FNum[i],
                  ib[i].OpenMode.V,&status);
    if (status) goto error_exit;
  }

  /* Read Images */
  for (i=1;i<BlockNum;i++) {
    printf("Reading image block %ld\n",i);
    if (verbose>0) printf("ReadImage(...,%ld,%ld,...);\n",i,FNum[i]);
    ReadImage(&cb,ib,i,FNum,ihb,&status);
    if (status) goto error_exit;
  }

  /* Check Sizes */
  for (i=1;i<BlockNum;i++) {
    printf("  ihb[%ld].Dim[0]=%ld, ihb[%ld].Dim[1]=%ld, ihb[%ld].Dim[2]=%ld\n",
      i,ihb[i].Dim[0], i,ihb[i].Dim[1],i, ihb[i].Dim[2]);
    if (i==1) Size = ihb[i].ImgByteLen;
    else {
      if (Size != ihb[i].ImgByteLen) {
        status = status_error;
        fprintf(stderr,"Incompatible image sizes\n");
        goto error_exit;
      }
    }
  }

  /* New Image */
  printf("Creating new image for output image block\n");
  if (verbose>0) printf("NewImage(...,%ld,%ld,...);\n",0,FNum[0]);
  NewImage(&cb,ib,0,FNum,ihb,&ihb[1],&status);
  if (status) goto error_exit;
  if (verbose>1) PrintImageHeaderBlock(&ihb[0]);

  /* Preset with 0 */
  if (verbose>0) printf("Clear Image\n");
  for (j=0;j<ihb[0].Dim[1]*ihb[0].Dim[2];j++) ihb[0].Data[j]=0;

  /* process Images */
  printf("Add input image blocks %ld to %ld\n",1,BlockNum-1);
  if (verbose>0) printf("Adding data Dim=%ld, Dim_1=%ld, Dim_2=%ld\n",
                   ihb[0].Dim[0], ihb[0].Dim[1], ihb[0].Dim[2]);
  for (j=0;j<ihb[0].Dim[1]*ihb[0].Dim[2];j++) {
    for (i=1;i<BlockNum;i++) ihb[0].Data[j]+=ihb[i].Data[j];
  }

  /* Write Output Image */
  printf("Writing output image block to %s (image %ld)\n",
    ib[0].FileName, ihb[0].ImNum);
  if (verbose>0) printf("WriteImage(...,%ld,...);\n",0);
  WriteImage(&cb,ib,0,ihb,&status);
  if (status) goto error_exit;

  /* Close Images */
  printf("Closing image blocks %ld to %ld and releasing allocated memory\n",
    0,BlockNum-1);
  for (i=0;i<BlockNum;i++) {
    if (verbose>0) printf("CloseImageFile(...,%ld,...);\n",i);
    CloseImageFile ( &cb, ib, i, &status);
    if (status) goto error_exit;
    if (verbose>0) printf("FreeImage(...,%ld,...);\n",i);
    FreeImage( &cb, i, ihb );
  }

  return( ReportSaxsStatus( status, 0 ) );

error_exit:
  ReportSaxsStatus( status, 0 );
  for (i=0;i<BlockNum;i++) {
    CloseImageFile ( &cb, ib, i, &tmp_status);
    FreeImage( &cb, i, ihb );
  }
  return( status );


} /* main */

