/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define TABLE_VERSION      "table : V1.1 Peter Boesecke 2007-04-19"
/*+++------------------------------------------------------------------------
NAME
   table --- routines to read text tables 

SYNOPSIS

   # include table.h

HISTORY
  2002-04-21 V1.0 
  2007-04-19 V1.1 PB table_LinIntXYDouble: checked that x1, x2 and index are
                     only used if they were explicitely set (found!=0),
                     -Wall compiler warnings resolved

DESCRIPTION

----------------------------------------------------------------------------*/
/******************************************************************************
* Include Files                                                               *
******************************************************************************/

# include "table.h"

/******************************************************************************
* Private Defines                                                             *
******************************************************************************/

# define TABLE_MAXCOLWIDTH 1024
# define TABLE_EPS 1e-30

/******************************************************************************
* Private Type Defs                                                           *
******************************************************************************/

typedef struct table_XYDataList { 
  TableXYData * xydata;
  struct table_XYDataList * Previous, * Next;
} TableXYDataList;

/******************************************************************************
* Private Variables                                                           *
******************************************************************************/

PRIVATE char comment_delimiter[TABLE_MAXLEN] = { '#', '\0' };
PRIVATE char line_delimiter   [TABLE_MAXLEN] = { '\n', '\0' };
PRIVATE char col_delimiter    [TABLE_MAXLEN] = { '\t', '\0' };
PRIVATE double table_empty = 0.0;

PRIVATE TableXYDataList * table_XYDataRoot = (TableXYDataList *) NULL;

/******************************************************************************
* Private Constants                                                           *
******************************************************************************/
PRIVATE const char * filename_default = "xy.txt";

/******************************************************************************
* Routines                                                                    *
******************************************************************************/

/*---------------------------------------------------------------------------
NAME
  table_newxydata --- allocate xydata
---------------------------------------------------------------------------*/
TableXYData *  table_newxydata( long buflen )
{ TableXYData * xydata;

  if (!(xydata = (TableXYData *) malloc( sizeof(TableXYData) )))
    return( xydata );

  xydata->XLabel = (char *) NULL;
  xydata->YLabel = (char *) NULL;
  if (!(xydata->X = (double *) malloc( sizeof(double)*buflen ))) {
    free(xydata); return( (TableXYData *) NULL );
  }
  if (!(xydata->Y = (double *) malloc( sizeof(double)*buflen ))) {
    free(xydata); free(xydata->X); return( (TableXYData *) NULL );
  }
  xydata->len    = 0l;
  xydata->buflen = buflen; 
  xydata->index  = 0l;

  return( xydata );

} // table_newxydata

/*---------------------------------------------------------------------------
NAME
  table_freexydata --- release xydata
---------------------------------------------------------------------------*/
int table_freexydata( TableXYData * xydata )
{ 
  if (xydata) {
    if (xydata->XLabel) free((char *) xydata->XLabel);
    if (xydata->YLabel) free((char *) xydata->YLabel);
    if (xydata->X)      free(xydata->X);
    if (xydata->Y)      free(xydata->Y);
    free(xydata);
  }

  return(0);

} // table_freexydata

/*+++------------------------------------------------------------------------
NAME
  table_newXYData --- create a new xy data 
 
SYNOPSIS
 
  TableXYData * table_newXYData( long buflen ); 
 
DESCRIPTION
 
  Creates a new XYData buffer for buflen xy-data pairs and inserts
  it into table_XYDataRoot list.
 
----------------------------------------------------------------------------*/
TableXYData * table_newXYData( long buflen )
{
  TableXYDataList * newxydatalist, * previous, * next;
  TableXYDataList ** proot = &table_XYDataRoot;
  
  TableXYData * newxydata = (TableXYData *) NULL;
 
  /* insert xy at root */
  previous = (TableXYDataList *) NULL;
  next = *proot;
 
    /* create new xy data */
    if (!(newxydatalist = (TableXYDataList *) malloc(sizeof(TableXYDataList)))) 
      return((TableXYData *) NULL);
    
    if (!(newxydatalist->xydata = table_newxydata( buflen )))
      return((TableXYData *) NULL); 

    /* insert newxydatalist before next */
    if (next) next->Previous = newxydatalist;
    newxydatalist->Next=next;
    newxydatalist->Previous=previous;
    if (previous) previous->Next=newxydatalist;
      else *proot = newxydatalist;
 
    next = newxydatalist;
 
  newxydata = next->xydata;
 
  return( newxydata );
 
} /* table_newXYData */
 
/*+++------------------------------------------------------------------------
NAME
  table_freeXYData --- releases xy data
 
SYNOPSIS
 
  int table_freeXYData( TableXYData * xydata ); 
 
DESCRIPTION
 
  Unlink and release an XYData buffer

RETURN VALUE

   0: OK
  -1: Error
 
----------------------------------------------------------------------------*/
int table_freeXYData( TableXYData * xydata )
{
  TableXYDataList *current, *previous, *next;
  TableXYDataList ** proot = &table_XYDataRoot;
 
  /* search xydata */
  next = *proot;
  while( next && (next->xydata!=xydata) ) next++;

  if ( next ) { // remove next 
 
    current  = next;
    previous = current->Previous;
    next     = current->Next;
 
    if (next) next->Previous = previous;
    if (previous) previous->Next = next;
      else *proot = next;
 
    table_freexydata( current->xydata );
    free( current );
 
    }
 
  return(0);
 
} // table_freeXYData
 
/*+++------------------------------------------------------------------------
NAME
  table_freeXYDataAll --- releases all xy data
 
SYNOPSIS
 
  int table_freeXYDataAll( void );
 
DESCRIPTION
 
  Releases all XYData buffer
 
RETURN VALUE
 
   0: OK
  -1: Error
 
----------------------------------------------------------------------------*/
int table_freeXYDataAll( void )
{
  TableXYDataList *current, *previous, *next;
  TableXYDataList ** proot = &table_XYDataRoot;
 
  /* search xydata */
  next = *proot;
 
  if ( next ) { // remove next
 
    current  = next;
    previous = current->Previous;
    next     = current->Next;
 
    if (next) next->Previous = previous;
    if (previous) previous->Next = next;
      else *proot = next;
 
    table_freexydata( current->xydata );
    free( current );
 
    }
 
  return(0);

} // table_freeXYDataAll

/*+++------------------------------------------------------------------------
NAME
  table_SkipComments --- skips comment lines 
 
SYNOPSIS
 
  int table_SkipComments( FILE * in );
 
DESCRIPTION

  Skips subsequently all lines that start with a comment delimiter. 
  The file pointer is positioned at the start of the first line that does not
  start with a comment character. 
 
----------------------------------------------------------------------------*/
int table_SkipComments( FILE * in ) 
{ int c;

  while (1) {
    c = fgetc(in);
    if (ferror(in)) return(-1);
    if (feof(in)) return(0);

    if (c==comment_delimiter[0]) {
      while (c!=line_delimiter[0]) {
        c = fgetc(in);
        if (ferror(in)) return(-1);
        if (feof(in)) return(0);
      } 
    } else {
      ungetc(c,in);
      if (ferror(in)) return(-1);
      if (feof(in)) return(0);
      break;
    }
  }

  return(0);

} // skip_comments

/*+++------------------------------------------------------------------------
NAME
  table_Init --- initializes the module table 
 
SYNOPSIS
 
  void table_Init( double empty,         const char * comment,
                   const char * limline, const char * limcol );
 
DESCRIPTION
 
  Initializes the module table and defines the comment character and
  line and column delimeters. Only items with string length > 0 are
  initialized.
  empty  : default value of an item
  comment: start of a comment (must be at start of a line)
  limline: line delimiter
  limcol : column delimiter
 
RETURN VALUES
 
  0 : successful
 -1 : error
 
----------------------------------------------------------------------------*/
void table_Init( double empty,  const char * comment,
                 const char * limline, const char * limcol )
{ size_t limlen;

  table_empty = empty;

  limlen = strlen(comment);
  limlen = (limlen<TABLE_MAXLEN)?limlen:TABLE_MAXLEN;
  if (limlen>0) {
    strncpy(comment_delimiter, comment, limlen);
    comment_delimiter[limlen] = '\0';
  }

  limlen = strlen(limline);
  limlen = (limlen<TABLE_MAXLEN)?limlen:TABLE_MAXLEN;
  if (limlen>0) {
    strncpy(line_delimiter, limline, limlen);
    line_delimiter[limlen] = '\0';
  }
  
  limlen = strlen(limcol);
  limlen = (limlen<TABLE_MAXLEN)?limlen:TABLE_MAXLEN;
  if (limlen>0) {
    strncpy(col_delimiter, limcol, limlen);
    col_delimiter[limlen] = '\0';
  }

} // table_Init

/*+++------------------------------------------------------------------------
NAME
  table_Eoln --- tests for end of line 
 
SYNOPSIS
 
  int table_Eoln ( FILE * in )
 
DESCRIPTION
 
  Tests the next character of in for eoln and returns 1 if it is eoln. 
  In case of error -1 is returned.

  The character is put back with ungetc.
 
RETURN VALUES
 
  0 : successful or eof
 -1 : error
 
----------------------------------------------------------------------------*/
int table_Eoln( FILE * in ) 
{ int c;
  c = fgetc(in);
  if (ferror(in)) return(-1);
  if (feof(in)) return(0);

  ungetc(c, in);
  if (ferror(in)) return(-1);

  if (c==line_delimiter[0])
    return(1);

  return(0);
  
} // table_Eoln

/*+++------------------------------------------------------------------------
NAME
  table_SkipLines --- skips a number of lines 

SYNOPSIS

  int table_SkipLines( FILE * in, long lines )

DESCRIPTION

  Reads the next abs(<lines>) line feeds from 'in' and positions the file 
  pointer after the last read line feed. Lines that start with a comment 
  delimter are skipped.

  If <lines> is positive comment delimiters are checked at start and after 
  a read line feed, if <lines> is negative the first line is skipped 
  without checking for a comment delimiter. 

RETURN VALUES

  0 : successful or eof
 -1 : error

----------------------------------------------------------------------------*/
int table_SkipLines( FILE * in, long lines )
{ int c;
  long i;

  if (lines >= 0 )
    if ( table_SkipComments( in ) )
      return(-1);

  lines=fabs(lines);

  for (i=0;i<lines;i++) {
    c = fgetc(in);
    if (ferror(in)) return(-1);
    if (feof(in)) return(0);

    while (c!=line_delimiter[0]) {
      c = fgetc(in);
      if (ferror(in)) return(-1);
      if (feof(in)) return(0);
    }

    if ( table_SkipComments( in ) )
       return(-1);

  }

  return(0);

} // table_SkipLines

/*+++------------------------------------------------------------------------
NAME
  table_SkipChars --- skips a number of characters 
 
SYNOPSIS
 
  int table_SkipChars( FILE * in, long skipchars )
 
DESCRIPTION
 
  Reads <skipchars> characters from 'in' and positions the file pointer after
  the end of the last skipped character.
 
RETURN VALUES
 
  0 : successful or eof
 -1 : error
 
----------------------------------------------------------------------------*/
int table_SkipChars( FILE * in, long skipchars )
{ long i;

  for (i=0;i<skipchars;i++) {
    fgetc(in);
    if (ferror(in)) return(-1);
    if (feof(in)) return(0);
  }

  return(0);

} //table_SkipChars

/*+++------------------------------------------------------------------------
NAME
  table_SkipCols --- skips a number of columns 
 
SYNOPSIS
 
  int table_SkipCols( FILE * in, long skipcols )
 
DESCRIPTION
 
  Reads <skipcols> columns from 'in' and positions the file pointer after
  the end of the last skipped column or before the next line delimiter if not
  enough columns are available
 
RETURN VALUES
 
  0 : successful or eof
 -1 : error
 
----------------------------------------------------------------------------*/
int table_SkipCols ( FILE * in, long skipcols  )
{ int c;
  long i;
 
  for (i=0;i<skipcols;i++) {
    c = fgetc(in);
    if (ferror(in)) return(-1);
    if (feof(in)) return(0);
 
    // check currently only a single character
    while ((c!=col_delimiter[0])&&(c!=line_delimiter[0])) { 
      c = fgetc(in);
      if (ferror(in)) return(-1);
      if (feof(in)) return(0);
    }
    if (c==line_delimiter[0]) {
      ungetc(c, in);
      if (ferror(in)) return(-1);
      break; // stop at end of line
    }
  }

  return(0);

} // table_SkipCols
 
/*+++------------------------------------------------------------------------
NAME
  table_ReadCol --- copies the next column element into buffer
 
SYNOPSIS
 
  char * table_ReadCol  (  FILE * in, char * buffer, size_t buflen ); 
 
DESCRIPTION
 
  Reads the next column from 'in' and positions the file pointer at  
  the start of the next column or before the next line delimiter if the line
  end is reached.
 
RETURN VALUES
 
  NULL : eof or error
  pointer to filled buffer
 
----------------------------------------------------------------------------*/
char * table_ReadCol  (  FILE * in, char * buffer, size_t buflen )
{ int c;
  long i=0;

  buffer[0] = '\0';

  c = fgetc(in);
  if (ferror(in)) return((char*) NULL);
  if (feof(in)) return(buffer);
 
  // check currently only a single character
  while (c!=col_delimiter[0]) {
    if (c==line_delimiter[0]) {
      ungetc(c, in);
      if (ferror(in)) return((char*) NULL);
      break;
    }
    if (i<buflen) buffer[i++] = (char) c;
    c = fgetc(in);
    if (ferror(in)) return((char*) NULL);
    if (feof(in)) break;
  }
  buffer[i] = '\0';

  return( buffer );

} // table_ReadCol


/*+++------------------------------------------------------------------------
NAME
  table_ReadXYRows --- reads xy-pairs from two rows 
 
SYNOPSIS

  int table_ReadXYRows( FILE * in, double bufferX[], double bufferY[],
                        long buflen, long posx, long posy, long maxcnt );
 
DESCRIPTION

  Reads maxcnt xy-pairs from in. The values are extracted from row 
  posx and posy. The values are converted to double and are written into
  bufferX and bufferY. Lines starting with a comment delimiter are skipped
  (except first line). The number of valid xy-pairs is returned.

  A maximum of buflen values are written into the double float buffers
  bufferX and bufferY. The length of both buffers must be buflen. If
  buflen is zero the buffers are not used and the function returns only
  the number of xy-pairs in the table. This can be used to allocate the 
  buffers dynamically.
 
  FILE * in
  double bufferX[]
  double bufferY[]
  long buflen
  long posx
  long posy
  long maxcnt
 
RETURN VALUES
 
   positive or zero: number of read lines (can be larger than buflen)
   negative        : error                                                                                                                    

----------------------------------------------------------------------------*/
int table_ReadXYRows( FILE * in, double bufferX[], double bufferY[], 
                      long buflen, long posx, long posy, long maxcnt )
{
  long pos1, pos2;
  long cnt;

  char buffer1[TABLE_MAXCOLWIDTH];

  double *B1, *B2;
  char *C1, *C2;
 
  float tmp;

  if (posx<posy) {
    pos1 = posx; pos2 = posy;
    B1 = bufferX; B2 = bufferY;
  } else {
    pos1 = posy; pos2 = posx;
    B1 = bufferY; B2 = bufferX;
  }

  /* first row */
  cnt = 0l;
  if (table_SkipLines (in, pos1-1  )) return(-1);
  if (feof(in)) return ( cnt );

  for (cnt=0;((0>maxcnt)||(cnt<maxcnt))&&(!table_Eoln(in));cnt++) {

    C1 = table_ReadCol  ( in, buffer1, (size_t) TABLE_MAXCOLWIDTH );
    if (!(C1)) return(-1);
    if (feof(in)) break;

    if (cnt<buflen) {
      if (B1) {
        if (sscanf(C1,"%g", &tmp)>0) {
          B1[cnt] = (double) tmp;
        } else B1[cnt] = table_empty;
      }
      if (B2) {
        if (pos1==pos2) B2[cnt] = B1[cnt]; else B2[cnt] = table_empty;
      }
    }
  }

  /* skip rest of the current line without checking it for a comment */
  if ((table_SkipLines( in, -1 ))) return(-1);
  if (feof(in)) return ( cnt );
 
  /* second row */
  if (pos1!=pos2) {
    cnt = 0l;
    if (table_SkipLines (in, pos2-pos1-1  )) return(-1);
    if (feof(in)) return ( cnt );

    for (cnt=0;((0>maxcnt)||(cnt<maxcnt))&&(!table_Eoln(in));cnt++) {

      C2 = table_ReadCol  ( in, buffer1, (size_t) TABLE_MAXCOLWIDTH );
      if (!(C2)) return(-1);
      if (feof(in)) break;

      if (cnt<buflen) {
         if (B2) {
           if (sscanf(C2,"%g", &tmp)>0) {
             B2[cnt] = (double) tmp;
           } else B2[cnt] = table_empty;
         }
      }
    } 

    /* skip rest of the current line without checking it for a comment */
    if ((table_SkipLines( in, -1 ))) return(-1);

  }

  return ( cnt );

} // table_ReadXYRows 

/*+++------------------------------------------------------------------------
NAME
  table_ReadXYColumns --- reads xy-pairs from two columns
 
SYNOPSIS
 
  int table_ReadXYColumns( FILE * in, double bufferX[], double bufferY[],
                           long buflen, long posx, long posy, long maxcnt );
 
DESCRIPTION
 
  Reads maxcnt xy-pairs from in. The values are extracted from column
  posx and posy. The values are converted to double and are written into 
  bufferX and bufferY. Lines starting with a comment delimiter are skipped
  (except first line). The number of valid xy-pairs is returned.
 
  A maximum of buflen values are written into the double float buffers
  bufferX and bufferY. The length of both buffers must be buflen. If
  buflen is zero the buffers are not used and the function returns only
  the number of xy-pairs in the table. This can be used to allocate the
  buffers dynamically.
 
  FILE * in
  double bufferX[]
  double bufferY[]
  long buflen
  long posx
  long posy
  long maxcnt
 
RETURN VALUES
 
   positive or zero: number of read lines (can be larger than buflen)
   negative        : error
 
----------------------------------------------------------------------------*/
int table_ReadXYColumns( FILE * in, double bufferX[], double bufferY[],
                         long buflen, long posx, long posy, long maxcnt )
{
  long pos1, pos2;
  long cnt=(long) 0;
 
  char buffer1[TABLE_MAXCOLWIDTH], buffer2[TABLE_MAXCOLWIDTH];
 
  char ** pC1, ** pC2;
  char * X, * Y;
 
  float tmp;

  if (posx<posy) {
    pos1 = posx; pos2 = posy;
    pC1 = &X; pC2 = &Y;
  } else {
    pos1 = posy; pos2 = posx;
    pC1 = &Y; pC2 = &X;
  }
 
  for (cnt=0;(0>maxcnt)||(cnt<maxcnt);cnt++) {
 
    if (feof(in)) break;
 
    /* first column */
    if (table_SkipCols (in, pos1-1  )) return(-1);
    if (feof(in)) break;
 
    *pC1 = table_ReadCol  ( in, buffer1, (size_t) TABLE_MAXCOLWIDTH );
    if (!(*pC1)) return(-1);
    if (feof(in)) break;
 
    /* second column */
    if (pos1!=pos2) {
      if (table_SkipCols (in, pos2-pos1-1  )) return(-1);
      if (feof(in)) break;
 
      *pC2 = table_ReadCol  ( in, buffer2, (size_t) TABLE_MAXCOLWIDTH );
      if (!(*pC2)) return(-1);
      if (feof(in)) break;
    } else *pC2 = *pC1;
 
    /* skip rest of the current line without checking it for a comment */
    if ((table_SkipLines( in, -1 ))) return(-1); 
 
    if (cnt<buflen) {
       if (bufferX) {
         if (sscanf(X,"%g", &tmp)>0) {
           bufferX[cnt] = (double) tmp;
         } else bufferX[cnt] = table_empty;
       }
       if (bufferY) {
         if (sscanf(Y,"%g", &tmp)>0) {
           bufferY[cnt] = (double) tmp;
         } else bufferY[cnt] = table_empty;
       }
    }
  }
 
  return ( cnt );
 
} // table_ReadXYColumns
 
/*+++------------------------------------------------------------------------
NAME
  table_ReadXYDouble --- reads an xy-pairs from a text table
 
SYNOPSIS
 
  int table_ReadXYDouble( double bufferX[], double bufferY[], long buflen,
                    const char * filename, long skiplines, long skipchars,
                    long posx, long posy, long maxcnt )
 
DESCRIPTION

  Reads maxcnt lines from filename and extracts column posx and 
  posy. The default filename is "xy.txt". The values are converted 
  to double and are written into xydata Lines starting with a comment 
  delimiter (default #) are skipped. The first skiplines 
  non-comment-lines are skipped. Additionally, the first skipchars 
  characters of the first non comment line are skipped. 
  If swap is set (!=0) the values are read from columns, if not from rows.
  The number of valid xy-pairs is returned. 
  If xydata is NULL only the number of x-pairs is returned.

  A maximum of buflen values are written into the double float buffers 
  bufferX and bufferY of xydata. The length of both buffers must be 
  buflen. If buflen is zero the buffers are not used and the function 
  returns only the required value of buflen to read the whole table. 
  This can be used to allocate dynamically the buffers.

  TableXYData * xydata
  const char * filename
  long skiplines
  long skipchars
  int  swap
  long posx
  long posy
  long maxcnt
 
RETURN VALUES
 
   positive or zero: number of read lines (can be larger than buflen) 
   negative        : error
 
----------------------------------------------------------------------------*/
int table_ReadXYDouble( TableXYData * xydata, const char * filename, 
                        long skiplines, long skipchars, int swap, 
                        long posx, long posy, long maxcnt ) 
{ long cnt=0;

  const char * fnam;
  FILE * in;
  double *bufferX, *bufferY; 
  long buflen = 0l;

  if ( (filename) && (strlen(filename)>0) ) 
    fnam = filename;
  else fnam = filename_default;

  in = fopen(fnam,"r");
  if (in == (FILE *) NULL) return(-1);

  // skip leading lines

  if (table_SkipLines( in, skiplines )) return(-1); 

  if (table_SkipChars( in, skipchars )) return(-1);

  // read table

  if ( xydata ) {
    bufferX = xydata->X; bufferY = xydata->Y; 
    buflen = xydata->buflen;
  } else {
    bufferX = (double*) NULL; bufferY = (double*) NULL; 
  }

  if (swap)
    cnt = table_ReadXYColumns( in, bufferX, bufferY, buflen, 
                               posx, posy, maxcnt );
  else
    cnt = table_ReadXYRows   ( in, bufferX, bufferY, buflen, 
                               posx, posy, maxcnt );

  if ( xydata ) xydata->len = cnt;

  fclose(in);

  return( cnt ); 

} /* table_ReadXYDouble */

/*---------------------------------------------------------------------------
NAME
  table_LinIntXYReset --- reset index
---------------------------------------------------------------------------*/
void table_LinIntXYReset( TableXYData * xydata )
{ xydata->index = 0l;
} /* table_LinIntXYReset */

/*+++------------------------------------------------------------------------
NAME
   table_LinIntXYDouble --- linear interpolation of xy-pairs 

SYNOPSIS

   double table_LinIntXYDouble( TableXYData * xydata, double s );

DESCRIPTION
   
  The x-values of xydata are scanned starting at the last used interval until 
the first interval [xydata->X[i],xydata->X[i+1]] that contains the value s. The 
following interpolated value is returned:

                                         xydata->Y[i+1]-xydata->Y[i]
     y = xydata->Y[i] + (s-xydata->X[i]) * -------------------------
                                         xydata->X[i+1]-xydata->X[i]

If s is equal to a tabulated x-value the corresponding y-value is returned.
The function table_LinIntXYReset resets the last used interval to [x[0],x[1]].

RETURN VALUE

   interpolated value

----------------------------------------------------------------------------*/
double table_LinIntXYDouble( TableXYData * xydata, double s )
{ double value = table_empty;
  double x1=0.0, x2=0.0, y1, y2;
  double tmp;
  long i, index=-1;
  int found=0;


  if (xydata->index<0l) xydata->index=0l;

  // from xydata->index to end of table
  if (xydata->index<xydata->len) x2 = xydata->X[xydata->index];
  for (i=xydata->index;(i<xydata->len-1)&&(!found);i++) {
    x1=x2; x2 = xydata->X[i+1];
    found = ((x1<=s)&&(s<=x2)) || ((x1>=s)&&(s>=x2));
    if (found) index=i;
  }

  if (!found) {
    // from 0 to xydata->index-1
    if (0<xydata->len) x2 = xydata->X[0];
    for (i=0;(i<xydata->index-1)&&(!found);i++) {
      x1=x2; x2 = xydata->X[i+1];
      found = ((x1<=s)&&(s<=x2)) || ((x1>=s)&&(s>=x2));
      if (found) index=i;
    }
  }

  if (found) {
    xydata->index = index;
    y1 = xydata->Y[index]; y2 = xydata->Y[index+1];
    tmp = x2-x1;
    if (fabs(tmp)>TABLE_EPS)
      value = y1 + (s-x1) * (y2-y1)/tmp;
    else
      value = y1;
  }

  return( value );

} /* table_LinIntXYDouble */

/*---------------------------------------------------------------------------
NAME
  table_printXYData --- reset index
---------------------------------------------------------------------------*/
void table_printXYData( FILE * out, TableXYData * xydata )
{ long i; 
  for (i=0;i<xydata->len;i++) {
    fprintf(out,"%ld : %g : %g \n",i,xydata->X[i],xydata->Y[i]);
  }
} /* table_printXYData */
