/****************************************************************************
*  Include                                                                  *
****************************************************************************/
# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <stdarg.h>
# include <string.h>
# include <ctype.h>
# include <limits.h>
# include <errno.h>
# include <fcntl.h>
# include <math.h>
# include <float.h>
  
# include "strlib.h"
# include "numio.h" 
# include "median.h"
# include "mean.h"

/******************************************************************************
* Type Defs                                                                   *
******************************************************************************/

typedef struct minmax_DataElement { 
  double X;
  struct minmax_DataElement * Previous, * Next;
} MinMaxDataElement;

typedef struct minmax_DataList {
  MinMaxDataElement * Element;
  MinMaxDataElement * Last; // pointer to last element
  long nelements;           // number of data elements
} MinMaxDataList;

MinMaxDataElement * append_minmax_element( MinMaxDataList * list, double value )
{ MinMaxDataElement * newelement=(MinMaxDataElement *) NULL;

  if (!list)
    goto append_minmax_element_error;

  if ( newelement = (MinMaxDataElement*) malloc(sizeof(MinMaxDataElement)) ) {
    MinMaxDataElement * current;

    newelement->X = value;

    current = list->Last; 

    // add newelement after current
    if (current) {
      newelement->Previous = current;
      newelement->Next = current->Next;
      current->Next = newelement; 
    } else {
      newelement->Previous = NULL;
      newelement->Next = NULL;
      list->Element = newelement;
    }

  }

  list->Last = newelement;
  list->nelements += 1;

  return( newelement );

append_minmax_element_error:

  if (newelement) free(newelement);

  return( NULL );

} // append_minmax_element

/*--------------------------------------------------------------------------
NAME

  new_minmax_list --- returns a pointer to a new MinMaxDataList

SYNOPSIS

  MinMaxDataList * new_minmax_list( void )

ARGUMENTS

  return MinMaxDataList * (o) : pointer to an empty minmax list or
                                NULL in case of error.

DESCRPTION

 Returns a pointer to the root of a new empty list.
--------------------------------------------------------------------------*/
MinMaxDataList * new_minmax_list( void )
{ MinMaxDataList * newlist = (MinMaxDataList*) NULL;

  if (newlist = (MinMaxDataList *) malloc ( sizeof(MinMaxDataList) )) {
    newlist->Element = (MinMaxDataElement *) NULL; 
    newlist->Last = (MinMaxDataElement *) NULL;
    newlist->nelements=0;
  }

  return( newlist );

} // new_minmax_list

/*--------------------------------------------------------------------------
NAME

  free_minmax_list --- frees the full list

SYNOPSIS

  int free_minmax_list( MinMaxDataList * list ):

ARGUMENTS

  returns 0

DESCRPTION

  Deallocates all elements of list and list itself. The list pointer is
  no longer valid and must be set to NULL. If list is NULL, nothing is
  done.
--------------------------------------------------------------------------*/
int free_minmax_list( MinMaxDataList * list ) 
{ if ( list ) {
    MinMaxDataElement *current, *next;
    // free all list elements
    next = list->Element;
    while ( next ) {
      current = next; 
      next = current->Next;
      free( current );
    }
    // free list root
    free( list );
  }
  
  return( 0 );

} // free_minmax_list

/*--------------------------------------------------------------------------
NAME

  print_minmax_list --- print list elements into a file 

SYNOPSIS

  int print_minmax_list( FILE * out, MinMaxDataList * list, int mode )

ARGUMENTS

  FILE * out            (i) : output
  MinMaxDataList * list (i) : list
  int mode              (i) : 0: 1 element per line
                              1: all elements into a single line

  returns 0

DESCRPTION

  Prints all list elements to out.
--------------------------------------------------------------------------*/
int print_minmax_list( FILE * out, MinMaxDataList * list, int mode )
{
 if ( list ) {
    MinMaxDataElement *current, *next;
    long index;


    switch (mode) {
      case 1:
        // print list information 
        fprintf(out,"%d: ",list->nelements);

        index=1;
        next = list->Element;
        while ( next ) {
          current = next;
          next = current->Next;
    
          fprintf(out,"%g ",index,current->X);
          index++;
        }
        fprintf(out,"\n");
        break;
      default:
        // print list information 
        fprintf(out,"n = %d\n",list->nelements);

        index=1;
        next = list->Element;
        while ( next ) {
          current = next;
          next = current->Next;

          fprintf(out,"%d : %g\n",index,current->X);
          index++;
        }
    }
  }

  return( 0 );

} // print_minmax_list

/*--------------------------------------------------------------------------
NAME

  minmax_list_length --- returns the number of elements in the list

SYNOPSIS

  long minmax_list_length(MinMaxDataList * list)

ARGUMENTS

  returns 0

DESCRPTION

  Returns the number of elements in the list.

--------------------------------------------------------------------------*/
long minmax_list_length(MinMaxDataList * list)
{ 
  return( list->nelements );
} // minmax_list_length

/*--------------------------------------------------------------------------
NAME

  minmax_table --- copies all list elements to buffer

SYNOPSIS

  double * minmax_table( double *buf, size_t buflen, MinMaxDataList * list );

ARGUMENTS

  Returns a pointer to the updated buffer or NULL in case of an error.

DESCRPTION

  Copies all list elements to buffer and returns a pointer to it. If list 
  is NULL or if it is empty NULL is returned.

--------------------------------------------------------------------------*/
double * minmax_table( double *buf, size_t buflen, MinMaxDataList * list )
{ MinMaxDataElement * pe;
  double *pb=buf;
  size_t u=(size_t) 0;

  if (!list) 
    goto minmax_table_error;

  if (!(pe = list->Element) )
    goto minmax_table_error;

  while (pe) {
    if (u>=buflen)
      goto minmax_table_error;
    *pb = pe->X;
    u++; pb++; pe = pe->Next;
  }

  return( buf );

minmax_table_error:

  return( NULL );

} // minmax_table

int print_minmax_table( FILE *out, double a[], long n ) {
  long i;
  fprintf(out,"Table (%d elements)\n",n);
  if (a) {
    for (i=0;i<n;i++) {
      fprintf(out, "a[%d] = %g\n", i, a[i] );
    }
    fprintf(out,"\n");
  }

  return(0);

} // print_minmax_table

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_minmax
#else
# define MAIN main
#endif

# define Usage "<minval> <maxval> (values are read from standard input)"

# define BUFLEN 2048
int MAIN (int argc, char *argv[])
{ static const char * Main_Error="ERROR: minmax";
  char buf[BUFLEN];                       // buffer for input command
  const char *pb;
  int errval;

  double value, minval, maxval;
  MinMaxDataList *list=NULL;
  long n;
  double *a=NULL;

  if (argc<=2) { printf("%s %s\n",argv[0],Usage); goto main_error; }

  if (argc>1) minval = num_str2double ( argv[1], NULL, &errval);
  if (errval) {
    printf("ERROR: num_str2double\n",pb);
    goto main_error;
  }

  if (argc>2) maxval = num_str2double ( argv[2], NULL, &errval);
  if (errval) {
    printf("ERROR: num_str2double\n",pb);
    goto main_error;
  }

  fgets(buf,BUFLEN, stdin);
  while(!feof(stdin)) {
    if (!(strlib_is_empty(buf))) {

      printf("-------------------------------------------------------\n");

      if (! (list = new_minmax_list()) )
        goto main_error;

      pb = buf;

      // num_str2double ( const char *str, const char **tail, int *perrval);
      while (!strlib_is_empty( pb )) {
        value = num_str2double ( pb, &pb, &errval);
        if (errval) {
          printf("ERROR: num_str2double rest=%s\n",pb);
          break;
        }
        append_minmax_element( list, value );
      }

      print_minmax_list( stdout, list, 1 );

      value = 0.0;
      n = minmax_list_length( list );
      if (n>0) {
        long n1;
        if ( a = (double *) malloc ( sizeof(double)*n ) ) {
          if ( minmax_table( a, n, list ) ) {
            print_minmax_table( stdout, a, n ); 
            value = dmedian ( a, n );
            printf("median = %g\n",value);
            value = mean ( a, n );
            printf("mean = %g\n",value);
            value = variance ( a, n, value );
            printf("variance = %g\n",value);
            value = minimum ( a, n );
            printf("minimum = %g\n",value);
            value = maximum ( a, n );
            printf("maximum = %g\n",value);

            n = minmaxfilter ( a, n, minval, maxval );
            print_minmax_table( stdout, a, n );
            value = dmedian ( a, n );
            printf("median = %g\n",value);
            value = mean ( a, n );
            printf("mean = %g\n",value);
            value = variance ( a, n, value );
            printf("variance = %g\n",value);
            value = minimum ( a, n );
            printf("minimum = %g\n",value);
            value = maximum ( a, n );
            printf("maximum = %g\n",value);
          }
          free(a);
        }
      }

      free_minmax_list( list );

    } // if not empty

    fgets(buf,BUFLEN, stdin);
  } /* while */

  return(0);

main_error:

  free_minmax_list( list );

  return(-1);

} /* MAIN */
