#! /opt/local/bin/python
# -*- coding: utf-8 -*-

# PyNX - Python tools for Nano-structures Crystallography
#   (c) 2016-present : ESRF-European Synchrotron Radiation Facility
#       authors:
#         Vincent Favre-Nicolin, favre@esrf.fr


import sys
import os
import glob
import time
import locale
import timeit
import warnings
import argparse

from ...utils import h5py
import numpy as np
from silx.io.dictdump import nxtodict

from .runner import PtychoRunner, PtychoRunnerScan, PtychoRunnerException, default_params as params0
from .parser import ActionPtychoMotors, ActionROI

helptext_epilog = """
Examples:

* ``pynx-ptycho-id16a-nf --data data.nx \
--algorithm analysis,ML**1000,DM**200,nbprobe=3,probe=1 --projection 1``

In this case the nx data file is an nxtomo-converted file from the
raw data files, and includes all frames (projections, dark and
flat/bright field). --projection is used to select the projection.

* ``pynx-ptycho-id16a-nf --data data.nx \
--algorithm ML**1000,DM**200,nbprobe=3,probe=1 \
 --use_direct_beam --projection="range(1,50)" \
 --multiscan_reuse_ptycho ML**500,probe=1``

Same as before but we are using the direct beam (flat/bright field)
as an additional constraint to stabilise the optimisation. Also,
we process a series of projections, and use a shorter algorithm
to go faster since we restart from the previous projection.

* ``pynx-ptycho-id16a-nf --data path/to/data.nx:entry0001 \
--algorithm analysis,ML**1000,DM**200,nbprobe=3,probe=1 \
--projection 1 --saveplot``

same as above, but give a full path, as well as a hdf5 entry in the
nexus file. Also, --saveplot will save a png plot at the end of
each optimisation.

Using OLD data format (up to 12/2025):

* ``pynx-ptycho-id16a-nf --data data.h5 --meta meta.h5 --dark dark000.h5\
--algorithm analysis,ML**100,DM**200,nbprobe=3,probe=1``

Using a reference (direct beam) frame, allows to use Paganin, and 
with use_direct_beam will provide an absolute scale for the probe:
    
* ``pynx-ptycho-id16a-nf --data data.h5 --meta meta.h5 --dark dark000.h5 --data_ref ref.h5\
--algorithm analysis,ML**100,DM**200,Paganin,probe=1\
--use_direct_beam``

"""

# NB: for id16 we start from a flat object (high energy, high transmission)
params_beamline = {
    'adu_scale': None,
    'data_ref': None,
    'delta_beta': None,
    'autocenter': False,
    'instrument': 'ESRF id16a',
    'maxsize': 10000,
    'meta': None,
    'near_field': True,
    'object': 'random,0.95,1,0,0.1',
    'padding': 0,
    'ptychomotors': None,
    'roi': 'full',
    'scan': 1,
    'sequence': None,
    'tomo_motor': None
}

default_params = params0.copy()
for k, v in params_beamline.items():
    default_params[k] = v


def _get_data_type(filename):
    """
    Test what type of data is given as input
    :param filename: the hdf5 data file
    :return: either 'legacy' (pre-bliss conversion of ID16A), 'nx' or 'bliss'
    """
    # TODO: check if this is the correct way to check this is a raw bliss data file
    if ':' in filename:
        # Remove hdf5 entry
        filename = filename.split(':')[0]
    with h5py.File(filename, 'r') as h:
        if 'publisher' in h.attrs:
            if h.attrs['publisher'] == 'bliss':
                return 'bliss'
        for entry in h.values():
            if 'definition' in entry.attrs:
                if entry.attrs['definition'].lower() == 'nxtomo':
                    return 'nx'
    return 'legacy'


class PtychoRunnerScanID16aNF(PtychoRunnerScan):
    def __init__(self, params, scan, timings=None):
        super(PtychoRunnerScanID16aNF, self).__init__(params, scan, timings=timings)
        self.h5_header_path = None
        self.tomo_metadata = {}  # to store the projection angle
        self.data_type = None  # bliss, nx or legacy
        self.proj_i0 = 0  # first frame of desired projection for raw bliss or NX data
        self.h5paths = {}  # paths to (meta)data in data files

    @staticmethod
    def _get_data_units(v):
        if 'units' in v.attrs:
            u = v.attrs['units']
            if u in ['um', 'µm']:
                return v[()] * 1e-6
            elif u in ['nm']:
                return v[()] * 1e-9
            elif u in ['mm']:
                return v[()] * 1e-3
            elif u in ['degree', '°']:
                return np.deg2rad(v[()])
        return v[()]

    def load_scan(self):
        data_filename = self.params['data']
        if '%' in data_filename:
            data_filename = data_filename % tuple(self.scan for i in range(data_filename.count('%')))
            print('data filename for scan #%d: %s' % (self.scan, data_filename))
        if ':' in data_filename:
            # An hdf5 entry was given
            data_filename, hentry = data_filename.split(':')
        else:
            hentry = None

        self.data_type = _get_data_type(data_filename)
        m = self.params['ptychomotors']
        if self.data_type == 'bliss':
            with h5py.File(data_filename, mode='r') as h:
                # First find the sequence
                seq0 = None
                if self.params['sequence'] is not None:
                    seq0 = self.params['sequence']
                    hseq0 = h[f"{seq0}.1"]
                    if 'NFP' not in hseq0['title'][()].decode():
                        raise PtychoRunnerException(
                            f"Entry {self.params['sequence']}.1 "
                            f"from {data_filename} does not look an NFP sequence "
                            f"(title: {hseq0['title'][()].decode()})")
                else:
                    for e in h:
                        if e['title'][()].decode() in ['NFP2D', 'NFP3D']:
                            seq = e
                            break
                if seq0 is None:
                    raise PtychoRunnerException(
                        "No sequence # given as input, and could not find"
                        "an NFP2D or 3D sequence in the file")

                # Find the entry with the relevant projection ('scan' in pynx)
                # TODO: also give access to the 'quali' scans
                subscan = None
                for i in seq['subscans/scan_numbers'][()]:
                    if 'projections' in h[f'{i}.1/title'][()].decode():
                        # TODO: get the relevant start index (self.proj_i0) of the desired projection
                        #  , and the number of frames.
                        # Motor names - 2 (x y) for 2D, 3 for ptycho-tomo (rot x y)
                        vmot = h[f'{i}.1/technique/proj/motors']
                        xmot, ymot = [v.decode() for v in vmot[-2:]]
                        if len(vmot) == 3:
                            # ptycho-tomo, we should have the rotation motor as well
                            ang_mot = vmot[0].decode()
                        else:
                            ang_mot = None
                        subscan = f'{i}.1'
                        break
                if subscan is None:
                    raise PtychoRunnerException("No subscan found with the requested projection")

                # Set paths in hdf5 file
                self.h5paths['x'] = f'{subscan}/measurement/{xmot}'
                self.h5paths['y'] = f'{subscan}/measurement/{ymot}'
                if ang_mot is None:
                    self.h5paths['angle'] = None
                else:
                    self.h5paths['angle'] = f'{subscan}/measurement/{xmot}'
                # TODO: energy, sample,...

        if self.data_type == 'nx':
            # bliss data file converted to NX
            with h5py.File(data_filename, mode='r') as h:
                if hentry is None:
                    hentry = list(h.keys())[0]
                else:
                    if hentry not in h.keys():
                        raise PtychoRunnerException(
                            f"Could not find entry '{hentry}' "
                            f"in file: {data_filename}\n"
                            f"Available entries are: {', '.join(h.keys())}")
                # Paths
                self.h5paths['entry'] = f'{hentry}'
                self.h5paths['image_key'] = f'{hentry}/data/image_key'
                self.h5paths['x'] = f'{hentry}/sample/x_translation'  # z ?
                self.h5paths['y'] = f'{hentry}/sample/y_translation'
                self.h5paths['angle'] = f'{hentry}/sample/rotation_angle'
                self.h5paths['x_pixel_size'] = f'{hentry}/sample/x_pixel_size'
                self.h5paths['y_pixel_size'] = f'{hentry}/sample/y_pixel_size'
                self.h5paths['distance'] = f'{hentry}/sample/propagation_distance'
                self.h5paths['nrj'] = f'{hentry}/instrument/beam/incident_energy'
                self.h5paths['sample_name'] = f'{hentry}/sample/name'
                self.h5paths['data'] = f'{hentry}/data/data'

                # Find the required projection frames
                k = h[self.h5paths['image_key']][()]
                idx = np.where(k == 0)[0]  # valid projections
                ang = h[f'{hentry}/sample/rotation_angle'][idx]
                c = (ang[:-1] - ang[1:]) == 0  # tomo angle changes
                # indices where angle change
                vchanges = np.concatenate(
                    ([0], np.where(c == False)[0] + 1, [len(ang)]))
                # Indices and number of frames per projection
                proj_nb = {idx[vchanges[i]]: vchanges[i + 1] - vchanges[i]
                           for i in range(len(vchanges) - 1)}

                # Isolate 'quali' projections
                # TODO: find a better way to isolate qualis
                k, v = list(proj_nb.keys()), np.array(list(proj_nb.values()))
                idx = np.where(v >= 25)[0]  # KLUDGE
                if len(idx) > 0:
                    quali_nb = {k[i]: v[i] for i in idx}
                    for i in idx:
                        proj_nb.pop(k[i])
                else:
                    quali_nb = {}

                print(f"Found {len(proj_nb)} projections and {len(quali_nb)} qualis")

                # Choose projection
                if self.scan is None:
                    self.scan = 1  # First projection (or 2D ptycho)
                elif self.scan == 0:
                    raise PtychoRunnerException(
                        "ID16A ptycho runner: --scan must either be >0 "
                        "(projection index) or <1 (to select one of the quali scans)")
                if self.scan > 0:
                    self.proj_i0 = list(proj_nb.keys())[self.scan - 1]  # First frame
                    proj_nb = proj_nb[self.proj_i0]  # number of frames
                else:
                    # Quali scan has been selected
                    self.proj_i0 = list(quali_nb.keys())[-self.scan - 1]  # First frame
                    proj_nb = quali_nb[self.proj_i0]  # number of frames

        if self.data_type in ['bliss', 'nx']:
            with h5py.File(data_filename, mode='r') as h:
                # We have the paths, get the (meta) data
                i0, nb = self.proj_i0, proj_nb
                print(f"Reading x positions from {data_filename}:{self.h5paths['x']}[{i0}:{i0 + nb}]")
                self.x = -self._get_data_units(h[self.h5paths['x']])[i0:i0 + nb]
                self.y = self._get_data_units(h[self.h5paths['y']])[i0:i0 + nb]
                if self.h5paths['angle'] is not None:
                    ang = self._get_data_units(h[self.h5paths['angle']])[i0]
                else:
                    ang = 0

        if self.data_type == 'legacy':
            # Old (up to 12/2025) data format converted from spec to hdf5
            data = h5py.File(data_filename, mode='r', enable_file_locking=False)
            xmot, ymot = m[0:2]
            for instrument in ['instrument', 'ESRF-ID16A']:
                for detector_name in ['Frelon', 'PCIe', 'ximea']:
                    s = f'entry_0000/{instrument}/{detector_name}/header'
                    if s in data:
                        self.h5_header_path = s
                        break
                if self.h5_header_path is not None:
                    break
            self.x = np.array([v for v in data[f'{self.h5_header_path}/{xmot}'][()].split()], dtype=np.float32)
            self.y = np.array([v for v in data[f'{self.h5_header_path}/{ymot}'][()].split()], dtype=np.float32)
            # Extract tomo angle
            motors = {}
            header = data[f'{self.h5_header_path}']

            for k, v in zip(header['motor_mne '][()].split(), header['motor_pos '][()].split()):
                motors[k.decode()] = float(v)
            ang = np.deg2rad(motors[self.params['tomo_motor']])

        self.tomo_metadata['angle'] = ang
        print(f"Tomography angle: {self.params['tomo_motor']}={np.rad2deg(ang):8.3f}")
        self.tomo_metadata['data'] = data_filename

        imgn = np.arange(len(self.x), dtype=np.int32)
        if self.params['moduloframe'] is not None:
            n1, n2 = self.params['moduloframe']
            idx = np.where(imgn % n1 == n2)[0]
            imgn = imgn.take(idx)
            self.x = self.x.take(idx)
            self.y = self.y.take(idx)

        if self.params['maxframe'] is not None:
            N = self.params['maxframe']
            if len(imgn) > N:
                print("MAXFRAME: only using first %d frames" % N)
                imgn = imgn[:N]
                self.x = self.x[:N]
                self.y = self.y[:N]

        imgn = np.array(imgn)

        if m is not None:
            if len(m) >= 3:
                x, y = self.x, self.y
                self.x, self.y = eval(m[-2]), eval(m[-1])

        if len(self.x) < 4:
            raise PtychoRunnerException("Less than 4 scan positions, is this a ptycho scan ?")

        self.imgn = imgn

    def load_data(self):
        data_filename = self.params['data']
        if '%' in data_filename:
            data_filename = data_filename % tuple(self.scan for i in range(data_filename.count('%')))
        if ':' in data_filename:
            # An hdf5 entry was given - already taken into account in h5paths
            data_filename = data_filename.split(':')[0]

        if self.data_type in ['bliss', 'nx']:
            with h5py.File(data_filename, mode='r') as h:
                if self.params['nrj'] is None:
                    self.params['nrj'] = float(h[self.h5paths['nrj']][()])
                    print("Energy read from the bliss/nx file: %6.3f keV" % self.params['nrj'])
                else:
                    print("Energy from the command line: %6.3f keV" % self.params['nrj'])

                if self.params['pixelsize'] is None:
                    self.params['pixelsize'] = \
                        self._get_data_units(h[self.h5paths['x_pixel_size']])
                    print("Effective pixel size: %12.3fnm [from bliss/nx file]" % (self.params['pixelsize'] * 1e9))
                else:
                    print("Effective pixel size: %12.3fnm [from command-line]" % (self.params['pixelsize'] * 1e9))

                # Detector distance
                if self.params['detectordistance'] is None:
                    self.params['detectordistance'] = \
                        self._get_data_units(h[self.h5paths['distance']])
                    print(f"Effective propagation distance (from bliss/nx): "
                          f"{self.params['detectordistance'] * 1e3:8.2f} mm")
                else:
                    print(f"Effective propagation distance (from command-line): "
                          f"{self.params['detectordistance'] * 1e3:8.3f} mm")

                self.sample_name = h[self.h5paths['sample_name']][()].decode()
                print(f"Sample name: {self.sample_name}")

                # Read all frames
                imgn = self.proj_i0 + self.imgn
                t0 = timeit.default_timer()
                print(f"Reading frames: {self.h5paths['data']}[{imgn[0]}:{imgn[-1]}+1]")
                self.raw_data = h[self.h5paths['data']][imgn.tolist()].astype(np.float32, copy=False)
                dt = timeit.default_timer() - t0
                print('Time to read all frames: %4.1fs [%5.2f Mpixel/s]' %
                      (dt, self.raw_data[0].size * len(self.raw_data) / 1e6 / dt))

                # image keys
                k = h[self.h5paths['image_key']][()]

                # Read flat - if needed
                if 'paganin' in self.params['algorithm'].lower() or \
                        'ctf' in self.params['algorithm'].lower() or \
                        self.params['use_direct_beam']:
                    idx = np.where(k == 1)[0]  # flat/bright field
                    d = h[self.h5paths['data']][idx].astype(np.float32, copy=False)
                    if d.ndim == 3:
                        self.data_ref = d.mean(axis=0)
                    else:
                        self.data_ref = d
                # Read dark
                idx = np.where(k == 2)[0]  # dark frames
                d = h[self.h5paths['data']][idx].astype(np.float32, copy=False)
                if d.ndim == 3:
                    self.raw_dark = d.mean(axis=0)
                else:
                    self.raw_dark = d

        else:
            data = h5py.File(data_filename, mode='r', enable_file_locking=False)

            meta = h5py.File(self.params['meta'], mode='r', enable_file_locking=False)
            # Entry name is obfuscated, so just take the first entry with 'entry' in the name
            meta_entry = None
            for k, v in meta.items():
                if 'entry' in k:
                    meta_entry = v
                    break

            # NXsample
            if 'sample' in meta_entry:
                try:
                    self.sample_nx_dict = nxtodict(meta_entry['sample'])
                    if 'name' in self.sample_nx_dict:
                        self.sample_name = self.sample_nx_dict['name'].tobytes().decode()
                except Exception:
                    warnings.warn("Something went wrong converting NXsample info - skipping")

            if False:
                date_string = data["entry_0000/start_time"][()]  # '2020-12-12T15:29:07Z'
                if 'Z' == date_string[-1]:
                    date_string = date_string[:-1]
                    # TODO: for python 3.7+, use datetime.isoformat()
                if sys.version_info > (3,) and isinstance(date_string, bytes):
                    date_string = date_string.decode('utf-8')  # Could also use ASCII in this case
                else:
                    date_string = str(date_string)
                pattern = '%Y-%m-%dT%H:%M:%S'

                try:
                    lc = locale._setlocale(locale.LC_ALL)
                    locale._setlocale(locale.LC_ALL, 'C')
                    epoch = int(time.mktime(time.strptime(date_string, pattern)))
                    locale._setlocale(locale.LC_ALL, lc)
                except ValueError:
                    print("Could not extract time from spec header, unrecognized format: %s, expected: %s" % (
                        date_string, pattern))

            if self.params['nrj'] is None:
                self.params['nrj'] = float(meta_entry["instrument/monochromator/energy"][()])
                print("Energy read from the meta file: %6.3f keV" % self.params['nrj'])
            else:
                print("Energy from the command line: %6.3f keV" % self.params['nrj'])

            if self.params['pixelsize'] is None:
                self.params['pixelsize'] = float(meta_entry["TOMO/pixelSize"][()]) * 1e-6
                print("Effective pixel size: %12.3fnm [from meta file]" % (self.params['pixelsize'] * 1e9))
            else:
                print("Effective pixel size: %12.3fnm [from command-line]" % (self.params['pixelsize'] * 1e9))

            if self.scan is None:
                self.scan = 0

            # Raw positioners positions, almost guaranteed not to be SI
            positioners = {}
            tmpn = meta_entry["sample/positioners/name"][()]
            tmpv = meta_entry["sample/positioners/value"][()]
            if isinstance(tmpn, np.bytes_):
                tmpn = tmpn.decode('ascii')
                tmpv = tmpv.decode('ascii')

            for k, v in zip(tmpn.split(), tmpv.split()):
                positioners[k] = float(v)

            if self.params['detectordistance'] is None:
                if f'{self.h5_header_path}/propagation_distance' in data:
                    self.params['detectordistance'] = \
                        1e-3 * float(data[f'{self.h5_header_path}/propagation_distance'][()])
                    print(f"Effective propagation distance (from {self.h5_header_path}/propagation_distance): "
                          "%12.8fm" % self.params['detectordistance'])
                else:
                    sx = 1e-3 * positioners['sx']
                    sx0 = 1e-3 * float(meta_entry["TOMO/sx0"][()])
                    z1 = sx - sx0
                    z12 = 1e-3 * float(meta_entry["PTYCHO/focusToDetectorDistance"][()])  # z1+z2
                    z2 = z12 - z1
                    print(sx * 1e3, sx0 * 1e3, z1 * 1e3, z12 * 1e3, z2 * 1e3)
                    self.params['detectordistance'] = z1 * z2 / z12
                    print("Effective propagation distance (computed): %12.8fm" % self.params['detectordistance'])

            # read all frames
            imgn = self.imgn
            t0 = timeit.default_timer()
            print("Reading frames:")
            self.raw_data = data["/entry_0000/measurement/data"][imgn.tolist()]
            dt = timeit.default_timer() - t0
            print('Time to read all frames: %4.1fs [%5.2f Mpixel/s]' %
                  (dt, self.raw_data[0].size * len(self.raw_data) / 1e6 / dt))

            # Read empty beam / reference images if necessary
            if self.params['data_ref'] is not None:
                # Assume this is an hdf5 file with a standard path for now
                with h5py.File(self.params['data_ref']) as h:
                    d = h['entry_0000/measurement/data'][()].astype(np.float32, copy=False)
                if d.ndim == 3:
                    self.data_ref = d.mean(axis=0)
                else:
                    self.data_ref = d

        if False:
            # DEBUG
            print("Sample positions in pixel units:")
            with np.printoptions(precision=3, suppress=False):
                tmpx = np.array(self.x / self.params['pixelsize'])
                tmpy = np.array(self.y / self.params['pixelsize'])
                print(tmpx - tmpx.mean())
                print(tmpy - tmpy.mean())
            import matplotlib.pyplot as plt
            plt.figure(999)
            plt.scatter(self.x, self.y)
            plt.tight_layout()
            plt.show()

        if self.params['adu_scale'] is not None:
            self.raw_data = self.raw_data.astype(np.float32, copy=False)
            self.raw_data *= self.params['adu_scale']
            if self.params['data_ref'] is not None:
                self.data_ref *= self.params['adu_scale']
            # Note: adu scale for dark is applied in _load_dark()

        self.load_data_post_process()


class PtychoRunnerID16aNF(PtychoRunner):
    """
    Class to process a series of scans with a series of algorithms, given from the command-line
    """

    def __init__(self, argv, params, *args, **kwargs):
        super(PtychoRunnerID16aNF, self).__init__(argv, default_params if params is None else params)
        self.PtychoRunnerScan = PtychoRunnerScanID16aNF

    @classmethod
    def make_parser(cls, default_par, description=None, script_name="pynx-ptycho-id16a-nf", epilog=None):
        if epilog is None:
            epilog = helptext_epilog
        if description is None:
            description = "Script to perform a ptychography analysis on NEAR FIELD data from ID16A@ESRF"
        p = default_par

        parser = super().make_parser(p, script_name, description, epilog)
        grp = parser.add_argument_group("ID16A (near field) parameters")
        grp.add_argument('--data', type=str, default=p['data'], required=True,
                         help="path to the data file, e.g. ``path/to/data.nx``."
                              "This can be a pattern, e.g. --data 'sample1_%%04d.h5', in which "
                              "case the field will be replaced by the scan/projection number.\n"
                              "An hdf5 entry can also be given using "
                              "``--data path/to/data.nx:entry0002``")

        class ActionScan(argparse.Action):
            """Argparse Action to check --scan is correctly used"""

            def __call__(self, parser_, namespace, values, option_string=None):
                vv = []
                for v in values:
                    v = v.replace('"', '').replace("'", "")
                    try:
                        if 'range(' in v or ',' in v:
                            vv += list(eval(v))
                        else:
                            vv.append(int(v))
                    except:
                        raise argparse.ArgumentTypeError(
                            f"--scan: failed converting '{v}' to a scan number or list of")
                setattr(namespace, self.dest, vv)

        # This will replace the main parser argument, to update the help and
        # allow this to be used with --projection
        grp.add_argument(
            '--projection', '--scan', dest='scan', default=None, nargs='+', action=ActionScan,
            help='Projection number e.g. ``--projection 5`` to extract a given '
                 'projection from a series (numbering starts at 1). '
                 'Alternatively a list or range of values can be given '
                 'using ``--projection 12 23 45`` or --projection "range(12,25)" '
                 '(note the quotes, necessary when using range with ()).')

        # grp.add_argument('--sequence', type=int, default=p['sequence'], required=False,
        #                 help="Sequence number in the bliss raw data, e.g. 1."
        #                      "If not present, the first entry called 'NFP2D' "
        #                      "or 'NFP3D' will be used.")

        grp.add_argument('--meta', type=str, default=p['meta'], required=False,
                         help='path to the bliss metadata file, e.g. path/to/meta.h5.\n'
                              'OBSOLETE: should not be needed for native bliss data from '
                              'January 2026 onwards')

        grp.add_argument('--data_ref', type=str, default=p['data_ref'],
                         help="path to hdf5 file with reference images (a.k.a. 'empty beam')"
                              "of the direct beam, which are needed for Paganin and CTF algorithms, "
                              "e.g. '--data_ref path/to/ref.h5'.\n"
                              "OBSOLETE: should not be needed for native bliss data from "
                              "January 2026 onwards.")

        grp.add_argument('--ptycho_motors', '--ptychomotors', type=str, default=p['ptychomotors'],
                         nargs='+', dest='ptychomotors', action=ActionPtychoMotors,
                         help="name of the two motors used for ptychography, optionally followed "
                              "by a mathematical expression to be used to calculate the actual "
                              "motor positions (axis convention, angle..) in meters. Values will be read "
                              "from the dat files:\n\n"
                              "* ``--ptychomotors=spy,spz,-x*1e-6,y*1e-6``  (the default)\n"
                              "* ``--ptychomotors pix piz``\n"
                              "Note that if the ``--xy=-y,x`` command-line argument is used, "
                              "it is applied _after_ this, using ``--ptychomotors=spy,spz,-x,y`` "
                              "is equivalent to ``--ptychomotors spy spz --xy=-x,y``\n\n"
                              "`Note that by default (01/2026-) this will read the scan "
                              "motors from the raw bliss file.")

        grp.add_argument('--delta_beta', type=float, default=p['delta_beta'],
                         help="delta/beta value, required for Paganin or CTF algorithms."
                              "Can also be set in the algorithm string")

        grp.add_argument('--adu_scale', type=float, default=p['adu_scale'],
                         help="optional scale factor by which the all data (including data_ref and dark) "
                              "will be multiplied in order to have photon counts.")

        grp.add_argument('--tomo_motor', type=str, default=p['tomo_motor'],
                         help='Name of the tomography motor -only used to export the rotation '
                              'angle in output files for ptycho-tomo.')

        grp.add_argument('--padding', type=int, default=p['padding'],
                         help=argparse.SUPPRESS)  # "padding value [DEPRECATED, does not work]")

        # Change const value
        grp.add_argument('--save_plot', '--saveplot', type=str,
                         default=False, const='object_phase', dest='saveplot',
                         choices=['object_phase', 'object_rgba'], nargs='?',
                         help='will save plot at the end of the optimization (png file).\n'
                              'A string can be given to specify if only the object phase '
                              '(default if only --saveplot is given) or rgba should be plotted.')

        # Experimental
        # grp.add_argument('--prefilter', type=int, default=0, nargs='?', const=200,
        #                  help=argparse.SUPPRESS)
        # "If --data_ref is supplied, it is possible to normalise "
        #      "all images by a low-passed filtered direct beam, "
        #      "in order to avoid intensity discontinuity at the periodic "
        #      "image boundaries.")
        grp.add_argument('--tapering', type=int, default=0, nargs='?', const=200,
                         help=argparse.SUPPRESS)
        # "Add a tapering (cosine/Tukey) window so that the intensities
        # "reach zero on the boundaries of the observed intensity data")`

        return parser

    def check_params_beamline(self):
        """
        Check if self.params includes a minimal set of valid parameters, specific to a beamline
        Returns: Nothing. Will raise an exception if necessary
        """
        data_type = _get_data_type(self.params['data'])
        if data_type is 'legacy':
            # Defaults  suitable for legacy datasets
            if self.params['ptychomotors'] is None:
                self.params['ptychomotors'] = ['spy', 'spz', '-x*1e-6', 'y*1e-6']
            if self.params['tomo_motor'] is None:
                self.params['tomo_motor'] = 'somega'
        if 'paganin' in self.params['algorithm'].lower() or 'ctf' in self.params['algorithm'].lower():
            if self.params['delta_beta'] is None and 'delta_beta:' not in self.params['algorithm'].lower():
                raise PtychoRunnerException('Need delta_beta=... for Paganin and CTF')
            if self.params['data_ref'] is None and data_type is 'legacy':
                raise PtychoRunnerException('Need data_ref=.../ref.h5 for Paganin or CTF')
        if self.params['dark'] is not None:
            if self.params['dark'].find('.h5') > 0 or self.params['dark'].find('.hdf5') > 0:
                if self.params['dark'][-3:] == '.h5' or self.params['dark'][-5:] == '.hdf5' \
                        and ":" not in self.params['dark']:
                    self.params['dark'] += ':/entry_0000/measurement/data'
                    print("dark: hdf5 path missing, adding it => dark=%s" % self.params['dark'])


def make_parser_sphinx():
    """Returns the argparse for sphinx documentation"""
    return PtychoRunnerID16aNF.make_parser(default_params)
