/** Apply the observed amplitude to a complex calculated array.
* Masked pixels (Iobs<0) are just multiplied by a scale factor.
* \param i: index of the considered pixel
* \param iobs: array of observed values, negative values are considered masked or free (no amplitude projection)
* \param dcalc: array of calculated complex amplitudes
* \param scale_in: scale factor to be applied to calculated intensities before comparison to observed intensities
* \param scale_out: scale factor to be applied to calculated intensities on output (for FFT scaling)
* \param zero_mask: if True (non-zero), all masked pixels amplitudes are set to zero
* \param confidence_interval_factor: a relaxation factor, with the projection of calculated amplitude being done
*        towards the limit of the poisson confidence interval. A value of 1 corresponds to a 50% confidence interval,
*        a value of 0 corresponds to a strict observed amplitude projection.
*/
void ApplyAmplitude(const int i, __global float *iobs, __global float2* dcalc,
                    const float scale_in, const float scale_out, const char zero_mask,
                    const float confidence_interval_factor)
{
  const float obs = iobs[i];
  const float scale = scale_in * scale_out;
  if(obs<-1e30f)
  { // Masked pixel
    if(zero_mask)
    {
      // Force all masked pixels to zero
      dcalc[i] = (float2)(0.0f,0.0f);
    }
    else
    {
      // keep calculated amplitudes for masked pixels, with an optional scale
      if(scale !=1.0f) dcalc[i] = (float2) (scale*dcalc[i].x , scale*dcalc[i].y);
    }
    return;
  }
  if(obs<-0.5f)
  { // Free pixel (-10) - we keep the complex amplitudes, with an optional scale.
    if(scale !=1.0f) dcalc[i] = (float2) (scale*dcalc[i].x , scale*dcalc[i].y);
    return;
  }

  const float2 dc=dcalc[i];
  const float adc = length(dc);
  const float icalc = adc * adc;

  // Confidence interval
  const float sig = native_sqrt(obs+1) * 0.675f * confidence_interval_factor;
  float aobs; // Amplitude in confidence interval, closest to calc
  if(fabs(obs - icalc) < sig) aobs = adc;
  else
  {
    if(icalc > obs) aobs = native_sqrt(obs + sig);
    else            aobs = native_sqrt(obs - sig);  // Can't have obs < sig here
  }

  const float a = aobs / fmax(adc, 1e-20f) * scale_out;
  dcalc[i] = (float2) (a*dc.x , a*dc.y);
}

/** Apply the observed amplitude to a complex calculated array, using already calculated intensities
* (taking into account PSF)
*/
void ApplyAmplitudeIcalc(const int i, __global float *iobs, __global float2* dcalc, __global float *icalc,
                         const float scale_in, const float scale_out, const char zero_mask,
                         const float confidence_interval_factor)
{
  const float obs = iobs[i];
  const float scale = scale_in * scale_out;
  if(obs<-1e30f)
  { // Masked pixel (-100)
    if(zero_mask)
    {
      // Force all masked pixels to zero
      dcalc[i] = (float2)(0.0f,0.0f);
    }
    else
    {
      // keep calculated amplitudes for masked pixels, with an optional scale
      if(scale !=1.0f) dcalc[i] = (float2) (scale*dcalc[i].x , scale*dcalc[i].y);
    }
    return;
  }
  if(obs<-0.5f)
  { // Free pixel (-10) - we keep the complex amplitudes, with an optional scale.
    if(scale !=1.0f) dcalc[i] = (float2) (scale*dcalc[i].x , scale*dcalc[i].y);
    return;
  }

  const float2 dc=dcalc[i];

  // KLUDGE: The GPU FFT-based convolution can produce negative values, which can be avoided by reverting to
  // non-convoluted values...
  float calc = icalc[i];
  if(calc < 1e-8f) calc = fmax(dot(dc, dc), icalc[i]);

  // Confidence interval
  const float sig = native_sqrt(obs+1) * 0.675f * confidence_interval_factor;
  float obs2; // Intensity in confidence interval, closest to calc
  if(fabs(obs - calc) < sig) obs2 = calc;
  else
  {
    if(calc > obs) obs2 = obs + sig;
    else           obs2 = obs - sig;  // Can't have obs < sig here
  }

  const float a = native_sqrt(obs2 / calc) * scale_out;
  dcalc[i] = (float2) (a*dc.x , a*dc.y);
}
