#!/usr/bin/env bash

# This script was written to install PyNX using conda - tested on MacOS 10.13.6 with miniconda

#####################################################################################################
## The following can be used to install conda
############################################# MacOs
#curl -O https://repo.anaconda.com/miniconda/Miniconda3-latest-MacOSX-x86_64.sh
#chmod +x Miniconda3-latest-MacOSX-x86_64.sh
#
## Conda install: accept license, choose directory to install, accept conda setup in ~/.bash_profile
#./Miniconda3-latest-MacOSX-x86_64.sh
#
#source ~/.bash_profile    # This may be different under linux, only needed once
#
############################################# Linux
# wget https://repo.anaconda.com/miniconda/Miniconda3-latest-Linux-x86_64.sh
#chmod +x Miniconda3-latest-Linux-x86_64.sh
#
## Conda install: accept license, choose directory to install, accept conda setup in ~/.bash_profile
#./Miniconda3-latest-Linux-x86_64.sh
#
#source ~/.bash_profile    # This may be different under linux, only needed once

echo $1

if [ -z $2 ];
then
  echo "No directory or python executable given for installation !";
  echo "Usage: install-pynx-conda-venv.sh ENVNAME PYNX_VERSION"
  echo "   with: DIRNAME the name of the new directory to create the python virtual environement, e.g. pynx-env"
  echo "         PYTHON_VERSION the python version, e.g. 3.7"
  echo "         PYNX_VERSION (optional) the git tag for the pynx version to be installed"
  echo "example: install-pynx-conda-venv.sh pynx-env 3.7"
  echo "example: install-pynx-conda-venv.sh pynx-3.6.2-py36 3.6 v3.6.2"
  exit
fi

echo
echo "#############################################################################################"
echo " Creating conda virtual environment"
echo "#############################################################################################"
echo

# create the conda virtual environement with necessary packages
conda create --yes -n $1 python=$2 pip numpy cython scipy matplotlib ipython notebook scikit-image h5py ipywidgets cmake

# Activate conda environment
conda activate $1


# Add packages from conda-forge
#conda install --yes -n py35-conda-env -c conda-forge hdf5plugin silx  fabio ipympl pyopencl
conda install --yes -n $1 -c conda-forge fabio ipympl pyopencl pytools
has_pyopencl=1

# Install packages using pip [conda silx requires opengl, pytools is not properly installed by conda ?]
pip install silx hdf5plugin pyqt5 pyopengl h5glance

# Is this a stable way to get the path to the environment ?
BASEDIR=`conda info | grep 'active env location' | awk '{print $5}'`
mkdir -p $BASEDIR/dev

echo
echo "#############################################################################################"
echo " Installing clfft"
echo "#############################################################################################"
echo
if [ $has_pyopencl -eq 1 ]; then

    # Install clfft, also in the virtualenv (recent versions allow prime numbers up to 13)
    # Recent versions (> v2.12.2) require cmake >=3.1)
    cd $BASEDIR/dev
    git clone https://github.com/clMathLibraries/clFFT.git --branch=v2.12.2
    cd clFFT
    mkdir build
    cd build
    cmake -DCMAKE_INSTALL_PREFIX:PATH=$BASEDIR/ ../src
    make all install

    # Do we need to set LD_LIBRARY_PATH or LIBRARY_PATH ? Assume conda is taking car of it..
else
     echo "pyopencl not available - not installing clfft"
fi


echo
echo "#############################################################################################"
echo " Installing gpyfft"
echo "#############################################################################################"
echo
if [ $has_pyopencl -eq 1 ]; then
    ## Install gpyfft - more tricky ! May require hand modifications in setup.py..
    cd $BASEDIR/dev
    git clone https://github.com/geggo/gpyfft.git
    cd gpyfft
    ## Get working version, gpyfft too often has incompatible changes
    git checkout 2c07fa8e7674757
    mv setup.py setup.py.ORIG
    awk '{sub("/Users/gregor/Devel/clFFT/src/library","'"$BASEDIR"'" "/lib/")};1' setup.py.ORIG > tmp1.py
    awk '{sub("/Users/gregor/Devel/","'"$BASEDIR"'" "/dev/")};1' tmp1.py > tmp2.py
    awk '{sub("/home/gregor/devel/","'"$BASEDIR"'" "/dev/")};1' tmp2.py > tmp1.py
    awk '{sub("/usr/local/lib64","'"$BASEDIR"'" "/lib/")};1' tmp1.py > tmp2.py
    # Not sure if next line needs to be adapted for conda
    #awk '{sub("os.path.join\\(CLFFT_DIR, \47src\47, \47include\47\\)", "os.path.join(CLFFT_DIR, \"src\", \"include\"), \"'"$BASEDIR"'" "/include/\"")};1' tmp2.py > setup.py
    mv tmp2.py setup.py

    rm -f tmp*.py
    python setup.py install
else
     echo "pyopencl not available - not installing gpyfft"
fi


echo
echo "#############################################################################################"
echo " Installing PyNX"
echo "#############################################################################################"
echo
cd $BASEDIR/dev
echo "Select method to download PyNX: git (if you have a https://gitlab.esrf.fr account) or ftp (no login required):"
select yn in "ftp" "git" "manual"; do
    case $yn in
        ftp ) curl -O http://ftp.esrf.fr/pub/scisoft/PyNX/pynx-latest.tar.bz2 ; tar -xjf pynx-latest.tar.bz2 ; break;;
        git ) git clone https://gitlab.esrf.fr/favre/PyNX.git pynx ; break;;
        manual ) echo "PyNX installation skipped-sould be manually installed (local git copy, etc..)" ; break;;
    esac
done
cd pynx

if [ -z $3 ];
    then echo "No tag given - using git pynx master head"
else
    git checkout tags/$3
fi

python setup.py install


echo
echo "#############################################################################################"
echo "Installing pyCUDA (optional)"
echo "#############################################################################################"
echo
# To also have CUDA support - use wheels if missing cuda.h for compilation
# Note: recent CUDA (>=8.0) require more recent scikit-cuda installation (from git), but older CUDA are incompatible
# with current scikit-cuda git...
# If scikit-cuda (old, from pip) gives an error, try pip install git+https://github.com/lebedov/scikit-cuda.git
if [[ $(command -v nvcc ) ]] ;
then
     pip install pycuda scikit-cuda --upgrade || echo "###\nCould not install pycuda - CUDA probably missing ?\n###\n"
else
  if [[ $(hostname -f) == debian8-devel.esrf.fr ]];
  then
     # Use precompiled wheels on debian8-devel because of missing missing cuda.h
     pip install /data/id01/inhouse/favre/pycuda-scikit-wheels/*.whl
  else
     echo "CUDA compiler (nvcc) no detected - not installing CUDA"
  fi
fi

echo
echo "#############################################################################################"
echo "Finished installation !"
echo "#############################################################################################"
echo
echo "To use the PyNX environment, use 'conda activate $1'"
echo
