# -*- coding: utf-8 -*-

# PyNX - Python tools for Nano-structures Crystallography
#   (c) 2016-present : ESRF-European Synchrotron Radiation Facility
#       authors:
#         Vincent Favre-Nicolin, favre@esrf.fr

from collections import OrderedDict
import time
import numpy as np


class PynxOrderedDict(OrderedDict):
    """
    OrderedDict with easy acces to the last value.
    """
    def last_value(self):
        if len(self) == 0:
            return None
        else:
            return self[next(reversed(self))]

    def last_key(self):
        if len(self) == 0:
            return None
        else:
            return next(reversed(self))


class History(dict):
    """
    History class to record optimization history. It is used to store the parameters like the algorithm,
    negative log-likelihood (llk), chi^2, resolution, cputime, walltime as a function of the cycle number.
    Not all values need be stored for all cycles.
    The default fields initialized as python OrderedDict (the key being the cycle number) are:
    
      - 'algorithm'
      - 'cputime'
      - 'walltime'
      - 'llk':
    
    A special field is last_cycle
    """
    def __init__(self):
        super(dict, self).__init__()
        for k in ['algorithm', 'cputime', 'walltime', 'llk']:
            self[k] = PynxOrderedDict()
        self['walltime'][0] = time.time()
        self['cputime'][0] = 0
        self.last_cycle = 0

    def add(self, cycle, **kwargs):
        """
        Store new values.
        Args:
            cycle: the current cycle
            **kwargs: e.g. llk=2e4, algorithm='ML-Poisson', cputime=17.2, walltime=32.2

        Returns:
            Nothing.
        """
        self.last_cycle = cycle
        for k, v in kwargs.items():
            self[k][cycle] = v


    def as_numpy_array(self,k, *args):
        """
        Get stored values for one or several keys (e.g. cputime, llk) as a couple of numpy arrays. There are additional keys, only values which are
        found for all keys are returned (i.e. the values must have been stored for the same cycle number).
        Args:
            k1: the desired key(s), e.g. 'cputime', 'llk',...
            *args: other desired keys
        Returns:
            a tuple of n+1 numpy vectors, the first with the cycle numbers, the others for the n keys supplied.
        """
        x1 = [v for v in self[k].keys()]
        y1 = [v for v in self[k].values()]
        if len(args) ==0:
            return np.array(x1, dtype=np.int32), np.array(y1)
        vx = []
        for k2 in args:
            vx.append([v for v in self[k2].keys()])
        vxy = [[] for k in range(len(args) + 2)]
        for i in x1:
            if np.array([i in self[k2].keys() for k2 in args]).all():
                vxy[0].append(i)
                vxy[1].append(self[k][i])
                for j in range(len(args)):
                    vxy[j+2].append(self[args[j]][i])
        vxy[0] = np.array(vxy[0], dtype=np.int32)
        for i in range(1, len(vxy)):
            vxy[i] = np.array(vxy[i])
        return vxy

