/** Kernels for updating an incoherent background in ptychography.
*
* Several approaches possible:
* - simple average diffrence between calc and obs, with a damping parameter biasing towards a lower background.
* - Marchesini et al approach from 'Augmented projections for ptychographic imaging', Inverse Problems 29 (2013), 115009.
*/

/** Compute the intermediate arrays for background update, from individual frames.
*
*/
__kernel
void BackgroundLoop(__global float2* vpsi, __global float* iobs, __global float* voldbackground,
                    __global float* vd, __global float* vd2, __global float* vz2, __global float* vdz2, const int npsi, const char firstpass)
{
  const unsigned long i2=get_global_id(0);  // Only loops over the background pixels (i2 < NXY)

  // Sum of calculated intensity from all modes and a stack of BLOCKSIZE frames
  float psi2 = 0;
  float dz2 = 0;
  float d2 = 0;
  float d = 0;
  for(int iz=0; iz<npsi ; iz++)
  {
    // Sum of calculated intensity from all modes
    float psi2z = 0;
    for(unsigned int mode=0 ; mode<NBMODE ; mode++)
    {
      const float2 ps = vpsi[i2 + mode*NXYZ + iz * NXY];
      psi2z += dot(ps,ps);
    }
    const float dd = iobs[i2 + iz * NXY] - voldbackground[i2];
    psi2 += psi2z;
    dz2 += dd * psi2z;
    d2 += dd * dd;
    d += dd;
  }
  if(firstpass)
  {
    vd  [i2] = d;
    vd2 [i2] = d2 ;
    vz2 [i2] = psi2;
    vdz2[i2] = dz2;
  }
  else
  {
    vd  [i2] += d;
    vd2 [i2] += d2 ;
    vz2 [i2] += psi2;
    vdz2[i2] += dz2;
  }
}

/** Final update of the background
*
*/
__kernel
void BackgroundUpdate(__global float* background, __global float* vd, __global float* vd2, __global float* vz2, __global float* vdz2,
                      const int nframes)
{
  const unsigned long i2=get_global_id(0);  // Only loops over the background pixels (i2 < NXY)

  const float eta = fmax(0.8f, vdz2[i2]/vd2[i2]);
  background[i2] = fmax(0.0f, background[i2] + (vd[i2] - vz2[i2] / eta) / nframes);
}

/** Final update of the background - Masked version
*
*/
__kernel
void BackgroundUpdateMask(__global float* background, __global float* vd, __global float* vd2, __global float* vz2, __global float* vdz2,
                          __global char* mask, const int nframes)
{
  const unsigned long i2=get_global_id(0);
  if(mask[i2] == 0)
  {
    BackgroundUpdate(background, vd, vd2, vz2, vdz2, nframes);
  }
}

/*
// Background loop, simple difference
__kernel
void BackgroundLoopSimple(__global float2* vpsi, __global float* iobs, __global float* vd, const int npsi, const char firstpass)
{
  const unsigned long i2=get_global_id(0);  // Only loops over the background pixels (i2 < NXY)

  // Difference between the observed and calculated intensity from all modes and a stack of BLOCKSIZE frames
  float d = 0;
  for(int iz=0; iz<npsi ; iz++)
  {
    // Sum of calculated intensity from all modes
    float psi2z = 0;
    for(unsigned int mode=0 ; mode<NBMODE ; mode++)
    {
      const float2 ps = vpsi[i2 + mode*NXYZ + iz * NXY];
      psi2z += dot(ps,ps);
    }
    const float dd = iobs[i2 + iz * NXY]  - psi2z;
    d += dd;
  }
  if(firstpass)
  {
    vd  [i2] = d;
  }
  else
  {
    vd  [i2] += d;
  }
}

/// Final update of the background
__kernel
void BackgroundUpdateSimple(__global float* background, __global float* vd, const int nframes)
{
  const unsigned long i2=get_global_id(0);  // Only loops over the background pixels (i2 < NXY)

  // Dampened change only if background increases ?
  //const float eps = (float) (vd[i2] > 0) * 0.5;
  const float eps = 0.5 ; //(float) (vd[i2] > 0) * 0.5;
  background[i2] = fmax((vd[i2] / nframes + eps * background[i2]) / (1 + eps), 0.0f);
}

/// Final update of the background - Masked version
__kernel
void BackgroundUpdateSimpleMask(__global float* background, __global float* vd, __global char* mask, const int nframes)
{
  const unsigned long i2=get_global_id(0);
  if(mask[i2] == 0)
  {
    BackgroundUpdateSimple(background, vd, nframes);
  }
}
*/